/***************************************************************************
                           cdir.cpp  -  description
                             -------------------
    begin                : Tue May 14 2002
    copyright            : (C) 2002-2005 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef WIN32
#include <unistd.h>
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_VFS_H
#include <sys/vfs.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#endif

#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <dclib/dcos.h>
#include "cdir.h"
#include "fsusage.h"
#include "platform.h"

/** */
CDir::CDir()
{
	sPath.Empty();
	sDrive.Empty();
}

/** */
CDir::CDir( CString path )
{
	SetPath(path);
}

/** */
CDir::~CDir()
{
}

/** */
CString CDir::Extension( CString file )
{
	int i;
	CString s;

	if ( (i = file.FindRev('.')) != -1 )
		s = file.Mid(i+1,file.Length()-i-1);

	return s;
}

/** */
CString CDir::Path()
{
#ifdef WIN32
	return ConvertSeparators(sDrive+DIRSEPARATOR+sPath);
#else
	return ConvertSeparators(sPath);
#endif
}

/** */
void CDir::SetPath( CString path )
{
	CString s;

	s = path;

	// convert separators to internal
	s = s.Replace('\\','/');

#ifdef WIN32
	// TODO: use _splitpath
	int i;

	// store drive letter
	if ( (i = s.Find(':',0)) == 1 )
	{
		sDrive = s.Left(i+1);
		s = s.Mid(i+1,s.Length()-i-1);

		// remove first separator
		if ( s.Find('/') == 0 )
		{
			s = s.Mid(1,s.Length()-1);
		}
	}
#endif

	// remove last '/'
	if ( ((1+s.FindRev('/')) == s.Length()) && (s.Length() > 1) )
	{
		sPath = s.Left( s.Length()-1 );
	}
	else
	{
		sPath = s;
	}
}

/** */
CString CDir::HomeDirPath()
{
	CString s;

#ifdef WIN32
	s = getenv("USERPROFILE");

	if ( s == "" )
	{
		char buffer[_MAX_PATH];

		/* Get the current working directory: */
		if( _getcwd( buffer, _MAX_PATH ) != NULL )
		{
			s = buffer;
		}
	}
#else
	s = getenv("HOME");
#endif
	if ( s == "" )
	{
		s = DIRSEPARATOR;
	}

	return s;
}

/** */
bool CDir::cd( CString path )
{
	sPath = "";

	if ( path == "." )
		return TRUE;
	if ( path == "" )
		return FALSE;

#ifdef WIN32
	if ( _access( path.Data(), F_OK | R_OK ) == 0 )
#else
	if ( access( path.Data(), F_OK | R_OK ) == 0 )
#endif
	{
		SetPath(path);
		return TRUE;
	}

	return FALSE;
}

/** */
CString CDir::DirName()
{
	int pos = sPath.FindRev('/');

	if( pos == -1 )
		return sPath;

	return sPath.Mid( pos + 1, sPath.Length() - 1 - pos );
}

/** */
bool CDir::ReadEntrys( FilterSpec filterspec, CList<CFileInfo> * list )
{
	DIR * dir;
	CFileInfo * FileInfo;
	dirent *file;
	CString s,d_name;
#ifdef WIN32
	struct _stat buf;
#else
	struct stat buf;
#endif

	if ( !list )
		return FALSE;

	list->Clear();

	if ( Path() == "" )
		return FALSE;

	dir = opendir( Path().Data() );

	if ( dir == 0 )
		return FALSE;

	while ( (file = readdir(dir)) != 0 )
	{
		d_name.Set(file->d_name,strlen(file->d_name));

		if ( (filterspec == Dirs) && (IsDir(d_name)) )
		{
			FileInfo = new CFileInfo();
			FileInfo->name = d_name;
			FileInfo->size = 0;
			FileInfo->m_bSymlink = FALSE;

#ifndef WIN32
			if ( GetLStat( d_name, &buf) )
				FileInfo->m_bSymlink = ((buf.st_mode&S_IFLNK)==S_IFLNK);
#endif

			list->Add(FileInfo);
		}
		else if ( (filterspec == Files) && (IsFile(d_name)) )
		{
			FileInfo = new CFileInfo();
			FileInfo->name = d_name;
			FileInfo->size = getFileSize(d_name);
			FileInfo->m_bSymlink = FALSE;

			if ( GetStat( d_name, &buf) )
			{
				FileInfo->st_a_time = buf.st_atime;
				FileInfo->st_c_time = buf.st_ctime;
				FileInfo->st_m_time = buf.st_mtime;

#ifndef WIN32
				if ( GetLStat( d_name, &buf) )
					FileInfo->m_bSymlink = ((buf.st_mode&S_IFLNK)==S_IFLNK);
#endif
			}

			list->Add(FileInfo);
		}
	}

	closedir(dir);

	return TRUE;
}

/** */
bool CDir::IsDir( CString s, bool rel )
{
#ifdef WIN32
	struct _stat buf;
#else
	struct stat buf;
#endif

	if ( GetStat(s,&buf,rel) )
	{
		if ((buf.st_mode&S_IFMT)&S_IFDIR)
		{
			return TRUE;
		}
	}

	return FALSE;
}

/** */
bool CDir::IsFile( CString s, bool rel )
{
#ifdef WIN32
	struct _stat buf;
#else
	struct stat buf;
#endif

	if ( GetStat(s,&buf,rel) )
	{
		if ((buf.st_mode&S_IFMT)&S_IFREG)
		{
			return TRUE;
		}
	}

	return FALSE;
}

/** */
CString CDir::ReadLink( CString string )
{
	CString s,st;
#ifndef WIN32
	char buf[1024];
	size_t size=1024;
	int i;

	st = SimplePath(string);

	if ( st == "" )
		return s;

	if ( (i=readlink(st.Data(),buf,size)) > 0 )
	{
		s.Set(buf,i);
	}
	else
	{
		perror("CDir::readlink: ");
	}
#endif

	return s;
}

/** */
#ifdef WIN32
bool CDir::GetStat( CString & s, struct _stat * buf, bool rel )
#else
bool CDir::GetStat( CString & s, struct stat * buf, bool rel )
#endif
{
	CString p;

	if ( !buf )
		return FALSE;

	if ( rel )
		p = Path()+DIRSEPARATOR+s;
	else
		p = s;

	if ( p == "" )
		return FALSE;

#ifdef WIN32
	if ( _stat( p.Data(), buf ) != 0 )
#else
	if ( stat( p.Data(), buf ) != 0 )
#endif
		return FALSE;

	return TRUE;
}

/** */
#ifdef WIN32
bool CDir::GetLStat( CString & s, struct _stat * buf, bool rel )
#else
bool CDir::GetLStat( CString & s, struct stat * buf, bool rel )
#endif
{
	CString p;

	if ( rel )
		p = Path()+DIRSEPARATOR+s;
	else
		p = s;

	if ( !buf )
		return FALSE;

	if ( p == "" )
		return FALSE;

#ifndef WIN32
	if ( lstat( p.Data(), buf ) != 0 )
		return FALSE;

	return TRUE;
#else
	return FALSE;
#endif
}

/** */
ulonglong CDir::getFileSize( CString s, bool rel )
{
	CString p;
#ifdef WIN32
	struct _stat buf;
#else
	struct stat buf;
#endif

	if ( rel )
		p = Path()+DIRSEPARATOR+s;
	else
		p = s;

	if ( p == "" )
		return 0;

#ifdef WIN32
	if ( _stat( p.Data(), &buf ) != 0 )
#else
	if ( stat( p.Data(), &buf ) != 0 )
#endif
	{
		//printf("CDir::getFileSize: stat: '%s' '%s'\n",strerror(errno),p.Data());
		return FALSE;
	}

	return buf.st_size;
}

/** */
void CDir::SplitPathFile( CString s, CString & path, CString & file )
{
	int i;

	path = "";
	file = "";

	if ( (i = s.FindRev('\\')) == -1 )
		i = s.FindRev('/'); // for unix-kind folders (never seen ...)

	if ( i == -1 )
	{
		file = s;
	}
	else
	{
		path = s.Left(i+1);
		file = s.Mid(i+1,s.Length()-i-1);
	}
}

/** */
CString CDir::CleanDirPath( CString path )
{
	CString s = path;
	long i=0;

	// convert first ../
	if ( s.Find( "../" ) == 0 )
	{
		s.Data()[0] = '/';
		s.Data()[1] = '/';
	}

	// convert '..' -> '//'
	while( (i=s.Find( "/.." ,i )) != -1 )
	{
		if ( (s.Data()[i+3] == 0) || (s.Data()[i+3] == '/') )
		{
			s.Data()[i+1] = '/';
			s.Data()[i+2] = '/';
		}
		i++;
	}

	return s;
}

/** */
CString CDir::ConvertSeparators( CString path )
{
	return path.Replace('/',DIRSEPARATOR);
}

/** */
CString CDir::SimplePath( CString path )
{
	CString s = path;
	CString t;
	char c;
	long i;

	i = 0;
	while( (i=s.Find('\\',i)) != -1 )
	{
		s.Data()[i] = '/';
		i++;
	}

	s = CleanDirPath(s);

	// remove double '/'
	for(i=0,t="",c=0;i<s.Length();i++)
	{
		if ( (c == '/') && (s.Data()[i] == '/') )
			continue;

		c = s.Data()[i];
		t += c;
	}

	s = ConvertSeparators(t);

	return s;
}

/** */
bool CDir::CreatePath( CString path )
{
	int i;
	CString s,s1;

	s  = SimplePath(path);
	s1 = "";

#ifdef WIN32
	if ( (i = s.Find(':',0)) != -1 )
	{
		sDrive = s.Left(i+1);
		s = s.Mid(i+1,s.Length()-i-1);
	}
#endif

	while ( s != "" )
	{
		if ( (i = s.Find(DIRSEPARATOR)) != -1 )
		{
			if (s1!="")
				s1 = s1 + DIRSEPARATOR + s.Left(i);
			else
				s1 = s.Left(i);
			s  = s.Mid(i+1,s.Length()-1-i);
		}
		else
		{
			if (s1!="")
				s1 = s1 + DIRSEPARATOR + s;
			else
				s1 = s;
			s  = "";
		}

		if ( s1 == "" )
		{
			continue;
		}

		if ( IsDir(s1) == FALSE )
		{
#ifdef WIN32
			if ( _mkdir((Path()+DIRSEPARATOR+s1).Data()) != 0 )
#else
			if ( mkdir((ConvertSeparators(sPath)+DIRSEPARATOR+s1).Data(),0777) != 0 )
#endif
			{
				if ( errno != EEXIST )
				{
					printf("mkdir Error: '%s' '%s'\n",strerror(errno),(sPath+DIRSEPARATOR+s1).Data());
				}

				return FALSE;
			}
		}
	}

	return TRUE;
}

/** */
bool CDir::FreeDiscSpace( CString path, ulonglong * res )
{
	if ( !res )
	{
		return FALSE;
	}

#ifdef WIN32
	ULARGE_INTEGER lpFreeBytesAvailableToCaller; // receives the number of bytes on
                                               // disk available to the caller
	ULARGE_INTEGER lpTotalNumberOfBytes;    // receives the number of bytes on disk
	ULARGE_INTEGER lpTotalNumberOfFreeBytes; // receives the free bytes on disk

	if ( GetDiskFreeSpaceEx( path.Data(), &lpFreeBytesAvailableToCaller,
				&lpTotalNumberOfBytes,
				&lpTotalNumberOfFreeBytes ) == TRUE )
	{
		*res = lpTotalNumberOfFreeBytes.QuadPart;
		return TRUE;
	}
	else
	{
		return FALSE;
	}
#else
	struct fs_usage fsp;

	if ( get_fs_usage(path.Data(),path.Data(),&fsp) == 0 )
	{
		// printf("ok %d\n",fsp.fsu_bavail_top_bit_set);
		*res = fsp.fsu_bavail*fsp.fsu_blocksize;
		return TRUE;
	}
	else
	{
		perror("FreeDiscSpace");
		return FALSE;
	}

#endif
}
