/* 
 * $Id: ctkwindow.c,v 1.37 2000/07/21 23:18:52 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ctk.h"
#include "ctkstack.h"
#include "ctkcolor.h"

gboolean ctk_window_button_press  (CtkWidget* object, CdkEventButton* event, gpointer data);
gboolean ctk_window_button_release(CtkWidget* object, CdkEventButton* event, gpointer data);
gboolean ctk_window_drag          (CtkWidget* object, CdkEventButton* event, gpointer data);
gboolean ctk_window_key_press     (CtkWidget* object, CdkEventKey*    event, gpointer data);

gpointer ctk_window_destroy(CtkObject *object);

/* CTK Window Initializer */

void ctk_window_init(CtkWindow *window)
{
	ctk_bin_init(&window->bin);
	CTK_OBJECT(window)->type = CtkTypeWindow;
	
	CTK_WIDGET(window)->set_min_size = &ctk_window_min_size;
	((CtkContainer*)window)->border_width = 1;

	window->resize_flag = FALSE;
	window->close_flag  = FALSE;
	window->move_flag   = FALSE;
	
	window->shadow = TRUE;
	window->position = CTK_WIN_POS_NONE;
	window->focus_widget = CTK_WIDGET(window);
	window->title = NULL;
	
	window->OPTIONS_MASK =
	    WINMASK_MOVABLE |
	    WINMASK_RESIZABLE |
	    WINMASK_CLOSABLE;
	
	window->modal = FALSE;

	/* Initialize Other Variables */
	((CtkWidget *)window)->sensitive = TRUE;
	((CtkWidget *)window)->width = 3;
	((CtkWidget *)window)->height = 2;
	((CtkWidget *)window)->orig_width = 3;
	((CtkWidget *)window)->orig_height = 2;	
	((CtkWidget *)window)->col = -1;
	((CtkWidget *)window)->row = -1;

	/* Also setup Colors */
	((CtkWidget *)window)->main_col = ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_LIGHT_GRAY);
	((CtkWidget *)window)->inverse_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_LIGHT_GRAY);
	((CtkWidget *)window)->shadow_col = ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_BLACK);
	((CtkWidget *)window)->title_col = ctk_calculate_palette(CTK_COLOR_BLUE,CTK_COLOR_LIGHT_GRAY);

	((CtkWidget *)window)->node = g_node_new((gpointer)((CtkWidget *)window));
	((CtkObject *)window)->destroy_func = ctk_window_destroy;

	ctk_signal_connect(CTK_OBJECT(window), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_window_key_press),      NULL);
	ctk_signal_connect(CTK_OBJECT(window), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_window_button_press),   NULL);
	ctk_signal_connect(CTK_OBJECT(window), "button_release_event",
			   CTK_SIGNAL_FUNC(&ctk_window_button_release), NULL);
	ctk_signal_connect(CTK_OBJECT(window), "ctk_drag",
			   CTK_SIGNAL_FUNC(&ctk_window_drag),           NULL);
}

/* CTK Window New - Constructor */
CtkWidget* ctk_window_new(CtkWindowType type)
{
	CtkWindow* window;
	
	/* Init CtkWindow */
	window = g_malloc(sizeof(CtkWindow));

	ctk_window_init(window);

	return((CtkWidget *)window);
}

gboolean ctk_window_key_press(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	CtkWindow* window;
	
	window = CTK_WINDOW(widget);
	
	switch (event->keyval)
	{
	case AK_ARROW_UP:
		if ((window->OPTIONS_MASK & WINMASK_MOVABLE) != 0 &&
			window->focus_widget == widget)
		{
			widget->row--;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case AK_ARROW_DOWN:
		if ((window->OPTIONS_MASK & WINMASK_MOVABLE) != 0 &&
			window->focus_widget == widget)
		{
			widget->row++;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case AK_ARROW_LEFT:
		if ((window->OPTIONS_MASK & WINMASK_MOVABLE) != 0 &&
			window->focus_widget == widget)
		{
			widget->col--;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case AK_ARROW_RIGHT:
		if ((window->OPTIONS_MASK & WINMASK_MOVABLE) != 0 &&
			window->focus_widget == widget)
		{
			widget->col++;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case (MODIFIER_CTRL | AK_ARROW_UP):
		if ((window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0 &&
				window->focus_widget == widget) {
			widget->height--;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case (MODIFIER_CTRL | AK_ARROW_DOWN):
		if ((window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0 &&
				window->focus_widget == widget &&
				(widget->height < arr_scr_y())) {
			widget->height++;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case (MODIFIER_CTRL | AK_ARROW_LEFT):
		if ((window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0 &&
				window->focus_widget == widget) {
			widget->width--;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	case (MODIFIER_CTRL | AK_ARROW_RIGHT):
		if ((window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0 &&
				window->focus_widget == widget &&
				(widget->width < arr_scr_x())) {
			widget->width++;
			ctk_size_mark_changed(widget);
		}
		return TRUE;
	}
	
	return FALSE;
}

gboolean ctk_window_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkWindow* window = CTK_WINDOW(widget);

      if (event->x == widget->col + widget->width - 2 && 
	  event->y == widget->row                     &&
	  (window->OPTIONS_MASK & WINMASK_CLOSABLE) != 0)
      {
            ctk_window_set_focus(window, widget);
	    window->close_flag = TRUE;
      }
      else if (event->x == widget->col + widget->width  - 1 && 
	       event->y == widget->row + widget->height - 1 &&
	       (window->OPTIONS_MASK & WINMASK_RESIZABLE) != 0)
      {
            ctk_window_set_focus(window, widget);
	    window->resize_flag = TRUE;
      }   
      else if ((window->OPTIONS_MASK & WINMASK_MOVABLE) != 0 &&
	       (event->x == widget->col                     || 
		event->x == widget->col + widget->width - 1 ||
		event->y == widget->row                     ||
		event->y == widget->row + widget->height - 1))
      {
            ctk_window_set_focus(window, widget);
	    window->move_flag = TRUE;
      }
      
      return TRUE;
}

gboolean ctk_window_button_release(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkWindow* window = CTK_WINDOW(widget);

      if (window->close_flag && 
	  event->x == widget->col + widget->width - 2 &&
	  event->y == widget->row)
      {
	    ctk_signal_emit_by_name(CTK_OBJECT(widget), "delete_event");
      }

      window->close_flag  = FALSE;
      window->resize_flag = FALSE;
      window->move_flag   = FALSE;

      return TRUE;
}

gboolean ctk_window_drag(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
      CtkWindow* window = CTK_WINDOW(widget);

      if (window->move_flag)
      {
	    widget->col += event->dx;
	    widget->row += event->dy;

	    ctk_size_mark_changed(widget);
	    return TRUE;
      }

      if (window->resize_flag)
      {
	    /* Don't resize past window - sizing won't fix this */
	    if (widget->col + widget->width + event->dx > ctk_scr_c)
	    {
		  widget->width = ctk_scr_c - widget->col;
	    }
	    else
	    {
		  widget->width += event->dx;
	    }
	    if (widget->row + widget->height + event->dy > ctk_scr_r)
	    {
		  widget->height = ctk_scr_r - widget->row;
	    }
	    else
	    {
		  widget->height += event->dy;
	    }

	    ctk_size_mark_changed(widget);
	    return TRUE;
      }

      return FALSE;
}

/* Set Window Title */
void ctk_window_set_title(CtkWindow* window, const gchar* title)
{
      if (!window) 
      {
	    ctk_close();
	    g_error("ctk_window_set_title: CtkWidget *widget = NULL!");
      }
	
      if (CTK_OBJECT(window)->type != CtkTypeWindow)
      {
	    ctk_close();
	    g_error("ctk_window_set_title: ((CtkObject *)widget->type) != CtkTypeWindow");
      }
	
      if (window->title)
	    g_free(window->title);

      if (title)
	    window->title = g_strdup(title);
      else
	    window->title = NULL;

      ctk_size_mark_changed(CTK_WIDGET(window));
}

void ctk_window_set_policy(CtkWindow* window, gboolean canShrink, 
	gboolean canGrow, gboolean autoShrink)
{
	// ignore canShrink - we can't!
	if (canGrow)
	{
		window->OPTIONS_MASK |= WINMASK_RESIZABLE;
	}
	else
	{
		window->OPTIONS_MASK &= ~WINMASK_RESIZABLE;
	}
	
	if (autoShrink)
	{
		window->OPTIONS_MASK |= WINMASK_AUTOSHRINK;
	}
	else
	{
		window->OPTIONS_MASK &= ~WINMASK_AUTOSHRINK;
	}
}

void ctk_window_set_autocenter(CtkWindow* window, gboolean center)
{
	if (center)
	{
		window->OPTIONS_MASK |= WINMASK_AUTOCENTER;
	}
	else
	{
		window->OPTIONS_MASK &= ~WINMASK_AUTOCENTER;
	}
}

void ctk_window_set_movable(CtkWindow* window, gboolean movable)
{
	if (movable)
	{
		window->OPTIONS_MASK |= WINMASK_MOVABLE;
	}
	else
	{
		window->OPTIONS_MASK &= ~WINMASK_MOVABLE;
	}
}

void ctk_window_set_closable(CtkWindow* window, gboolean closable)
{
	if (closable)
	{
		window->OPTIONS_MASK |= WINMASK_CLOSABLE;
	}
	else
	{
		window->OPTIONS_MASK &= ~WINMASK_CLOSABLE;
	}
}

/* Set Window Modal */
void ctk_window_set_modal(CtkWindow* window, gboolean modal)
{
	if (!window)
	    return;
	
	window->modal = modal;
	ctk_size_mark_changed(CTK_WIDGET(window));
}

/* Set Window Position */
void ctk_window_set_position(CtkWindow* window, CtkWindowPosition position)
{
	if (!window)
	    return;
	
	window->position = position;
	ctk_size_mark_changed(CTK_WIDGET(window));
}

CtkWindow* ctk_window_get_focus_window(void)
{
	if (!focus_stack || !*focus_stack)
	    return NULL;
	
	return CTK_WINDOW(focus_stack[0]->data);
}

/* Destroy Data */
gpointer ctk_window_destroy(CtkObject* object)
{
	if (((CtkWindow *)object)->title)
	    g_free(((CtkWindow *)object)->title);

	return NULL;
}

/* Never call ctk_lose/gain_focus - use ctk_main_remove/set_widget_focus_window */
void ctk_lose_focus()
{
      static gboolean running = FALSE;
      CtkWindow* window = ctk_window_get_focus_window();

      if (running) 
	    return;

      if (window)
      {
	    running = TRUE;
	    ctk_event_emit_by_name(CTK_OBJECT(window->focus_widget), "focus_out_event");
	    running = FALSE;
      }
}

void ctk_give_focus()
{
      static gboolean running = FALSE;
      CtkWindow* window = ctk_window_get_focus_window();

      if (running) 
	    return;

      if (window)
      {
	    running = TRUE;
	    ctk_event_emit_by_name(CTK_OBJECT(window->focus_widget), "focus_in_event");
	    running = FALSE;
      }
}

/* Set the Window Focus */
void ctk_window_set_focus(CtkWindow* window, CtkWidget* widget)
{
      ctk_assert(window != NULL, "Attempt to set focus in null window");
      ctk_assert(widget != NULL, "Attempt to set focus on a null widget");
      ctk_assert(ctk_window_container(widget) == window,
             "Attempt to set focus of window to something outside of it");
      ctk_assert(!widget->destroyed, "Attempt to focus a destroyed widget");

      if (window->focus_widget != widget) 
      {
	    if (ctk_window_get_focus_window() == window)
		  ctk_lose_focus();
            
	    window->focus_widget = widget;
	    
	    if (ctk_window_get_focus_window() == window)
		  ctk_give_focus();

	    ctk_draw_mark_changed(widget);
      }
}

void ctk_window_raise(CtkWindow* window)
{
      if (ctk_window_get_focus_window() != window)
      {
	    ctk_lose_focus();
	    stack_pop (CTK_WIDGET(window)->node);
	    stack_push(CTK_WIDGET(window)->node);
	    ctk_give_focus();

	    ctk_draw_mark_changed(CTK_WIDGET(window));
      }
}

CtkWindow* ctk_window_container(CtkWidget* widget)
{
      while (widget->node->parent)
      {
	    widget = CTK_WIDGET(widget->node->parent->data);
      }

      if (CTK_CHECK_TYPE(widget, CtkTypeWindow)) return CTK_WINDOW(widget);
      return NULL;
}

void ctk_window_min_size(CtkWidget* widget)
{
      CtkWindow* window = CTK_WINDOW(widget);
      
      ctk_bin_min_size(widget);
      
      if (window->title)
      {
            gint min = strlen(window->title) + 7;
            if (widget->min_width < min)
                  widget->min_width = min;
      }
}
