/*
 * $Id: cap_proc.c,v 1.1.1.1.4.4 2001/01/16 07:38:25 agmorgan Exp $
 *
 * Copyright (c) 1997-8 Andrew G Morgan <morgan@linux.kernel.org>
 *
 * This file deals with setting capabilities on processes.
 */

#include "libcap.h"

static void _libcap_cappid(cap_t cap_d, pid_t pid)
{
    cap_d->head.version = _libcap_kernel_version;

    _cap_debug("version: %x, features: %x\n",
	       cap_d->head.version, cap_d->features);

#if (_LINUX_CAPABILITY_VERSION == 0x19980330)

    if (_libcap_kernel_version == 0x19980330) {
	cap_d->head.pid = pid;
    }
    _cap_debug("pid: %d\n", cap_d->head.pid);

#else /* holds for 0x20000603 and 0x20010113 */

    if (_libcap_kernel_version == 0x19980330) {
	cap_d->head.type = pid;          /* a backward compatibility hack */
    } else {
	cap_d->head.type = CAP_USERHEADER_PID;
	cap_d->head.u.pid = pid;
    }
    _cap_debug("type: %d\npid: %d\n", cap_d->head.type, cap_d->head.u.pid);

#endif

    _cap_debugcap("effective = ", &cap_d->set.effective);
    _cap_debugcap("inheritable = ", &cap_d->set.inheritable);
    _cap_debugcap("permitted = ", &cap_d->set.permitted);
}


cap_t cap_get_proc(void)
{
    cap_t result;

    /* allocate a new capability set */
    result = cap_init();
    if (result) {
	_cap_debug("getting current process' capabilities");

	_libcap_cappid(result, 0);

	/* fill the capability sets via a system call */
	if (capget(&result->head, &result->set)) {
	    cap_free(result);
	    result = NULL;
	}
    }

    if (result && (_libcap_kernel_features & CAP_FEATURE_TO_32)) {
	if (_libcap_n1_to_n2(result, 1, __CAP_BLKS)) {
	    cap_free(result);
	    result = NULL;
	}
    }

    return result;
}

int cap_set_proc(cap_t cap_d)
{
    int retval;

    if (!good_cap_t(cap_d)) {
	errno = EINVAL;
	return -1;
    }

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	retval = _libcap_n1_to_n2(cap_d, __CAP_BLKS, 1);
    }

    _cap_debug("setting process capabilities");
    _libcap_cappid(cap_d, 0);

    retval = capset(&cap_d->head, &cap_d->set);

    if (_libcap_kernel_features & CAP_FEATURE_TO_32) {
	(void) _libcap_n1_to_n2(cap_d, 1, __CAP_BLKS);
    }

    return retval;
}

/* the following functions are not required by POSIX */

/* read the caps on a specific process */

int capgetp(pid_t pid, cap_t cap_d)
{
    int error;

    if (!good_cap_t(cap_d)) {
	_cap_debug("failed check for good cap_t");
	errno = EINVAL;
	return -1;
    }

    _cap_debug("getting process capabilities for proc %d", pid);

    _libcap_cappid(cap_d, pid);
    error = capget(&cap_d->head, &cap_d->set);
    _libcap_cappid(cap_d, 0);

    if ((!error) && (_libcap_kernel_features & CAP_FEATURE_TO_32)) {
	if (_libcap_n1_to_n2(cap_d, 1, __CAP_BLKS)) {
	    cap_free(cap_d);
	    cap_d = NULL;
	    error = -1;
	}
    }

    return error;
}

#ifdef CAP_USERHEADER_X_WITH_E

static int map_cap_flag(cap_flag_t set, int *flavor)
{
    switch (set) {
    case CAP_EFFECTIVE:
	*flavor = CAP_USERHEADER_X_WITH_E;
	break;

    case CAP_INHERITABLE:
	*flavor = CAP_USERHEADER_X_WITH_I;
	break;

    case CAP_PERMITTED:
	*flavor = CAP_USERHEADER_X_WITH_P;
	break;

    default:
	_cap_debug("illegal cap_flag_t = %d", set);
	return 0;
    }

    return 1;
}

/* get the current cap_bset */

cap_t cap_get_bound(cap_flag_t set)
{
    int cap_flavor;
    cap_t result;

    if (!map_cap_flag(set, &cap_flavor)) {
	errno = EINVAL;
	return NULL;
    }

    /* allocate a new capability set */
    result = cap_init();
    if (result) {
	_cap_debug("getting current process' bounding set");

	result->head.type = cap_flavor;

	/* fill the capability sets via a system call */
	if (capget(&result->head, &result->set)) {
	    cap_free(result);
	    result = NULL;
	}
    }

    return result;
}

/* set the current cap_bset */

int cap_set_bound(cap_t cap_d, cap_flag_t set)
{
    int cap_flavor;

    if (!good_cap_t(cap_d)) {
	errno = EINVAL;
	return -1;
    }

    if (!map_cap_flag(set, &cap_flavor)) {
	errno = EINVAL;
	return -1;
    }

    _cap_debug("setting current process' bounding set");

    cap_d->head.type = cap_flavor;

    return capset(&cap_d->head, &cap_d->set);
}

#endif /* CAP_USERHEADER_X_WITH_E */
