/* 
 * $Id: term_iface.c,v 1.24 2000/08/28 19:05:10 terpstra Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>
#include <sys/ioctl.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "term_iface.h"
#include "arr.h"
#include "input.h"
#include "tinfo.h"
#include "types.h"
#include "misc.h"
#include "cap_offset.h"
#include "tt_sprintf.h"

/* padding_chr is used for delays on older terminals (instead of a usleep(), or
 * something like that.  This is the correct behavior.).
 */
char padding_chr;

/* tparm_buf is the buffer used for writing strings from terminfo/termcap format
 * to something that can be sent to the terminal.  It is of arbitrary length and
 * can be overflowed relatively easily if your sysadmin creates dangerous
 * terminfo files.  Unfortunately, dynamic memory allocation for something like
 * this would be far too slow.
 */
char tparm_buf[TERM_BUFSIZE];

int
arr_init_term(term)
	struct arr_term *term;
{
	struct winsize ws;
	char *ptr;

	if (tinfo_init((ptr = getenv("TERM")) != NULL ? ptr : "dumb") < 0)
		kill_tc_entry();

	if (ioctl(1, TIOCGWINSZ, &ws) < 0) {
		if (((term->co = tinfo_getnum(tc_entry, OFFSET_CO)) < 0) ||
		    ((term->li = tinfo_getnum(tc_entry, OFFSET_LI)) < 0)) {
			fprintf(stderr, "Warning: Using 80x25 terminal size\n");
			sleep(1);

			term->co = 25;
			term->li = 80;
		}
	}
	else {
		term->co = ws.ws_col;
		term->li = ws.ws_row;
	}

	if ((*tinfo_getstr(tc_entry, OFFSET_CM)) == NUL) {
		--term->co;
		--term->li;
	}

	if (tinfo_getflag(tc_entry, OFFSET_NP) == 0)
		padding_chr = 0;
	else
		padding_chr = *tinfo_getstr(tc_entry, OFFSET_PC);

	if (arr_term_acsc(term) < 0)
		return -1;

	return 0;
}

int
arr_term_acsc(term)
	struct arr_term *term;
{
	u_char *t_ptr;

	if (!(term->acs_map = MALLOC(acs_map_t, sizeof_acs_default))) {
		perror("Unable to allocate memory chunk");
		return -1;
	}

	memcpy(term->acs_map, acs_default, 
		sizeof_acs_default * sizeof(acs_map_t));

	if ((t_ptr = (u_char *)tinfo_getstr(tc_entry, OFFSET_AC))) {
		for (; *t_ptr != NUL; t_ptr += 2) {
			term->acs_map[*t_ptr - '+'].chr = t_ptr[1];
			term->acs_map[*t_ptr - '+'].ext = 1;
		}

		TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_EA));
	}

	if ((*tinfo_getstr(tc_entry, OFFSET_CM)) != NUL)
		TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_TI));

	return 0;
}

int
arr_acs_char(chr)
	byte_t chr;
{
	int i;
	int order_table[] = {
		TC_HORZLINE,
		TC_VERTLINE,
		TC_LRCORNER,
		TC_URCORNER,
		TC_ULCORNER,
		TC_LLCORNER,
		TC_RTEE,
		TC_LTEE,
		TC_UTEE,
		TC_DTEE,
		TC_PLUS,
		TC_RARROW,
		TC_LARROW,
		TC_UARROW,
		TC_DARROW,
		TC_BULLET,
		TC_DIAMOND,
		TC_DEGREE,
		TC_PLUSMINUS,
		TC_LESSEQUAL,
		TC_MOREEQUAL,
		TC_NOTEQUAL,
		TC_PI,
		TC_STERLING,
		TC_BLOCK,
		TC_CHKBOARD,
		TC_SQBOARD,
		TC_LANTERN,
		TC_SCANL1,
		TC_SCANL3,
		TC_SCANL7,
		TC_SCANL9
	};

	for (i = 0; i < sizeof(order_table) / sizeof(order_table[0]); ++i)
		if (arr_scr->term.acs_map[order_table[i] - 256].chr == chr)
			return order_table[i];

	return 0;
}

void
term_reset(void)
{
	term_buffer(tinfo_getstr(tc_entry, OFFSET_ME));
	term_buffer(tinfo_getstr(tc_entry, OFFSET_CL));
	term_buffer(tinfo_getstr(tc_entry, OFFSET_VE));
	term_buffer(tinfo_getstr(tc_entry, OFFSET_AE));

	if ((*tinfo_getstr(tc_entry, OFFSET_CM)) != NUL)
		TERM_NOTHROTTLE_WRITE(tinfo_getstr(tc_entry, OFFSET_TE));

	TERM_FLUSH();
}
