/*
 *  inf.c
 *  mod_musicindex
 *
 *  $Id: inf.c,v 1.27 2004/07/04 17:32:10 varenet Exp $
 *
 *  Created by Thibaut VARENE on Thu Mar 20 2003.
 *  Copyright (c) 2003-2004 Regis BOUDIN
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1, or (at your option)
 *  any later version.
 *
 */

/**
 * @file 
 * Sorting routines.
 *
 * @author Regis Boudin
 * @version $Revision: 1.27 $
 * @date 2003-2004
 *
 * @todo Regis, please document (eventually).
 * @todo review int types
 */

#include "inf.h"

/**
 * Sorts the list of musical entries.
 *
 * This is a fully home-made quicksort implementation. It places directory
 * entries first, then songs accordingly to the configuration.
 *
 * @param base The first entry of the portion to sort
 * @param end The last entry of the portion to sort
 * @param conf The configuration structure
 *
 * @return The first entry of the sorted list
 */
mu_ent *quicksort(mu_ent *base, mu_ent *end, mu_config *conf) 
{
	mu_ent *prem;
	mu_ent *deux = base;
	/* The first entry will be used as reference */
	mu_ent *refe = base;

	/* If we have 0 or one entry, sorting is very easy */
	if ((base == end) || (base->next == end))
		return base;

	/* We will begin comparisons with the second entry of the list */
	prem = base->next;

	do {
		/* If the current entry is not correctly placed,
		move it before the reference */
		if (inf_global(prem, refe, conf) < 0) {
			deux->next = prem->next;
			prem->next = base;
			base = prem;
			prem = deux->next;
		/* Otherwise, just continue working with the next one */
		} else {
			deux = prem;
			prem = prem->next;
		}
	} while (prem != end);

	/* We now have two lists, A & B, separated by a value V,
	considering A < V < B . Just run the same quicksort on A and B */
	base = quicksort(base, refe, conf);
	refe->next = quicksort(refe->next, end, conf);

	return base;
}

/**
 * Sets sort order function pointers.
 *
 * @param conf MusicIndex configuration paramaters struct.
 */
void set_fctptrs(mu_config *conf)
{
	unsigned short i;

	for (i = 0; i < ARG_NUMBER; i++) {
		switch (conf->order[i]) {
			case SB_ARTIST:
				conf->order_functions[i] = inf_by_artist;
				break;
			case SB_ALBUM:
				conf->order_functions[i] = inf_by_album;
				break;
			case SB_TITLE:
				conf->order_functions[i] = inf_by_title;
				break;
			case SB_TRACK:
				conf->order_functions[i] = inf_by_track;
				break;
			case SB_POSN:
				conf->order_functions[i] = inf_by_posn;
				break;
			case SB_LENGTH:
				conf->order_functions[i] = inf_by_length;
				break;
			case SB_BITRATE:
				conf->order_functions[i] = inf_by_bitrate;
				break;
			case SB_FILETYPE:
				conf->order_functions[i] = inf_by_filetype;
				break;
			case SB_FILENAME:
				conf->order_functions[i] = inf_by_file;
				break;
			case SB_URI:
				conf->order_functions[i] = inf_by_uri;
				break;
			case SB_DATE:
				conf->order_functions[i] = inf_by_date;
				break;
			case SB_SIZE:
				conf->order_functions[i] = inf_by_size;
				break;
			case SB_GENRE:
				conf->order_functions[i] = inf_by_genre;
				break;
			case SB_DIR:
				conf->order_functions[i] = inf_by_dir;
				break;
			case SB_RANDOM:
				conf->order_functions[i] = inf_by_rand;
				break;
			case SB_MTIME:
				conf->order_functions[i] = inf_by_mtime;
				break;
			default:
				conf->order_functions[i] = inf_by_uri;
		}
	}
	if (i == ARG_NUMBER)
		i--;
	/* Force inf_by_uri as last comparison */
	conf->order_functions[i] = inf_by_uri;
}

short inf_by_rand(mu_ent *first, mu_ent *second)
{
	return (rand()-(RAND_MAX/2));
}

short inf_by_track(mu_ent *first, mu_ent *second)
{
	return (first->track - second->track);
}

short inf_by_posn(mu_ent *first, mu_ent *second)
{
	return (first->posn - second->posn);
}

short inf_by_date(mu_ent *first, mu_ent *second)
{
	return (first->date - second->date);
}

short inf_by_length(mu_ent *first, mu_ent *second)
{
	/* length is long */
	return ((first->length < second->length) ? -1 : 1);
}

short inf_by_bitrate(mu_ent *first, mu_ent *second)
{
	/* bitrate is long */
	return ((first->bitrate < second->bitrate) ? -1 : 1);
}

short inf_by_size(mu_ent *first, mu_ent *second)
{
	/* size is long */
	return ((first->size < second->size) ? -1 : 1);
}

/**
 * Sorts by mtime.
 * Sort order is inverted here, since it wouldn't make sense
 * to have "oldest" entries first.
 * @param first first
 * @param second second
 * @return order
 */
short inf_by_mtime(mu_ent *first, mu_ent *second)
{
	/* mtime is long */
	return ((first->mtime < second->mtime) ? 1 : -1);
}

short inf_by_artist(mu_ent *first, mu_ent *second)
{
	if ((!first->artist) || (!second->artist)) {
		if (first->artist)
			return -1;
		else if (second->artist)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->artist, second->artist);
}

short inf_by_album(mu_ent *first, mu_ent *second)
{
	if ((!first->album) || (!second->album)) {
		if (first->album)
			return -1;
		else if (second->album)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->album, second->album);
}

short inf_by_title(mu_ent *first, mu_ent *second)
{
	if ((!first->title) || (!second->title)) {
		if (first->title)
			return -1;
		else if (second->title)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->title, second->title);
}

short inf_by_filetype(mu_ent *first, mu_ent *second)
{
	return (first->filetype - second->filetype);
}

short inf_by_file(mu_ent *first, mu_ent *second)
{
	return strcasecmp(first->file, second->file);
}

short inf_by_uri(mu_ent *first, mu_ent *second)
{
	return strcmp(first->uri, second->uri);
}

short inf_by_genre(mu_ent *first, mu_ent *second)
{
	if ((!first->genre) || (!second->genre)) {
		if (first->genre)
			return -1;
		else if (second->genre)
			return 1;
		else
			return 0;
	}
	return strcasecmp(first->genre, second->genre);
}

short inf_by_dir(mu_ent *first, mu_ent *second)
{
	char *one = first->uri;
	char *two = second->uri;
	unsigned char cone = 'a', ctwo = 'a';
	unsigned short i;
	
	for (i=0; one[i] == two[i]; i++)
		continue;
	
	for (; ((cone!='\0') && (cone!='/')) || ((ctwo!='\0') && (ctwo!='/')); i++) {
		if (((one[i]=='\0')||(one[i]=='/'))&&(cone!='\0')&&(cone!='/'))
			cone = one[i];
		
		if (((two[i]=='\0')||(two[i]=='/'))&&(ctwo!='\0')&&(ctwo!='/'))
			ctwo = two[i];
	}
	
	return ((short)cone - (short)ctwo);
}

/**
 * Returns the "difference" between 2 files.
 *
 * If one entry is a dir, put it first. If both are, sort them by alpahbetic
 * order. Then, compare according to each different parameter designed.
 *
 * @param first The first entry
 * @param second The second entry
 * @param conf The configuration structure
 *
 * @return The difference between entries (<0 means order will be inverted)
 */
short inf_global(mu_ent *first, mu_ent *second, mu_config *conf)
{
	short result, i;

	if ((first->filetype == FT_DIR) || (second->filetype == FT_DIR)) {
		if ((result = second->filetype - first->filetype))
			return (result);
		else
			return (inf_by_file(first, second));
	}

	for (i = 0; i < ARG_NUMBER; i++) {
		if (conf->order_functions[i]) {
			if ((result = (conf->order_functions[i](first, second))))
				return result;
		}
	}
	return 1;
}
