//LabPlot : CapabilityListDialog.cc

#include "CapabilityListDialog.h"
#include "defs.h"

#ifdef HAVE_GSL
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_fit.h>
#include <gsl/gsl_blas.h>
#endif

struct capdata {
	int n;
	double * x;
	double * y;
	double * sigma;
};

#ifdef HAVE_GSL
int gaussian_f (const gsl_vector * v, void *params, gsl_vector * f) {
	int n = ((struct capdata *)params)->n;
	double *x = ((struct capdata *)params)->x;
	double *y = ((struct capdata *)params)->y;
	double *sigma = ((struct capdata *) params)->sigma;

	double A = gsl_vector_get (v, 0);
	double x0 = gsl_vector_get (v, 1);
	double s = gsl_vector_get (v, 2);

	for (int i = 0; i < n; i++) {
		/* Model Yi = A * exp(-lambda * i) + b */
		double Yi = A * exp (-(x[i]-x0)*(x[i]-x0)/(2*s*s));
		gsl_vector_set (f, i, (Yi - y[i])/sigma[i]);
	}

	return GSL_SUCCESS;
}

int gaussian_df (const gsl_vector * v, void *params, gsl_matrix * J) {
	int n = ((struct capdata *)params)->n;
	double *x = ((struct capdata *)params)->x;
	double *sigma = ((struct capdata *) params)->sigma;

	double A = gsl_vector_get (v, 0);
	double x0 = gsl_vector_get (v, 1);
	double s = gsl_vector_get (v, 2);

	for (int i = 0; i < n; i++) {
		/* Jacobian matrix J(i,j) = dfi / dxj, */
		/* where fi = (Yi - yi)/sigma[i],      */
		/*	Yi = A * exp(-lambda * i) + b  */
		/* and the xj are the parameters (A,lambda,b) */
		double e = exp(-(x[i]-x0)*(x[i]-x0)/(2*s*s));
		gsl_matrix_set (J, i, 0, e/sigma[i]);
		gsl_matrix_set (J, i, 1, A * (x[i]-x0)/(s*s) * e/sigma[i]);
		gsl_matrix_set (J, i, 2, A *(x[i]-x0)*(x[i]-x0)/(s*s*s) * e/sigma[i]);
	}
	return GSL_SUCCESS;
}

int gaussian_fdf (const gsl_vector * x, void *params, gsl_vector * f, gsl_matrix * J) {
	gaussian_f (x, params, f);
	gaussian_df (x, params, J);

	return GSL_SUCCESS;
}

void CapabilityListDialog::print_state (int iter, gsl_multifit_fdfsolver * s) {
	QString text;
	text += "iter : "+QString::number(iter) + " | x = ";
	for (int i=0;i<3;i++)
		text += QString::number(gsl_vector_get (s->x, i))+" ";
	text += "|f(x)| = " + QString::number(gsl_blas_dnrm2 (s->f));

	infote->append(text);
}
#endif

CapabilityListDialog::CapabilityListDialog(MainWin *m, const char *name)
	: ListDialog(m, name)
{
	kdDebug()<<"CapabilityListDialog()"<<endl;
	setCaption(i18n("Capability Dialog"));
	KConfig *config = mw->Config();
	config->setGroup( "Capability" );

	QTabWidget *tw = new QTabWidget(vbox);
	// a tab for every operation ?
	QVBox *tab1 = new QVBox(tw);

	QHBox *hb = new QHBox(tab1);
	new QLabel(i18n("Bins : "),hb);
	hb = new QHBox(tab1);
	binni = new KIntNumInput(config->readNumEntry("Bins",100),hb);
	binni->setRange(1,INF,1,false);

	hb = new QHBox(tab1);
	new QLabel(i18n("Lower Specification Limit (LSL) : "),hb);
	hb = new QHBox(tab1);
	lslle = new KLineEdit(QString::number(config->readDoubleNumEntry("LSL",0)),hb);
	lslle->setValidator(new QDoubleValidator(lslle));
	hb = new QHBox(tab1);
	new QLabel(i18n("Upper Specification Limit (USL) : "),hb);
	hb = new QHBox(tab1);
	uslle = new KLineEdit(QString::number(config->readDoubleNumEntry("USL",1)),hb);
	uslle->setValidator(new QDoubleValidator(uslle));

	hb = new QHBox(tab1);
	histcb = new QCheckBox(i18n("Show Histogram"),hb);
	histcb->setChecked(config->readBoolEntry("HistEnabled",true));
	fitcb = new QCheckBox(i18n("Show Gaussian"),hb);
	fitcb->setChecked(config->readBoolEntry("FitEnabled",true));

	if(p==0) {
		histcb->setChecked(false);
		fitcb->setChecked(false);
		histcb->hide();
		fitcb->hide();
	}

	// a text edit for the result
	infote = new QTextEdit(vbox);
#if QT_VERSION > 0x030005
	infote->setTextFormat( Qt::LogText );
#else
	infote->setTextFormat( Qt::PlainText );
#endif

	Style *style=0;
	Symbol *symbol=0;
	QVBox *styletab=0;
	if(p && p->getPlot(p->API())->Type() == PSURFACE) {
		styletab = surfaceStyle(tw,true);
	}
	else if(p) {
		hb = new QHBox(tab1);
		labelcb = new QCheckBox(i18n("add label"),hb);
		labelcb->setChecked(config->readBoolEntry("LabelEnabled",true));
		styletab = simpleStyle(tw, style, symbol);
	}

	tw->addTab(tab1,i18n("Capability Analysis"));
	if(p) {
		tw->addTab(styletab,i18n("Style"));
		sheetcb->setCurrentItem(sheetcb->count()-2);	// new worksheet
	}

	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));

	setMinimumWidth((int) fmax(500,vbox->minimumSizeHint().width()));
	setMinimumHeight((int) fmax(450,gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height()));
	resize(minimumSize());
}

void CapabilityListDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Capability" );

	config->writeEntry("Bins",binni->value());
	config->writeEntry("LSL",lslle->text().toDouble());
	config->writeEntry("USL",uslle->text().toDouble());
	config->writeEntry("HistEnabled",histcb->isChecked());
	config->writeEntry("FitEnabled",fitcb->isChecked());
	if(p)
		config->writeEntry("LabelEnabled",labelcb->isChecked());
}

int CapabilityListDialog::apply_clicked() {
	kdDebug()<<"CapabilityListDialog::apply_clicked()"<<endl;
#ifdef HAVE_GSL
	int bins = binni->value();

	// get Graph2D object
	Graph2D *g=0;
	if(s)
		g = s->getGraph2D();
	else if (p) {
		GraphList *gl = p->getPlot(p->API())->getGraphList();
		int item = (int) (lv->itemPos(lv->currentItem())/lv->currentItem()->height());
		GRAPHType gtype = gl->getType(item);

		if (gtype == GRAPH2D)
			g = gl->getGraph2D(item);
		else
			return -1;
	}

	Style *style = 0;
	Symbol *symbol = 0;
	if(p) {
		style = new Style((StylesType)cb2->currentItem(),color->color(),filled->isChecked(),fcolor->color(),
				  width->value(),pencb->currentItem(),brushcb->currentItem());
		style->setBoxWidth(boxwidth->value());
		style->setAutoBoxWidth(autobox->isChecked());
		style->setPointsSorting(sortpointscb->isChecked());
		symbol = new Symbol((SType)symbolcb->currentItem(),scolor->color(),ssize->value(),
				     (FType)symbolfillcb->currentItem(),sfcolor->color(),sbrushcb->currentItem());
	}

	if(g == 0) {
		KMessageBox::error(this,i18n("No data found."));
		return -1;
	}

	int n=0;
	Point *data=0;
	QString label;

	n = g->Number();
	data = g->Data();
	LRange yrange = g->Range(1);
	double rmin = yrange.rMin(), rmax = yrange.rMax();
	label = g->getLabel()->simpleTitle();

	if(rmax-rmin == 0) {
		KMessageBox::error(this,i18n("Histogram can not be made because data is constant."));
		return -1;
	}

	// make histogram of bins
	gsl_histogram *h = gsl_histogram_alloc (bins);
	gsl_histogram_set_ranges_uniform (h, rmin, rmax);

	for (int i = 0;i<n;i++)
		gsl_histogram_increment(h,data[i].Y());

	double xmin=0, xmax=1, ymin=0, ymax=1;
	Point *ptr = new Point[bins];
	for (int i = 0;i<bins;i++) {
		double x = rmin + i*(rmax-rmin)/(bins-1);
		double y = gsl_histogram_get (h,i);

		ptr[i].setPoint(x,y);
	}

	mw->calculateRanges2D(ptr,bins,&xmin,&xmax,&ymin,&ymax);

	// create the new Graph
	LRange range[2];
	range[0] = LRange(xmin,xmax);
	range[1] = LRange(ymin,ymax);

	QString histfun = QString(i18n("histogram of ") + label);
	Graph2D *histg = new Graph2D(histfun,histfun,range,SSPREADSHEET,P2D,style,symbol,ptr,bins);
	gsl_histogram_free (h);

	// fit gaussian
	int nrp=3, N=bins;

	gsl_matrix *covar = gsl_matrix_alloc (nrp, nrp);
	double x[N], y[N], sigma[N];
	struct capdata d = { N, x, y, sigma};
	double x_init[3] = { 1.0, (xmin+xmax)/2.0, 1.0 };
	gsl_vector_view v = gsl_vector_view_array (x_init, nrp);

	gsl_multifit_function_fdf f;
	f.f = &gaussian_f;
	f.df = &gaussian_df;
	f.fdf = &gaussian_fdf;
	f.n = N;
	f.p = nrp;
	f.params = &d;

	for(int i=0;i<N;i++) {
		x[i]=ptr[i].X();
		y[i]=ptr[i].Y();
		sigma[i] = 1.0;
	}

	const gsl_multifit_fdfsolver_type *T = gsl_multifit_fdfsolver_lmsder;
	gsl_multifit_fdfsolver *s = gsl_multifit_fdfsolver_alloc (T, N, nrp);
	gsl_multifit_fdfsolver_set (s, &f, &v.vector);


	int iter=0, status;
	print_state (iter, s);
	do {
		iter++;
		status = gsl_multifit_fdfsolver_iterate (s);

		//infote->append(QString("status = ")+gsl_strerror(status));
		print_state (iter, s);

		if (status) break;

		status = gsl_multifit_test_delta (s->dx, s->x,1e-4, 1e-4);
	} while (status == GSL_CONTINUE && iter < 500);

	gsl_multifit_covar (s->J, 0.0, covar);

#define FIT(i) gsl_vector_get(s->x, i)
#define ERR(i) sqrt(gsl_matrix_get(covar,i,i))

	QString text;
	text += "A = "+QString::number(FIT(0))+" +/- "+QString::number(ERR(0))+"\n";
	text += "x0 = "+QString::number(FIT(1))+" +/- "+QString::number(ERR(1))+"\n";
	text += "sigma = "+QString::number(FIT(2))+" +/- "+QString::number(ERR(2))+"\n";
	double chi = gsl_blas_dnrm2(s->f);

	text += "chi^2 = "+QString::number(chi*chi)+"\n";
	infote->append(text);

	xmin=0, xmax=1, ymin=0, ymax=1;
	Point *ptr2 = new Point[bins];
	for (int i = 0;i<bins;i++) {
		double x = rmin + i*(rmax-rmin)/(bins-1);
		double y = FIT(0)*exp(-(x-FIT(1))*(x-FIT(1))/(2*FIT(2)*FIT(2)));

		ptr2[i].setPoint(x,y);
	}

	mw->calculateRanges2D(ptr2,bins,&xmin,&xmax,&ymin,&ymax);

	// create the new Graph
	LRange range2[2];
	range2[0] = LRange(xmin,xmax);
	range2[1] = LRange(ymin,ymax);

	QString fitfun = QString(i18n("fit of histogram ") + label);
	Graph2D *fitg = new Graph2D(fitfun,fitfun,range2,SSPREADSHEET,P2D,style,symbol,ptr2,bins);

	double lsl = lslle->text().toDouble(), usl = uslle->text().toDouble();
	double cpl = (FIT(1)-lsl)/(3.0*FIT(2)), cpu = (usl-FIT(1))/(3.0*FIT(2));
	text = "Cp = "+QString::number((usl-lsl)/(6.0*FIT(2)))+"\n";
	text += "Cpl = "+QString::number(cpl)+"\n";
	text += "Cpu = "+QString::number(cpu)+"\n";
	text += "Cpk = "+QString::number(fmin(cpl,cpu))+"\n";
	infote->append(text);
	infote->scrollToBottom();

	//TODO : spreadsheet : select destination
	int item=0;
	if(p != 0) {
		item = sheetcb->currentItem();

		if(histcb->isChecked())
			mw->addGraph2D(histg,item);
		if(fitcb->isChecked())
			mw->addGraph2D(fitg,item);
	}

	// set region
	if(Worksheet *ws = mw->activeWorksheet()) {
		if(Plot *plot = ws->getPlot(ws->API())) {
			plot->enableRegion();
			plot->setRegion(lsl,usl);
			if(labelcb->isChecked()) {
				Label *label = new Label(text.replace(QChar('\n'),QString("<br>")), QFont("TImes",10));
				label->setPosition(0.13,0.1);
				label->setBoxed(true);
				ws->setLabel(0,label);
			}
			ws->updatePixmap();
		}
	}


	if(p) updateList();

#endif
	return 0;
}
