/* icoutils - A set of programs dealing with MS Windows icons and cursors.
 * Copyright (C) 1998-1999 Oskar Liljeblad
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "wrestool.h"

static void *extract_group_icon_cursor_resource(WinLibrary *, WinResource *, char *, int *, boolean);

void
extract_resources_callback (WinLibrary *fi, WinResource *wr,
                            WinResource *type_wr, WinResource *name_wr,
                            WinResource *lang_wr)
{
	int size;
	boolean free_it;
	void *memory;
	char *outname;
	FILE *out;

	memory = extract_resource(fi, wr, &size, &free_it, type_wr->id, (lang_wr == NULL ? NULL : lang_wr->id), arg_raw);
	free_it = FALSE;
	if (memory == NULL) {
		/* extract resource has printed error */
		return;
	}

	/* determine where to extract to */
	outname = get_destination_name(fi, type_wr->id, name_wr->id, (lang_wr == NULL ? NULL : lang_wr->id));
	if (outname == NULL) {
		out = stdout;
	} else {
		out = fopen(outname, "wb");
		if (out == NULL) {
			P_OBJ_SYS_ERROR(outname);
			goto cleanup;
		}
	}

	/* write the actual data */
	fwrite(memory, size, 1, out);
	
	cleanup:
	if (free_it)
		free(memory);
	if (out != NULL && out != stdout)
		fclose(out);
}

/* extract_resource:
 *   Extract a resource, returning pointer to data.
 */
void *
extract_resource (WinLibrary *fi, WinResource *wr, int *size,
                  boolean *free_it, char *type, char *lang, boolean raw)
{
	char *str;
	int intval;
	
	/* just return pointer to data if raw */
	if (raw) {
		*free_it = FALSE;
		/* get_resource_entry will print possible error */
		return get_resource_entry(fi, wr, size);
	}

	/* find out how to extract */
	str = type;
	if (str != NULL && str2int(STRIP_RES_ID_FORMAT(str), &intval)) {
		if (intval == (int) RT_GROUP_ICON) {
			*free_it = TRUE;
			return extract_group_icon_cursor_resource(fi, wr, lang, size, TRUE);
		}
		if (intval == (int) RT_GROUP_CURSOR) {
			*free_it = TRUE;
			return extract_group_icon_cursor_resource(fi, wr, lang, size, FALSE);
		}
	}

	P_OBJ_ERROR(fi->name, "don't know how to extract resource, try `--raw'\n");
	return NULL;
}

/* extract_group_icon_resource:
 *   Create a complete RT_GROUP_ICON resource, that can be written to
 *   an `.ico' file without modifications. Returns an allocated
 *   memory block that should be freed with free() once used.
 *
 *   `root' is the offset in file that specifies the resource.
 *   `base' is the offset that string pointers are calculated from.
 *   `ressize' should point to an integer variable where the size of
 *   the returned memory block will be placed.
 *   `is_icon' indicates whether resource to be extracted is icon
 *   or cursor group.
 */
static void *
extract_group_icon_cursor_resource(WinLibrary *fi, WinResource *wr, char *lang,
                                   int *ressize, boolean is_icon)
{
	Win32CursorIconDir *icondir;
	Win32CursorIconFileDir *fileicondir;
	char *memory;
	int c, size, offset;

	/* get resource data and size */
	icondir = (Win32CursorIconDir *) get_resource_entry(fi, wr, &size);
	if (icondir == NULL) {
		/* get_resource_entry will print error */
		return NULL;
	}

	/* calculate total size of output file */
	RETURN_IF_BAD_POINTER(NULL, icondir->count);
	offset = size = sizeof(Win32CursorIconFileDir)
	  + icondir->count * sizeof(Win32CursorIconFileDirEntry);
	for (c = 0 ; c < icondir->count ; c++) {
		RETURN_IF_BAD_POINTER(NULL, icondir->entries[c]);
		verbose_printf("%d. bytes_in_res=%ld width=%d height=%d planes=%d bit_count=%d\n", c,
			icondir->entries[c].bytes_in_res,
			(is_icon ? icondir->entries[c].res_info.icon.width : icondir->entries[c].res_info.cursor.width),
			(is_icon ? icondir->entries[c].res_info.icon.height : icondir->entries[c].res_info.cursor.height),
			icondir->entries[c].plane_count,
			icondir->entries[c].bit_count);

		size += icondir->entries[c].bytes_in_res;

		/* cursor resources have two additional WORDs that contain
		 * hotspot info */
		if (!is_icon)
			size -= sizeof(uint16_t)*2;
	}
	*ressize = size;

	verbose_printf("group_icon icons=%d size=%d\n", icondir->count, size);

	/* allocate that much memory */
	memory = malloc(size);
	if (memory == NULL)
		malloc_failure();
	fileicondir = (Win32CursorIconFileDir *) memory;

	/* transfer Win32CursorIconDir structure members */
	fileicondir->reserved = icondir->reserved;
	fileicondir->type = icondir->type;
	fileicondir->count = icondir->count;

	/* transfer each cursor/icon: Win32CursorIconDirEntry and data */
	for (c = 0 ; c < icondir->count ; c++) {
		int level;
		char name[14];
		WinResource *fwr;
		char *data;
	
		/* copy ICONDIRENTRY (not including last dwImageOffset) */
		memcpy(&fileicondir->entries[c], &icondir->entries[c],
			sizeof(Win32CursorIconFileDirEntry)-sizeof(uint32_t));

		/* special treatment for cursors */
		if (!is_icon) {
			fileicondir->entries[c].width = icondir->entries[c].res_info.cursor.width;
			fileicondir->entries[c].height = icondir->entries[c].res_info.cursor.height / 2;
			fileicondir->entries[c].color_count = 0;
			fileicondir->entries[c].reserved = 0;
		}

		/* set image offset and increase it */
		fileicondir->entries[c].dib_offset = offset;

		/* find the corresponding icon resource */
		name[0] = '-';
		int2str(icondir->entries[c].res_id, &name[1], 13);

		fwr = find_resource(fi, (is_icon ? "-3" : "-1"), name, lang, &level);
		if (fwr == NULL) {
			error_printf("%s: could not find `%s' in `%s' resource.\n",
			 	fi->name, &name[1], (is_icon ? "group_icon" : "group_cursor"));
			return NULL;
		}

		/* get data and size of that resource */
		data = get_resource_entry(fi, fwr, &size);
		if (data == NULL) {
			/* get_resource_entry has printed error */
			return NULL;
		}
		
		if (size < icondir->entries[c].bytes_in_res) {
			error_printf("%s: mismatch of size in icon resource `%s' and group\n",
			  fi->name, name);
			return NULL;
		}

		/* transfer resource into file memory */
		if (is_icon) {
			memcpy(&memory[offset], data, icondir->entries[c].bytes_in_res);
		} else {
			fileicondir->entries[c].hotspot_x = ((uint16_t *) data)[0];
			fileicondir->entries[c].hotspot_y = ((uint16_t *) data)[1];
			memcpy(&memory[offset], data+sizeof(uint16_t)*2,
				   icondir->entries[c].bytes_in_res-sizeof(uint16_t)*2);
			offset -= sizeof(uint16_t)*2;
		}

		/* increase the offset pointer */
		offset += icondir->entries[c].bytes_in_res;
	}

	return (void *) memory;
}
