// ParPort.h

/* Copyright (C) 2000-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#ifndef PARPORT_H
#define PARPORT_H

#include "ExMgr.h"
#include <stdio.h>

#if defined(PAR_PLATFORM_LINUX)
	#include "ioWrapper.h"
#elif defined(PAR_PLATFORM_FREEBSD)
	#include <unistd.h>
	#include <fcntl.h>
    extern "C" {
	#include <machine/cpufunc.h>
    }
#else
	#error Undefined platform
#endif

#if 0	// Change to 1 to enable LOG_INFO messages.
	#define PARPORT_LOG_INFO LOG_INFO
#else
	#define PARPORT_LOG_INFO(args...)
#endif


class ParPort {
	// Constructor, destructor, dump():
    public:
	/* Bank 0: 0x3BC, 0x378, 0x278 for ISA, may differ for PCI.
	 * Bank 1: 0x400 above bank 0 for ISA, may differ for PCI. */
	ParPort(int baseaddr0=0x378,int baseaddr1=0x778,
	    int portType=PORTTYPE_UNKNOWN);
	~ParPort() {
		terminate();
		removeIoPortAccess();
	}
	void dump(void);

	// Port type:
    public:		// CONST
	static const int PORTTYPE_NONE		=ERROR;
	static const int PORTTYPE_UNKNOWN	=0;
	static const int PORTTYPE_SPP		=1;
	static const int PORTTYPE_BPP		=2;
	static const int PORTTYPE_ECP		=3;	/* No HW assist. */
	static const int PORTTYPE_ECPHW		=4;
    protected:		// ATTRIB
	int portType;
    protected:
	void setPortType(int value,int force=0) {
		if (portType==PORTTYPE_UNKNOWN || force)
			portType=value;
	}
    public:
	int getPortType(void) { return portType; }
	int portTypeIsValid(void) {
		return portType>PORTTYPE_UNKNOWN;
	}
	int portTypeIsBidirectional(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>PORTTYPE_SPP);
	}
	int portTypeIsEcp(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>=PORTTYPE_ECP);
	}
	int portTypeIsEcpWithHwAssist(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>=PORTTYPE_ECPHW);
	}

	// Timeouts:
    public:		// CONST
	enum TimevalIndex {
		TIMEVAL_COMPAT_SETUP_DELAY=0,
		TIMEVAL_COMPAT_STROBE_DELAY,
		TIMEVAL_COMPAT_HOLD_DELAY,
		TIMEVAL_ECP_SETUP_DELAY,
		TIMEVAL_ECP_POST_HTR_DELAY,
		TIMEVAL_SIGNAL_TIMEOUT,
		TIMEVAL_BUSY_TIMEOUT,
		TIMEVAL_EPP_TERM_DELAY,
		TIMEVAL_COUNT
	};
    protected:		// ATTRIB
	struct timeval timevalArray[TIMEVAL_COUNT];
    protected:
	struct timeval *lookupTimeval(TimevalIndex index) {
		return &timevalArray[index];
	}
	void dumpTimeval(TimevalIndex index,char *name) {
		printf("%s(%d)={tv_sec=%ld,tv_usec=%ld}\n",
			name,index,
			timevalArray[index].tv_sec,
			timevalArray[index].tv_usec);
	}
    public:
	void getTimeval(TimevalIndex index,int *pSec,int *pUsec) {
		struct timeval *src=lookupTimeval(index);
		if (pSec) *pSec=src->tv_sec;
		if (pUsec) *pUsec=src->tv_usec;
	}
	void getTimeval(TimevalIndex index,struct timeval *dest) {
		struct timeval *src=lookupTimeval(index);
		if (dest) {
			dest->tv_sec=src->tv_sec;
			dest->tv_usec=src->tv_usec;
		}
	}
	void setTimeval(TimevalIndex index,int sec,int usec) {
		struct timeval *dest=lookupTimeval(index);
		dest->tv_sec=sec;
		dest->tv_usec=usec;
	}
	void setTimeval(TimevalIndex index,struct timeval *src) {
		struct timeval *dest=lookupTimeval(index);
		dest->tv_sec=src->tv_sec;
		dest->tv_usec=src->tv_usec;
	}
    protected:
	void initTimevals(void) {
		setTimeval(TIMEVAL_COMPAT_SETUP_DELAY,0,100);
		setTimeval(TIMEVAL_COMPAT_STROBE_DELAY,0,100);
		setTimeval(TIMEVAL_COMPAT_HOLD_DELAY,0,100);
		setTimeval(TIMEVAL_ECP_SETUP_DELAY,0,0);
		setTimeval(TIMEVAL_ECP_POST_HTR_DELAY,0,10000);
		setTimeval(TIMEVAL_SIGNAL_TIMEOUT,1,0);
		setTimeval(TIMEVAL_BUSY_TIMEOUT,1,0);
		setTimeval(TIMEVAL_EPP_TERM_DELAY,0,100);
	}

	// I/O port register access (PC-style assumed for efficiency):
    protected:		// CONST
	enum {
		SUBREG_DATA=0,
		SUBREG_STATUS,
		SUBREG_CONTROL,
		SUBREG_COUNT
	};
	enum ParPortRegister {
		REG_SPP_DATA=0,
		REG_SPP_STATUS,
		REG_SPP_CONTROL,
		REG_ECP_DATA,
		REG_ECP_STATUS,
		REG_ECP_CONTROL,
		REG_COUNT
	};
    protected:		// ATTRIB
	int regaddr[REG_COUNT];
#ifdef PAR_PLATFORM_FREEBSD
	int iofd;
#endif
    protected:
	void regInit(int baseaddr0,int baseaddr1) {
		regaddr[REG_SPP_DATA   ]=baseaddr0+SUBREG_DATA   ;
		regaddr[REG_SPP_STATUS ]=baseaddr0+SUBREG_STATUS ;
		regaddr[REG_SPP_CONTROL]=baseaddr0+SUBREG_CONTROL;
		regaddr[REG_ECP_DATA   ]=baseaddr1+SUBREG_DATA   ;
		regaddr[REG_ECP_STATUS ]=baseaddr1+SUBREG_STATUS ;
		regaddr[REG_ECP_CONTROL]=baseaddr1+SUBREG_CONTROL;
	}
	int regRead(ParPortRegister reg) {
		// LOGD_ASSERT(portTypeIsEcp() || reg<REG_ECP_DATA,0,0,0);
		#if (defined(PAR_PLATFORM_LINUX) || \
		     defined(PAR_PLATFORM_FREEBSD))
			return inb(regaddr[reg]);
		#else
			#error Undefined platform
		#endif
	}
	void regWrite(ParPortRegister reg,int value) {
		// LOGD_ASSERT(portTypeIsEcp() || reg<REG_ECP_DATA,0,0,0);
		#if defined(PAR_PLATFORM_LINUX)
			outb(value,regaddr[reg]);
		#elif defined(PAR_PLATFORM_FREEBSD)
			outb(regaddr[reg],value);
		#else
			#error Undefined platform
		#endif
	}
	int getIoPortAccess(void) {
		#if defined(PAR_PLATFORM_LINUX)
			return iopl(3);
		#elif defined(PAR_PLATFORM_FREEBSD)
			return (iofd=open("/dev/io",O_RDWR));
		#else
			#error Undefined platform
		#endif
	}
	int removeIoPortAccess(void) {
		#if defined(PAR_PLATFORM_LINUX)
			return iopl(0);
		#elif defined(PAR_PLATFORM_FREEBSD)
			return close(iofd);
		#else
			#error Undefined platform
		#endif
	}

	// SPP control register:
    protected:		// CONST
	static const int CONTROL_NSTROBE	=0x01;
	static const int CONTROL_NAUTOFD	=0x02;
	static const int CONTROL_NINIT		=0x04;
	static const int CONTROL_NSELECTIN	=0x08;
	static const int CONTROL_INT_ENABLE	=0x10;	/* Not used. */
	static const int CONTROL_REVERSE_DATA	=0x20;
	/* Transparently handle PC-parport-inverted register bits: */
	static const int CONTROL_INVERTED=
		(CONTROL_NSTROBE|CONTROL_NAUTOFD|CONTROL_NSELECTIN);
    protected:
	int controlRead(void) {
		return (regRead(REG_SPP_CONTROL)^CONTROL_INVERTED);
	}
	void controlWrite(int value) {
		regWrite(REG_SPP_CONTROL,value^CONTROL_INVERTED);
	}
	void controlSetClear(int event,int set,int clear) {
		PARPORT_LOG_INFO(cEXBP,0,
			"controlSetClear(event=%d,set=0x%2.2X,clear=0x%2.2X)\n",
			event,set,clear);
		controlWrite((controlRead()&(~clear))|set);
#if 0 && defined(JD_DEBUGLITE)
		int r=controlRead();
		if ((r&(set|clear))!=set) {
		    LOG_WARN(cEXBP,0,
			"controlSetClear(event=%d,set=0x%2.2X,clear=0x%2.2X): "
			"incorrect control register=0x%2.2X!\n",
			event,set,clear,controlRead());
		}
#endif
	}
	void eppGetSetClearLines(int isAddress,int *set,int *clear) {
		if (!isAddress) {
			*set=CONTROL_NSELECTIN;
			*clear=CONTROL_NAUTOFD;
		} else {
			*set=CONTROL_NAUTOFD;
			*clear=CONTROL_NSELECTIN;
		}
	}

	// SPP status register:
    protected:		// CONST
	static const int STATUS_NFAULT		=0x08;
	static const int STATUS_SELECT		=0x10;
	static const int STATUS_PERROR		=0x20;
	static const int STATUS_NACK		=0x40;
	static const int STATUS_BUSY		=0x80;
	/* Transparently handle PC-parport-inverted register bits: */
	static const int STATUS_INVERTED	=(STATUS_BUSY);
    protected:		// ATTRIB
	PolledTimer statusWaitTimer;
    protected:
	int statusRead(void) {
		return (regRead(REG_SPP_STATUS)^STATUS_INVERTED);
	}
	int statusReadLowNibble(void) {
		int s=statusRead();
		return ((s>>3)&0x07)|((s>>4)&0x08);
	}
	int statusReadHighNibble(void) {
		int s=statusRead();
		return ((s<<1)&0x70)|(s&0x80);
	}
	int statusTestSetClear(int set,int clear) {
		return ((statusRead()&(set|clear))==set);
	}
    public:
	int statusReverseDataIsAvailable(void) {
		return (statusTestSetClear(0,STATUS_NFAULT) ||
			(modeIsEcpReverse() && ecpHwAssistIsEnabled() &&
			 !ecpFifoIsEmpty()>0));
	}
    protected:
	int statusWaitSetClear(int event,TimevalIndex timeout,
	    int set,int clear) {
		PARPORT_LOG_INFO(cEXBP,0,"statusWaitSetClear(event=%d,"
			"set=0x%2.2X,clear=0x%2.2X)\n",
			event,set,clear);

		if (statusTestSetClear(set,clear)) return OK;

		statusWaitTimer.setTimeout(lookupTimeval(timeout));
	    while (!statusTestSetClear(set,clear)) {
		if (event==43 && !statusReverseDataIsAvailable()) {
			return ERROR;
		}

		if (statusWaitTimer.isTimedOut()) {
			LOG_WARN(cEXBP,0,	// No error for event 23.
				"statusWaitSetClear(event=%d) timed out, "
				"expected set/clear=0x%2.2X/0x%2.2X, "
				"got=0x%2.2X, currentMode=0x%3.3X!\n",
				event,set,clear,statusRead(),currentMode);
			return ERROR;
		}
	    }

		return OK;
	}

	// SPP and ECP data registers:
    protected:		// ATTRIB
	int ecpFifoSize;
    protected:
	int dataRead(void) {
		return regRead(REG_SPP_DATA);
	}
	void dataWrite(int value) {
		regWrite(REG_SPP_DATA,value);
	}
	int dataTest(int value) {
		dataWrite(value);
		if (dataRead()!=value) return ERROR;
		return OK;
	}
	int ecpDataRead(void) {
		return regRead(REG_ECP_DATA);
	}
	void ecpDataWrite(int value) {
		regWrite(REG_ECP_DATA,value);
	}

	// ECP control and status registers:
    protected:		// CONST
	static const int ECP_CONTROL_MODE_UNIDIRECTIONAL	=0x00;
	static const int ECP_CONTROL_MODE_BIDIRECTIONAL		=0x20;
	static const int ECP_CONTROL_MODE_FAST_CENTRONICS	=0x40;
	static const int ECP_CONTROL_MODE_ECP			=0x60;
	static const int ECP_CONTROL_MODE_EPP			=0x80;
	static const int ECP_CONTROL_MODE_TEST			=0xC0;
	static const int ECP_CONTROL_MODE_CONFIGURATION		=0xE0;
	static const int ECP_CONTROL_MODE_MASK			=0xE0;
	static const int ECP_CONTROL_ALWAYS_SET			=0x14;
	static const int ECP_CONTROL_FULL			=0x02;
	static const int ECP_CONTROL_EMPTY			=0x01;
    protected:		// ATTRIB
	int ecpConfigA;
	int ecpConfigB;
    protected:
	int ecpControlRead(void) {
		return regRead(REG_ECP_CONTROL);
	}
	void ecpControlWrite(int value) {
		regWrite(REG_ECP_CONTROL,value|ECP_CONTROL_ALWAYS_SET);
	}
	int ecpStatusRead(void) {
		return regRead(REG_ECP_STATUS);
	}
	int ecpControlGetMode(void) {
		if (portTypeIsEcp()) {
			return ecpControlRead()&ECP_CONTROL_MODE_MASK;
		}
		if (portTypeIsBidirectional()) {
			return ECP_CONTROL_MODE_BIDIRECTIONAL;
		}
		return ECP_CONTROL_MODE_UNIDIRECTIONAL;
	}
	int ecpControlSetMode(int mode) {
		if (portTypeIsEcp()) {
			ecpControlWrite(mode);
			return OK;
		}
		if (portTypeIsBidirectional()) {
			if (mode>ECP_CONTROL_MODE_BIDIRECTIONAL) return ERROR;
			return OK;
		}
		if (mode>ECP_CONTROL_MODE_UNIDIRECTIONAL) return ERROR;
		return OK;
	}
	int ecpHwAssistIsEnabled(void) {
		int mode=ecpControlGetMode();
		return (mode==ECP_CONTROL_MODE_ECP ||
			mode==ECP_CONTROL_MODE_FAST_CENTRONICS);
	}
	int ecpFifoIsHosed(int r) {
		if ((r&(ECP_CONTROL_FULL|ECP_CONTROL_EMPTY))==
		     (ECP_CONTROL_FULL|ECP_CONTROL_EMPTY)) {
			LOG_ERROR(cEXBP,0,cCauseBadState,
				"ECP FIFO overrun or underrun!\n");
			return 1;
		}
		return 0;
	}
	int ecpFifoIsFull(void) {
		int r=ecpControlRead();
		if (ecpFifoIsHosed(r)) return ERROR;
		return r&ECP_CONTROL_FULL;
	}
	// Used only by writeEcpHw():
	int ecpFifoWaitForNotFull(TimevalIndex timeout) {
		if (!ecpFifoIsFull()) return OK;
		statusWaitTimer.setTimeout(lookupTimeval(timeout));
		while (42) {
			int r=ecpFifoIsFull();
			if (!r) break;
			if (r==ERROR || statusWaitTimer.isTimedOut()) {
				LOG_WARN(cEXBP,0,
					"ecpFifoWaitForNotFull timed out!\n");
				return ERROR;
			}
		}
		return OK;
	}
	// Used only by readEcpHw():
	int ecpFifoWaitForFull(TimevalIndex timeout) {
		if (ecpFifoIsFull()) return OK;
		statusWaitTimer.setTimeout(lookupTimeval(timeout));
		while (42) {
			int r=ecpFifoIsFull();
			if (r>0) break;
			if (r==ERROR || statusWaitTimer.isTimedOut()) {
				LOG_WARN(cEXBP,0,
					"ecpFifoWaitForFull timed out!\n");
				return ERROR;
			}
		}
		return OK;
	}
	int ecpFifoIsEmpty(void) {
		int r=ecpControlRead();
		if (ecpFifoIsHosed(r)) return ERROR;
		return r&ECP_CONTROL_EMPTY;
	}
	// Used only by readEcpHw():
	int ecpFifoWaitForNotEmpty(TimevalIndex timeout) {
		if (!ecpFifoIsEmpty()) return OK;
		statusWaitTimer.setTimeout(lookupTimeval(timeout));
		while (42) {
			int r=ecpFifoIsEmpty();
			if (!r) break;
			if (r==ERROR || statusWaitTimer.isTimedOut()) {
				LOG_WARN(cEXBP,0,
					"ecpFifoWaitForNotEmpty timed out!\n");
				return ERROR;
			}
		}
		return OK;
	}
	// Used only by negotiate(), terminate(), and ecpFwdToRev():
	int ecpFifoWaitForEmpty(TimevalIndex timeout) {
		if (!ecpHwAssistIsEnabled() || ecpFifoIsEmpty()>0) return OK;
		statusWaitTimer.setTimeout(lookupTimeval(timeout));
		while (42) {
			int r=ecpFifoIsEmpty();
			if (r>0) break;
			if (r==ERROR || statusWaitTimer.isTimedOut()) {
				LOG_WARN(cEXBP,0,
					"ecpFifoWaitForEmpty timed out!\n");
				return ERROR;
			}
		}
		return statusWaitSetClear(32,TIMEVAL_BUSY_TIMEOUT,
			0,
			STATUS_BUSY);
	}
	int ecpConfigRead(void) {
		int r=ERROR;
		int old=ecpControlGetMode();
		ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);
		if (ecpControlSetMode(ECP_CONTROL_MODE_CONFIGURATION)!=ERROR) {
			ecpConfigA=ecpDataRead();
			ecpConfigB=ecpStatusRead();
			ecpControlSetMode(ECP_CONTROL_MODE_UNIDIRECTIONAL);
			r=OK;
		}
		ecpControlSetMode(old);
		return r;
	}

	// Low-level 1284 mode/direction negotiation:
    public:		// CONST
	static const int MODE_COMPAT		=0x100;
	static const int MODE_REVERSE		=0x200;	/* Only for ECP. */
	static const int MODE_HW_ASSIST		=0x400; /* Compat or ECP. */
	static const int MODE_1284_MASK		=0x0FF;
	static const int MODE_NIBBLE		=0x000;
	static const int MODE_BYTE		=0x001;	/* Not supported. */
	static const int MODE_2			=0x002;	/* Not supported. */
	static const int MODE_DEVICE_ID		=0x004;
	static const int MODE_DOT3		=0x008;	/* 1284.3 bit. */
	static const int MODE_ECP		=0x010;
	static const int MODE_RLE		=0x020;	/* Not supported. */
	static const int MODE_EPP		=0x040;
	static const int MODE_EXT_LINK		=0x080;	/* Not supported. */
	static const int MODE_BOUNDED_ECP	=(MODE_ECP|MODE_DOT3);
	static const int MODE_CHANNELIZED_NIBBLE=(MODE_DOT3);
    protected:		// ATTRIB
	int currentMode;
    public:
	static int modeIsCompat(int mode) {
		return (mode&MODE_COMPAT);
	}
	int modeIsCompat(void) {
		return modeIsCompat(currentMode);
	}
	static int modeIsNibble0(int mode) {
		return ((mode&MODE_1284_MASK)==MODE_NIBBLE);
	}
	int modeIsNibble0(void) {
		return modeIsNibble0(currentMode);
	}
	static int modeIsNibble(int mode) {
		return (!(mode&(~(MODE_DEVICE_ID|MODE_DOT3))));
	}
	int modeIsNibble(void) {
		return modeIsNibble(currentMode);
	}
	static int modeIsEcp(int mode) {
		return (mode&MODE_ECP);
	}
	int modeIsEcp(void) {
		return modeIsEcp(currentMode);
	}
	static int modeIsBoundedEcp(int mode) {
		return ((mode&MODE_BOUNDED_ECP)==MODE_BOUNDED_ECP);
	}
	int modeIsBoundedEcp(void) {
		return modeIsBoundedEcp(currentMode);
	}
	static int modeIsEcpForward(int mode) {
		return ((mode&(MODE_ECP|MODE_REVERSE))==MODE_ECP);
	}
	int modeIsEcpForward(void) {
		return modeIsEcpForward(currentMode);
	}
	static int modeIsEcpReverse(int mode) {
		return ((mode&(MODE_ECP|MODE_REVERSE))==
			(MODE_ECP|MODE_REVERSE));
	}
	int modeIsEcpReverse(void) {
		return modeIsEcpReverse(currentMode);
	}
	int modeIsHwAssist(int mode) {
		return (portTypeIsEcpWithHwAssist() && (mode&MODE_HW_ASSIST));
	}
	int modeIsHwAssist(void) {
		return modeIsHwAssist(currentMode);
	}
	int modeIsBoundedEcpReverseHwAssist(int mode) {
		return (portTypeIsEcpWithHwAssist() &&
			((mode&(MODE_BOUNDED_ECP|MODE_REVERSE|MODE_HW_ASSIST))
			 ==(MODE_BOUNDED_ECP|MODE_REVERSE|MODE_HW_ASSIST)));
	}
	int modeIsBoundedEcpReverseHwAssist(void) {
		return modeIsBoundedEcpReverseHwAssist(currentMode);
	}
	static int modeIsEpp(int mode) {
		return (mode&MODE_EPP);
	}
	int modeIsEpp(void) {
		return modeIsEpp(currentMode);
	}
	int negotiate(int mode);
	int terminate(int mode=0);
	int ecpFwdToRev(int mode=0);
	int ecpRevToFwd(int mode=0);

	// Forward data transfers:
    protected:		// CONST
	static const int ECP_CHANNEL		=0x080;
	static const int DEFAULT_HTR_COUNT	=4;
    protected:		// ATTRIB
	int currentChannel;
	int ecpChannelSetNeeded;
	int htrCount;
    public:
	int getHtrCount(void) { return htrCount; }
	void setHtrCount(int htrCount) { SETTHIS(htrCount); }
    protected:
	int writeEcp(const unsigned char *buffer,int len,int isCommand);
	int writeEcpHw(const unsigned char *buffer,int len,int isCommand) {
		int countup=0;

	      // Normal data:
	    if (!isCommand /* || modeIsCompat() */ ) {
	      while (len>0) {
		if (ecpFifoWaitForNotFull(TIMEVAL_BUSY_TIMEOUT)==ERROR) break;
		ecpDataWrite(*buffer);
		buffer++;
		countup++;
		len--;
	      }
		return countup;
	    }

	      // ECP command bytes:
	      while (len>0) {
		if (ecpFifoWaitForNotFull(TIMEVAL_BUSY_TIMEOUT)==ERROR) break;
		dataWrite(*buffer);
		buffer++;
		countup++;
		len--;
	      }
		return countup;
	}
	int writeEpp(const unsigned char *buffer,int len,int isAddress);
    public:
	int write(const unsigned char *buffer,int len,int extra=0) {
		if (modeIsEcpForward()) {
			if (ecpChannelSetNeeded && !extra) {
				unsigned char channel=
					currentChannel|ECP_CHANNEL;
				// This is recursive:
				if (write(&channel,1,1)!=1) return ERROR;
				ecpChannelSetNeeded=0;
			}
			if (!ecpHwAssistIsEnabled()) {
				return writeEcp(buffer,len,extra);
			}
			return writeEcpHw(buffer,len,extra);
		}
		if (modeIsEpp()) {
			return writeEpp(buffer,len,extra);
		}
		LOG_ERROR(cEXBP,0,cCauseBadState,
			"write: invalid mode=0x%3.3X!\n",currentMode);
		return ERROR;
	}
    protected:
	/* The SIIG "Duet Parallel ISA" (and no telling what else) has a bug
	 * where exiting HW-assisted ECP mode after sending a command byte
	 * (with no data bytes in between) causes nAutoFd to be stuck at 0.
	 * Therefore, don't send channel-address bytes until right before
	 * sending regular data.  If that turns out to have problems, then
	 * we'll have to revert to doing this with software signalling right
	 * after negotiating into ECP mode. */
	STATUS writeEcpChannel(int channel) {
		if (channel!=currentChannel) {
			currentChannel=channel;
			ecpChannelSetNeeded=1;
		}
		return OK;
	}
    public:
	STATUS writeEppAddress(unsigned char addr) {
		if (writeEpp(&addr,1,1)!=1) return ERROR;
		return OK;
	}
	STATUS writeEppAddress(unsigned char addr,unsigned char data) {
		if (writeEppAddress(addr)==ERROR) return ERROR;
		if (writeEpp(&data,1,0)!=1) return ERROR;
		return OK;
	}

	// Reverse data transfers:
    protected:
	int readNibble(unsigned char *buffer,int len);
	int readEcpSw(unsigned char *buffer,int len);
	int readEcpHw(unsigned char *buffer,int len) {
		int countup=0;

	    while (len>0) {
		if (ecpFifoWaitForNotEmpty(TIMEVAL_BUSY_TIMEOUT)==ERROR) break;
		*buffer=ecpDataRead();
		buffer++;
		countup++;
		len--;
	    }

		return countup;
	}
	int readEcp(unsigned char *buffer,int len) {
		/* Don't use hardware-assisted non-bounded-ECP mode for
		 * transfers shorter than the FIFO size. */
		int firstLen=len-ecpFifoSize;
		if (!modeIsHwAssist() || firstLen<0) {
			return readEcpSw(buffer,len);
		}

		/* Event 43: nAck=0. */
		int r=statusWaitSetClear(43,TIMEVAL_BUSY_TIMEOUT,
			0,
			STATUS_NACK);
		if (r==ERROR) return 0;

		/* Enable hardware-assisted ECP mode. */
		ecpControlSetMode(ECP_CONTROL_MODE_ECP);
		/* Let the hardware have control of nAutoFd. */
		controlSetClear(-44,
			CONTROL_NAUTOFD,
			0);

		/* Read all but the last FIFO-length's worth of data. */
		int countup=readEcpHw(buffer,firstLen);
		/* Let the FIFO fill up with the last chunk of data. */
		if (countup==firstLen) {
			ecpFifoWaitForFull(TIMEVAL_BUSY_TIMEOUT);
		}

		/* Don't let the hardware latch in any more bytes. */
		controlSetClear(-144,
			0,
			CONTROL_NAUTOFD);

		if (countup==firstLen) {
			countup+=readEcpHw(buffer+countup,ecpFifoSize);
		}

		/* Disable hardware-assisted ECP mode. */
		ecpControlSetMode(ECP_CONTROL_MODE_BIDIRECTIONAL);

		return countup;
	}
	int readBoundedEcpHw(unsigned char *buffer,int len) {
		if (!ecpHwAssistIsEnabled()) {
			/* Event 43: nAck=0. */
			int r=statusWaitSetClear(43,TIMEVAL_BUSY_TIMEOUT,
				0,
				STATUS_NACK);
			if (r==ERROR) return 0;

			/* Enable hardware-assisted ECP mode. */
			ecpControlSetMode(ECP_CONTROL_MODE_ECP);
			/* Let the hardware have control of nAutoFd. */
			controlSetClear(-44,
				CONTROL_NAUTOFD,
				0);
		}
		return readEcpHw(buffer,len);
	}
	int readEpp(unsigned char *buffer,int len,int isAddress);
    public:
	int read(unsigned char *buffer,int len,int extra=0) {
		if (modeIsEcpReverse()) {
			/* Hardware assistance is always OK for bounded
			 * ECP mode, which we enable in ecpFwdToRev(). */
			if (modeIsBoundedEcpReverseHwAssist()) {
				return readBoundedEcpHw(buffer,len);
			}
			return readEcp(buffer,len);
		}
		if (modeIsNibble()) {
			return readNibble(buffer,len);
		}
		if (modeIsEpp()) {
			return readEpp(buffer,len,extra);
		}
		LOG_ERROR(cEXBP,0,cCauseBadState,
			"read: invalid mode=0x%3.3X!\n",currentMode);
		return ERROR;
	}
	char *getDeviceID(void);

	// High-level ECP/nibble mode/direction negotiation:
    protected:		// ATTRIB
	int forwardMode;
	int reverseMode;
	int desiredChannel;
    public:
	int setModes(int forwardMode,int reverseMode) {
		SETTHIS(forwardMode);
		SETTHIS(reverseMode);
		terminate();
		if (negotiate(forwardMode)==ERROR) {
			LOG_ERROR(cEXBP,0,cCausePeriphError,
				"setModes: negotiate(0x%2.2X) failed!\n",
				forwardMode);
			return ERROR;
		}
		return OK;
	}
	int setChannel(int desiredChannel) {
		return writeEcpChannel(SETTHIS(desiredChannel));
	}
	int startReverse(void) {
		if (modeIsEcp(reverseMode)) {
			return ecpFwdToRev(reverseMode);
		}
		if (terminate()==ERROR ||
		    negotiate(reverseMode)==ERROR) {
			return ERROR;
		}
		return OK;
	}
	int finishReverse(void) {
		if (modeIsEcp(reverseMode)) {
			return ecpRevToFwd(forwardMode);
		}
		if (terminate()==ERROR ||
		    negotiate(forwardMode)==ERROR ||
		    writeEcpChannel(desiredChannel)==ERROR) {
			return ERROR;
		}
		return OK;
	}
};

#endif
