// ExMlcTransport.cpp

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <stdio.h>
#ifndef EX_TRANSPORT_UNIX_PORT
#include <bp/ex/ExFileNum.h>
#define cFileNum CFILENUM(cModExio,cFileNum_transport_ExMlcTransport)
#include <lm/LoggingManager.h>
#endif

#include <ExMlcTransport.h>

/*****************************************************************************\
|* class ExMlcTransportChannel
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

// const int ExMlcTransportChannel::FLAG_ALLOCATED;
// const int ExMlcTransportChannel::FLAG_OPEN;
// const int ExMlcTransportChannel::FLAG_DEACTIVATING;
// const int ExMlcTransportChannel::FLAG_????;
const int ExMlcTransportChannel::FLAG_LOCAL_OPEN_PENDING;
const int ExMlcTransportChannel::FLAG_LOCAL_CLOSE_PENDING;
const int ExMlcTransportChannel::FLAG_LOCAL_CREDIT_PENDING;
const int ExMlcTransportChannel::FLAG_LOCAL_CREDIT_REQUEST_PENDING;
const int ExMlcTransportChannel::FLAG_REMOTE_CLOSE_PENDING;
const int ExMlcTransportChannel::FLAG_LOCAL_CLOSE_REQUESTED;
const int ExMlcTransportChannel::DEFAULT_FORWARD_CREDIT_REQUEST;
const int ExMlcTransportChannel::DEFAULT_MAX_OUTSTANDING_CREDIT;
const int ExMlcTransportChannel::DEFAULT_MUSHER_FIRST_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_MUSHER_NEXT_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_GUSHER_FIRST_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_GUSHER_NEXT_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_MISER_FIRST_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_MISER_NEXT_CREDIT_REQUEST_DELAY;
const int ExMlcTransportChannel::DEFAULT_GUSHER_PIGGYBACK_CREDIT_COUNT;
const int ExMlcTransportChannel::DEFAULT_GUSHER_CREDIT_COUNT;
const int ExMlcTransportChannel::DEFAULT_MISER_CREDIT_REQUEST_COUNT;
const int ExMlcTransportChannel::DEFAULT_REVERSE_CREDIT_HEARTBEAT_DELAY;

/*---------------------------------------------------------------------------*\
|* Macros:
\*---------------------------------------------------------------------------*/

#define REV_REQUEST_VALIDATE_CHANNEL_OPEN(functionName,mlcError,dot4ReplyCall) \
	if (!isOpen()) { \
		LOG_ERROR(cEXTP,0,cCausePeriphError, \
			functionName "(port=%d,channel=%d): not open!\n", \
			port,channel); \
		if (pMlcTransport->revisionIsMlc()) { \
			pCommandChannel->mlcSendError(mlcError); \
		} else /* if (pMlcTransport->revisionIsDot4()) */ { \
			dot4ReplyCall; \
		} \
		return; \
	}

#define REV_REPLY_VALIDATE_LOCAL_REQUEST_PENDING(setFlag,functionName) \
	if (!testFlagsSetClear(setFlag,0)) { \
		LOG_ERROR(cEXTP,0,cCausePeriphError, \
			functionName "(port=%d,channel=%d): unexpected!\n", \
			port,channel); \
		pCommandChannel->sendError(psid,ssid, \
			ExMlcTransport::MLC_ERROR_REPLY_WITHOUT_REQUEST, \
			ExMlcTransport::DOT4_ERROR_REPLY_WITHOUT_REQUEST); \
		return; \
	} \
	setClearFlags(0,setFlag);

#define REV_REPLY_VALIDATE_SUCCESSFUL(functionName) \
	if (result!=ExMlcTransport::RESULT_SUCCESS) { \
		LOG_ERROR(cEXTP,0,cCausePeriphError, \
			functionName "(port=%d,channel=%d): " \
			"result=%d!\n", \
			port,channel,result); \
		pMgr->exClose(ExMlcTransport:: \
			REASON_RECEIVED_UNSUCCESSFUL_REPLY); \
		return; \
	}

#define CREDIT_OVERFLOW(functionName) \
    do { \
	LOG_ERROR(cEXTP,0,cCausePeriphError, \
		functionName "(port=%d,channel=%d): " \
		"credit overflow!\n", \
		port,channel); \
	pCommandChannel->sendError( \
		getPrimarySocket(),getSecondarySocket(), \
		ExMlcTransport::MLC_ERROR_CREDIT_OVERFLOW, \
		ExMlcTransport::DOT4_ERROR_PIGGYBACKED_CREDIT_OVERFLOW); \
    } while (0)

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, dump, handleMsg:
\*---------------------------------------------------------------------------*/

ExMlcTransportChannel::ExMlcTransportChannel(ExMlcTransport *pMlcTransport,
    int channel,int localSocket,ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
    ExMlcCommandChannel *pCommandChannel,int final=1):
      ExTransportChannel(pMlcTransport,channel,localSocket,pMgr,
       pPhysicalPort,0) {
	openingAnySetFlags|=FLAG_LOCAL_OPEN_PENDING;
	closingAnySetFlags|=FLAG_LOCAL_CLOSE_PENDING|FLAG_REMOTE_CLOSE_PENDING;

	SETTHIS(pMlcTransport);
	SETTHIS(pCommandChannel);
	disableCreditCommands=0;

	pForwardCreditRequestMsg=pMgr->getFreeMsg();
	pForwardCreditRequestMsg->setType(
		(ExMsgType)MSG_CREDIT_REQUEST_TIMEOUT);
	pForwardCreditRequestTimer=new ExWatchdogTimer(this,
		pForwardCreditRequestMsg,
		DEBUG_STRING("pForwardCreditRequestTimer"));

	pReverseCreditHeartbeatMsg=pMgr->getFreeMsg();
	pReverseCreditHeartbeatMsg->setType(
		(ExMsgType)MSG_CREDIT_HEARTBEAT);
	pReverseCreditHeartbeatTimer=new ExWatchdogTimer(this,
		pReverseCreditHeartbeatMsg,
		DEBUG_STRING("pReverseCreditHeartbeatTimer"));

	if (final) {
		reset(RESET_STARTUP);
		registerStuff("ExMlcTransportChannel");
	}
}

ExMlcTransportChannel::~ExMlcTransportChannel(void) {
	pForwardCreditRequestTimer->removeMsg();
	delete pForwardCreditRequestTimer;
	pMgr->returnMsg(pForwardCreditRequestMsg);

	pReverseCreditHeartbeatTimer->removeMsg();
	delete pReverseCreditHeartbeatTimer;
	pMgr->returnMsg(pReverseCreditHeartbeatMsg);
}

void ExMlcTransportChannel::reset(int rtype) {
	int timeout;

	if (rtype==RESET_CLOSE) {
		uncreditCreditedBuffers();
	}

	ExTransportChannel::reset(rtype);

	pForwardCreditRequestTimer->cancel();
	pReverseCreditHeartbeatTimer->cancel();

	if (rtype!=RESET_OPEN &&
	    rtype!=RESET_CLOSE &&
	    rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		timeout=DEFAULT_REVERSE_CREDIT_HEARTBEAT_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_REVERSE_CREDIT_HEARTBEAT_DELAY,&timeout);
		pReverseCreditHeartbeatTimer->setDelay(timeout);

		musherFirstCreditRequestDelay=
			DEFAULT_MUSHER_FIRST_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_MUSHER_FIRST_CREDIT_REQUEST_DELAY,
			&musherFirstCreditRequestDelay);

		musherNextCreditRequestDelay=
			DEFAULT_MUSHER_NEXT_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_MUSHER_NEXT_CREDIT_REQUEST_DELAY,
			&musherNextCreditRequestDelay);

		gusherFirstCreditRequestDelay=
			DEFAULT_GUSHER_FIRST_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_GUSHER_FIRST_CREDIT_REQUEST_DELAY,
			&gusherFirstCreditRequestDelay);

		gusherNextCreditRequestDelay=
			DEFAULT_GUSHER_NEXT_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_GUSHER_NEXT_CREDIT_REQUEST_DELAY,
			&gusherNextCreditRequestDelay);

		miserFirstCreditRequestDelay=
			DEFAULT_MISER_FIRST_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_MISER_FIRST_CREDIT_REQUEST_DELAY,
			&miserFirstCreditRequestDelay);

		miserNextCreditRequestDelay=
			DEFAULT_MISER_NEXT_CREDIT_REQUEST_DELAY;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_MISER_NEXT_CREDIT_REQUEST_DELAY,
			&miserNextCreditRequestDelay);

		gusherPiggybackCreditCount=
			DEFAULT_GUSHER_PIGGYBACK_CREDIT_COUNT;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_GUSHER_PIGGYBACK_CREDIT_COUNT,
			&gusherPiggybackCreditCount);

		gusherCreditCount=
			DEFAULT_GUSHER_CREDIT_COUNT;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_GUSHER_CREDIT_COUNT,
			&gusherCreditCount);

		miserCreditRequestCount=
			DEFAULT_MISER_CREDIT_REQUEST_COUNT;
		tknobGetWorkingValue(port,
			EX_KNOB_MLC_MISER_CREDIT_REQUEST_COUNT,
			&miserCreditRequestCount);

		workaroundReverseCreditLoss=0;
	}

	if (rtype!=RESET_CLOSE &&
	    rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		maxForwardPacketSize=0;
		maxReversePacketSize=0;

		forwardCreditRequest=DEFAULT_FORWARD_CREDIT_REQUEST;
		tknobGetWorkingValue(port,EX_KNOB_MLC_FORWARD_CREDIT_REQUEST,
			&forwardCreditRequest);
		forwardMaxOutstandingCredit=DEFAULT_MAX_OUTSTANDING_CREDIT;
		tknobGetWorkingValue(port,EX_KNOB_MLC_MAX_OUTSTANDING_CREDIT,
			&forwardMaxOutstandingCredit);
		forwardCredit.reset();
		lastCreditRequestGotUsNowhere=0;
		reverseMaxOutstandingCredit=DEFAULT_MAX_OUTSTANDING_CREDIT;
		reverseBuffersPerPacket=0;	/* Set later. */
		if (rtype!=RESET_OPEN) uncreditedBuffers=bufferCount;
		reverseCreditToGrant=0;
		reverseCredit.reset();

		countSendPiggybackCredit.reset();
		countHandlePiggybackCredit.reset();
		countSendCredit.reset();
		countHandleCredit.reset();
		countHandleCreditAfterCreditRequest.reset();
		countSendCreditRequest.reset();
		countHandleCreditRequest.reset();
		countSendEmptyCreditRequestReply.reset();
		countHandleEmptyCreditRequestReply.reset();
	}
}

void ExMlcTransportChannel::registerStuff(char *title) {
	ExTransportChannel::registerStuff(title);

#if 0
	nettestRegister(NETTEST_TYPE_INT32,
		&disableCreditCommands,
		"%s[%d,%d].disableCreditCommands",
		title,port,channel);
#endif
	nettestRegister(NETTEST_TYPE_INT32,
		pForwardCreditRequestTimer->pointToDelay(),
		"%s[%d,%d].forwardCreditRequestTimer.delay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		pReverseCreditHeartbeatTimer->pointToDelay(),
		"%s[%d,%d].reverseCreditHeartbeatTimer.delay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&musherFirstCreditRequestDelay,
		"%s[%d,%d].musherFirstCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&musherNextCreditRequestDelay,
		"%s[%d,%d].musherNextCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&gusherFirstCreditRequestDelay,
		"%s[%d,%d].gusherFirstCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&gusherNextCreditRequestDelay,
		"%s[%d,%d].gusherNextCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&miserFirstCreditRequestDelay,
		"%s[%d,%d].miserFirstCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&miserNextCreditRequestDelay,
		"%s[%d,%d].miserNextCreditRequestDelay",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&gusherPiggybackCreditCount,
		"%s[%d,%d].gusherPiggybackCreditCount",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&gusherCreditCount,
		"%s[%d,%d].gusherCreditCount",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&miserCreditRequestCount,
		"%s[%d,%d].miserCreditRequestCount",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&workaroundReverseCreditLoss,
		"%s[%d,%d].workaroundReverseCreditLoss",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&maxForwardPacketSize,
		"%s[%d,%d].maxForwardPacketSize",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&maxReversePacketSize,
		"%s[%d,%d].maxReversePacketSize",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&forwardCreditRequest,
		"%s[%d,%d].forwardCreditRequest",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&forwardMaxOutstandingCredit,
		"%s[%d,%d].forwardMaxOutstandingCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		forwardCredit.pointToCurrent(),
		"%s[%d,%d].forwardCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&lastCreditRequestGotUsNowhere,
		"%s[%d,%d].lastCreditRequestGotUsNowhere",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseMaxOutstandingCredit,
		"%s[%d,%d].reverseMaxOutstandingCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseBuffersPerPacket,
		"%s[%d,%d].reverseBuffersPerPacket",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&uncreditedBuffers,
		"%s[%d,%d].uncreditedBuffers",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseCreditToGrant,
		"%s[%d,%d].reverseCreditToGrant",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		reverseCredit.pointToCurrent(),
		"%s[%d,%d].reverseCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countSendPiggybackCredit.pointToCount(),
		"%s[%d,%d].countSendPiggybackCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countHandlePiggybackCredit.pointToCount(),
		"%s[%d,%d].countHandlePiggybackCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countSendCredit.pointToCount(),
		"%s[%d,%d].countSendCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countHandleCredit.pointToCount(),
		"%s[%d,%d].countHandleCredit",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countHandleCreditAfterCreditRequest.pointToCount(),
		"%s[%d,%d].countHandleCreditAfterCreditRequest",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countSendCreditRequest.pointToCount(),
		"%s[%d,%d].countSendCreditRequest",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countHandleCreditRequest.pointToCount(),
		"%s[%d,%d].countHandleCreditRequest",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countSendEmptyCreditRequestReply.pointToCount(),
		"%s[%d,%d].countSendEmptyCreditRequestReply",
		title,port,channel);
	nettestRegister(NETTEST_TYPE_INT32,
		countHandleEmptyCreditRequestReply.pointToCount(),
		"%s[%d,%d].countHandleEmptyCreditRequestReply",
		title,port,channel);
}

#ifdef JD_DEBUGLITE
void ExMlcTransportChannel::dump(void) {
	ExTransportChannel::dump();
	printf("----------------\n");
	printf("pMlcTransport=0x%8.8X\n",
		(int)pMlcTransport);
	printf("pCommandChannel=0x%8.8X\n",
		(int)pCommandChannel);
	printf("disableCreditCommands=%d\n",
		disableCreditCommands);
	printf("pForwardCreditRequestTimer=0x%8.8X\n",
		(int)pForwardCreditRequestTimer);
	printf("pForwardCreditRequestMsg=0x%8.8X\n",
		(int)pForwardCreditRequestMsg);
	printf("pReverseCreditHeartbeatTimer=0x%8.8X\n",
		(int)pReverseCreditHeartbeatTimer);
	printf("pReverseCreditHeartbeatMsg=0x%8.8X\n",
		(int)pReverseCreditHeartbeatMsg);
	printf("musherFirstCreditRequestDelay=%d\n",
		musherFirstCreditRequestDelay);
	printf("musherNextCreditRequestDelay=%d\n",
		musherNextCreditRequestDelay);
	printf("gusherFirstCreditRequestDelay=%d\n",
		gusherFirstCreditRequestDelay);
	printf("gusherNextCreditRequestDelay=%d\n",
		gusherNextCreditRequestDelay);
	printf("miserFirstCreditRequestDelay=%d\n",
		miserFirstCreditRequestDelay);
	printf("miserNextCreditRequestDelay=%d\n",
		miserNextCreditRequestDelay);
	printf("gusherPiggybackCreditCount=%d\n",
		gusherPiggybackCreditCount);
	printf("gusherCreditCount=%d\n",
		gusherCreditCount);
	printf("miserCreditRequestCount=%d\n",
		miserCreditRequestCount);
	printf("workaroundReverseCreditLoss=%d\n",
		workaroundReverseCreditLoss);
	printf("maxForwardPacketSize=%d\n",
		maxForwardPacketSize);
	printf("maxReversePacketSize=%d\n",
		maxReversePacketSize);
	printf("forwardCreditRequest=%d\n",
		forwardCreditRequest);
	printf("forwardMaxOutstandingCredit=%d\n",
		forwardMaxOutstandingCredit);
	printf("forwardCredit: ");
		forwardCredit.dump();
	printf("lastCreditRequestGotUsNowhere=%d\n",
		lastCreditRequestGotUsNowhere);
	printf("reverseMaxOutstandingCredit=%d\n",
		reverseMaxOutstandingCredit);
	printf("reverseBuffersPerPacket=%d\n",
		reverseBuffersPerPacket);
	printf("uncreditedBuffers=%d\n",
		uncreditedBuffers);
	printf("reverseCreditToGrant=%d\n",
		reverseCreditToGrant);
	printf("reverseCredit: ");
		reverseCredit.dump();
	printf("countSendPiggybackCredit=%d\n",
		countSendPiggybackCredit.get());
	printf("countHandlePiggybackCredit=%d\n",
		countHandlePiggybackCredit.get());
	printf("countSendCredit=%d\n",
		countSendCredit.get());
	printf("countHandleCredit=%d\n",
		countHandleCredit.get());
	printf("countHandleCreditAfterCreditRequest=%d\n",
		countHandleCreditAfterCreditRequest.get());
	printf("countSendCreditRequest=%d\n",
		countSendCreditRequest.get());
	printf("countHandleCreditRequest=%d\n",
		countHandleCreditRequest.get());
	printf("countSendEmptyCreditRequestReply=%d\n",
		countSendEmptyCreditRequestReply.get());
	printf("countHandleEmptyCreditRequestReply=%d\n",
		countHandleEmptyCreditRequestReply.get());
	printf("isGusher=%d\n",
		isGusher());
}
#endif

void ExMlcTransportChannel::handleMsg(ExMsg *pMsg) {
	switch (pMsg->getType()) {
	   case MSG_CREDIT_REQUEST_TIMEOUT:
		if (!pForwardCreditRequestTimer->isCancelled()) {
			scheduleCreditRequest(SOURCE_HANDLE_MSG);
		}
		pForwardCreditRequestTimer->setMsg(pForwardCreditRequestMsg);
		break;

	   case MSG_CREDIT_HEARTBEAT:
		if (!pReverseCreditHeartbeatTimer->isCancelled()) {
			scheduleCredit(SOURCE_HANDLE_MSG);
		}
		pReverseCreditHeartbeatTimer->
			setMsg(pReverseCreditHeartbeatMsg);
		break;

	   default:
		ExTransportChannel::handleMsg(pMsg);
	}
}

/*---------------------------------------------------------------------------*\
|* Allocate channel:
\*---------------------------------------------------------------------------*/

void ExMlcTransportChannel::allocate(ExService *pService,SCD scd,
    int forwardDataPriority,int minBuffersRequired,int benefitOfMoreBuffers,
    int reverseDataBufferSize) {
	ExTransportChannel::allocate(pService,scd,forwardDataPriority,
		minBuffersRequired,benefitOfMoreBuffers,reverseDataBufferSize);
}

void ExMlcTransportChannel::grantBuffers(int count) {
	ExTransportChannel::grantBuffers(count);
	uncreditedBuffers+=count;
}

/*---------------------------------------------------------------------------*\
|* Set remote socket, open:
\*---------------------------------------------------------------------------*/

void ExMlcTransportChannel::setRemoteSocket_ts(int remoteSocket,
    int maxForwardDatalen,int maxReverseDatalen) {
	int state;

	if (!pMlcTransport->revisionIsMlc()) {
sendOK:
		sendRemoteSocketSetResponse(OK);
		return;
	}

	if (pMlcTransport->remsockLookup(remoteSocket,&state,0,0)!=ERROR) {
		if (state==ExMlcTransport::REMSOCK_STATE_CONFIGURED)
			goto sendOK;
		LOG_ASSERT(state==ExMlcTransport::REMSOCK_STATE_CONFIGURING,
			cEXTP,0,cCauseBadState,"");
		return;
	}

	if (pMlcTransport->remsockAdd(remoteSocket)==ERROR) {
		sendRemoteSocketSetResponse(S_hlio_BP_SOCKET_TABLE_FULL);
		return;
	}

	int maxForwardPacketSize=maxForwardDatalen+ExMlcTransport::LEN_HEADER;
	int maxReversePacketSize=maxReverseDatalen+ExMlcTransport::LEN_HEADER;
	pCommandChannel->sendConfigSocket(remoteSocket,
		pMlcTransport->pickPrimarySocket(
			maxForwardPacketSize,maxReversePacketSize),
		pMlcTransport->pickSecondarySocket(
			maxForwardPacketSize,maxReversePacketSize));
}

void ExMlcTransportChannel::open_ts(int maxForwardDatalen,
    int maxReverseDatalen) {
	int status,state;

	setClearFlags(FLAG_LOCAL_OPEN_PENDING,0);

	if (pMlcTransport->revisionIsMlc()) {
		status=pMlcTransport->remsockLookup(remoteSocket,&state,
			&maxForwardPacketSize,&maxReversePacketSize);
		LOG_ASSERT(status==OK &&
			state==ExMlcTransport::REMSOCK_STATE_CONFIGURED,
			cEXTP,0,cCauseBadState,"");
		setDatalensFromPacketSizes();

		pCommandChannel->sendOpenChannel(
			getPrimarySocket(),getSecondarySocket(),0,0,
			forwardMaxOutstandingCredit,grantReverseCredit());

	} else /* if (pMlcTransport->revisionIsDot4()) */ {
		int maxForwardPacketSize=
			maxForwardDatalen+ExMlcTransport::LEN_HEADER;
		int maxReversePacketSize=
			maxReverseDatalen+ExMlcTransport::LEN_HEADER;

		pCommandChannel->sendOpenChannel(
			getPrimarySocket(),getSecondarySocket(),
			pMlcTransport->pickPrimarySocket(
				maxForwardPacketSize,maxReversePacketSize),
			pMlcTransport->pickSecondarySocket(
				maxForwardPacketSize,maxReversePacketSize),
			forwardMaxOutstandingCredit,0);
	}
}

void ExMlcTransportChannel::handleOpenChannelReply(int result,
    int psid,int ssid,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int maxOutstandingCredit,int credit) {
	int status=OK;
	REV_REPLY_VALIDATE_LOCAL_REQUEST_PENDING(
		FLAG_LOCAL_OPEN_PENDING,"handleOpenChannelReply");

	if (result!=ExMlcTransport::RESULT_SUCCESS) {
	    if (pMlcTransport->revisionIsMlc()) {
		switch (result) {
		   case ExMlcTransport::MLC_RESULT_TOO_MANY_CONNECTIONS:
			status=S_hlio_BP_CONNECTION_REFUSED;
			break;
		   case ExMlcTransport::MLC_RESULT_INSUFFICIENT_RESOURCES:
			status=S_hlio_BP_INSUFFICIENT_RESOURCES;
			break;
		   default:
			status=S_hlio_BP_UNKNOWN_ERROR;
		}

	    } else /* if (pMlcTransport->revisionIsDot4()) */ {
		switch (result) {
		   case ExMlcTransport::DOT4_RESULT_INSUFFICIENT_RESOURCES:
			status=S_hlio_BP_INSUFFICIENT_RESOURCES;
			break;
		   case ExMlcTransport::DOT4_RESULT_CONNECTION_REFUSED:
			status=S_hlio_BP_CONNECTION_REFUSED;
			break;
		   case ExMlcTransport::DOT4_RESULT_NO_SERVICE_ON_SOCKET:
			status=S_hlio_BP_UNSUPPORTED_SOCKET;
			break;
		   default:
			LOG_ERROR(cEXTP,0,cCausePeriphError,
				"handleOpenChannelReply(port=%d,channel=%d): "
				"result=%d!\n",port,channel,result);
			int shouldWeDeactivate=0;
			tknobGetWorkingValue(port,
				EX_KNOB_MLC_DEACTIVATE_ON_UNKNOWN_OPEN_ERROR,
				&shouldWeDeactivate);
			if (shouldWeDeactivate) {
				pMgr->exClose(ExMlcTransport::
					REASON_OPEN_CHANNEL_FAILED);
			}
			status=S_hlio_BP_UNKNOWN_ERROR;
		}
	    }
	    setClosed();
	    goto done;
	}

	setOpen();

	/* It's OK to do this after setOpen(), because part 1 doesn't
	 * attempt to send any data. */
	forwardCredit.preset(credit);
	if (handleForwardCreditPart1(credit,0)==ERROR) {
		CREDIT_OVERFLOW("handleOpenChannelReply");
		return;
	}

	/* Do this after setOpen() to make sure we send credit. */
	reverseMaxOutstandingCredit=maxOutstandingCredit;
	if (pMlcTransport->revisionIsDot4()) {
		maxForwardPacketSize=pMlcTransport->pickLocalSocket(
			maxPriToSecPacketSize,maxSecToPriPacketSize);
		maxReversePacketSize=pMlcTransport->pickRemoteSocket(
			maxPriToSecPacketSize,maxSecToPriPacketSize);
		setDatalensFromPacketSizes();
	}
done:
	sendChannelOpenResponse(status);
}

void ExMlcTransportChannel::setDatalensFromPacketSizes(void) {
	const int LEN_HEADER=ExMlcTransport::LEN_HEADER;

	/* Warn of invalid packet sizes (1<=x<=5, or both zero). */
	if (!packetSizesAreValid(maxForwardPacketSize,maxReversePacketSize)) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"setDatalensFromPacketSizes(port=%d,channel=%d): "
			"invalid maxForwardPacketSize=%d, "
			"maxReversePacketSize=%d!\n",
			port,channel,maxForwardPacketSize,
			maxReversePacketSize);
	}

	maxForwardDatalen=0;
	if (maxForwardPacketSize>LEN_HEADER)
		maxForwardDatalen=maxForwardPacketSize-LEN_HEADER;
	maxReverseDatalen=0;
	if (maxReversePacketSize>LEN_HEADER)
		maxReverseDatalen=maxReversePacketSize-LEN_HEADER;

	reverseBuffersPerPacket=maxReversePacketSize/reverseDataBufferSize;
	if (!reverseBuffersPerPacket ||
	    maxReversePacketSize%reverseDataBufferSize)
		reverseBuffersPerPacket++;
	reverseBuffersPerPacket+=pTransport->getOverheadBufferCount();

	creditReturnedBuffers();
	reverseCredit.preset(reverseCreditToGrant+reverseCredit.get());
}

/*---------------------------------------------------------------------------*\
|* Close:
\*---------------------------------------------------------------------------*/

void ExMlcTransportChannel::close_ts(void) {
	/* It's not necessary to check the queue as well, because
	 * the service doesn't close us until its queue is empty. */
	if (currentForwardBuffer) {
		LOG_INFO(cEXTP,0,"close_ts(port=%d,channel=%d): "
			"deferring until queue empty.\n",port,channel);
		setClearFlags(FLAG_LOCAL_CLOSE_REQUESTED,0);
		return;
	}
	setClearFlags(0,FLAG_LOCAL_CLOSE_REQUESTED);
	setClearFlags(FLAG_LOCAL_CLOSE_PENDING,0);

	/* TODO: Handle this properly if also closing remotely! */

	pCommandChannel->sendCloseChannel(getPrimarySocket(),
		getSecondarySocket());
}

void ExMlcTransportChannel::handleCloseChannelReply(int result,
    int psid,int ssid) {
	REV_REPLY_VALIDATE_LOCAL_REQUEST_PENDING(
		FLAG_LOCAL_CLOSE_PENDING,"handleCloseChannelReply");

	REV_REPLY_VALIDATE_SUCCESSFUL("handleCloseChannelReply");

	setClosed();
	sendChannelCloseResponse(OK);
}

void ExMlcTransportChannel::handleCloseChannel(int psid,int ssid) {
	REV_REQUEST_VALIDATE_CHANNEL_OPEN("handleCloseChannel",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CLOSE,
		pCommandChannel->sendCloseChannelReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid));

	/* Make sure this works when local Credit or CreditRequest
	 * is still pending. */
	int test=0;
	tknobGetWorkingValue(port,EX_KNOB_MLC_TEST_FLUSH_CLOSE_CHANNEL_REPLY,
		&test);
	if (test&1) {
		scheduleCreditRequest(SOURCE_HANDLE_MSG);
	}
	if (test&2) {
		scheduleCredit(SOURCE_HANDLE_MSG);
	}

	/* TODO: Handle this properly if also closing locally! */

	setClearFlags(FLAG_REMOTE_CLOSE_PENDING,0);

	/* TODO: Make sure service handles this properly:
	 * - refrain from sending any more forward data
	 * - notify application properly
	 * - bitbucket additional forward data if app does send more
	 */
	sendChannelCloseResponse(S_hlio_BP_CLOSED_BY_PERIPHERAL);

	/* TODO: Flush buffered forward data.  Or let service do it? */

	flushCloseChannelReply();
}

void ExMlcTransportChannel::flushCloseChannelReply(void) {
	if (!testFlagsSetClear(FLAG_REMOTE_CLOSE_PENDING,0)) {
		return;
	}

	if (getFlags()&(FLAG_LOCAL_CREDIT_PENDING|
	     FLAG_LOCAL_CREDIT_REQUEST_PENDING)) {
		LOG_INFO(cEXTP,0,
			"flushCloseChannelReply(port=%d,channel=%d): "
			"waiting for Credit[Request]Reply.\n",port,channel);
		return;
	}

	pCommandChannel->sendCloseChannelReply(ExMlcTransport::RESULT_SUCCESS,
		getPrimarySocket(),getSecondarySocket());

	setClearFlags(0,FLAG_REMOTE_CLOSE_PENDING);
	setClosed();
}

/*---------------------------------------------------------------------------*\
|* Forward data flow:
\*---------------------------------------------------------------------------*/

void ExMlcTransportChannel::noMoreForwardData(void) {
	flushCloseChannel();
}

ExMlcTransportChannel::IsGusher ExMlcTransportChannel::isGusher(void) {
	if (countHandlePiggybackCredit.get()>=gusherPiggybackCreditCount ||
	    countHandleCredit.get()>=gusherCreditCount) {
		return IS_GUSHER;
	}

	if (countSendCreditRequest.get()<miserCreditRequestCount) {
		return IS_MUSHER;
	}

	return IS_MISER;
}

STATUS ExMlcTransportChannel::addOverhead(ExBdr **ppBdr) {
	ExMlcTransport::Header *header;
	ExBdr *pHeaderBdr=pMlcTransport->getHeaderBuffer(&header);
	if (!pHeaderBdr) return ERROR;

	/* These are only for the first data BDR in a chain. */
	ExBdr *pDataBdr=*ppBdr;
	unsigned char *data=bdrGetData(pDataBdr);
	int datalen=pDataBdr->getDataLength();

	pHeaderBdr->setDataLength(ExMlcTransport::LEN_HEADER);
	pHeaderBdr->setNext(pDataBdr);
	*ppBdr=pHeaderBdr;

	int packetSize,bdrCount;
	measureBDR(pHeaderBdr,&packetSize,&bdrCount);

	header->psid=getPrimarySocket();
	header->ssid=getSecondarySocket();
	BEND_SET_WORD(header->length,packetSize);
	header->credit=grantReverseCredit(0xFF,data,datalen);
	if (header->credit) {
		countSendPiggybackCredit.increment();
	}
	header->credit=prepareToSendPiggybackCredit(header->credit);
	/* Copy control field from BDR into header. */
	header->control=pDataBdr->getTransportHeaderFlags();

	LOG_INFO(cEXTP,0,
		"addOverhead(port=%d,channel=%d): psid=0x%2.2X, ssid=0x%2.2X, "
		"length=%d, credit=%d, control=0x%2.2X.\n",
		port,channel,header->psid,header->ssid,
		BEND_GET_WORD(header->length),header->credit,header->control);

	return OK;
}

void ExMlcTransportChannel::removeOverhead(ExBdr **ppBdr) {
	ExBdr *pHeaderBdr=*ppBdr;
	ExBdr *pDataBdr=pHeaderBdr->getNext();
	pHeaderBdr->setNext(0);
	*ppBdr=pDataBdr;
	pMlcTransport->returnHeaderBuffer(pHeaderBdr);
}

int ExMlcTransportChannel::grabForwardCredit(ExBdr *pBdr) {
	if (forwardCredit.grab()==ERROR) {
		scheduleCreditRequest(SOURCE_GRAB_FORWARD_CREDIT);
		return ERROR;
	}

	return 1;
}

void ExMlcTransportChannel::scheduleCreditRequest(SourceEnum source) {
	if (testFlagsSetClear(FLAG_LOCAL_CREDIT_REQUEST_PENDING,0)) {
		LOG_INFO(cEXTP,0,
			"scheduleCreditRequest(port=%d,channel=%d,source=%d): "
			"local credit request pending.\n",
			port,channel,source);
		return;
	}

	if (source==SOURCE_CANCEL) {
		pForwardCreditRequestTimer->cancel();
		return;
	}
	if (!isOpen() || isClosing()) {
		LOG_INFO(cEXTP,0,
			"scheduleCreditRequest(port=%d,channel=%d,source=%d): "
			"closing or not open.\n",
			port,channel,source);
		scheduleCreditRequest(SOURCE_CANCEL);	/* This is recursive. */
		return;
	}

	if (pForwardCreditRequestTimer->isStarted()) {
		LOG_INFO(cEXTP,0,
			"scheduleCreditRequest(port=%d,channel=%d,source=%d): "
			"pForwardCreditRequestTimer is started.\n",
			port,channel,source);
		return;
	}

	if (disableCreditCommands) {
		/* LOG_INFO(cEXTP,0,
			"scheduleCreditRequest(port=%d,channel=%d,source=%d): "
			"credit commands disabled.\n",
			port,channel,source); */
		return;
	}

	IsGusher g=isGusher();
	int musherDelay=musherFirstCreditRequestDelay;
	int gusherDelay=gusherFirstCreditRequestDelay;
	int miserDelay=miserFirstCreditRequestDelay;
	int delay;

	switch (source) {
	   case SOURCE_HANDLE_FORWARD_CREDIT:
		musherDelay=musherNextCreditRequestDelay;
		gusherDelay=gusherNextCreditRequestDelay;
		miserDelay=miserNextCreditRequestDelay;

	   case SOURCE_GRAB_FORWARD_CREDIT:
		g=isGusher();
		if (g==IS_GUSHER) {
			delay=gusherDelay;
		} else if (g==IS_MUSHER) {
			delay=musherDelay;
		} else /* if (g==IS_MISER) */ {
			delay=miserDelay;
		}
		LOG_INFO(cEXTP,0,"scheduleCreditRequest(port=%d,channel=%d,"
			"source=%d): isGusher=%d, delay=%d.\n",
			port,channel,source,g,delay);
		pForwardCreditRequestTimer->start(delay);
		break;

	   case SOURCE_HANDLE_MSG:
		setClearFlags(FLAG_LOCAL_CREDIT_REQUEST_PENDING,0);

		countSendCreditRequest.increment();
		pCommandChannel->sendCreditRequest(
			getPrimarySocket(),getSecondarySocket(),
			forwardCreditRequest,
			forwardMaxOutstandingCredit);
		break;

	   default:
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
	}
}

void ExMlcTransportChannel::handleCreditRequestReply(int result,
    int psid,int ssid,int credit,int cmdinfo) {
	REV_REPLY_VALIDATE_LOCAL_REQUEST_PENDING(
		FLAG_LOCAL_CREDIT_REQUEST_PENDING,"handleCreditRequestReply");

	REV_REPLY_VALIDATE_SUCCESSFUL("handleCreditRequestReply");

	lastCreditRequestGotUsNowhere=(!credit);
	if (!credit) {
		countHandleEmptyCreditRequestReply.increment();
	}
	if (handleForwardCredit(credit,cmdinfo)==ERROR) {
		CREDIT_OVERFLOW("handleCreditRequestReply");
		return;
	}

	flushCloseChannelReply();
}

void ExMlcTransportChannel::handleCredit(int psid,int ssid,int credit,
    int cmdinfo) {
	if (lastCreditRequestGotUsNowhere) {
		countHandleCreditAfterCreditRequest.increment();
		if (credit) lastCreditRequestGotUsNowhere=0;

	} else {
		countHandleCredit.increment();
	}

	REV_REQUEST_VALIDATE_CHANNEL_OPEN("handleCredit",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CREDIT,
		pCommandChannel->sendCreditReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid));

	int part1Result=handleForwardCreditPart1(credit,cmdinfo);
	if (part1Result==ERROR) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleCredit(port=%d,channel=%d): "
			"credit overflow!\n",
			port,channel);
		if (pMlcTransport->revisionIsMlc()) {
			pCommandChannel->mlcSendError(
				ExMlcTransport::MLC_ERROR_CREDIT_OVERFLOW);
		} else {
			pCommandChannel->sendCreditReply(
				ExMlcTransport::DOT4_RESULT_CREDIT_OVERFLOW,
				psid,ssid);
		}
		return;
	}

	pCommandChannel->sendCreditReply(ExMlcTransport::RESULT_SUCCESS,
		psid,ssid);

	handleForwardCreditPart2(part1Result);
}

int ExMlcTransportChannel::handlePiggybackCreditPart1(int credit,
    unsigned char *data,int datalen,int cmdinfo) {
	if (!credit) {
		/* Prevent handleForwardCreditPart2() from doing anything,
		 * such as scheduleCreditRequest(), when we get no
		 * piggyback credit. */
		return 1;
	}

	countHandlePiggybackCredit.increment();

	/* The caller is responsible for logging an error message
	 * and sending an Error packet. */
	return handleForwardCreditPart1(credit,cmdinfo);
}

int ExMlcTransportChannel::handleForwardCreditPart1(int credit,int cmdinfo) {
	int oldCredit=forwardCredit.get();
	if (forwardCredit.increment(credit)==ERROR) return ERROR;
	return oldCredit;
}

STATUS ExMlcTransportChannel::handleForwardCreditPart2(int oldCredit) {
	if (oldCredit==ERROR) return ERROR;

	int newCredit=forwardCredit.get();

	if (!oldCredit) {
		/* If we finally got some credit, then cancel the
		 * CreditRequest watchdog timer and try to send more data. */
		if (newCredit) {
			scheduleCreditRequest(SOURCE_CANCEL);
			forwardDataAvailable();

		/* But if we still don't have credit, then
		 * maybe we should ask (again) for some eventually. */
		} else {
			scheduleCreditRequest(SOURCE_HANDLE_FORWARD_CREDIT);
		}
	}

	return OK;
}

/*---------------------------------------------------------------------------*\
|* Reverse data flow:
\*---------------------------------------------------------------------------*/

void ExMlcTransportChannel::reverseDataReceived_ts(ExBdr *pHeaderBdr,
    int status) {
	int mlcError,dot4Error;
	ExMlcTransport::Header *header=ExMlcTransport::bdrGetHeader(pHeaderBdr);
	int part1Result;

	/* Count bytes and BDRs in chain. */
	int packetSize,bdrCount;
	measureBDR(pHeaderBdr,&packetSize,&bdrCount);

	ExBdr *pDataBdr=pHeaderBdr;
	unsigned char *data=0;
	int datalen=packetSize-ExMlcTransport::LEN_HEADER;

	/* Log what we got. */
	LOG_INFO(cEXTP,0,"reverseDataReceived_ts(port=%d,channel=%d,"
		"pHeaderBdr=0x%8.8X,status=%d): packetSize=%d, bdrCount=%d.\n",
		port,channel,pHeaderBdr,status,packetSize,bdrCount);

	/* Make sure header length field reflects amount of data read. */
	if (BEND_GET_WORD(header->length)!=packetSize) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"header length=%d != packetSize=%d!\n",
			port,channel,BEND_GET_WORD(header->length),packetSize);
		mlcError=ExMlcTransport::MLC_ERROR_DATA_LINK_ERROR;
		dot4Error=ExMlcTransport::DOT4_ERROR_MALFORMED_PACKET;
		goto abort;
	}

	/* Make sure this channel is open. */
	if (!isOpen()) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"not open!\n",port,channel);
		mlcError=ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_DATA;
		dot4Error=ExMlcTransport::DOT4_ERROR_UNOPENED_CHANNEL_DATA;
		goto abort;
	}

	/* Make sure packet isn't bigger than negotiated or implicit
	 * packet size for this channel. */
	if (packetSize>maxReversePacketSize) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"packetSize=%d > maxReversePacketSize=%d!\n",
			port,channel,packetSize,maxReversePacketSize);
		mlcError=ExMlcTransport::MLC_ERROR_PACKET_TOO_BIG;
		dot4Error=ExMlcTransport::DOT4_ERROR_PACKET_TOO_BIG;
		goto abort;
	}

	/* Figure out where the data starts.
	 * If header and data are separate, or if this is an empty packet... */
	if ((int)pHeaderBdr->getDataLength()<=ExMlcTransport::LEN_HEADER) {
		/* The chain gets split later. */
		pDataBdr=pHeaderBdr->getNext();

	/* Else if header and data are together... */
	} else {
		pDataBdr->unprependData(ExMlcTransport::LEN_HEADER);
	}
	if (pDataBdr) data=bdrGetData(pDataBdr);

	/* Make sure we have (or don't need) credit for this packet. */
	if (grabReverseCredit(data,datalen)==ERROR) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"uncredited packet!\n",
			port,channel);
		mlcError=ExMlcTransport::MLC_ERROR_RECEIVED_UNCREDITED_PACKET;
		dot4Error=ExMlcTransport::DOT4_ERROR_RECEIVED_UNCREDITED_PACKET;
		goto abort;
	}

	/* Handle piggyback credit.  Raise error if credit overflow. */
	part1Result=handlePiggybackCreditPart1(header->credit,data,datalen,0);
	if (part1Result==ERROR) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"piggyback credit overflow!\n",
			port,channel);
		mlcError=ExMlcTransport::MLC_ERROR_CREDIT_OVERFLOW;
		dot4Error=ExMlcTransport::
			DOT4_ERROR_PIGGYBACKED_CREDIT_OVERFLOW;
	}

	/* Make sure this packet didn't use too many buffers,
	 * which would screw up our buffer<->credit accounting. */
	if (bdrCount>reverseBuffersPerPacket) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"reverseDataReceived_ts(port=%d,channel=%d): "
			"bdrCount=%d > reverseBuffersPerPacket=%d!\n",
			port,channel,bdrCount,reverseBuffersPerPacket);
		mlcError=ExMlcTransport::MLC_ERROR_DATA_LINK_ERROR;
		dot4Error=ExMlcTransport::DOT4_ERROR_MALFORMED_PACKET;
		goto abort;
	}

	/* Copy control field from header into BDR. */
	pDataBdr->setTransportHeaderFlags(header->control);

	/* Send empty packets on up the stack. */
	if (!pDataBdr) {
		pDataBdr=pHeaderBdr;

	/* If header is in a separate BDR and this is not an empty packet,
	 * then remove it from the chain. */
	} else if (pDataBdr!=pHeaderBdr) {
		pHeaderBdr->setNext(0);
		pHeaderBdr->returnBuffer();
		bdrCount--;
	}

	/* Credit back forfeited and header buffers. */
	creditReturnedBuffers(reverseBuffersPerPacket-bdrCount);

	/* If there is still data, then mark which transport channel to
	 * return the chain to, and send the data to the service. */
	if (pDataBdr) {
		markBDR(pDataBdr,&countReverseBuffers);
		serviceReverseDataReceived(pDataBdr,status,data,datalen);
	}

	/* If we got piggyback credit on a channel blocked on credit,
	 * then try to unblock it now. */
	handleForwardCreditPart2(part1Result);

	return;

abort:
	/* Return BDR chain. */
	while (pHeaderBdr) {
		pDataBdr=pHeaderBdr->getNext();
		pHeaderBdr->setNext(0);
		pHeaderBdr->returnBuffer();
		pHeaderBdr=pDataBdr;
	}

	/* Enqueue error packet. */
	pCommandChannel->sendError(header->psid,header->ssid,
		mlcError,dot4Error);
}

int ExMlcTransportChannel::grabReverseCredit(unsigned char *data,int datalen) {
	if (reverseCredit.grab()==ERROR) {
		return ERROR;
	}

	scheduleCredit(SOURCE_GRAB_REVERSE_CREDIT);

	return 1;
}

void ExMlcTransportChannel::returnBufferNotification(ExBdr *pBdr) {
	/* Assert that this isn't a BDR chain. */
	LOG_ASSERT(!pBdr->getNext(),cEXTP,0,cCauseBadParm,"");

#ifdef JD_DEBUGLITE
	countReverseBufferReturns.increment();
#endif
	creditReturnedBuffers(1);
}

/* TODO: Add asserts in various places to validate credit management. */
void ExMlcTransportChannel::creditReturnedBuffers(int returnedBuffers=0) {
	uncreditedBuffers+=returnedBuffers;
	if ((!isOpen() || isClosing()) && !isOpening()) {
		LOG_INFO(cEXTP,0,
			"creditReturnedBuffers(port=%d,channel=%d,count=%d): "
			"closing or not open, and not opening.\n",
			port,channel,returnedBuffers);
		return;
	}
	reverseCreditToGrant+=uncreditedBuffers/reverseBuffersPerPacket;
	uncreditedBuffers%=reverseBuffersPerPacket;

	scheduleCredit(SOURCE_CREDIT_RETURNED_BUFFERS);
}

void ExMlcTransportChannel::uncreditCreditedBuffers(void) {
	int credit=reverseCreditToGrant+reverseCredit.get();
	reverseCreditToGrant=0;
	reverseCredit.set(0);

	int buffers=credit*reverseBuffersPerPacket;
	uncreditedBuffers+=buffers;

	LOG_INFO(cEXTP,0,"uncreditCreditedBuffers(port=%d,channel=%d): "
		"converted credit=%d to buffers=%d, uncreditedBuffers=%d.\n",
		port,channel,credit,buffers,uncreditedBuffers);
}

int ExMlcTransportChannel::grantReverseCredit(int max=0xFFFF,
    unsigned char *data=0,int datalen=0) {
	int credit=reverseCreditToGrant;
	if (credit>max) credit=max;
	reverseCreditToGrant-=credit;
	reverseCredit.increment(credit);
	if (credit) {
		scheduleCredit(SOURCE_CANCEL);	/* This could be recursive. */
	}
	return credit;
}

int ExMlcTransportChannel::prepareToSendPiggybackCredit(int credit) {
	return credit;
}

void ExMlcTransportChannel::ungrantReverseCredit(void) {
	int credit=reverseCredit.get();
	reverseCredit.set(0);
	reverseCreditToGrant+=credit;

#ifdef JD_DEBUGLITE
	if (credit) {
	    LOG_INFO(cEXTP,0,"ungrantReverseCredit(port=%d,channel=%d): "
		"ungranted reverseCredit=%d, now reverseCreditToGrant=%d.\n",
		port,channel,credit,reverseCreditToGrant);
	}
#endif
}

void ExMlcTransportChannel::scheduleCredit(SourceEnum source) {
	/* We check below if Credit is already pending. */

	if (source==SOURCE_CANCEL) {
		pReverseCreditHeartbeatTimer->cancel();
		return;
	}
	if (!isOpen() || isClosing()) {
		LOG_INFO(cEXTP,0,
			"scheduleCredit(port=%d,channel=%d,source=%d): "
			"closing or not open.\n",
			port,channel,source);
		scheduleCredit(SOURCE_CANCEL);	/* This is recursive. */
		return;
	}

	/* We don't care if the timer is already started. */

	if (disableCreditCommands) {
		/* LOG_INFO(cEXTP,0,
			"scheduleCredit(port=%d,channel=%d,source=%d): "
			"credit commands disabled.\n",
			port,channel,source); */
		return;
	}

	int now=0;
	int later=0;

	switch (source) {
	   case SOURCE_GRAB_REVERSE_CREDIT:
	   case SOURCE_HANDLE_CREDIT_REPLY:
		if (reverseCredit.isEmpty()) later=1;
		break;

	   case SOURCE_CREDIT_RETURNED_BUFFERS:
		if (!reverseCreditToGrant) {
			/* Do nothing. */
		} else if (reverseCreditToGrant>=reverseCredit.get()) {
			now=1;
		} else {
			later=1;
		}
		break;

	   case SOURCE_HANDLE_MSG:
		now=1;
		break;

	   default:
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
	}

	LOG_INFO(cEXTP,0,"scheduleCredit(port=%d,channel=%d,source=%d): "
		"now=%d, later=%d, reverseCredit=%d, ToGrant=%d.\n",
		port,channel,source,now,later,
		reverseCredit.get(),reverseCreditToGrant);

	if (now) {
		if (testFlagsSetClear(FLAG_LOCAL_CREDIT_PENDING,0)) {
			LOG_INFO(cEXTP,0,
				"scheduleCredit(port=%d,channel=%d,source=%d): "
				"local credit pending!\n",
				port,channel,source);
			goto laterInstead;
		}
		setClearFlags(FLAG_LOCAL_CREDIT_PENDING,0);

		countSendCredit.increment();
		pCommandChannel->sendCredit(
			getPrimarySocket(),getSecondarySocket(),
			grantReverseCredit());

	} else if (later) {
laterInstead:
		pReverseCreditHeartbeatTimer->startIfIdle();
	}
}

void ExMlcTransportChannel::handleCreditReply(int result,int psid,int ssid) {
	REV_REPLY_VALIDATE_LOCAL_REQUEST_PENDING(
		FLAG_LOCAL_CREDIT_PENDING,"handleCreditReply");

	REV_REPLY_VALIDATE_SUCCESSFUL("handleCreditReply");

	scheduleCredit(SOURCE_HANDLE_CREDIT_REPLY);

	flushCloseChannelReply();
}

void ExMlcTransportChannel::handleCreditRequest(int psid,int ssid,
    int creditRequested,int maxOutstandingCredit) {
	countHandleCreditRequest.increment();

	REV_REQUEST_VALIDATE_CHANNEL_OPEN("handleCreditRequest",
		ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CREDIT,
		pCommandChannel->sendCreditRequestReply(
			ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN,
			psid,ssid,0));

	int credit=0;

	/* Don't grant credit if we're closing. */
	if (isClosing()) {
		LOG_INFO(cEXTP,0,
			"handleCreditRequest(port=%d,channel=%d): closing.\n",
			port,channel);

	} else if (!disableCreditCommands) {
		credit=grantReverseCredit();

		if (!credit && workaroundReverseCreditLoss) {
			credit=reverseCredit.get();
#ifdef JD_DEBUGLITE
			if (credit) {
			    LOGD_ERROR(cEXTP,0,cCausePeriphError,
				"handleCreditRequest(port=%d,channel=%d): "
				"re-granting lost reverse credit=%d!\n",
				port,channel,credit);
			}
#endif
		}
	}
	/* We violate the MLC spec here in that we try to grant credit
	 * even if the peripheral requested zero credits (a ping packet). */

	if (!credit) {
		countSendEmptyCreditRequestReply.increment();
	}
	pCommandChannel->sendCreditRequestReply(ExMlcTransport::RESULT_SUCCESS,
		psid,ssid,credit);
}

/*****************************************************************************\
|* class ExMlcTransport
\*****************************************************************************/

/*---------------------------------------------------------------------------*\
|* static const member definitions:
\*---------------------------------------------------------------------------*/

const int ExMlcTransport::FIRST_DYNAMIC_SOCKET;
const int ExMlcTransport::REVISION_NONE;
const int ExMlcTransport::REVISION_DOT4;
const int ExMlcTransport::REVISION_PRE_DOT4;
const int ExMlcTransport::REVISION_MLC;
const int ExMlcTransport::RESULT_SUCCESS;
const int ExMlcTransport::MLC_RESULT_TOO_MANY_CONNECTIONS;
const int ExMlcTransport::MLC_RESULT_INSUFFICIENT_RESOURCES;
const int ExMlcTransport::MLC_RESULT_BAD_SOCKET;
const int ExMlcTransport::MLC_RESULT_UNKNOWN_COMMAND;
const int ExMlcTransport::MLC_ERROR_BAD_COMMAND_PACKET_LENGTH;
const int ExMlcTransport::MLC_ERROR_RECEIVED_UNCREDITED_PACKET;
const int ExMlcTransport::MLC_ERROR_REPLY_WITHOUT_REQUEST;
const int ExMlcTransport::MLC_ERROR_CREDIT_OVERFLOW;
const int ExMlcTransport::MLC_ERROR_NOT_INITIALIZED;
const int ExMlcTransport::MLC_ERROR_PACKET_TOO_BIG;
const int ExMlcTransport::MLC_ERROR_DATA_LINK_ERROR;
const int ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_DATA;
const int ExMlcTransport::MLC_ERROR_DUPLICATE_CONFIG_SOCKET;
const int ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CLOSE;
const int ExMlcTransport::MLC_ERROR_OPEN_UNCONFIG_SOCKET;
const int ExMlcTransport::MLC_ERROR_UNOPENED_CHANNEL_CREDIT;
const int ExMlcTransport::MLC_ERROR_UNKNOWN_RESULT_VALUE;
const int ExMlcTransport::MLC_ERROR_SOCKET_DOMAIN_MISMATCH;
const int ExMlcTransport::MLC_ERROR_INVALID_PACKET_SIZE;
const int ExMlcTransport::MLC_ERROR_SOCKET_ALREADY_OPEN;
const int ExMlcTransport::MLC_ERROR_INVALID_ENTER_COMMAND;
const int ExMlcTransport::MLC_ERROR_INVALID_REVISION;
const int ExMlcTransport::DOT4_RESULT_UNABLE_TO_INITIALIZE;
const int ExMlcTransport::DOT4_RESULT_INVALID_REVISION;
const int ExMlcTransport::DOT4_RESULT_COMMAND_CHANNEL_CLOSE;
const int ExMlcTransport::DOT4_RESULT_INSUFFICIENT_RESOURCES;
const int ExMlcTransport::DOT4_RESULT_CONNECTION_REFUSED;
const int ExMlcTransport::DOT4_RESULT_CHANNEL_ALREADY_OPEN;
const int ExMlcTransport::DOT4_RESULT_CREDIT_OVERFLOW;
const int ExMlcTransport::DOT4_RESULT_CHANNEL_NOT_OPEN;
const int ExMlcTransport::DOT4_RESULT_NO_SERVICE_ON_SOCKET;
const int ExMlcTransport::DOT4_RESULT_SERVICE_LOOKUP_FAILED;
const int ExMlcTransport::DOT4_RESULT_SIMULTANEOUS_INIT;
const int ExMlcTransport::DOT4_RESULT_INVALID_PACKET_SIZE;
const int ExMlcTransport::DOT4_RESULT_BOTH_PACKET_SIZES_ZERO;
const int ExMlcTransport::DOT4_RESULT_UNSUPPORTED_CREDIT_MODE;
const int ExMlcTransport::DOT4_ERROR_MALFORMED_PACKET;
const int ExMlcTransport::DOT4_ERROR_RECEIVED_UNCREDITED_PACKET;
const int ExMlcTransport::DOT4_ERROR_REPLY_WITHOUT_REQUEST;
const int ExMlcTransport::DOT4_ERROR_PACKET_TOO_BIG;
const int ExMlcTransport::DOT4_ERROR_UNOPENED_CHANNEL_DATA;
const int ExMlcTransport::DOT4_ERROR_UNKNOWN_RESULT_VALUE;
const int ExMlcTransport::DOT4_ERROR_PIGGYBACKED_CREDIT_OVERFLOW;
const int ExMlcTransport::DOT4_ERROR_UNKNOWN_COMMAND;
const int ExMlcTransport::DOT4_ERROR_PACKET_DIRECTION_ZERO_DATALEN;
const int ExMlcTransport::LEN_HEADER;
const int ExMlcTransport::REMSOCK_STATE_UNCONFIGURED;
const int ExMlcTransport::REMSOCK_STATE_CONFIGURING;
const int ExMlcTransport::REMSOCK_STATE_CONFIGURED;
const int ExMlcTransport::GRC_STATE_READY_FOR_HEADER;
const int ExMlcTransport::GRC_STATE_READY_FOR_DATA;
const int ExMlcTransport::DEFAULT_MLC_FORWARD_DATA_TIMEOUT;
const int ExMlcTransport::DEFAULT_TRY_REVISION_MLC;
const int ExMlcTransport::DEFAULT_TRY_REVISION_PRE_DOT4;
const int ExMlcTransport::DEFAULT_TRY_REVISION_DOT4;
const int ExMlcTransport::DEFAULT_MAX_REMOTE_SOCKETS;

/*---------------------------------------------------------------------------*\
|* Constructor, destructor, dump:
\*---------------------------------------------------------------------------*/

ExMlcTransport::ExMlcTransport(ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
    int forwardTransactionCount,int channelCount,int final=1):
      ExTransport(pMgr,pPhysicalPort,forwardTransactionCount,channelCount+1,0) {

	/* For now, we will leave this at the default of zero. */
	// overheadBufferCount=1;

	/* Override forward transaction count to the number of forward
	 * headers we want to allocate. */
	int maxForwardTransactionCount=(channelCount+1)*(cMaxWriteIrps+1);
	if (forwardTransactionCount<=0 ||
	    forwardTransactionCount>maxForwardTransactionCount) {
		forwardTransactionCount=maxForwardTransactionCount;
	}
	forwardTransactionCounter.resetMax(forwardTransactionCount);

	/* Allocate buffer pool for forward headers. */
	pForwardHeaderPool=new ExBufferPool((void *)this,
		LEN_HEADER,forwardTransactionCount,
		pMgr->getPortNumber(),pMgr->getBufferPoolMgr());

	maxRemoteSockets=DEFAULT_MAX_REMOTE_SOCKETS;
	exTknobGetInit(EX_KNOB_MLC_INIT_MAX_REMOTE_SOCKETS,&maxRemoteSockets);
	remoteSocketArray=new RemoteSocket[maxRemoteSockets];
	LOG_ASSERT(remoteSocketArray,cEXTP,0,cCauseNoMem,"");

	if (final) {
		reset(RESET_STARTUP);
		registerStuff("ExMlcTransport");
	}
}

void ExMlcTransport::initChannelArray(void) {
	ExTransport::initChannelArray();

	/* Set up type-correct alias for MLC channel array,
	 * which was allocated in the base version of this function. */
	mlcChannelArray=(ExMlcTransportChannel **)channelArray;
}

ExTransportChannel *ExMlcTransport::newChannel(int channel) {
	/* Channel 0 is the command channel. */
	if (!channel) {
		pCommandChannel=new ExMlcCommandChannel(this,channel,
			pMgr,pPhysicalPort,channelCount);
		return pCommandChannel;
	}

	/* Other channels are data channels with local socket IDs starting
	 * at FIRST_DYNAMIC_SOCKET and increasing by 1. */
	return new ExMlcTransportChannel(this,channel,
		channel-1+FIRST_DYNAMIC_SOCKET,
		pMgr,pPhysicalPort,pCommandChannel);
}

ExMlcTransport::~ExMlcTransport(void) {
	delete pForwardHeaderPool;
}

void ExMlcTransport::reset(int rtype) {
	ExTransport::reset(rtype);

	/* initChannelArray() is called by ExTransport::reset(). */

	if (rtype!=RESET_START_DEACTIVATE &&
	    rtype!=RESET_FINISH_DEACTIVATE) {
		/* resetReverseCount() is called in the base class. */
		reverseDataStopped=0;

		unsetRevision();

		remsockReset();
	}
}

void ExMlcTransport::registerStuff(char *title) {
	ExTransport::registerStuff(title);

	nettestRegister(NETTEST_TYPE_INT32,
		&grcState,
		"%s[%d].grcState",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&reverseDataStopped,
		"%s[%d].reverseDataStopped",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&tryDot4,
		"%s[%d].tryDot4",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&tryPreDot4,
		"%s[%d].tryPreDot4",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&tryMlc,
		"%s[%d].tryMlc",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&requestedRevision,
		"%s[%d].requestedRevision",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&revision,
		"%s[%d].revision",
		title,port);
	nettestRegister(NETTEST_TYPE_INT32,
		&maxRemoteSockets,
		"%s[%d].maxRemoteSockets",
		title,port);
	/* TODO: Add ability to inspect remoteSocketArray? */
}

#ifdef JD_DEBUGLITE
void ExMlcTransport::dump(void) {
	int i;

	ExTransport::dump();
	printf("----------------\n");
	printf("pForwardHeaderPool=0x%8.8X\n",
		(int)pForwardHeaderPool);
	printf("mlcChannelArray=0x%8.8X\n",
		(int)mlcChannelArray);
	printf("pCommandChannel=0x%8.8X\n",
		(int)pCommandChannel);
	printf("grcState=%d\n",
		grcState);
	printf("reverseDataStopped=%d\n",
		reverseDataStopped);
	printf("tryDot4=%d\n",
		tryDot4);
	printf("tryPreDot4=%d\n",
		tryPreDot4);
	printf("tryMlc=%d\n",
		tryMlc);
	printf("requestedRevision=0x%2.2X\n",
		requestedRevision);
	printf("revision=0x%2.2X\n",
		revision);
	printf("maxRemoteSockets=%d\n",
		maxRemoteSockets);
    for (i=0;i<maxRemoteSockets;i++) {
	if (remoteSocketArray[i].state==REMSOCK_STATE_UNCONFIGURED) continue;
	printf("remsock[%d]: state=%d, socketID=%d, fwdPS=%d, revPS=%d\n",
		i,remoteSocketArray[i].state,
		remoteSocketArray[i].socketID,
		remoteSocketArray[i].maxForwardPacketSize,
		remoteSocketArray[i].maxReversePacketSize);
    }
}
#endif

/*---------------------------------------------------------------------------*\
|* Activate:
\*---------------------------------------------------------------------------*/

void ExMlcTransport::transportResetComplete_ts(void) {
	TCD tcd;
	int timeout=DEFAULT_MLC_FORWARD_DATA_TIMEOUT;

	tcd=allocateChannel(0,0,0,0,0);
	LOG_ASSERT(tcd==pCommandChannel,cEXTP,0,cCauseBadState,"");

	tryMlc=DEFAULT_TRY_REVISION_MLC;
	tknobGetWorkingValue(port,EX_KNOB_MLC_TRY_REVISION_MLC,&tryMlc);
	tryPreDot4=DEFAULT_TRY_REVISION_PRE_DOT4;
	tknobGetWorkingValue(port,EX_KNOB_MLC_TRY_REVISION_PRE_DOT4,
		&tryPreDot4);
	tryDot4=DEFAULT_TRY_REVISION_DOT4;
	tknobGetWorkingValue(port,EX_KNOB_MLC_TRY_REVISION_DOT4,&tryDot4);

	tknobGetWorkingValue(port,EX_KNOB_MLC_FORWARD_DATA_TIMEOUT,&timeout);
	pForwardDataTimer->setDelay(timeout);

	sendInit();
}

void ExMlcTransport::sendInit(int nextRevision) {
	/* If we already unsuccessfully tried a revision,
	 * then skip to the next possible revision. */
	switch (requestedRevision) {
	   case REVISION_NONE:
		break;
	   case REVISION_DOT4:
		goto skipDot4;
	   case REVISION_PRE_DOT4:
		goto skipPreDot4;
	   case REVISION_MLC:
		goto skipMlc;
	   default:
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,
			"sendInit(port=%d): "
			"invalid requestedRevision=0x%2.2X!\n",
			port,requestedRevision);
	}

	if (tryDot4 && nextRevision>=REVISION_DOT4) {
		requestedRevision=REVISION_DOT4;
		goto send;
	}
skipDot4:

	/* LaserJet 3150 uses old 1284.4 revision. */
	if (tryPreDot4 && nextRevision>=REVISION_PRE_DOT4) {
		requestedRevision=REVISION_PRE_DOT4;
		goto send;
	}
skipPreDot4:

	if (tryMlc && nextRevision>=REVISION_MLC) {
		requestedRevision=REVISION_MLC;
		goto send;
	}
skipMlc:

	LOG_ERROR(cEXTP,0,cCausePeriphError,
		"sendInit(port=%d): exhausted all possible revisions!\n",
		port);
	pMgr->exClose(REASON_INIT_REVISION_REFUSED);
	return;

send:
	pCommandChannel->sendInit(requestedRevision);
}

STATUS ExMlcTransport::handleInitReply(int result,int revision) {
	/* Complain about but ignore second InitReply. */
	if (revisionIsSet()) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleInitReply(port=%d): "
			"ignoring extra InitReply!\n",
			port);
		return ERROR;
	}

	/* Complain but ignore if Init hasn't been sent yet. */
	if (requestedRevision==REVISION_NONE) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleInitReply(port=%d): "
			"Init hasn't been sent yet!\n",
			port);
		return ERROR;
	}

	/* Handle unsuccessful InitReply. */
	/* If the problem is with the requested revision, then try a
	 * different revision. */
	if (result==DOT4_RESULT_INVALID_REVISION) {
		sendInit(revision);
		return ERROR;
	}
	/* Otherwise, refuse to do business with this peripheral. */
	if (result!=RESULT_SUCCESS) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleInitReply(port=%d): result=%d!\n",
			port,result);
		pMgr->exClose(REASON_RECEIVED_UNSUCCESSFUL_REPLY);
		return ERROR;
	}

	/* Complain about successful InitReply for wrong revision. */
	if (revision!=requestedRevision) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleInitReply(port=%d,revision=0x%2.2X): "
			"requested revision=0x%2.2X!\n",
			port,revision,requestedRevision);
		if (revision<requestedRevision) {
			/* If the InitReply revision was less than what we
			 * requested (i.e. MLC vs 1284.4), then take that
			 * as a hint of what we should try next. */
			sendInit(revision);
		}
		return ERROR;
	}

	/* Handle successful InitReply for requested revision. */
	SETTHIS(revision);

	/* Tell data manager if we're in MLC or 1284.4. */
	if (revisionIsMlc()) {
		dmSetUint32(cKeyBpUsbCurrentMode,bpUsbCurrentModeMlc);
	} else /* if (revisionIsDot4()) */ {
		dmSetUint32(cKeyBpUsbCurrentMode,bpUsbCurrentModeDot4);
	}

	sendWaitingForInitialRequests();

	/* Returning OK tells the command channel to cancel the reply timer. */
	return OK;
}

void ExMlcTransport::handleInit(int revision) {
	if (revisionIsMlc()) {
		pCommandChannel->mlcSendError(MLC_ERROR_INVALID_ENTER_COMMAND);

	} else /* if (revisionIsDot4()) */ {
		pCommandChannel->sendInitReply(
			DOT4_RESULT_UNABLE_TO_INITIALIZE,revision);
	}
}

/*---------------------------------------------------------------------------*\
|* Deactivate:
\*---------------------------------------------------------------------------*/

void ExMlcTransport::handleExit(void) {
	/* TODO: Wait until outstanding transactions are completed,
	 * and don't initiate any more. */
	pCommandChannel->sendExitReply(RESULT_SUCCESS);
}

void ExMlcTransport::handleExitReply(int result) {
	pMgr->exClose(REASON_RECEIVED_EXIT_REPLY);
}

void ExMlcTransport::handleError(int psid,int ssid,int error) {
	if (error==MLC_ERROR_INVALID_REVISION &&
	    !revisionIsSet() && requestedRevision!=REVISION_MLC) {
		sendInit(REVISION_MLC);

	} else {
		pMgr->exClose(REASON_RECEIVED_ERROR);
	}
}

/*---------------------------------------------------------------------------*\
|* Remote socket lookup:
\*---------------------------------------------------------------------------*/

void ExMlcTransport::lookupRemoteSocket_ts(ExLookup *pLookupRequest) {
	if (revisionIsMlc()) {
		return ExTransport::lookupRemoteSocket_ts(pLookupRequest);
	}

	switch (pLookupRequest->getLastSet()) {
	   case ExLookup::LAST_SET_SERVICE_NAME:
		pCommandChannel->sendGetSocketID(
			pLookupRequest->getServiceName());
		break;

	   case ExLookup::LAST_SET_SOCKET_ID:
		pCommandChannel->sendGetServiceName(
			pLookupRequest->getSocketID());
		break;

	   default:
		LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,"");
	}
}

void ExMlcTransport::handleGetSocketIDReply(int result,int socketID,
    char *serviceName) {
	handleLookupReply(ExLookup::LAST_SET_SERVICE_NAME,
		result,socketID,serviceName);
}

void ExMlcTransport::handleGetServiceNameReply(int result,int socketID,
    char *serviceName) {
	handleLookupReply(ExLookup::LAST_SET_SOCKET_ID,
		result,socketID,serviceName);
}

void ExMlcTransport::handleLookupReply(int lastSet,int result,
    int socketID,char *serviceName) {
	int status=OK;
#ifdef JD_DEBUGLITE
	char *msg=0,*infomsg=0;
#endif

	ExLookup *pLookupRequest=lookupQueue.pop();
	if (!pLookupRequest) {
#ifdef JD_DEBUGLITE
		msg="no request pending";
#endif

	} else if (lastSet!=pLookupRequest->getLastSet()) {
#ifdef JD_DEBUGLITE
		msg="wrong reply";
#endif
		status=S_hlio_BP_PERIPHERAL_ERROR;

	} if (result!=RESULT_SUCCESS) {
#ifdef JD_DEBUGLITE
		infomsg="lookup failed";
#endif
		switch (result) {
		   case DOT4_RESULT_NO_SERVICE_ON_SOCKET:
			/* Some peripherals incorrectly return this result
			 * for failed socketID->serviceName lookups. */
#ifdef JD_DEBUGLITE
			infomsg="lookup failed (wrong result code)";
#endif
		   case DOT4_RESULT_SERVICE_LOOKUP_FAILED:
			status=S_hlio_BP_SERVICE_LOOKUP_FAILED;
			break;
		   default:
#ifdef JD_DEBUGLITE
			msg="lookup failed (unknown result code)";
#endif
			status=S_hlio_BP_UNKNOWN_ERROR;
		}

	} else if (lastSet==ExLookup::LAST_SET_SERVICE_NAME) {
		if (strcmp(serviceName,pLookupRequest->getServiceName())) {
#ifdef JD_DEBUGLITE
			msg="wrong serviceName";
#endif
			status=S_hlio_BP_PERIPHERAL_ERROR;

		} else {
			pLookupRequest->setSocketID(socketID,1);
		}

	} else /* if (lastSet==ExLookup::LAST_SET_SOCKET_ID) */ {
		if (socketID!=pLookupRequest->getSocketID()) {
#ifdef JD_DEBUGLITE
			msg="wrong socketID";
#endif
			status=S_hlio_BP_PERIPHERAL_ERROR;

		} else {
			pLookupRequest->setServiceName(serviceName,0,1);
		}
	}

#ifdef JD_DEBUGLITE
	if (msg) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"handleLookupReply(port=%d,lastSet=%d,result=%d,"
			"socketID=%d,serviceName=<%s>): %s!\n",
			port,lastSet,result,socketID,serviceName,msg);
	} else if (infomsg) {
		LOG_INFO(cEXTP,0,
			"handleLookupReply(port=%d,lastSet=%d,result=%d,"
			"socketID=%d,serviceName=<%s>): %s!\n",
			port,lastSet,result,socketID,serviceName,infomsg);
	}
#endif

	if (pLookupRequest) {
		pLookupRequest->setStatus(status);
		sendRemoteSocketLookupResponse(pLookupRequest);
	}
}

void ExMlcTransport::handleGetSocketID(char *serviceName) {
	pCommandChannel->sendGetSocketIDReply(
		DOT4_RESULT_SERVICE_LOOKUP_FAILED,0,serviceName);
}

void ExMlcTransport::handleGetServiceName(int socketID) {
	pCommandChannel->sendGetServiceNameReply(
		DOT4_RESULT_SERVICE_LOOKUP_FAILED,socketID,"");
}

/*---------------------------------------------------------------------------*\
|* Set remote socket:
\*---------------------------------------------------------------------------*/

void ExMlcTransport::remsockReset(void) {
	int i;

	for (i=0;i<maxRemoteSockets;i++) {
		remoteSocketArray[i].state=REMSOCK_STATE_UNCONFIGURED;
		remoteSocketArray[i].socketID=0;
		remoteSocketArray[i].maxForwardPacketSize=0;
		remoteSocketArray[i].maxReversePacketSize=0;
	}
}

STATUS ExMlcTransport::remsockLookup(int socketID,int *pState,
    int *pMaxForwardPacketSize,int *pMaxReversePacketSize) {
	LOG_ASSERT(revisionIsMlc(),cEXTP,0,cCauseBadState,"");

	int i;

	for (i=0;i<maxRemoteSockets;i++) {
		if (remoteSocketArray[i].state!=REMSOCK_STATE_UNCONFIGURED &&
		    remoteSocketArray[i].socketID==socketID) {
			if (pState) *pState=remoteSocketArray[i].state;
			if (pMaxForwardPacketSize) *pMaxForwardPacketSize=
				remoteSocketArray[i].maxForwardPacketSize;
			if (pMaxReversePacketSize) *pMaxReversePacketSize=
				remoteSocketArray[i].maxReversePacketSize;
			return OK;
		}
	}

	/* Don't log an error here.
	 * This is how we know the socket needs to be configured. */
	return ERROR;
}

STATUS ExMlcTransport::remsockAdd(int socketID) {
	LOG_ASSERT(revisionIsMlc(),cEXTP,0,cCauseBadState,"");

	int i,firstFree=-1;

	for (i=0;i<maxRemoteSockets;i++) {
		if (remoteSocketArray[i].state==REMSOCK_STATE_UNCONFIGURED) {
			if (firstFree<0) firstFree=i;
		} else if (remoteSocketArray[i].socketID==socketID) {
			LOG_ERROR_FATAL(cEXTP,0,cCauseBadParm,
				"remsockAdd(port=%d,socketID=%d): "
				"already in table!\n",
				port,socketID);
			return ERROR;
		}
	}

	if (firstFree<0) {
		LOG_ERROR(cEXTP,0,cCauseNoHandles,
			"remsockAdd(port=%d,socketID=%d): "
			"table is full!\n",
			port,socketID);
		return ERROR;
	}

	remoteSocketArray[firstFree].state=REMSOCK_STATE_CONFIGURING;
	remoteSocketArray[firstFree].socketID=socketID;
	remoteSocketArray[firstFree].maxForwardPacketSize=0;
	remoteSocketArray[firstFree].maxReversePacketSize=0;

	return OK;
}

void ExMlcTransport::handleConfigSocketReply(int result,int socketID,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int statusLevel) {
	LOG_ASSERT(revisionIsMlc(),cEXTP,0,cCauseBadState,"");

	int status=OK;
	int maxForwardPacketSize=
		pickLocalSocket(maxPriToSecPacketSize,maxSecToPriPacketSize);
	int maxReversePacketSize=
		pickRemoteSocket(maxPriToSecPacketSize,maxSecToPriPacketSize);

	if (result!=RESULT_SUCCESS) {
		remsockDelete(socketID);
		status=S_hlio_BP_UNSUPPORTED_SOCKET;
	} else if (remsockUpdate(socketID,maxForwardPacketSize,
		   maxReversePacketSize)==ERROR) {
		status=S_hlio_BP_UNKNOWN_ERROR;
	} else {
		status=OK;
	}

	for (int channel=0;channel<channelCount;channel++) {
		TCD tcd=mlcChannelArray[channel];
		if (tcd->getRemoteSocket()==socketID)
			tcd->sendRemoteSocketSetResponse(status);
	}
}

STATUS ExMlcTransport::remsockUpdate(int socketID,
    int maxForwardPacketSize,int maxReversePacketSize) {
	LOG_ASSERT(revisionIsMlc(),cEXTP,0,cCauseBadState,"");

	int i;

	for (i=0;i<maxRemoteSockets;i++) {
		if (remoteSocketArray[i].state!=REMSOCK_STATE_UNCONFIGURED &&
		    remoteSocketArray[i].socketID==socketID) {
			if (remoteSocketArray[i].state!=
			    REMSOCK_STATE_CONFIGURING) {
				LOG_ERROR(cEXTP,0,cCausePeriphError,
					"remsockUpdate(port=%d,socketID=%d): "
					"already configured!\n",
					port,socketID);
				return ERROR;
			}

			remoteSocketArray[i].state=REMSOCK_STATE_CONFIGURED;
			remoteSocketArray[i].maxForwardPacketSize=
				maxForwardPacketSize;
			remoteSocketArray[i].maxReversePacketSize=
				maxReversePacketSize;
			return OK;
		}
	}

	LOG_ERROR(cEXTP,0,cCausePeriphError,
		"remsockUpdate(port=%d,socketID=%d): not in table!\n",
		port,socketID);
	return ERROR;
}

STATUS ExMlcTransport::remsockDelete(int socketID) {
	LOG_ASSERT(revisionIsMlc(),cEXTP,0,cCauseBadState,"");

	int i;

	for (i=0;i<maxRemoteSockets;i++) {
		if (remoteSocketArray[i].state!=REMSOCK_STATE_UNCONFIGURED &&
		    remoteSocketArray[i].socketID==socketID) {
			if (remoteSocketArray[i].state!=
			    REMSOCK_STATE_CONFIGURING) {
				LOG_ERROR(cEXTP,0,cCausePeriphError,
					"remsockDelete(port=%d,socketID=%d): "
					"already configured!\n",
					port,socketID);
				return ERROR;
			}

			remoteSocketArray[i].state=REMSOCK_STATE_UNCONFIGURED;
			remoteSocketArray[i].maxForwardPacketSize=0;
			remoteSocketArray[i].maxReversePacketSize=0;
			return OK;
		}
	}

	LOG_ERROR(cEXTP,0,cCausePeriphError,
		"remsockDelete(port=%d,socketID=%d): not in table!\n",
		port,socketID);
	return ERROR;
}

/*---------------------------------------------------------------------------*\
|* Peripheral-initiated opens (not supported):
\*---------------------------------------------------------------------------*/

void ExMlcTransport::handleOpenChannel(int psid,int ssid,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int maxOutstandingCredit,int credit) {
	int result;

	if (revisionIsMlc()) {
		result=MLC_RESULT_TOO_MANY_CONNECTIONS;
	} else /* if (revisionIsDot4()) */ {
		result=DOT4_RESULT_NO_SERVICE_ON_SOCKET;
	}

	pCommandChannel->sendOpenChannelReply(result,psid,ssid,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		maxOutstandingCredit,credit);
}

void ExMlcTransport::handleConfigSocket(int socketID,
    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
    int statusLevel) {
	pCommandChannel->sendConfigSocketReply(MLC_RESULT_BAD_SOCKET,socketID,
		maxPriToSecPacketSize,maxSecToPriPacketSize,
		statusLevel);
}

/*---------------------------------------------------------------------------*\
|* Reverse data flow:
\*---------------------------------------------------------------------------*/

void ExMlcTransport::resetReverseCount(void) {
	grcState=GRC_STATE_READY_FOR_HEADER;
}

STATUS ExMlcTransport::getReverseCount(char *lastData,int lastLength,
    int *pCount,int *pFlags) {
	if (grcState==GRC_STATE_READY_FOR_HEADER) {
		if (reverseDataStopped) {
			*pCount=0;
			*pFlags=GRC_FLAG_END_OF_TRANSACTION;
		} else {
			*pCount=LEN_HEADER;
			*pFlags=(GRC_FLAG_RECEIVE_DATA_BEFORE_NEXT_REQUEST|
				 GRC_FLAG_ALIGN_NEXT_DATA);
			grcState=GRC_STATE_READY_FOR_DATA;
		}

	} else /* if (grcState==GRC_STATE_READY_FOR_DATA) */ {
		if (!lastData || lastLength<LEN_HEADER) {
			LOG_ERROR(cEXTP,0,cCausePeriphError,
				"getReverseCount(port=%d): "
				"invalid header length=%d!\n",
				port,lastLength);
			return ERROR;
		}
		Header *header=(Header *)lastData;
		*pCount=BEND_GET_WORD(header->length)-LEN_HEADER;
		*pFlags=GRC_FLAG_END_OF_TRANSACTION;
		if (!overheadBufferCount) {
			*pFlags|=GRC_FLAG_APPEND_TO_PREVIOUS_BUFFER;
		}
		grcState=GRC_STATE_READY_FOR_HEADER;
	}

	return OK;
}

ExTransportChannel *ExMlcTransport::lookupChannel(ExBdr *pBdr) {
	Header *header=bdrGetHeader(pBdr);
	int len=pBdr->getDataLength();

	if (!len) {
		LOG_INFO(cEXTP,0,"lookupChannel(port=%d,pBdr=0x%8.8X): "
			"empty packet!\n",port,pBdr);
		return 0;
	}

	if (len<LEN_HEADER) {
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"lookupChannel(port=%d,pBdr=0x%8.8X): "
			"partial header, length=%d!\n",
			port,pBdr,len);
#ifdef JD_DEBUGLITE
		for (int i=0;i<len;i++) {
			LOGD_ERROR(cEXTP,0,cCausePeriphError,
				"partial header byte[%d]=0x%2.2X!\n",
				i,((unsigned char *)header)[i]);
		}
#endif
		pCommandChannel->sendError(0,0,
			MLC_ERROR_DATA_LINK_ERROR,
			DOT4_ERROR_MALFORMED_PACKET);
		return 0;
	}

	LOG_INFO(cEXTP,0,"Received header on port=%d: "
		"psid=0x%2.2X, ssid=0x%2.2X, length=%d, "
		"credit=%d, control=0x%2.2X.\n",
		port,header->psid,header->ssid,
		BEND_GET_WORD(header->length),
		header->credit,header->control);

	TCD tcd=lookupChannel(header->psid,header->ssid);
	if (!tcd) {
		pCommandChannel->sendError(header->psid,header->ssid,
			MLC_ERROR_UNOPENED_CHANNEL_DATA,
			DOT4_ERROR_UNOPENED_CHANNEL_DATA);
	}
	return tcd;
}

ExMlcTransportChannel *ExMlcTransport::lookupChannel(int psid,int ssid) {
	if (psid==ExMlcCommandChannel::COMMAND_SOCKET &&
	    ssid==ExMlcCommandChannel::COMMAND_SOCKET)
		return pCommandChannel;

	int localSocket=pickLocalSocket(psid,ssid);
	int channel=localSocket-FIRST_DYNAMIC_SOCKET+1;
	int remoteSocket=pickRemoteSocket(psid,ssid);

	if (channel<1 || channel>=channelCount) {
abort:
		LOG_ERROR(cEXTP,0,cCausePeriphError,
			"lookupChannel(port=%d,psid=%d,ssid=%d): invalid!\n",
			port,psid,ssid);
		return 0;
	}

	ExMlcTransportChannel *tcd=mlcChannelArray[channel];
	if (remoteSocket!=tcd->getRemoteSocket()) goto abort;

	return tcd;
}
