// ParPort.h

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#ifndef PARPORT_H
#define PARPORT_H

#include "ExMgr.h"

#define ERROR_NO_DATA ERROR	/* TODO: Remove. */

class ParPort {
    protected:		// ATTRIB
	int portType;
    public:
	static const int PORTTYPE_NONE		=ERROR;
	static const int PORTTYPE_UNKNOWN	=0;
	static const int PORTTYPE_SPP		=1;
	static const int PORTTYPE_BPP		=2;
	static const int PORTTYPE_ECP		=3;	/* No HW assist. */
	static const int PORTTYPE_ECPHW		=4;
	int getPortType(void) { return portType; }
    protected:
	void setPortType(int value,int force=0) {
		if (portType==PORTTYPE_UNKNOWN || force)
			portType=value;
	}
    public:
	int portTypeIsValid(void) {
		return portType>PORTTYPE_UNKNOWN;
	}
	int portTypeIsBidirectional(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>PORTTYPE_SPP);
	}
	int portTypeIsEcp(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>=PORTTYPE_ECP);
	}
	int portTypeIsEcpWithHwAssist(void) {
		return (portType==PORTTYPE_UNKNOWN ||
			portType>=PORTTYPE_ECPHW);
	}

    protected:		// ATTRIB
	int debug;
    public:
	int getDebug(void) { return debug; }
	void setDebug(int debug) { SETTHIS(debug); }
	void dump(void);

    public:
	enum TimevalIndex {
		TIMEVAL_SETUP_DELAY=0,
		TIMEVAL_STROBE_DELAY,
		TIMEVAL_HOLD_DELAY,
		TIMEVAL_ECP_SETUP_DELAY,
		TIMEVAL_ECP_POST_HTR_DELAY,
		TIMEVAL_SIGNAL_TIMEOUT,
		TIMEVAL_BUSY_TIMEOUT,
		TIMEVAL_REVERSE_POLL_RATE,
		TIMEVAL_COUNT
	};
    protected:		// ATTRIB
	struct timeval timevalArray[TIMEVAL_COUNT];
    protected:
	struct timeval *lookupTimeval(TimevalIndex index) {
		if (index<0 || index>=TIMEVAL_COUNT) return 0;
		return &timevalArray[index];
	}
	void dumpTimeval(TimevalIndex index,char *name);
    public:
	int getTimeval(TimevalIndex index,int *pSec,int *pUsec) {
		struct timeval *src=lookupTimeval(index);
		if (!src) return ERROR;

		if (pSec) *pSec=src->tv_sec;
		if (pUsec) *pUsec=src->tv_usec;

		return OK;
	}
	int getTimeval(TimevalIndex index,struct timeval *dest) {
		struct timeval *src=lookupTimeval(index);
		if (!src) return ERROR;

		if (dest) {
			dest->tv_sec=src->tv_sec;
			dest->tv_usec=src->tv_usec;
		}

		return OK;
	}
	int setTimeval(TimevalIndex index,int sec,int usec) {
		struct timeval *dest=lookupTimeval(index);
		if (!dest) return ERROR;

		dest->tv_sec=sec;
		dest->tv_usec=usec;

		return OK;
	}
	int setTimeval(TimevalIndex index,struct timeval *src) {
		struct timeval *dest=lookupTimeval(index);
		if (!dest) return ERROR;

		dest->tv_sec=src->tv_sec;
		dest->tv_usec=src->tv_usec;

		return OK;
	}

    public:
	ParPort(int portType=PORTTYPE_UNKNOWN);
    protected:
	void postConstructor(void);
	void preDestructor(void);
    public:
	virtual ~ParPort();

    protected:		// ATTRIB
	int dead;
    public:
	void reset(void);
    protected:
	void die(void) { dead=1; }	/* TODO: Use this. */
    public:
	int isDead(void) { return dead; }

	/* For the sake of efficiency, assume parallel port with PC-style
	 * register set.  Access details are defined in a derived class. */
    protected:
	static const int BANK_SPP		=0;
	static const int BANK_ECP		=1;
	static const int MAX_BANKS		=2;
	static const int REG_DATA		=0;
	static const int REG_STATUS		=1;
	static const int REG_CONTROL		=2;
	virtual int regRead(int bank,int reg)=0;
	virtual void regWrite(int bank,int reg,int value)=0;
	/* TODO: Routines to reserve/unreserve port. */

	/* I find it very annoying that some PC parport bits are inverted
	 * relative to the 1284 spec, so our control/status read/write
	 * routines will handle the inversion for us. */
    protected:
	static const int CONTROL_NSTROBE	=0x01;
	static const int CONTROL_NAUTOFD	=0x02;
	static const int CONTROL_NINIT		=0x04;
	static const int CONTROL_NSELECTIN	=0x08;
	static const int CONTROL_INT_ENABLE	=0x10;	/* Not used. */
	static const int CONTROL_REVERSE_DATA	=0x20;
	static const int CONTROL_INVERTED	=(CONTROL_NSTROBE|
						  CONTROL_NAUTOFD|
						  CONTROL_NSELECTIN);
	int controlRead(void) {
		return (regRead(BANK_SPP,REG_CONTROL)^CONTROL_INVERTED);
	}
	void controlWrite(int value) {
		regWrite(BANK_SPP,REG_CONTROL,value^CONTROL_INVERTED);
	}
	void controlSetClear(int set,int clear) {
		controlWrite((controlRead()&(~clear))|set);
	}
	void controlSetClear(int event,int set,int clear);

    protected:		// ATTRIB
	PolledTimer statusWaitTimer;
    protected:
	static const int STATUS_NFAULT		=0x08;
	static const int STATUS_SELECT		=0x10;
	static const int STATUS_PERROR		=0x20;
	static const int STATUS_NACK		=0x40;
	static const int STATUS_BUSY		=0x80;
	static const int STATUS_INVERTED	=(STATUS_BUSY);
	int statusRead() {
		return (regRead(BANK_SPP,REG_STATUS)^STATUS_INVERTED);
	}
	int statusReadNibble() {
		int s=statusRead();
		return ((s>>3)&0x07)|((s>>4)&0x08);
	}
	int statusTestSetClear(int set,int clear) {
		return ((statusRead()&(set|clear))==set);
	}
    public:
	int statusReverseDataIsAvailable(void) {
		return statusTestSetClear(0,STATUS_NFAULT);
	}
    protected:
	int statusWaitSetClear(int event,TimevalIndex timeout,
	    int set,int clear);

    protected:
	int dataRead(void) {
		return regRead(BANK_SPP,REG_DATA);
	}
	void dataWrite(int value) {
		regWrite(BANK_SPP,REG_DATA,value);
	}
	int dataTest(int value) {
		dataWrite(value);
		if (dataRead()!=value) return ERROR;
		return OK;
	}

    protected:
	int ecpDataRead(void) {
		return regRead(BANK_ECP,REG_DATA);
	}
	void ecpDataWrite(int value) {
		regWrite(BANK_ECP,REG_DATA,value);
	}
	int ecpStatusRead(void) {
		return regRead(BANK_ECP,REG_STATUS);
	}

    protected:		// ATTRIB
	int ecpConfigA;
	int ecpConfigB;
    protected:
	int ecpConfigRead(void);

    protected:
	static const int ECP_CONTROL_MODE_UNIDIRECTIONAL	=0x00;
	static const int ECP_CONTROL_MODE_BIDIRECTIONAL		=0x20;
	static const int ECP_CONTROL_MODE_FAST_CENTRONICS	=0x40;
	static const int ECP_CONTROL_MODE_ECP			=0x60;
	static const int ECP_CONTROL_MODE_TEST			=0xC0;
	static const int ECP_CONTROL_MODE_CONFIGURATION		=0xE0;
	static const int ECP_CONTROL_MODE_MASK			=0xE0;
	static const int ECP_CONTROL_ALWAYS_SET			=0x14;
	static const int ECP_CONTROL_FULL			=0x02;
	static const int ECP_CONTROL_EMPTY			=0x01;
	int ecpControlRead(void) {
		return regRead(BANK_ECP,REG_CONTROL);
	}
	void ecpControlWrite(int value) {
		regWrite(BANK_ECP,REG_CONTROL,value|ECP_CONTROL_ALWAYS_SET);
	}
	int ecpControlGetMode(void);
	int ecpControlSetMode(int mode);
	int ecpHwAssistIsEnabled(void) {
		int mode=ecpControlGetMode();
		return (mode==ECP_CONTROL_MODE_FAST_CENTRONICS ||
			mode==ECP_CONTROL_MODE_ECP);
	}
	int ecpFifoIsFull(void) {
		return ecpControlRead()&ECP_CONTROL_FULL;
	}
	int ecpFifoIsEmpty(void) {
		return ecpControlRead()&ECP_CONTROL_EMPTY;
	}
	int ecpFifoWaitNotFull(TimevalIndex timeout);
	int ecpFifoWaitNotEmpty(TimevalIndex timeout);
	int ecpFifoWaitEmpty(TimevalIndex timeout);

    public:
	static const int MODE_COMPAT		=0x100;
	static const int MODE_REVERSE		=0x200;	/* Only for ECP. */
	static const int MODE_HW_ASSIST		=0x400; /* Compat or ECP. */
	static const int MODE_NIBBLE		=0x000;
	static const int MODE_BYTE		=0x001;	/* Not supported. */
	static const int MODE_2			=0x002;	/* Not supported. */
	static const int MODE_DEVICE_ID		=0x004;
	static const int MODE_DOT3		=0x008;	/* 1284.3 bit. */
	static const int MODE_ECP		=0x010;
	static const int MODE_RLE		=0x020;	/* Not supported. */
	static const int MODE_EPP		=0x040;	/* Not supported. */
	static const int MODE_EXT_LINK		=0x080;	/* Not supported. */
	static const int MODE_BOUNDED_ECP	=(MODE_ECP|MODE_DOT3);
	static const int MODE_CHANNELIZED_NIBBLE=(MODE_DOT3);
    protected:		// ATTRIB
	int currentMode;
	int currentChannel;
    public:
	int isCompat(int mode) {
		return (mode&MODE_COMPAT);
	}
	int isCompat(void) {
		return isCompat(currentMode);
	}
	int isNibble(int mode) {
		return (!(mode&(~(MODE_DEVICE_ID|MODE_DOT3))));
	}
	int isNibble(void) {
		return isNibble(currentMode);
	}
	int isEcp(int mode) {
		return (mode&MODE_ECP);
	}
	int isEcp(void) {
		return isEcp(currentMode);
	}
	int isBoundedEcp(int mode) {
		return ((mode&(MODE_ECP|MODE_DOT3))==(MODE_ECP|MODE_DOT3));
	}
	int isBoundedEcp(void) {
		return isBoundedEcp(currentMode);
	}
	int isEcpForward(int mode) {
		return ((mode&(MODE_ECP|MODE_REVERSE))==MODE_ECP);
	}
	int isEcpForward(void) {
		return isEcpForward(currentMode);
	}
	int isEcpReverse(int mode) {
		return ((mode&(MODE_ECP|MODE_REVERSE))==
			(MODE_ECP|MODE_REVERSE));
	}
	int isEcpReverse(void) {
		return isEcpReverse(currentMode);
	}

    protected:
	int negotiate(int mode);
	int terminate(int mode=0);
	int ecpFwdToRev(int mode=0);
	int ecpRevToFwd(int mode=0);

    protected:		// ATTRIB
	int htrCount;
    public:
	int getHtrCount(void) { return htrCount; }
	void setHtrCount(int htrCount) { SETTHIS(htrCount); }

    protected:
	int writeCompatByte(int c);
	int writeEcpByte(int c);
	int writeHwByte(int c);
    public:
	int writeByte(int c);
	int writeByte(int c,int count);
	int write(unsigned char *buffer,int len);

    protected:
	static const int ECP_COMMAND		=0x100;
	static const int ECP_CHANNEL		=0x080;
	static const int ECP_COMMAND_MASK	=0x07F;
    protected:
	int writeEcpChannel(int channel) {
		if (channel==currentChannel) return OK;
		currentChannel=channel;
		return writeByte((channel&ECP_COMMAND_MASK)|
			ECP_CHANNEL|ECP_COMMAND);
	}

    protected:
	int readNibble(int index);
	int readEcpByte(void);
	int readHwByte(void);
    public:
	int readByte(void);
	int read(unsigned char *buffer,int len);
	int flushReverseData(void);

    protected:
	int readDeviceIDLength(void);
    public:
	unsigned char *getDeviceID();

    protected:		// ATTRIB
	int forwardMode;
	int reverseMode;
	int channel;
    public:
	int setModes(int forwardMode,int reverseMode);
	int setChannel(int channel) {
		return writeEcpChannel(SETTHIS(channel));
	}
	int startReverse(void);
	int finishReverse(void);
};

#if defined(PAR_PLATFORM_LINUX)
	#include "ioWrapper.h"
#elif defined(PAR_PLATFORM_FREEBSD)
	#include <unistd.h>
	#include <fcntl.h>
    extern "C" {
	#include <machine/cpufunc.h>
    }
#else
	/* TODO: Handle these more gracefully! */
	#error Undefined platform
#endif

class IoParPort: public ParPort {
    protected:
	/* Bank 0: 0x3BC, 0x378, 0x278.
	 * Bank 1: 0x400 above bank 0 for ISA, may differ for PCI. */
	int baseaddr[MAX_BANKS];

#ifdef PAR_PLATFORM_FREEBSD
	int iofd;
#endif

	int getIoPortAccess(void) {
#if defined(PAR_PLATFORM_LINUX)
		return iopl(3);

#elif defined(PAR_PLATFORM_FREEBSD)
		return (iofd=open("/dev/io",O_RDWR));

#else
#error Undefined platform
#endif
	}
	int removeIoPortAccess(void) {
#if defined(PAR_PLATFORM_LINUX)
		return iopl(0);

#elif defined(PAR_PLATFORM_FREEBSD)
		return close(iofd);

#else
#error Undefined platform
#endif
	}
	virtual int regRead(int bank,int reg) {
#if defined(PAR_PLATFORM_LINUX)
		return inb(baseaddr[bank]+reg);

#elif defined(PAR_PLATFORM_FREEBSD)
		return inb(baseaddr[bank]+reg);

#else
#error Undefined platform
#endif
	}
	virtual void regWrite(int bank,int reg,int value) {
#if defined(PAR_PLATFORM_LINUX)
		outb(value,baseaddr[bank]+reg);

#elif defined(PAR_PLATFORM_FREEBSD)
		outb(baseaddr[bank]+reg,value);

#else
#error Undefined platform
#endif
	}

    public:
	IoParPort(int baseaddr0,int baseaddr1,int portType=PORTTYPE_UNKNOWN):
	    ParPort(portType) {
		baseaddr[0]=baseaddr0;
		baseaddr[1]=baseaddr1;

		if (getIoPortAccess()==ERROR) {
			LOG_ERROR_FATAL(cEXBP,0,cCauseFuncFailed,
				"getIoPortAccess failed!\n");
		}

		postConstructor();
	}
	virtual ~IoParPort() {
		preDestructor();

		removeIoPortAccess();
	}
};

#endif
