//
// Filename: xojpanel.cpp
//
// Implementation of the XojPanel widget.

/* LICENSE
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
*/

/*
  Copyright (C) circa 1998 Andreas Fester
  Copyright (C) 2000, 2001 Joe Piolunek

  This is an expanded version of the original
  xojpanel application released by Andreas Fester.

  Thanks to David Paschal for contributing a patch to allow xojpanel to exit
  gracefully if the peripheral does not support LCD message requests.

  David Paschal also did the ojlib->PTAL conversion.
*/

//
#include "xojpanel.h"
#include "hpoj_mini.xpm"
#include "hpojlcd.xpm"
#include "hpoj_lcdmon.xpm"
//
#include <qapplication.h>
#include <qtimer.h>
#include <qpainter.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qstring.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
//

/* SET LCD FONT ************************************/
// If you want to change the default font
// (as appears in LCD), do it here.
// Uncomment ( remove the "//" ) one line for
// your choice of font and one line for your preferred
// fontsize. Choosing a fixed-width font will usually
// give better results. You can specify a font not listed
// here. If you comment out all of the font choices below,
// Qt will attempt to use your local system's font setting.
//
// Note: earlier versions of Qt may not like the font
// you specify. You may need to experiment a little to get
// the results you want.
//
// Fixed-width fonts
#define DEFAULT_FONT "Courier" // May look a little rough.
// #define DEFAULT_FONT "Adobe-courier"
// #define DEFAULT_FONT "clean"
// #define DEFAULT_FONT "dec-terminal"
// #define DEFAULT_FONT "misc-fixed"
// #define DEFAULT_FONT "quicktype mono"
// #define DEFAULT_FONT "Courier new"
// #define FONTSIZE 12
//#define FONTSIZE 14
#define FONTSIZE 16

/***************************************************/

// Enables the use of the Qt 'QChar' character class.
// Useful if Unicode support is desired. As far as I
// know, it is not needed.
// Define only if building with qt-2.0 or higher.
#ifndef HAVE_QT2
//#define HAVE_QT2
#endif


// Scroll speed and delay options, LCD text color.
const unsigned int PRINTER_POLL_REPEAT_INTERVAL = 500;	// in ms. (val incr. = speed decr.)
const unsigned int SCROLL_INTERVAL = 333;		// in ms. (val incr. = speed decr.)
const unsigned int SCROLL_START_DELAY = 3;		// Number of  'scrolls' skipped.
const unsigned int SCROLL_END_DELAY = 2;		// Number of  'scrolls' skipped.
const unsigned int MAX_LCD_CHARS = 20;			// Number of characters in LCD.
const QColor *LCD_TEXT_COLOR = new QColor(0, 0, 0);          // Color of LCD text.
/**************************************************************************************/

/**
 * XojPanel class constructor.
 * Creates the XojPanel widget. xojpanel is a simple graphical app
 * that shows the HP printer's current LCD window contents, scrolling
 * them if necessary.
 */
XojPanel::XojPanel(int argc, char **argv, QWidget *parent, const char *name )
		: QWidget(parent,name)
{
//printf( "XojPanel class constructor called." );

	// Parse command line and set up PTAL.
	if (argc!=2) {
		/* TODO: Use default device name from PTAL? */
		printf("Syntax: %s <dev>\n"
			"Where <dev> is the PTAL device name, such as \"mlc:par:0\"\n"
			,*argv);
		exit(1);
	}
	if (ptalInit()==PTAL_ERROR) {
		printf("Failed to initialize PTAL!\n");
		exit(1);
	}
	dev=ptalDeviceOpen(argv[1]);
	if (!dev) {
		printf("Failed to open device \"%s\"!\n",argv[1]);
		exit(1);
	}
	if (ptalPmlOpen(dev)==PTAL_ERROR) {
		printf("Failed to open PML on device \"%s\"!\n",argv[1]);
		exit(1);
	}
	/* TODO: Define these OIDs in a common include file. */
	objLine1=objHpLine1=ptalPmlAllocateID(dev,"\x1\x1\x2\x14\x2\x1\x1");
	objLine2=objHpLine2=ptalPmlAllocateID(dev,"\x1\x1\x2\x14\x2\x2\x1");
	objSpmLine1=ptalPmlAllocateID(dev,"\x2\x10\x5\x1\x2\x1\x1");
	objSpmLine2=ptalPmlAllocateID(dev,"\x2\x10\x5\x1\x2\x1\x2");

	getPrinterLCDMessages(); // Makes the messages available at start.
	createInterface( argv[1] ); // Sets up appearance of the xojpanel widget.
	createTimers();// Creates any QTimers we need.
}

/**
 * Enables immediate *current* text display on restoring/uncovering
 * window. Re-implements QWidget::paintEvent()
 * @param event Pointer to the paintEvent
 */
void
XojPanel::paintEvent( QPaintEvent *event )
{
//printf( " XojPanel::paintEvent( QPaintEvent * )\n" );

	lcdGraphic->setPixmap( *pm );
}


/**
 * Closes the xojpanel application when a window 'exit' or 'close'
 * button is pushed. Reimplemented from QWidget::closeEvent()
 * @param closeEvent Pointer to the xojpanel object's closeEvent
 */
void
XojPanel::closeEvent( QCloseEvent *closeEvent )
{
//printf( "void XojPanel::closeEvent()\n\n" );

	ptalDone();
	closeEvent->accept(); // This is the main widget, so app will be terminated.
}

/**
 * Sets up xojpanel's graphical interface.
 * @param deviceName Identifies device in titlebar.
 */
void
XojPanel::createInterface( QString deviceName )
{
//printf("void XojPanel::createInterface( void )\n\n" );

	//QString titlebarText = "OfficeJet Status:  ";
	//titlebarText.append( "[ " + deviceName + " ]" );
	QString titlebarText = "OfficeJet Status";
	titlebarText.append( ":   " + deviceName );

	this->setCaption( titlebarText );
	this->setIcon(QPixmap(( const char**) hpoj_mini_xpm));

	#ifdef DEFAULT_FONT
	#ifdef FONTSIZE
	QFont f = QFont( DEFAULT_FONT, FONTSIZE );
	this->setFont( f );
	f.setFixedPitch( true );
	f.setStyleHint( QFont::AnyStyle );
	#endif
	#endif

	this->setBackgroundColor( QColor( 190, 190, 190 )); // 190, 190, 190
	this->setMinimumSize( 330, 150 ); // 330, 150
	this->setMaximumSize( 330, 150 ); // 330, 150

	createLCD();
	createCoverGraphic();
}


/**
 * Places a 254x40 pixel "LCD" background pixmap on window.
 */
void
XojPanel::createLCD( void )
{
//printf( "XojPanel::createLCD( void )\n" );

	lcdPixmap = new QPixmap((const char**)hpojlcd_xpm);
	pm = new QPixmap((const char**)hpojlcd_xpm);

	// Create the LCD Graphic.
	lcdGraphic = new QLabel( this, "lcdGraphic",(uint)0 );
	lcdGraphic->setPixmap( *lcdPixmap );

	//actual size of graphic - 254x40
	lcdGraphic->setGeometry( 38, 90, 254, 40 ); // 38, 90 254, 40

	updateLCDPixmap();// get text from printer and display it.
}


/**
 * Places a 290x70 pixel decorative label on the window.
 */
void
XojPanel::createCoverGraphic( void )
{
//printf( "bool createCoverGraphic( void )\n\n");

	QLabel *coverGraphic = new QLabel( this, "label", (uint)0);
	coverGraphic->setPixmap(QPixmap((const char**)hpoj_lcdmon_xpm ));
	coverGraphic->setGeometry( 20, 7, 290, 70 ); // 20, 7, 290, 70

	// xojpanel background color is (rgb) 190,190,190
	coverGraphic->setBackgroundColor( QColor( 190, 190, 190 ));
}


/**
 * Creates timers for such things as retrieving
 * printer LCD messages and scroll speed.
 */
void
XojPanel::createTimers( void ) // Timer settings at top of file.
{
//printf( "XojPanel::createTimers( void )\n" );

	//create a timer for polling printer for new lcd messages.
	QTimer *getLCDMessagesTimer = new QTimer( this );
	connect( getLCDMessagesTimer, SIGNAL(timeout()), SLOT(getPrinterLCDMessages()) );
	getLCDMessagesTimer->start( PRINTER_POLL_REPEAT_INTERVAL ); // in milliseconds.

	// Create a timer for lcd repainting.
	// Controls scroll speed.
	QTimer *scrollTimer = new QTimer( this );
	connect( scrollTimer, SIGNAL(timeout()), SLOT(updateLCDPixmap()) );
	scrollTimer->start( SCROLL_INTERVAL ); // in milliseconds.
}


/**
 * Retrieves printer's LCD messages, then immediately sends
 * them to stripRightSpace() to have any trailing rightspace
 * removed. Timer controlled SLOT.
 */
void
XojPanel::getPrinterLCDMessages( void )
{
//printf( "void XojPanel::getPrinterLCDMessages( void )\n" );

	int r1,r2;
	char buffer[PTAL_PML_MAX_VALUE_LEN];

	// read the current contents of the printer's LCD
retry:
	r1=ptalPmlRequestGet(objLine1,0);
	r2=ptalPmlRequestGet(objLine2,0);
	if (r1==PTAL_ERROR || r2==PTAL_ERROR) {
		if (objLine1!=objSpmLine1) {
			objLine1=objSpmLine1;
			objLine2=objSpmLine2;
			goto retry;
		}

		r1=ptalPmlGetStatus(objLine1);
		r2=ptalPmlGetStatus(objLine2);
		printf("Errors while executing PML requests: "
			"0x%2.2X, 0x%2.2X\n",r1,r2);
		if (r1==PTAL_PML_ERROR_UNKNOWN_OBJECT_IDENTIFIER) {
			printf("This model doesn't seem to support "
				"LCD readback.\n");
		}
		exit(1);
	}
	ptalPmlGetStringValue(objLine1,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	Line1=buffer;
	ptalPmlGetStringValue(objLine2,0,buffer,PTAL_PML_MAX_VALUE_LEN);
	Line2=buffer;

/*
	Some printf() statements may not be secure.
	Use this form when dealing with QStrings:
	//printf("Line1 = %s", (const char*)Line1 );

	// SCROLL and WHITESPACE STRIPPING STRING TESTS
	// Scroll test is in this section. Whitespace stripping test is farther down.
	printf( "\n in XojPanel::getPrinterLCDMessages()\n" );
	printf( "\nOriginal lines (endmarks not part of orig.string):\n" );
	printf("new Line1 -->%s<--end\n",(const char*)Line1);
	printf("new Line2 -->%s<--end\n",(const char*)Line2 );

	printf( "\nSubstitute lines below if used:\n" );
	//Line1 = "Short Line 1 test.";
	//Line1 = "Fourscore and seven years ago, our forefathers had no MTV.";
	//Line2 = "Short test line 2.";
	//Line2 = "Long test line 2    spacer   spacer      end.";

	//Line1 = "Open the pod bay door please, HAL.      ";
	//Line2 = "   I'm sorry Dave, I can't do that.      ";
*/

/*
	// You can create a custom string here that includes unprintable characters.
	// String will be shown in xojpanel's lcd window if printable.
	// The array needs to be null-terminated, otherwise you get garbage added.
	char asciiArray[] = { 132, 0 };
	printf( "%s \n", asciiArray );
	Line2 = asciiArray;

	printf("new Line1 -->%s<--end\n",(const char*)Line1);
	printf("new Line2 -->%s<--end\n",(const char*)Line2 );
	printf("End of original line substitutes.\n" );
*/
	// Send both lines to be stripped of any whitespace. The Oj 600 has been
	// found to output a mix of spaces and control characters at the end of
	// text messages that prompt for user panel-button input.
	stripRightSpace( &Line1 );
	stripRightSpace( &Line2 );
/*
	// Shows strings with rightspace removed. Echoed to command line.
	printf( "Strings with rightspace removed--:\n" );
	printf("new Line1 -->%s<--end\n",(const char*)Line1);
	printf("new Line2 -->%s<--end\n",(const char*)Line2 );

	// End of whitespace stripping test.
*/
}


/**
 * Removes whitespace from the trailing end of
 * LCD messages retrieved from the printer.
 * @param string Pointer to the QString to be stripped.
 */
void
XojPanel::stripRightSpace( QString * string )
{
//printf("XojPanel::stripRightSpace( QString * string )");

	unsigned int length = string->QString::length();
	unsigned int unwantedChars = 0;

	if( (string->isEmpty()) || (string->isNull()) ) return; // Don't strip an empty string.

/*
	// If the printer is outputting ascii characters,
	// this test will tell you their numeric codes.
	// The ascii codes will be printed to the command line.
	printf( "\nvoid XojPanel::stripRightsSpace( QString * string )\n" );
	printf( "string = %s<-end\n",(const char*)*string );
	printf( "Characters before stripping rightspace are: \n" );
	for( unsigned int i = 0; i < string->length(); i++ ) {
		printf("%c = ", (const char)*string->mid(i, 1));
		unsigned int c;
		c = *string->mid( i, 1);
		printf( "%d\n",c );
	}

	printf( "Length of string before stripping = %d \n", string->length() );
*/
	#ifndef HAVE_QT2
	// Counts unprintable characters on right end of string.
	// This section must be used if building with Qt versions < 2.x . Can
	// be used with any Qt version if you don't need unicode support. Handles only ascii.
	while(isgraph( *string->mid( (length - unwantedChars) - 1, 1 ) ) == 0 ){
		unwantedChars++; // Count unwanted chars on right end of string.

		// Don't cycle through again if the whole string
		// is counted for removal. It results in a seg fault.
		if(unwantedChars >= length ) break;
	}
	#endif

	#ifdef HAVE_QT2
	// Only use this section if building with
	// qt-2.0 or later. Will handle unicode or ascii.
	QChar c;
	c = *string->mid( (length - unwantedChars) - 1, 1 );
	//Counts unprintable characters on right end of string.
	while( ( c.isSpace() ) || (!(c.isPrint() ))) {
		unwantedChars++;

		// Don't cycle through again if the whole string
		// is counted for removal. It results in a segfault.
		if(unwantedChars >= length ) break;

		c = *string->mid( (length - unwantedChars) - 1, 1 );

	}
	#endif

	//printf( "unwantedChars = %d\n", unwantedChars );

	// Removes the rightspace characters after all are found
	string->remove( length - unwantedChars, unwantedChars ); // index, #chars.
/*
	printf( "Stripped any rightspace. Displaying chars in new string.\n" );
	for( unsigned int i = 0; i < string->length(); i++ ) {
		printf("%c = ", (const char)*string->mid(i, 1));
		unsigned int c;
		c = *string->mid( i, 1);
		printf( "%d\n", c );
	}

	printf( "Length of string after stripping: %d\n", string->length() );
*/
}


/**
 * Checks current LCD messages, looks to see if the contents
 * are new, then sends the information to displayLCDMessage().
 * Timer controlled SLOT.
 */
void
XojPanel::updateLCDPixmap( void )
{
//printf( "void XojPanel::updateLCDPixmap( void )\n" );

	static bool newLine1Message = true;
	static bool newLine2Message = true;

	// For local copies of the printer's LCD text strings.
	static QString *Line1Copy = new QString;
	static QString *Line2Copy = new QString;

	if(strcmp( Line1, *Line1Copy ) == 0 ) { // No new message from printer.
		newLine1Message = false;
	}else{ // New message from printer.
		newLine1Message = true;
		*Line1Copy = Line1; // We're using a local copy of Line1 and Line2.
	}

	if(strcmp( Line2, *Line2Copy ) == 0 ) {
		newLine2Message = false;
	}else{
		newLine2Message = true;
		*Line2Copy = Line2; // We're using a local copy of Line1 and Line2.
	}

	displayLCDMessage( Line1Copy, Line2Copy, newLine1Message, newLine2Message );
}


/**
 * Displays the printer's LCD message text
 * on the pixmap, scrolling if necessary.
 * If the contents of the peripheral's LCD line
 * are new, the corresponding string is updated
 * and scrolled from the beginning.
 * @param firstLine Pointer to rightspace-stripped
 *      version of LCD Line1 contents
 * @param secondLine Pointer to rightspace-stripped
 *      version of LCD Line2 contents
 * @param newLine1 True if Line1 contains new contents; else false.
 * @param newLine2 True if Line2 contains new contents; else false.
 */
void
XojPanel::displayLCDMessage( QString * firstLine, QString * secondLine,
							bool newLine1, bool newLine2 )
{
/*
printf("XojPanel::displayLCDMessage( QString * firstLine, QString * secondLine,"
						"bool newLine1, bool newLine2");
*/

	// Places text on pixmap. Scrolls if necessary.
	// Called by updateLCDPixmap (SLOT).
	static QString *line_1; line_1 = firstLine; // Stripped of rightspace already.
	static QString *line_2; line_2 = secondLine;// Stripped of rightspace already.

	static bool newLine1Message; newLine1Message = newLine1;
	static bool newLine2Message; newLine2Message = newLine2;

	int x = 10;		// Initial horizontal coordinate for text drawing.
	int y_Line1 = 18; 	// Vertical coordinate for line 1 text draw.
	int y_Line2 = 32; 	// Vertical coordinate for line 2 text draw.

	bool line1Painted = false;
	bool line2Painted = false;
	static unsigned int line1ScrollStartDelay = 0;
	static unsigned int line1ScrollEndDelay = 0;
	static unsigned int line2ScrollStartDelay = 0;
	static unsigned int line2ScrollEndDelay = 0;
	static unsigned int beginAtLine1Char = 0; // For scrolling line 1
	static unsigned int beginAtLine2Char = 0; // For scrolling line 2

	QString *line1Segment = new QString; // These seem to need deleting after use
	QString *line2Segment = new QString; // is over. Memory leaks appear if not done.

	*pm = *lcdPixmap; // Create a text-free copy of the LCD background pixmap

	QPainter* p = new QPainter; // Draw the LCD text contents into the text-free pixmap.
	p->begin( pm );             // Delete p after drawing is done.
	p->setPen(*LCD_TEXT_COLOR); // Color of LCD text.
	p->setFont( font() ); //QApplication::font() returns default font for this app.

	// Paint LCD line 1 text.

	// Paints the whole line if short.
	if( line_1->length() < (MAX_LCD_CHARS + 1) ) { // It's a short line.
		drawLCDTextLine( p, line_1, x, y_Line1 );
		line1Painted = true;
	}

	// It's a long line, so we do something a little more complicated.
	if( line_1->length() > MAX_LCD_CHARS ) { // It's a long line. Advance text one char position.
		if( newLine1 ) {
			line1ScrollStartDelay = SCROLL_START_DELAY;
			beginAtLine1Char = 0; // Since it's a new line, start at the start.
		}

		line1Painted = false;

		if( ( line1ScrollStartDelay == 0 ) && ( line1ScrollEndDelay == 0 ) ) { //no delays
			*line1Segment = line_1->mid( beginAtLine1Char , MAX_LCD_CHARS ); // Select midstring.
			drawLCDTextLine( p, line1Segment, x, y_Line1 ); // Paint it.
			line1Painted = true;
			beginAtLine1Char++;

			if( ( beginAtLine1Char + MAX_LCD_CHARS) > line_1->length() ) { // End of line. Delays needed.
			/*
			printf( " beginAtLine1Char = %d\n", beginAtLine1Char );
			printf( "line_1->length() = %d\n", line_1->length() );
			*/

				line1ScrollEndDelay = SCROLL_END_DELAY; // Wait at the end of the line.
				//printf("line1ScrollEndDelay-- = %d\n", line1ScrollEndDelay );

				beginAtLine1Char = 0;// For rescroll later.
				//printf("line1ScrollEndDelay-- = %d\n", line1ScrollEndDelay );
			}
		}

		if( !line1Painted && (line1ScrollEndDelay > 0) ) { // Delay at end of line.
			line1ScrollStartDelay = 0;

			/*
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );//test
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			*/

			*line1Segment = line_1->mid( (line_1->length() - MAX_LCD_CHARS), line_1->length() );
			drawLCDTextLine( p, line1Segment, x, y_Line1 );// End-of-line.
			line1Painted = true;
			line1ScrollEndDelay--;

			if( line1ScrollEndDelay < 1 ) { //after the '--' above.
				line1ScrollStartDelay = SCROLL_START_DELAY;
			}

			/*
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			*/
		}

		if( !line1Painted && (line1ScrollStartDelay > 0)) { // Delay at beginning of line.

			//printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );

			line1ScrollStartDelay--;
			line1ScrollEndDelay = 0;

			/*
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			*/

			*line1Segment = line_1->mid( 0, MAX_LCD_CHARS ); // First 16 characters of message.
			drawLCDTextLine( p, line1Segment, x, y_Line1 ); // Draw start of line.

			/*
			printf( "line1ScrollStartDelay = %d\n", line1ScrollStartDelay );
			printf( "line1ScrollEndDelay = %d\n", line1ScrollEndDelay );
			*/

			line1Painted = true;
		}
	}

	if( !line1Painted ) printf( "Error: Line1 has not been painted.\n" );

	// Paint LCD line 2 text.

	// For LCD line 2. Paints the whole line if short.
	if( line_2->length() < (MAX_LCD_CHARS + 1) ) { //It's a short line
		drawLCDTextLine( p, line_2, x, y_Line2 );  // Paint it.
		line2Painted = true;
	}

	// It's a long line, so we do something a little more complicated.
	  if( line_2->length() > MAX_LCD_CHARS ) { // It's a long line. Advance text one char. position.

		if( newLine2 ) {
			line2ScrollStartDelay = SCROLL_START_DELAY;
			beginAtLine2Char = 0;
		}

		line2Painted = false;

		if( ( line2ScrollStartDelay == 0 ) && ( line2ScrollEndDelay == 0 ) ) { //no delays
			*line2Segment = line_2->mid( beginAtLine2Char , MAX_LCD_CHARS ); // Select midstring.
			drawLCDTextLine( p, line2Segment, x, y_Line2 ); // Paint it.
			line2Painted = true;
			beginAtLine2Char++;

			if( ( beginAtLine2Char + MAX_LCD_CHARS) > line_2->length() ) { //end of line. Delays needed.

				/*
				printf( " beginAtLine2Char = %d\n", beginAtLine2Char );
				printf( "line_2->length() = %d\n", line_2->length() );
				*/

				beginAtLine2Char = 0;// Set midstring to beginning of line.

				/*
				printf("line2ScrollEndDelay-- = %d\n", line2ScrollEndDelay );
				*/

				line2ScrollEndDelay = SCROLL_END_DELAY;

				/*
				printf("line2ScrollEndDelay-- = %d\n", line2ScrollEndDelay );
				*/
			}
		}

		if( !line2Painted && (line2ScrollEndDelay > 0) ) { // Delay at end of line.
			line2ScrollStartDelay = 0;

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			*/

			*line2Segment = line_2->mid( (line_2->length() - MAX_LCD_CHARS), line_2->length() );
			drawLCDTextLine( p, line2Segment, x, y_Line2 );// End-of-line.
			line2Painted = true;
			line2ScrollEndDelay--;

			if( line2ScrollEndDelay < 1 ) { //due to the '--' above.
				line2ScrollStartDelay = SCROLL_START_DELAY;
			}

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			*/
		}

		if( !line2Painted && (line2ScrollStartDelay > 0)) { // Delay at beginning of line.

			/*
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

			line2ScrollStartDelay--;
			line2ScrollEndDelay = 0;

			/*
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

		        *line2Segment = line_2->mid( 0, MAX_LCD_CHARS ); // First 16 characters of message.
			drawLCDTextLine( p, line2Segment, x, y_Line2 ); // Draw start of line.

			/*
			printf( "line2ScrollStartDelay = %d\n", line2ScrollStartDelay );
			printf( "line2ScrollEndDelay = %d\n", line2ScrollEndDelay );
			*/

			line2Painted = true;
		}
	}

	if( !line2Painted ) printf( "Error: Line2 has not been painted.\n" );

	delete line1Segment;// plugs memory leak.
	delete line2Segment;// plugs memory leak.

	p->end(); // We're finished painting.
	delete p; // plugs a memory leak

	// Copy the updated pixmap to the LCD.
	bitBlt( lcdGraphic, 0, 0, pm );
}


/**
 * Paints line of text onto pixmap. The QPainter
 * for this is created in displayLCDMessages().
 * @param p Pointer to the QPainter that redraws the LCD background
 * @param line Pointer to the QString that needs to be redrawn
 * @param x Horizontal offset of the first character
 * @param y Vertical offset of the first character
 */
void
XojPanel::drawLCDTextLine( QPainter * p, QString * line, int x, int y )
{
//printf( "void drawLCDTextLine( QPainter * p, QString * line, int x, int y )\n" );

	unsigned int i;
	
	if( line->length() < (MAX_LCD_CHARS + 1) ) { // It's short enough to paint all at once.
		for (i = 0;  i < line->length();  i++, x += 12)
			p->drawText( x, y, line->mid(i, 1), 1);
	}
	else {
		printf( "Error: in \"void drawLCDTextLine(...)\": Line length exceeds ");
		printf( "%i characters.\n", MAX_LCD_CHARS );
		printf( "line = ->%s<-end\n", (const char*)line );
	}

}


/**
 * Declares xojpanel object and begins execution.
 */
int main( int argc, char **argv )
{
	QApplication app(argc,argv);

	XojPanel *xojpanel = new XojPanel(argc,argv);
	app.setMainWidget( xojpanel );
	xojpanel->show();

	int ret = app.exec();
	return ret;

}

// End of file
