/* ptal-hp -- PTAL command-line HP peripheral configuration/status client */

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */
/* JPEG manipulation code based on imaging code provided by Mark Overton
 * in the HP All-in-One division (AiO) in San Diego. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/wait.h>
#include "ptal.h"

enum {
	RETCODE_SUCCESS=0,
	RETCODE_NEW_PAGE,
	RETCODE_NO_PAGE,

	RETCODE_SYNTAX_ERROR,
	RETCODE_DEVICE_ERROR
};

void printOneLine(ptalPmlObject_t obj) {
	int symbolSet,len,i,start;
	unsigned char buffer[PTAL_PML_MAX_VALUE_LEN+1];

	len=ptalPmlGetStringValue(obj,&symbolSet,buffer,
		PTAL_PML_MAX_VALUE_LEN);
	if (len==PTAL_ERROR) {
		printf("(line unavailable)\n");
		return;
	}

	/* Reduce len if there's a null before it. */
	if (len>strlen((char *)buffer)) len=strlen((char *)buffer);

	/* Convert non-printable characters to whitespace. */
	for (i=0;i<len;i++) if (buffer[i]<' ') buffer[i]=' ';

	/* Strip leading and trailing whitespace. */
	for (start=0;start<len && isspace(buffer[start]);start++);
	for (;len>start && isspace(buffer[len-1]);len--);

	/* Make sure the string is null-terminated in the right place. */
	buffer[len]=0;

#define PRINT_BRACKETS 0

	printf(
#if PRINT_BRACKETS
	"<"
#endif
	"%s"
#if PRINT_BRACKETS
	">"
#endif
	"\n",buffer+start);
}

int handleDevice(ptalDevice_t dev,int arg,int argc,char **argv) {
	ptalPmlObject_t obj=ptalPmlAllocate(dev);
	int value;

	printf("Model name:        ");
	ptalPmlSetID(obj,"\x1\x1\x3\x2");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	printf("Model number:      ");
	ptalPmlSetID(obj,"\x1\x1\x3\x1");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	printf("Serial number:     ");
	ptalPmlSetID(obj,"\x1\x1\x3\x3");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	printf("Firmware version:  ");
	ptalPmlSetID(obj,"\x1\x1\x3\x6");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	printf("Firmware datecode: ");
	ptalPmlSetID(obj,"\x1\x1\x3\x5");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	printf("Total RAM size:    ");
	ptalPmlSetID(obj,"\x1\x1\x2\x15");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else if (ptalPmlGetIntegerValue(obj,0,&value)==PTAL_ERROR) {
		printf("(value unavailable)\n");
	} else {
		printf("%d bytes\n",value);
	}

	ptalPmlDeallocate(obj);
	return RETCODE_SUCCESS;
}

int handleDisplay(ptalDevice_t dev,int arg,int argc,char **argv) {
	ptalPmlObject_t objLine1=ptalPmlAllocateID(dev,
		"\x1\x1\x2\x14\x2\x1\x1");
	ptalPmlObject_t objLine2=ptalPmlAllocateID(dev,
		"\x1\x1\x2\x14\x2\x2\x1");
	ptalPmlObject_t objSpmLine1=ptalPmlAllocateID(dev,
		"\x2\x10\x5\x1\x2\x1\x1");
	ptalPmlObject_t objSpmLine2=ptalPmlAllocateID(dev,
		"\x2\x10\x5\x1\x2\x1\x2");
	int r1,r2;

	r1=ptalPmlRequestGet(objLine1,0);
	r2=ptalPmlRequestGet(objLine2,0);
	if (r1!=PTAL_ERROR || r2!=PTAL_ERROR) {
		if (r1!=PTAL_ERROR) printOneLine(objLine1);
		if (r2!=PTAL_ERROR) printOneLine(objLine2);

	} else {
		r1=ptalPmlRequestGet(objSpmLine1,0);
		r2=ptalPmlRequestGet(objSpmLine2,0);
		if (r1!=PTAL_ERROR || r2!=PTAL_ERROR) {
			if (r1!=PTAL_ERROR) printOneLine(objSpmLine1);
			if (r2!=PTAL_ERROR) printOneLine(objSpmLine2);

		} else {
			printf("(unavailable)\n");
		}
	}

	ptalPmlDeallocate(objLine1);
	ptalPmlDeallocate(objLine2);
	ptalPmlDeallocate(objSpmLine1);
	ptalPmlDeallocate(objSpmLine2);

	return RETCODE_SUCCESS;
}

/* Ignore the weekday since some devices report it incorrectly. */
#define PRINT_WEEKDAY 0

int handleClock(ptalDevice_t dev,int arg,int argc,char **argv) {
	int retcode;
	ptalPmlObject_t objClock=ptalPmlAllocateID(dev,"\x1\x1\x2\x11");
	ptalPmlObject_t objDateFormat=ptalPmlAllocateID(dev,"\x1\x1\x1\x16");
	ptalPmlObject_t objTimeFormat=ptalPmlAllocateID(dev,"\x1\x1\x2\x1C");
	time_t tt=time(0);
	struct tm *tlocal=localtime(&tt);
	struct {
		unsigned char year;
		unsigned char month;
		unsigned char day;
		unsigned char weekday;
		unsigned char hour;
		unsigned char minute;
		unsigned char second;
	} pmlClock;
	enum {
		MMDDYY=1,
		DDMMYY=2,
		YYMMDD=3
	} dateFormat;
	enum {
		hour12=1,
		hour24=2
	} timeFormat;
	int month,year,hour,ampm;
	char *smonth[]={"???","Jan","Feb","Mar","Apr","May","Jun",
		"Jul","Aug","Sep","Oct","Nov","Dec"};
#if PRINT_WEEKDAY
	int weekday;
	char *sweekday[]={"???","Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
#endif

	while (argc) {
		if (!strcmp(*argv,"-set")) {
			pmlClock.year=tlocal->tm_year%100;
			pmlClock.month=tlocal->tm_mon+1;
			pmlClock.day=tlocal->tm_mday;
			pmlClock.weekday=tlocal->tm_wday+1;
			pmlClock.hour=tlocal->tm_hour;
			pmlClock.minute=tlocal->tm_min;
			pmlClock.second=tlocal->tm_sec;
/* setClock: */
			ptalPmlSetValue(objClock,PTAL_PML_TYPE_BINARY,
				(char *)&pmlClock,sizeof(pmlClock));
			ptalPmlRequestSet(objClock);

#if 0
		} else if (!strcmp(*argv,"-unset")) {
			pmlClock.year=0;
			pmlClock.month=1;
			pmlClock.day=1;
			pmlClock.weekday=0;
			pmlClock.hour=0;
			pmlClock.minute=0;
			pmlClock.second=0;
			goto setClock;
#endif

		} else if (!strcmp(*argv,"-mmddyy")) {
			dateFormat=MMDDYY;
setDateFormat:
			ptalPmlSetIntegerValue(objDateFormat,
				PTAL_PML_TYPE_ENUMERATION,dateFormat);
			ptalPmlRequestSet(objDateFormat);

		} else if (!strcmp(*argv,"-ddmmyy")) {
			dateFormat=DDMMYY;
			goto setDateFormat;

		} else if (!strcmp(*argv,"-yymmdd")) {
			dateFormat=YYMMDD;
			goto setDateFormat;

		} else if (!strcmp(*argv,"-12hour")) {
			timeFormat=hour12;
setTimeFormat:
			ptalPmlSetIntegerValue(objTimeFormat,
				PTAL_PML_TYPE_ENUMERATION,timeFormat);
			ptalPmlRequestSet(objTimeFormat);

		} else if (!strcmp(*argv,"-24hour")) {
			timeFormat=hour24;
			goto setTimeFormat;

		} else {
			PTAL_LOG_ERROR(
"Syntax: ... clock [<params>...]\n"
"Valid params:\n"
"\t-set                      -- Set clock to PC clock\n"
/* "\t-unset                    -- Unset clock\n" */
"\t-mmddyy, -ddmmyy, -yymmdd -- Set date format\n"
"\t-12hour, -24hour          -- Set time format\n"
				);
			retcode=RETCODE_SYNTAX_ERROR;
			goto abort;
		}

		argc--; argv++;
	}

	if (ptalPmlRequestGet(objDateFormat,0)==PTAL_ERROR ||
	    ptalPmlGetIntegerValue(objDateFormat,0,(int *)&dateFormat)==
	     PTAL_ERROR) {
		dateFormat=MMDDYY;
	}

	if (ptalPmlRequestGet(objTimeFormat,0)==PTAL_ERROR ||
	    ptalPmlGetIntegerValue(objTimeFormat,0,(int *)&timeFormat)==
	     PTAL_ERROR) {
		timeFormat=hour12;
	}

	printf("Device clock: ");
	if (ptalPmlRequestGet(objClock,0)==PTAL_ERROR ||
	    ptalPmlGetValue(objClock,0,(char *)&pmlClock,sizeof(pmlClock))==
	     PTAL_ERROR) {
		printf("(unavailable)\n");

	} else {
#if PRINT_WEEKDAY
		weekday=pmlClock.weekday;
		if (weekday<1 || weekday>7) weekday=0;
		printf("%s ",sweekday[weekday]);
#endif

		month=pmlClock.month;
		if (month<1 || month>12) month=0;
		year=1900+((tlocal->tm_year/100)*100)+(pmlClock.year%100);
		if (dateFormat==YYMMDD) {
			printf("%d-%s-%d ",year,smonth[month],pmlClock.day);
		} else if (dateFormat==DDMMYY) {
			printf("%d-%s-%d ",pmlClock.day,smonth[month],year);
		} else /* if (dateFormat==MMDDYY) */ {
			printf("%s-%d-%d ",smonth[month],pmlClock.day,year);
		}

		hour=pmlClock.hour;
		if (timeFormat==hour24) {
			printf("%d:%2.2d:%2.2d\n",
				hour,pmlClock.minute,pmlClock.second);
		} else /* if (timeFormat==hour12) */ {
			ampm=(hour>=12);
			hour%=12;
			if (!hour) hour=12;
			printf("%d:%2.2d:%2.2d %s\n",
				hour,pmlClock.minute,pmlClock.second,
				!ampm?"AM":"PM");
		}
	}

	retcode=RETCODE_SUCCESS;
abort:

	ptalPmlDeallocate(objClock);
	ptalPmlDeallocate(objDateFormat);
	ptalPmlDeallocate(objTimeFormat);

	return retcode;
}

int handleScanStatus(ptalDevice_t dev,int arg,int argc,char **argv) {
	ptalPmlObject_t obj=ptalPmlAllocate(dev);
	int value;

	printf("Scanner status:              ");
	ptalPmlSetID(obj,"\x1\x2\x2\x2\x1");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else if (ptalPmlGetIntegerValue(obj,0,&value)==PTAL_ERROR) {
		printf("(value unavailable)\n");
	} else if (!value) {
		printf("ready\n");
	} else {
		printf("value=%d",value);
		if (value&1) printf(", unknown error");
		if (value&2) printf(", invalid media size");
		if (value&4) printf(", feeder open");
		if (value&8) printf(", feeder jam");
		if (value&16) printf(", feeder empty");
		printf("\n");
	}

	printf("Number of brightness levels: ");
	ptalPmlSetID(obj,"\x1\x2\x2\x2\x2");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else if (ptalPmlGetIntegerValue(obj,0,&value)==PTAL_ERROR) {
		printf("(value unavailable)\n");
	} else {
		printf("%d\n",value);
	}

	printf("Scan resolution range:       ");
	ptalPmlSetID(obj,"\x1\x2\x2\x2\x3");
	if (ptalPmlRequestGet(obj,0)==PTAL_ERROR) {
		printf("(unavailable)\n");
	} else {
		printOneLine(obj);
	}

	ptalPmlDeallocate(obj);
	return RETCODE_SUCCESS;
}

/* Since the firmware does not supply tables in its header,
 * the tables used in the firmware are supplied below.  */

/*____________________________________________________________________________
 |                                                                            |
 | Zigzag of Normal Quantization Tables                                       |
 |____________________________________________________________________________|
*/ 

static const unsigned char orig_lum_quant[64] = {
	 16,  11,  12,  14,  12,  10,  16,  14,
	 13,  14,  18,  17,  16,  19,  24,  40,
	 26,  24,  22,  22,  24,  49,  35,  37,
	 29,  40,  58,  51,  61,  60,  57,  51,
	 56,  55,  64,  72,  92,  78,  64,  68,
	 87,  69,  55,  56,  80, 109,  81,  87,
	 95,  98, 103, 104, 103,  62,  77, 113,
	121, 112, 100, 120,  92, 101, 103,  99
};

static const unsigned char orig_chrom_quant[64] = {
	17,  18,  18,  24,  21,  24,  47,  26,
	26,  47,  99,  66,  56,  66,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99,
	99,  99,  99,  99,  99,  99,  99,  99
};

/*____________________________________________________________________________
 |                                                                            |
 | Huffman Tables                                                             |
 |____________________________________________________________________________|
*/
 
static const unsigned char lum_DC_counts[16] = {
	0x00, 0x01, 0x05, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
 
static const unsigned char lum_DC_values[12] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b
};
 
static const unsigned char chrom_DC_counts[16] = {
	0x00, 0x03, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00
};
 
static const unsigned char chrom_DC_values[12] = {
	0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b
};
 
static const unsigned char lum_AC_counts[16] = {
	0x00, 0x02, 0x01, 0x03, 0x03, 0x02, 0x04, 0x03,
	0x05, 0x05, 0x04, 0x04, 0x00, 0x00, 0x01, 0x7d
};

static const unsigned char lum_AC_values[162] = {
	0x01, 0x02, 0x03, 0x00, 0x04, 0x11, 0x05, 0x12,
	0x21, 0x31, 0x41, 0x06, 0x13, 0x51, 0x61, 0x07,
	0x22, 0x71, 0x14, 0x32, 0x81, 0x91, 0xa1, 0x08,
	0x23, 0x42, 0xb1, 0xc1, 0x15, 0x52, 0xd1, 0xf0,
	0x24, 0x33, 0x62, 0x72, 0x82, 0x09, 0x0a, 0x16,
	0x17, 0x18, 0x19, 0x1a, 0x25, 0x26, 0x27, 0x28,
	0x29, 0x2a, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
	0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
	0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59,
	0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
	0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
	0x7a, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89,
	0x8a, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
	0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
	0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6,
	0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3, 0xc4, 0xc5,
	0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2, 0xd3, 0xd4,
	0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xe1, 0xe2,
	0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea,
	0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xf9, 0xfa
};
 
static const unsigned char chrom_AC_counts[16] = {
	0x00, 0x02, 0x01, 0x02, 0x04, 0x04, 0x03, 0x04,
	0x07, 0x05, 0x04, 0x04, 0x00, 0x01, 0x02, 0x77
};

static const unsigned char chrom_AC_values[162] = {
	0x00, 0x01, 0x02, 0x03, 0x11, 0x04, 0x05, 0x21,
	0x31, 0x06, 0x12, 0x41, 0x51, 0x07, 0x61, 0x71,
	0x13, 0x22, 0x32, 0x81, 0x08, 0x14, 0x42, 0x91,
	0xa1, 0xb1, 0xc1, 0x09, 0x23, 0x33, 0x52, 0xf0,
	0x15, 0x62, 0x72, 0xd1, 0x0a, 0x16, 0x24, 0x34,
	0xe1, 0x25, 0xf1, 0x17, 0x18, 0x19, 0x1a, 0x26,
	0x27, 0x28, 0x29, 0x2a, 0x35, 0x36, 0x37, 0x38,
	0x39, 0x3a, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
	0x49, 0x4a, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
	0x59, 0x5a, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
	0x69, 0x6a, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
	0x79, 0x7a, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
	0x88, 0x89, 0x8a, 0x92, 0x93, 0x94, 0x95, 0x96,
	0x97, 0x98, 0x99, 0x9a, 0xa2, 0xa3, 0xa4, 0xa5,
	0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xb2, 0xb3, 0xb4,
	0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xc2, 0xc3,
	0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xd2,
	0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda,
	0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9,
	0xea, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8,
	0xf9, 0xfa
};

static const unsigned char hpojTSeriesToneMap[]={
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x01,0x02,0x02,0x04,0x05,0x06,0x07,
	0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,
	0x0F,0x10,0x11,0x11,0x12,0x13,0x14,0x15,
	0x17,0x18,0x1A,0x1B,0x1C,0x1C,0x1D,0x1E,
	0x1F,0x20,0x21,0x22,0x23,0x24,0x25,0x26,
	0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,
	0x2F,0x30,0x31,0x32,0x33,0x34,0x35,0x36,
	0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,
	0x3F,0x40,0x41,0x42,0x43,0x44,0x45,0x46,
	0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,
	0x4E,0x4F,0x50,0x51,0x52,0x53,0x54,0x55,
	0x56,0x57,0x58,0x5A,0x5B,0x5C,0x5D,0x5E,
	0x5F,0x60,0x61,0x62,0x63,0x64,0x65,0x66,
	0x67,0x69,0x6A,0x6A,0x6B,0x6C,0x6D,0x6E,
	0x6F,0x70,0x71,0x72,0x73,0x74,0x75,0x76,
	0x77,0x78,0x78,0x79,0x7A,0x7B,0x7C,0x7D,
	0x7E,0x7F,0x80,0x81,0x82,0x83,0x84,0x85,
	0x86,0x87,0x88,0x89,0x8A,0x8A,0x8B,0x8C,
	0x8D,0x8E,0x8F,0x90,0x91,0x92,0x92,0x93,
	0x94,0x95,0x96,0x97,0x98,0x99,0x99,0x9A,
	0x9B,0x9C,0x9D,0x9E,0x9F,0xA0,0xA1,0xA2,
	0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,
	0xAB,0xAB,0xAC,0xAD,0xAE,0xAF,0xB0,0xB1,
	0xB2,0xB3,0xB4,0xB5,0xB6,0xB7,0xB8,0xB9,
	0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,0xC0,0xC0,
	0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,
	0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xD0,0xD1,
	0xD2,0xD4,0xD5,0xD6,0xD8,0xD9,0xDA,0xDC,
	0xDD,0xDE,0xE0,0xE1,0xE2,0xE4,0xE5,0xE7,
	0xE8,0xE9,0xEB,0xEC,0xEE,0xEF,0xF0,0xF2,

	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x01,0x02,0x03,0x04,0x05,
	0x06,0x07,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,
	0x0F,0x10,0x11,0x12,0x13,0x14,0x15,0x16,
	0x17,0x18,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
	0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,
	0x29,0x2A,0x2B,0x2C,0x2E,0x2F,0x30,0x31,
	0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,
	0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,0x41,0x42,
	0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,
	0x4B,0x4C,0x4D,0x4D,0x4E,0x4F,0x50,0x51,
	0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,
	0x5A,0x5B,0x5D,0x5E,0x5F,0x60,0x61,0x62,
	0x63,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,
	0x6C,0x6D,0x6E,0x6F,0x70,0x71,0x72,0x73,
	0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,
	0x7C,0x7C,0x7D,0x7E,0x7F,0x80,0x81,0x82,
	0x83,0x84,0x85,0x86,0x86,0x87,0x88,0x89,
	0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,0x90,0x91,
	0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,
	0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,0xA0,0xA1,
	0xA2,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,
	0xA9,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,0xB0,
	0xB1,0xB2,0xB3,0xB3,0xB4,0xB5,0xB6,0xB7,
	0xB8,0xB9,0xBA,0xBB,0xBC,0xBD,0xBD,0xBE,
	0xBF,0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,
	0xC6,0xC7,0xC8,0xC9,0xCA,0xCC,0xCD,0xCE,
	0xCF,0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xD6,
	0xD7,0xD8,0xD9,0xDA,0xDC,0xDD,0xDE,0xDF,
	0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
	0xE8,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,0xF0,
	0xF1,0xF2,0xF3,0xF4,0xF6,0xF7,0xF8,0xF9,

	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x01,0x02,0x03,0x04,0x05,
	0x06,0x07,0x08,0x09,0x0A,0x0C,0x0D,0x0E,
	0x0F,0x10,0x11,0x11,0x12,0x14,0x15,0x16,
	0x17,0x18,0x19,0x1A,0x1B,0x1D,0x1E,0x1F,
	0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,
	0x28,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,0x30,
	0x31,0x32,0x33,0x35,0x36,0x37,0x38,0x39,
	0x3A,0x3C,0x3D,0x3E,0x3F,0x40,0x41,0x42,
	0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,
	0x4C,0x4D,0x4E,0x4F,0x50,0x51,0x51,0x52,
	0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,
	0x5C,0x5D,0x5F,0x60,0x61,0x63,0x64,0x65,
	0x66,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,
	0x6F,0x70,0x71,0x72,0x73,0x74,0x75,0x76,
	0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E,
	0x7F,0x80,0x81,0x82,0x83,0x84,0x85,0x86,
	0x87,0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,
	0x8F,0x90,0x92,0x93,0x94,0x95,0x96,0x97,
	0x98,0x99,0x9A,0x9B,0x9C,0x9E,0x9F,0xA0,
	0xA1,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,
	0xA9,0xAB,0xAC,0xAD,0xAE,0xAF,0xB0,0xB1,
	0xB2,0xB3,0xB4,0xB6,0xB7,0xB7,0xB8,0xB8,
	0xB8,0xB9,0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,
	0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,
	0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,
	0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,
	0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,
	0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
	0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,
	0xF0,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,
	0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF
};

/*____________________________________________________________________________
 |               |                                                            |
 | scale_q_table | scales a q-table according to the q-factors                |
 |_______________|____________________________________________________________|
*/
void scale_q_table(int dc_q_factor,int ac_q_factor,int ident,
    unsigned char *out) {
	static const int Q_DEFAULT=20;
	static const int FINAL_DC_INDEX=9;
	int i,val;
	int q=dc_q_factor;
	const unsigned char *in=orig_lum_quant;
	if (ident) in=orig_chrom_quant;
 
	for (i=0; i<64; i++) {
		val = ((*in++)*q + Q_DEFAULT/2) / Q_DEFAULT;
		if (val < 1)   val = 1;
		if (val > 255) val = 255;
		*out++ = (unsigned char)val;
		if (i == FINAL_DC_INDEX) {
			q = ac_q_factor;
		}
	}
}

/* Make sure signatureString is plenty smaller than LEN_OUTPUT_PAD! */
static const char signatureString[]=
	"ptal-hp scan, http://hpoj.sourceforge.net";

#define MYWRITEBUF(data,datalen) \
	do { \
		if (inlen+datalen>inmax) { \
			PTAL_LOG_ERROR("jpeg: Attempted buffer overrun, " \
				"inlen=%d, datalen=%d, inmax=%d!\n", \
				inlen,datalen,inmax); \
		} \
		memcpy(in+inlen,(char *)(data),datalen); \
		inlen+=datalen; \
	} while(0)
#define MYWRITE(data) MYWRITEBUF(&(data),sizeof(data))
#define MYWRITESTR(s) MYWRITEBUF(&(s),strlen(s))

#include "xjpg_dec.c"

void jpegStart(int filefd,int *pJpegFd,int rcCmdline,int rcDelta) {
	int pipefds[2],jpegfds[2];	/* [0]=read, [1]=write. */
	int pipefd,jpegfd,r,imax,i,j,x,y,len,done=0,ateof=0,inHeader=1;
	DWORD inmax,inmin,outmin,inlen=0,inused,innext,outlen=0,outused,outthis;
	IP_IMAGE_TRAITS inTraits,outTraits;
	IP_XFORM_HANDLE hxform;
	char *inbuf,*in,*outbuf=0;
	int rowCount,rcSof=0,rcDnl,rcMfpdtf,rcCountup,rcSofOffset;
	struct {
		unsigned char soi[2];		/* 0xFF, 0xD8 */

		unsigned char app1[2];		/* 0xFF, 0xE1 */
		unsigned char app1Length[2];	/* 0x00, 0x12 */
		unsigned char height[2];
		unsigned char width[2];
		unsigned char xres[2];
		unsigned char yres[2];
		unsigned char ac_q_factor;
		unsigned char numComponents;	/* 1=gray, 3=color */
		unsigned char xSampleFactors[2];
		unsigned char ySampleFactors[2];
		unsigned char dc_q_factor;
		unsigned char reserved;		/* 0x00 */
	} inputHeader;
	struct outputSoiApp0_t {
		unsigned char soi[2];		/* 0xFF, 0xD8 */

		unsigned char app0[2];		/* 0xFF, 0xE0 */
		unsigned char app0Length[2];	/* 0x00, 0x10 */
		unsigned char jfif[5];		/* "JFIF\0" */
		unsigned char majorVersion;	/* 0x01 */
		unsigned char minorVersion;	/* 0x00 */
		unsigned char units;		/* 0x01 = DPI */
		unsigned char xres[2];
		unsigned char yres[2];
		unsigned char xthumb;		/* 0x00 */
		unsigned char ythumb;		/* 0x00 */
	} outputSoiApp0,*pInputSoiApp0=(struct outputSoiApp0_t *)&inputHeader;
	struct {
		unsigned char com[2];		/* 0xFF, 0xFE */
		unsigned char comLength[2];	/* 0x00, 0x?? */
	} outputCom;
	struct {
		unsigned char sof0[2];		/* 0xFF, 0xC0 */
		unsigned char sof0Length[2];	/* 0x00, 0x?? */
		unsigned char eight;		/* 0x08 */
		unsigned char height[2];
		unsigned char width[2];
		unsigned char numComponents;	/* 1=gray, 3=color */
	} outputSof0Part1;
	struct {
		unsigned char iComponent;
		unsigned char xySampleFactors;
		unsigned char isNotFirstComponent;
	} outputSofComponent;
	struct {
		unsigned char dqt[2];		/* 0xFF, 0xDB */
		unsigned char dqtLength[2];	/* 0x00, 0x43 */
		unsigned char ident;		/* 0=lum., 1=chrom. */
		unsigned char elements[64];
	} outputDqt;
	struct {
		unsigned char dht[2];		/* 0xFF, 0xC4 */
		unsigned char dhtLength[2];	/* 0x00, 0x?? */
	} outputDhtPart1;
	struct {
		unsigned char hclass_ident;
		unsigned char counts[16];
		/* Variable-length huffval table follows. */
	} outputDhtPart2;
	static const struct {
		unsigned char hclass_ident;
		const unsigned char *counts;
		const unsigned char *huffval;
	} dhtInfo[4]={
		{0x00,lum_DC_counts,lum_DC_values},
		{0x10,lum_AC_counts,lum_AC_values},
		{0x01,chrom_DC_counts,chrom_DC_values},
		{0x11,chrom_AC_counts,chrom_AC_values}
	};
	int dhtCountCounts[4];
	struct {
		unsigned char sos[2];		/* 0xFF, 0xDA */
		unsigned char sosLength[2];	/* 0x00, 0x?? */
		unsigned char numComponents;	/* 1=gray, 3=color */
	} outputSosPart1;
	struct {
		unsigned char iComponent;
		unsigned char x00x11;		/* (i==0 ? 0x00 : 0x11) */
	} outputSosComponent;
	struct {
		unsigned char zero1;		/* 0x00 */
		unsigned char sixtythree;	/* 0x3F */
		unsigned char zero2;		/* 0x00 */
	} outputSosPart2;

	/* Fork off a child process to handle the JPEG data, which is
	 * passed via anonymous pipes. */
	if (pipe(pipefds)<0) {
		PTAL_LOG_ERROR("jpeg: pipe(pipefds) failed!\n");
		return;
	}
	if (pipe(jpegfds)<0) {
		PTAL_LOG_ERROR("jpeg: pipe(jpegfds) failed!\n");
		return;
	}
	r=fork();
	if (r<0) {
		PTAL_LOG_ERROR("jpeg: fork() failed!\n");
		return;
	} else if (r) {
		if (dup2(pipefds[1],filefd)<0) {
			PTAL_LOG_ERROR("jpeg: dup2() failed!\n");
			return;
		}
		close(pipefds[1]);
		close(pipefds[0]);
		*pJpegFd=jpegfds[1];
		close(jpegfds[0]);
		return;
	}
	pipefd=pipefds[0];
	close(pipefds[1]);
	jpegfd=jpegfds[0];
	close(jpegfds[1]);

	/* Open JPEG decoder. */
	r=jpgDecode_openXform(&hxform);
	if (r!=IP_DONE) {
		PTAL_LOG_ERROR("jpeg: jpgDecode_openXform "
			"returns 0x%4.4X!\n",r);
		goto abort;
	}

	/* Allocate JPEG decoder input buffer. */
	r=jpgDecode_getHeaderBufSize(hxform,&inmax);
	if (r!=IP_DONE) {
		PTAL_LOG_ERROR("jpeg: jpgDecode_getHeaderBufSize "
			"returns 0x%4.4X!\n",r);
		goto abort;
	}
	PTAL_LOG_DEBUG("jpeg: jpgDecode_getHeaderBufSize(inmax=%d) "
		"returns 0x%4.4X.\n",inmax,r);
	in=inbuf=malloc(inmax+IP_FUDGE_FACTOR);

	/* Read and validate APP1 (OfficeJet short header) record.
	 * If we actually get a JFIF standard APP0 record, then
	 * skip parsing the header. */
	r=read(pipefd,(char *)&inputHeader,sizeof(inputHeader));
	if (r!=sizeof(inputHeader)) {
		PTAL_LOG_ERROR("jpeg: read(inputHeader) returns %d!\n",r);
		if (r<=0) goto abort;
		goto skipParse;
	}
	if (inputHeader.soi[0]!=0xFF || inputHeader.soi[1]!=0xD8 ||
	    inputHeader.app1[0]!=0xFF || inputHeader.app1[1]!=0xE1 ||
	    inputHeader.app1Length[0] || inputHeader.app1Length[1]!=0x12) {
		if (pInputSoiApp0->soi[0]==0xFF &&
		    pInputSoiApp0->soi[1]==0xD8 &&
		    pInputSoiApp0->app0[0]==0xFF &&
		    pInputSoiApp0->app0[1]==0xE0 &&
		    pInputSoiApp0->app0Length[0]==0x00 &&
		    pInputSoiApp0->app0Length[1]==sizeof(outputSoiApp0)-4 &&
		    !memcmp(pInputSoiApp0->jfif,"JFIF",5)) {
			PTAL_LOG_DEBUG(
				"jpeg: Encountered standard JFIF header.\n");
		} else {
			PTAL_LOG_ERROR("jpeg: Unrecognized inputHeader!\n");
		}
skipParse:
		MYWRITEBUF(&inputHeader,r);
		goto skippedParse;
	}

	/* Write standard JPEG/JFIF records... */

	/* Start Of Image record. */
	outputSoiApp0.soi[0]=0xFF;
	outputSoiApp0.soi[1]=0xD8;
	/* APP0 (JFIF header) record. */
	outputSoiApp0.app0[0]=0xFF;
	outputSoiApp0.app0[1]=0xE0;
	outputSoiApp0.app0Length[0]=0x00;
	outputSoiApp0.app0Length[1]=sizeof(outputSoiApp0)-4;
	strcpy((char *)outputSoiApp0.jfif,"JFIF");
	outputSoiApp0.majorVersion=0x01;
	outputSoiApp0.minorVersion=0x00;
	outputSoiApp0.units=0x01;
	outputSoiApp0.xres[0]=inputHeader.xres[0];
	outputSoiApp0.xres[1]=inputHeader.xres[1];
	outputSoiApp0.yres[0]=inputHeader.yres[0];
	outputSoiApp0.yres[1]=inputHeader.yres[1];
	outputSoiApp0.xthumb=0;
	outputSoiApp0.ythumb=0;
	MYWRITE(outputSoiApp0);

	/* Comment record. */
	outputCom.com[0]=0xFF;
	outputCom.com[1]=0xFE;
	outputCom.comLength[0]=0;
	outputCom.comLength[1]=sizeof(outputCom)-2+strlen(signatureString);
	MYWRITE(outputCom);
	MYWRITESTR(signatureString);

	/* Start Of Frame record. */
	outputSof0Part1.sof0[0]=0xFF;
	outputSof0Part1.sof0[1]=0xC0;
	outputSof0Part1.sof0Length[0]=0x00;
	outputSof0Part1.sof0Length[1]=sizeof(outputSof0Part1)-2+
		inputHeader.numComponents*sizeof(outputSofComponent);
	outputSof0Part1.eight=0x08;
	outputSof0Part1.height[0]=inputHeader.height[0];
	outputSof0Part1.height[1]=inputHeader.height[1];
	outputSof0Part1.width[0]=inputHeader.width[0];
	outputSof0Part1.width[1]=inputHeader.width[1];
	outputSof0Part1.numComponents=inputHeader.numComponents;
	MYWRITE(outputSof0Part1);
	x=BEND_GET_SHORT(inputHeader.xSampleFactors);
	y=BEND_GET_SHORT(inputHeader.ySampleFactors);
	for (i=0;i<inputHeader.numComponents;i++) {
		outputSofComponent.iComponent=i;
		outputSofComponent.xySampleFactors=
			(((x>>(4*(3-i)))&0x0F)<<4) | ((y>>(4*(3-i)))&0x0F);
		outputSofComponent.isNotFirstComponent=(!i?0:1);
		MYWRITE(outputSofComponent);
	}

	/* Define Quantization Table record. */
	outputDqt.dqt[0]=0xFF;
	outputDqt.dqt[1]=0xDB;
	outputDqt.dqtLength[0]=0x00;
	outputDqt.dqtLength[1]=sizeof(outputDqt)-2;
	imax=inputHeader.numComponents>1?1:0;
	for (i=0;i<=imax;i++) {
		outputDqt.ident=i;	/* Upper nibble=0 for 8-bit table. */
		scale_q_table(inputHeader.dc_q_factor,inputHeader.ac_q_factor,
			i,outputDqt.elements);
		MYWRITE(outputDqt);
	}
	imax=inputHeader.numComponents>1?4:2;
	x=sizeof(outputDhtPart1)-2;
	for (i=0;i<imax;i++) {
		dhtCountCounts[i]=0;
		x+=sizeof(outputDhtPart2);
		for (j=0;j<16;j++) {
			y=dhtInfo[i].counts[j];
			dhtCountCounts[i]+=y;
			x+=y;
		}
	}
	outputDhtPart1.dht[0]=0xFF;
	outputDhtPart1.dht[1]=0xC4;
	BEND_SET_SHORT(outputDhtPart1.dhtLength,x);
	MYWRITE(outputDhtPart1);
	for (i=0;i<imax;i++) {
		outputDhtPart2.hclass_ident=dhtInfo[i].hclass_ident;
		memcpy(outputDhtPart2.counts,dhtInfo[i].counts,16);
		MYWRITE(outputDhtPart2);
		MYWRITEBUF(dhtInfo[i].huffval,dhtCountCounts[i]);
	}

	/* Start Of Scan record. */
	imax=inputHeader.numComponents;
	outputSosPart1.sos[0]=0xFF;
	outputSosPart1.sos[1]=0xDA;
	outputSosPart1.sosLength[0]=0;
	outputSosPart1.sosLength[1]=sizeof(outputSosPart1)-2+
		imax*sizeof(outputSosComponent)+sizeof(outputSosPart2);
	outputSosPart1.numComponents=imax;
	MYWRITE(outputSosPart1);
	for (i=0;i<imax;i++) {
		outputSosComponent.iComponent=i;
		outputSosComponent.x00x11=(!i?0x00:0x11);
		MYWRITE(outputSosComponent);
	}
	outputSosPart2.zero1=0;
	outputSosPart2.sixtythree=63;
	outputSosPart2.zero2=0;
	MYWRITE(outputSosPart2);

skippedParse:
	/* Our main loop that reads data, runs it through the JPEG
	 * decoder to process the header and count the lines (but
	 * discarding the semi-decompressed data), and writes the
	 * still-compressed data to the output file. */
	do {
		/* Since the JPEG decoder requires at least a certain
		 * amount of input data, shift down any undersized
		 * remaining data to the beginning of the buffer. */
		if (inlen<inmin && in+inlen==inbuf+inmax) {
			if (inlen>0) {
				PTAL_LOG_DEBUG("jpeg: Shifting down "
					"%d bytes in input buffer.\n",inlen);
				memmove(inbuf,in,inlen);
			}
			in=inbuf;
		}

		/* Read more data there's room at the end. */
		if (in==inbuf) {
			/* We have to read in a loop because each read
			 * completes after each write on the pipe. */
			while (inlen<inmax) {
				len=inmax-inlen;
				r=read(pipefd,inbuf+inlen,len);
				PTAL_LOG_DEBUG("jpeg: read(len=%d) "
					"returns %d, inmax=%d, inlen=%d.\n",
					len,r,inmax,inlen);
				if (r<=0) {
					ateof=1;
					break;
				} else {
					inlen+=r;
				}
			}
		} else if (ateof && inlen<=0) {
			in=0;
		}

		/* Header processing: */
		if (inHeader) {
			r=jpgDecode_getActualTraits(hxform,
				inlen,in,&inused,&innext,
				&inTraits,&outTraits);
			PTAL_LOG_DEBUG("jpeg: jpgDecode_getActualTraits("
				"inlen=%d,inused=%d,innext=%d) returns 0x%4.4X.\n",
				inlen,inused,innext,r);
			if (r&(IP_INPUT_ERROR|IP_FATAL_ERROR)) {
				PTAL_LOG_ERROR("jpeg: "
					"jpgDecode_getActualTraits "
					"returns 0x%4.4X!\n",r);
				goto abort;
			}
			if (r&IP_DONE) {
				rcSof=outTraits.lNumRows;
				r=jpgDecode_getActualBufSizes(hxform,
					&inmin,&outmin);
				PTAL_LOG_DEBUG("jpeg: "
					"jpgDecode_getActualBufSizes("
					"inmin=%d,outmin=%d) returns 0x%4.4X.\n",
					inmin,outmin,r);
				if (r&(IP_INPUT_ERROR|IP_FATAL_ERROR)) {
					PTAL_LOG_ERROR("jpeg: "
						"jpgDecode_getActualBufSizes "
						"returns 0x%4.4X!\n",r);
					goto abort;
				}
				outbuf=malloc((outlen=outmin)+IP_FUDGE_FACTOR);
				inHeader=0;
			}

		/* Data processing: */
		} else {
			r=jpgDecode_convert(hxform,inlen,in,&inused,&innext,
				outlen,outbuf,&outused,&outthis);
			PTAL_LOG_DEBUG("jpeg: jpgDecode_convert("
				"inlen=%d,inused=%d,innext=%d,outlen=%d,"
				"outused=%d,outthis=%d) returns 0x%4.4X.\n",
				inlen,inused,innext,outlen,outused,outthis,r);
			if (r&(IP_INPUT_ERROR|IP_FATAL_ERROR)) {
				PTAL_LOG_ERROR("jpeg: jpgDecode_convert "
					"returns 0x%4.4X!\n",r);
				goto abort;
			}
			if (r&IP_DONE) done=1;
		}

		/* Write to the output file any input data "consumed" by
		 * the JPEG decoder. */
		if (inused && in) {
			r=write(filefd,in,inused);
			if (r!=inused) {
				PTAL_LOG_ERROR("jpeg: write(len=%d) "
					"returns %d!\n",inused,r);
				goto abort;
			}
			in+=inused;
			inlen-=inused;
		}
	} while (!done);

	/* Get row count information recorded by the JPEG decoder. */
	r=jpgDecode_getRowCountInfo(hxform,&rcCountup,&rcDnl,&rcSofOffset);
	if (r!=IP_DONE) {
		PTAL_LOG_ERROR("jpeg: jpgDecode_getRowCountInfo"
			"returns 0x%4.4X!\n",r);
		goto abort;
	}

	/* Look for the parent process to send us a possible row count. */
	if (read(jpegfd,(char *)&rcMfpdtf,sizeof(int))!=sizeof(int)) {
		rcMfpdtf=0;
	}

	/* Pick which rowCount we'll pay attention to. */
	if (rcCmdline>=0 && rcCmdline<=0xFFFF) {
		rowCount=rcCmdline;
#if 0	/* Disabled because rcSof and rcMfpdtf are too big on LJ 3200. */
	} else if (rcSof>0 && rcSof<=0xFFFF) {
		rowCount=rcSof;
	} else if (rcDnl>0 && rcDnl<=0xFFFF) {
		rowCount=rcDnl;
	} else if (rcMfpdtf>0 && rcMfpdtf<=0xFFFF) {
		rowCount=rcMfpdtf;
#endif
	} else /* if (rcCountup>0 && rcCountup<=0xFFFF) */ {
		rowCount=rcCountup;
	}
	rowCount+=rcDelta;
	PTAL_LOG_DEBUG("jpeg: rcCmdline=%d, rcSof=%d, rcDnl=%d, rcMfpdtf=%d, "
		"rcCountup=%d,\n"
		"      rcDelta=%d, rowCount=%d, rcSofOffset=0x%4.4X.\n",
		rcCmdline,rcSof,rcDnl,rcMfpdtf,rcCountup,rcDelta,rowCount,
		rcSofOffset);

	/* Hack the rowCount into the SOF header. */
	PTAL_LOG_ERROR("jpeg: Setting image length=%d.\n",rowCount);
	if (lseek(filefd,rcSofOffset,SEEK_SET)==-1) {
		PTAL_LOG_ERROR("jpeg: lseek(SOF) failed!\n");
		goto abort;
	}
	BEND_SET_SHORT(outbuf,rowCount);
	r=write(filefd,outbuf,2);
	if (r!=2) {
		PTAL_LOG_ERROR("jpeg: write(SOF) returns %d!\n",r);
		goto abort;
	}

abort:
	exit(0);
}

void jpegEnd(int jpegfd,int rcMfpdtf) {
	write(jpegfd,(char *)&rcMfpdtf,sizeof(rcMfpdtf));
}

void jpegDone(int *fd,int *fdJpeg) {
	int status;

	close(*fd);
	*fd=PTAL_ERROR;
	if (*fdJpeg!=PTAL_ERROR) {
		close(*fdJpeg);
		*fdJpeg=PTAL_ERROR;

		wait(&status);
	}
}

#define GETOBJ_OPTIONAL(obj) ptalPmlRequestGet(obj,0)
#define GETOBJ_REQUIRED(obj,name) \
	do { \
		if (GETOBJ_OPTIONAL(obj)==PTAL_ERROR) { \
			PTAL_LOG_ERROR("Error 0x%2.2X getting %s!\n", \
				ptalPmlGetStatus(obj),name); \
			goto abort; \
		} \
	} while(0)
#define SETOBJ_OPTIONAL(obj) ptalPmlRequestSet(obj)
#define SETOBJ_REQUIRED(obj,name) \
	for (i=0;;i++) { \
		if (SETOBJ_OPTIONAL(obj)!=PTAL_ERROR) break; \
		r=ptalPmlGetStatus(obj); \
		if (i>20 || r!=PTAL_PML_ERROR_ACTION_CAN_NOT_BE_PERFORMED_NOW) { \
			PTAL_LOG_ERROR("Error 0x%2.2X setting %s!\n", \
				ptalPmlGetStatus(obj),name); \
			goto abort; \
		} \
		PTAL_LOG_DEBUG("Retrying set of %s.\n",name); \
		sleep(2); \
	}
#define SETOBJ_SCAN_UPLOAD(state) \
	do { \
		uploadState=state; \
		ptalPmlSetIntegerValue(objUploadState, \
			PTAL_PML_TYPE_ENUMERATION,uploadState); \
		SETOBJ_REQUIRED(objUploadState,"upload state"); \
	} while(0)

#define WAITOBJ_SCAN_UPLOAD(currentState,newState1,newState2) \
	for (i=0;;i++) { \
		GETOBJ_REQUIRED(objUploadState,"upload state"); \
		ptalPmlGetIntegerValue(objUploadState,0,(int *)&uploadState); \
		if (uploadState==newState1 || uploadState==newState2) break; \
		if (i>10 || uploadState!=currentState) { \
			PTAL_LOG_ERROR("Scan upload state is %d (i=%d" \
				/* ", from=%d, to=%d or %d" */ \
				")!\n", \
				uploadState,i,currentState, \
				newState1,newState2); \
			goto abort; \
		} \
		PTAL_LOG_DEBUG("Scan upload state is still %d (i=%d).\n", \
			uploadState,i); \
		sleep(1); \
	}

#define OPEN_SCAN_CHANNEL() \
	do { \
		if (ptalChannelOpen(chan)==PTAL_ERROR) { \
			PTAL_LOG_ERROR("Error opening scan channel!\n"); \
			goto abort; \
		} \
	} while(0)

#define LEN_OUTPUT_PAD 100

#define READ(buffer,count) \
	do { \
		PTAL_LOG_DEBUG("Reading %d bytes at offset=0x%8.8X.\n", \
			count,offset); \
		r=ptalChannelReadTimeout(chan,(unsigned char *)buffer,count,pReadTimeout,pReadTimeout); \
		if (r!=count) { \
			PTAL_LOG_ERROR("ptalChannelRead returns %d!\n",r); \
			goto abort; \
		} \
		if (fdMfpdtf>=0) write(fdMfpdtf,buffer,count); \
		blockRemaining-=count; \
		rasterRemaining-=count; \
		offset+=count; \
	} while(0)
#define VAR_READ(count) \
	do { \
		PTAL_LOG_DEBUG("Reading %d bytes at offset=0x%8.8X.\n", \
			count,offset); \
		r=ptalChannelReadTimeout(chan,buffer,count,pReadTimeout,pReadTimeout); \
		if (r<=0) { \
			PTAL_LOG_ERROR("ptalChannelRead returns %d!\n",r); \
			goto abort; \
		} \
		if (r!=count) { \
			PTAL_LOG_DEBUG("Actually read %d bytes.\n",r); \
			count=r; \
		} \
		if (fdMfpdtf>=0) write(fdMfpdtf,buffer,count); \
		blockRemaining-=count; \
		rasterRemaining-=count; \
		offset+=count; \
	} while(0)

#define LEN_BUFFER 4096

int handleScan(ptalDevice_t dev,int arg,int argc,char **argv) {
	/* PML objects and values: */

	ptalPmlObject_t objUploadTimeout=ptalPmlAllocateID(dev,
		"\x1\x1\x1\x12");
	int uploadTimeout=45;		/* TODO: Adjust? */

	ptalPmlObject_t objContrast=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x1");
	int contrast=50;

	ptalPmlObject_t objResolution=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x2");
	int defres=100,xres,yres;
	int bwdefres=0;
	struct {
		unsigned char x[4];
		unsigned char y[4];
	} resolution;

	ptalPmlObject_t objPixelDataType=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x3");
	enum {
		BW=1,
		BWHT=2,
		GRAY=8,
		COLOR=24
	} defaultPixelDataType=COLOR,pixelDataType=defaultPixelDataType;
	char *sDefaultPixelDataType;

	ptalPmlObject_t objCompression=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x4");
	enum {
		NO_COMPRESSION=1,
		DEFAULT_COMPRESSION=2,
		MH_COMPRESSION=3,
		MR_COMPRESSION=4,
		MMR_COMPRESSION=5,
		JPEG_COMPRESSION=6
	} compression=NO_COMPRESSION;

	ptalPmlObject_t objCompressionFactor=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x5");
	int compressionFactor=0;

	ptalPmlObject_t objUploadError=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x6");
	int uploadError=0;

	ptalPmlObject_t objUploadState=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\xC");
	enum {
		IDLE=1,
		START=2,
		ACTIVE=3,
		ABORTED=4,
		DONE=5,
		NEWPAGE=6
	} uploadState=IDLE;

	ptalPmlObject_t objAbcThresholds=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\xE");
	struct {
		unsigned char ceiling;
		unsigned char floor;
	} abcThresholds={ 0xE6,0x00 };

	ptalPmlObject_t objSharpeningCoefficient=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\xF");
	int sharpeningCoefficient=0x37;

	ptalPmlObject_t objNeutralClipThresholds=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x1F");
	struct {
		unsigned char luminance;
		unsigned char chrominance;
	} neutralClipThresholds={ 0xFA,0x05 };

	ptalPmlObject_t objToneMap=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x20");
	/* The tone map is stored as a global const. */

	ptalPmlObject_t objCopierReduction=ptalPmlAllocateID(dev,
		"\x1\x5\x1\x4");
	int copierReduction=100;

#if 0
	/* TODO: Use these if necessary: */
	ptalPmlObject_t objScanToken=ptalPmlAllocateID(dev,
		"\x1\x1\x1\x19");
	ptalPmlObject_t objColorCorrectionMatrix=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x7");
	ptalPmlObject_t objLeftMargin=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x10");
	ptalPmlObject_t objRightMargin=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x11");
	ptalPmlObject_t objHeight=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x32");
	ptalPmlObject_t objPadImage=ptalPmlAllocateID(dev,
		"\x1\x2\x2\x1\x36");
#endif

	/* General variables for this function: */
	int retcode=RETCODE_DEVICE_ERROR,futureRetcode=retcode,abortState=0;
	unsigned char buffer[LEN_BUFFER];

	/* Input: PTAL scan channel: */
	int openFirst=0;
	int closeBetweenPages=0;
	int prematureStateChange=0;
	ptalChannel_t chan=ptalChannelAllocate(dev);
	/* TODO: Adjust? */
	struct timeval readTimeout={30,0},*pReadTimeout=&readTimeout;
	struct timeval shortReadTimeout={1,0},*pShortReadTimeout;
	int maxShortReadTimeouts=30;

	/* Output: file(s) and related settings: */
	int keepOnError=0;
	int multiPage=0;
	int pageNumber=1;
	int pageDigits=2;
	int pageCount=0;
	char *outputFilenameBase=0;
	char *dot;
	int dotlen;
	char *outputFilenamePattern=0;
	char *outputFilename=0;
	int saveMfpdtfInput=0;
	char *mfpdtfFilename=0;
	int fd=PTAL_ERROR,fdMfpdtf=PTAL_ERROR,fdJpeg=PTAL_ERROR;
	int jpegOnlyGrayColor=0,jpegAddHeader=1,rcCmdline=-1,rcDelta=0;

	/* MFPDTF-related variables: */
	enum {
		RASTER_BITMAP=0,
		RASTER_GRAYMAP=1,
		RASTER_MH=2,
		RASTER_MR=3,
		RASTER_MMR=4,
		RASTER_RGB=5,
		RASTER_YCC411=6,
		RASTER_JPEG=7,
		RASTER_PCL=8,
		RASTER_NOT=9
	} expectedEncoding=RASTER_NOT;
	struct {
		unsigned char blockLength[4];	/* Includes header(s). */
		unsigned char headerLength[2];	/* Just header(s). */
		unsigned char dataType;
		unsigned char pageFlags;
	} fixedHeader;
#define FHFLAG_NEW_PAGE 1
#define FHFLAG_END_PAGE 2
#define FHFLAG_NEW_DOCUMENT 4
#define FHFLAG_END_DOCUMENT 8
#define FHFLAG_END_STREAM 16
	struct {
		unsigned char ID;
	} record;
#define ID_START_PAGE 0
#define ID_RASTER_DATA 1
#define ID_END_PAGE 2
	struct {
		unsigned char encoding;
		unsigned char pageNumber[2];
		struct {
			unsigned char pixelsPerRow[2];
			unsigned char bitsPerPixel[2];
			unsigned char rowsThisPage[4];
			unsigned char xres[4];
			unsigned char yres[4];
		} black,color;
	} startPage;
	struct {
		unsigned char unused;
		unsigned char byteCount[2];
	} rasterData;
	struct {
		unsigned char unused[3];
		struct {
			unsigned char numberOfRows[4];
		} black,color;
	} endPage;
	int blockLength,headerLength,blackRows,colorRows,rcMfpdtf;
	int offset=0,blockRemaining,rasterRemaining,datalen,i,r;

	/* Special-case some peripherals with different defaults. */
	if (ptalDeviceGetDeviceIDString(dev,(char *)buffer,
	     LEN_BUFFER)!=PTAL_ERROR) {
		if (strstr((char *)buffer,"LaserJet")) {
			/* The LaserJets require open-first/start-second. */
			openFirst=1;

			/* The LaserJet 1100A doesn't support color. */
			if (strstr((char *)buffer,"LaserJet 1100")) {
				defaultPixelDataType=GRAY;
			}

		} else if (strstr((char *)buffer,"OfficeJet") ||
		    strstr((char *)buffer,"Printer/Scanner/Copier 300")) {
			/* The OfficeJets require start-first/open-second. */
			openFirst=0;

			/* The OfficeJets can only do JPEG gray/color scans,
			 * and need the header to be added in software. */
			jpegOnlyGrayColor=1;
			jpegAddHeader=1;

			/* The OfficeJets can only scan in BW/BWHT at 300dpi. */
			bwdefres=300;

			/* Exceptions: */
			/* For the 500 series, default to
			 * "-gray -res 300 -raw". */
			if (strstr((char *)buffer,"OfficeJet Series 500")) {
				defaultPixelDataType=GRAY;
				defres=300;
				jpegOnlyGrayColor=0;

			/* For the 300 series, default to "-bw -res 300". */
			/* TODO: Extend this to C2890A and LX. */
			/* TODO: Figure out why I can't open the scan
			 * channel unless I slow things down by turning
			 * on ptal-mlcd's debug messages. */
			/* TODO: Handle the older data transfer format
			 * used by these models. */
			} else if (strstr((char *)buffer,"OfficeJet Series 300")) {
				defaultPixelDataType=BW;
				defres=300;
				jpegOnlyGrayColor=0;
			}

		} else {
			PTAL_LOG_ERROR("Unknown device!\n");
			goto abort;
		}
	}
	pixelDataType=defaultPixelDataType;
	switch (defaultPixelDataType) {
	   case BW:
		sDefaultPixelDataType="bw"; break;
	   case BWHT:
		sDefaultPixelDataType="bwht"; break;
	   case GRAY:
		sDefaultPixelDataType="gray"; break;
	   case COLOR:
		sDefaultPixelDataType="color"; break;
	   default:
		sDefaultPixelDataType="???";
	}
	/* Apply default resolution to both X and Y axes. */
	xres=yres=defres;

	/* Parse remainder of command line. */
	while (argc) {
		if (!strcmp(*argv,"-bw")) {
			pixelDataType=BW;
bwres:
			if (bwdefres && (xres!=bwdefres || yres!=bwdefres)) {
				PTAL_LOG_ERROR("Setting default resolution "
					"to %d.\n",bwdefres);
				xres=yres=bwdefres;
			}

		} else if (!strcmp(*argv,"-bwht")) {
			pixelDataType=BWHT;
			goto bwres;

		} else if (!strcmp(*argv,"-gray") || !strcmp(*argv,"-grey")) {
			pixelDataType=GRAY;

		} else if (!strcmp(*argv,"-color")) {
			pixelDataType=COLOR;

		} else if (!strcmp(*argv,"-contrast")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			contrast=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-res")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			xres=yres=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-xres")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			xres=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-yres")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			yres=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-raw")) {
			compression=NO_COMPRESSION;
			jpegOnlyGrayColor=0;

		} else if (!strcmp(*argv,"-jpeg")) {
			compression=JPEG_COMPRESSION;

		} else if (!strcmp(*argv,"-compfactor")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			compressionFactor=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-length")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			if (**argv=='+') {
				rcDelta=strtol((*argv)+1,0,0);
				rcCmdline=-1;
			} else if (**argv=='-') {
				rcDelta=-strtol((*argv)+1,0,0);
				rcCmdline=-1;
			} else {
				rcDelta=0;
				rcCmdline=strtol(*argv,0,0);
			}

		} else if (!strcmp(*argv,"-o")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			outputFilenameBase=*argv;

		} else if (!strcmp(*argv,"-keep")) {
			keepOnError=1;

		} else if (!strcmp(*argv,"-multi")) {
			multiPage=1;

		} else if (!strcmp(*argv,"-pagenum")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pageNumber=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-pagenumdigits")) {
			argc--; argv++; if (argc<=0) goto syntaxError;
			pageDigits=strtol(*argv,0,0);

		} else if (!strcmp(*argv,"-mh")) {
			compression=MH_COMPRESSION;
			jpegOnlyGrayColor=0;

		} else if (!strcmp(*argv,"-mr")) {
			compression=MR_COMPRESSION;
			jpegOnlyGrayColor=0;

		} else if (!strcmp(*argv,"-mmr")) {
			compression=MMR_COMPRESSION;
			jpegOnlyGrayColor=0;

		} else if (!strcmp(*argv,"-mfpdtf")) {
			saveMfpdtfInput=1;

		} else if (!strcmp(*argv,"-openfirst")) {
			openFirst=1;

		} else if (!strcmp(*argv,"-opensecond")) {
			openFirst=0;

		} else if (!strcmp(*argv,"-parsejpeg")) {
			jpegAddHeader=1;

		} else if (!strcmp(*argv,"-noparsejpeg")) {
			jpegAddHeader=0;

		} else if (!strcmp(*argv,"-closebetweenpages")) {
			closeBetweenPages=1;

		} else if (!strcmp(*argv,"-noclosebetweenpages")) {
			closeBetweenPages=0;

		} else {
syntaxError:
			PTAL_LOG_ERROR(
"Syntax: ... scan [<settings>...]\n"
"Valid settings:\n"
"  -bw, -bwht, -gray, -color       -- Scan mode (default=%s)\n"
"  -contrast <contrast>            -- Contrast (0-100, default=50)\n"
"  -res <xy>, -xres <x>, -yres <y> -- Resolution (default=%d DPI)\n"
"  -raw, -jpeg                     -- Compression (default varies)\n"
"  -compfactor <n>                 -- JPEG compression factor (0-100, default=0)\n"
"  -length [+|-]<n>                -- JPEG image length or +/- delta\n"
"  -o <filename[.ext]>             -- Output file (default=out.pnm or out.jpg)\n"
"  -keep                           -- Don't delete output file on error\n"
"  -multi                          -- Scan multiple pages\n"
"  -pagenum <n>                    -- First page number (default=1)\n"
"  -pagenumdigits <n>              -- Digits for page number (default=2)\n"
/* Hide these debugging/development options:
"  -mh, -mr, -mmr                  -- Unsupported compression types\n"
"  -mfpdtf                         -- Save MFPDTF stream\n"
"  -openfirst, -opensecond         -- Reorders scan start and open steps\n"
"  -[no]parsejpeg                  -- Overrides JPEG parsing\n"
"  -[no]closebetweenpages          -- Overrides closing between pages\n"
*/
"\n"
				,sDefaultPixelDataType,defres);

			/* Query/display scanner status while we're at it. */
			handleScanStatus(dev,arg,argc,argv);

			retcode=RETCODE_SYNTAX_ERROR;
			goto abort;
		}
		argc--; argv++;
	}

	/* Reset the scanner. */
	SETOBJ_SCAN_UPLOAD(IDLE);

	/* Set upload timeout. */
	ptalPmlSetIntegerValue(objUploadTimeout,PTAL_PML_TYPE_SIGNED_INTEGER,
		uploadTimeout);
	SETOBJ_OPTIONAL(objUploadTimeout);

	/* Set pixel data type. */
	ptalPmlSetIntegerValue(objPixelDataType,PTAL_PML_TYPE_ENUMERATION,
		pixelDataType);
	SETOBJ_REQUIRED(objPixelDataType,"pixel data type");

	/* Set contrast. */
	ptalPmlSetIntegerValue(objContrast,PTAL_PML_TYPE_SIGNED_INTEGER,
		contrast);
	SETOBJ_REQUIRED(objContrast,"contrast");

	/* Set resolution. */
	BEND_SET_LONG(resolution.x,xres<<16);
	BEND_SET_LONG(resolution.y,yres<<16);
	ptalPmlSetValue(objResolution,PTAL_PML_TYPE_BINARY,
		(char *)&resolution,sizeof(resolution));
	SETOBJ_REQUIRED(objResolution,"resolution");
	if (ptalPmlGetStatus(objResolution)==
	     PTAL_PML_OK_NEAREST_LEGAL_VALUE_SUBSTITUTED) {
		PTAL_LOG_ERROR("Using nearest supported resolution.\n");
	}

	/* Set compression. */
	if (jpegOnlyGrayColor && compression!=JPEG_COMPRESSION &&
	    (pixelDataType==GRAY || pixelDataType==COLOR)) {
		PTAL_LOG_ERROR("Saving in JPEG instead of PNM format.\n");
		compression=JPEG_COMPRESSION;
	}
	ptalPmlSetIntegerValue(objCompression,PTAL_PML_TYPE_ENUMERATION,
		compression);
	if (compression==JPEG_COMPRESSION) {
		SETOBJ_REQUIRED(objCompression,"compression");
	} else {
		SETOBJ_OPTIONAL(objCompression);
	}

	/* Determine expectedEncoding (depends on compression setting). */
	if (pixelDataType==COLOR) {
		expectedEncoding=RASTER_RGB;
	} else if (pixelDataType==GRAY) {
		expectedEncoding=RASTER_GRAYMAP;
	} else {
		expectedEncoding=RASTER_BITMAP;
	}
	if (compression==JPEG_COMPRESSION) {
		expectedEncoding=RASTER_JPEG;
	}

	/* Set compression factor. */
	ptalPmlSetIntegerValue(objCompressionFactor,
		PTAL_PML_TYPE_SIGNED_INTEGER,compressionFactor);
	if (compression==JPEG_COMPRESSION) {
		SETOBJ_REQUIRED(objCompressionFactor,"compression factor");
	} else {
		SETOBJ_OPTIONAL(objCompressionFactor);
	}

	/* Set Automatic Background Control thresholds. */
	ptalPmlSetValue(objAbcThresholds,PTAL_PML_TYPE_BINARY,
		(char *)&abcThresholds,sizeof(abcThresholds));
	SETOBJ_OPTIONAL(objAbcThresholds);
	/* Ignore errors here. */

	/* Set sharpening coefficient. */
	ptalPmlSetIntegerValue(objSharpeningCoefficient,
		PTAL_PML_TYPE_SIGNED_INTEGER,sharpeningCoefficient);
	SETOBJ_OPTIONAL(objSharpeningCoefficient);
	/* Ignore errors here. */

	/* Set neutral clip thresholds. */
	ptalPmlSetValue(objNeutralClipThresholds,PTAL_PML_TYPE_BINARY,
		(char *)&neutralClipThresholds,sizeof(neutralClipThresholds));
	SETOBJ_OPTIONAL(objNeutralClipThresholds);
	/* Ignore errors here. */

	/* Set tone map if supported (needed for T series, breaks others). */
	if (GETOBJ_OPTIONAL(objToneMap)!=PTAL_ERROR) {
		ptalPmlSetValue(objToneMap,PTAL_PML_TYPE_BINARY,
			(char *)hpojTSeriesToneMap,3*256);
		SETOBJ_REQUIRED(objToneMap,"tone map");
	}

	/* Set copier reduction. */
	ptalPmlSetIntegerValue(objCopierReduction,PTAL_PML_TYPE_SIGNED_INTEGER,
		copierReduction);
	SETOBJ_OPTIONAL(objCopierReduction);

	/* Determine output filename pattern. */
	if (!outputFilenameBase) {
		outputFilenameBase="out";
	}
	outputFilenamePattern=malloc(strlen(outputFilenameBase)+100);
	dot=rindex(outputFilenameBase,'.');
	dotlen=strlen(outputFilenameBase);
	if (dot) {
		dotlen=dot-outputFilenameBase;
	}
	memcpy(outputFilenamePattern,outputFilenameBase,dotlen);
	outputFilenamePattern[dotlen]=0;

	if (multiPage) {
		sprintf(outputFilenamePattern+dotlen,
			"-%%%d.%dd",pageDigits,pageDigits);
	}

	if (dot &&
	    ((compression==JPEG_COMPRESSION && strcasecmp(dot,".pnm")) ||
	     (compression!=JPEG_COMPRESSION && strcasecmp(dot,".jpg")))) {
		strcat(outputFilenamePattern,dot);
	} else {
		if (compression==JPEG_COMPRESSION) {
			strcat(outputFilenamePattern,".jpg");
		} else {
			strcat(outputFilenamePattern,".pnm");
		}
	}
	outputFilename=malloc(strlen(outputFilenamePattern)+100);

	/* Prepare for open, which will happen either before or after
	 * starting the scan. */
	ptalChannelSetRemoteService(chan,PTAL_STYPE_SCAN,0,0);
	/* The OfficeJet T series needs a minimum of about 8K for reverse. */
	ptalChannelSetPacketSizes(chan,256,8192);

nextPage:
	retcode=RETCODE_DEVICE_ERROR;
	abortState=0;

	/* Open scan channel if we need to do it before starting the scan. */
	if (openFirst && (!pageCount || closeBetweenPages)) {
		OPEN_SCAN_CHANNEL();
	}

	/* Start scanning. */
	if (!prematureStateChange) SETOBJ_SCAN_UPLOAD(START);
	prematureStateChange=0;
	/* Look for a confirmation that the scan started or failed. */
	WAITOBJ_SCAN_UPLOAD(START,ACTIVE,ACTIVE);

	/* Verify compression. */
	if (GETOBJ_OPTIONAL(objCompression)!=PTAL_ERROR) {
		ptalPmlGetIntegerValue(objCompression,0,&r);
		if (r!=compression) {
			PTAL_LOG_ERROR("Failed to set compression, "
				"current value=%d!\n",r);
			goto abort;
		}
	}

	/* Open scan channel if we need to do it after starting the scan. */
	if (!openFirst && (!pageCount || closeBetweenPages)) {
		OPEN_SCAN_CHANNEL();
	}

	/* Open output file(s). */
	sprintf(outputFilename,outputFilenamePattern,pageNumber);
	PTAL_LOG_ERROR("Scanning page %d to file \"%s\".\n",
		pageNumber,outputFilename);
	fd=open(outputFilename,O_CREAT|O_RDWR|O_TRUNC,0600);
	if (fd<0) {
		PTAL_LOG_ERROR("Error creating output file!\n");
		goto abort;
	}

	if (saveMfpdtfInput) {
		/* Open MFPDTF output file if requested (for debugging). */
		mfpdtfFilename=malloc(strlen(outputFilename)+20);
		if (!mfpdtfFilename) {
			PTAL_LOG_ERROR("Error mallocing mfpdtfFilename!\n");
			goto abort;
		}
		sprintf(mfpdtfFilename,"%s.mfpdtf",outputFilename);
		fdMfpdtf=creat(mfpdtfFilename,0600);
		if (fdMfpdtf<0) {
			PTAL_LOG_ERROR("Error creating file %s!\n",
				mfpdtfFilename);
			goto abort;
		}
		free(mfpdtfFilename);
		mfpdtfFilename=0;
	}

	pShortReadTimeout=0;
    do {
	/* Get ready to read fixed header. */
	if (pShortReadTimeout) {
		PTAL_LOG_DEBUG("Awaiting fixed header.\n");
		i=0;
		while (42) {
			r=1;
			if (ptalChannelSelect(chan,&r,0,0,
			     pShortReadTimeout)>0 && r) {
				break;
			}
			GETOBJ_REQUIRED(objUploadState,"upload state");
			ptalPmlGetIntegerValue(objUploadState,0,
				(int *)&uploadState);
			if (uploadState==DONE) {
				PTAL_LOG_DEBUG("Prematurely reached "
					"DONE state.\n");
				SETOBJ_SCAN_UPLOAD(IDLE);
				futureRetcode=RETCODE_SUCCESS;
				prematureStateChange=1;
				/* Hopefully we'll get an end-of-page
				 * right after this! */

			} else if (uploadState==NEWPAGE) {
				PTAL_LOG_DEBUG("Prematurely reached "
					"NEWPAGE state.\n");
				if (multiPage) {
					SETOBJ_SCAN_UPLOAD(START);
				} else {
					SETOBJ_SCAN_UPLOAD(IDLE);
				}
				futureRetcode=RETCODE_NEW_PAGE;
				prematureStateChange=1;
				/* Hopefully we'll get an end-of-page
				 * right after this! */

			} else if (uploadState!=ACTIVE) {
				PTAL_LOG_ERROR("Unexpected upload state=%d!\n",
					uploadState);
				goto abort;
			}

			i++;
			if (i>=maxShortReadTimeouts) {
				PTAL_LOG_ERROR("Timed out waiting for data!\n");
				goto abort;
			}
		}
	}

	/* Read fixed header. */
	PTAL_LOG_DEBUG("Reading fixed header.\n");
	READ(&fixedHeader,sizeof(fixedHeader));
	blockLength=LEND_GET_LONG(fixedHeader.blockLength);
	headerLength=LEND_GET_SHORT(fixedHeader.headerLength);

	/* Read and discard variant header (if any). */
	datalen=headerLength-sizeof(fixedHeader);
	if (datalen>0) {
		PTAL_LOG_DEBUG("Reading variant header (%d bytes).\n",datalen);
		if (datalen>LEN_BUFFER) {
			PTAL_LOG_ERROR("Oversized variant header=%d!\n",
				datalen);
#if 0
			datalen=LEN_BUFFER;
#else
			goto abort;
#endif
		}
		VAR_READ(datalen);
	}

	/* Read records contained within this block. */
	blockRemaining=blockLength-headerLength;
	PTAL_LOG_DEBUG("blockRemaining=%d.\n",blockRemaining);
	while (blockRemaining) {
		/* Figure out what kind of record it is,
		 * then read the rest of the record. */
		READ(&record,sizeof(record));

		if (record.ID==ID_RASTER_DATA) {
			PTAL_LOG_DEBUG("Reading raster data record.\n");
			READ(&rasterData,sizeof(rasterData));

			/* Read and save the subsequent image data. */
			i=rasterRemaining=LEND_GET_SHORT(rasterData.byteCount);
			if (!rasterRemaining) rasterRemaining=0x10000;
			PTAL_LOG_DEBUG("Raster data byte count=%d (old=%d).\n",
				rasterRemaining,i);
			while (rasterRemaining) {
				datalen=rasterRemaining;
				if (datalen>LEN_BUFFER) datalen=LEN_BUFFER;
				PTAL_LOG_DEBUG("Reading %d bytes of "
					"raster data.\n",datalen);
				VAR_READ(datalen);
				write(fd,buffer,datalen);
			}

		} else if (record.ID==ID_START_PAGE) {
			pShortReadTimeout=&shortReadTimeout;

			PTAL_LOG_DEBUG("Reading start page record.\n");
			READ(&startPage,sizeof(startPage));
			if (startPage.encoding!=expectedEncoding) {
				PTAL_LOG_ERROR("MFPDTF encoding field=%d, "
					"expected=%d!\n",
					startPage.encoding,expectedEncoding);
			}

		    if (compression==JPEG_COMPRESSION) {
			/* Start JPEG-handling child process. */
			if (jpegAddHeader) {
				jpegStart(fd,&fdJpeg,rcCmdline,rcDelta);
			}

		    } else {
			/* Write pnm header.  Leave room for image size
			 * fields, which we write after end of page. */
			if (pixelDataType==COLOR) {
				i='6';
			} else if (pixelDataType==GRAY) {
				i='5';
			} else {
				i='4';
			}
			sprintf(buffer,"P%c\n# %s",i,signatureString);
			for (i=strlen((char *)buffer);i<LEN_OUTPUT_PAD;i++) {
				buffer[i]=' ';
			}
			write(fd,(char *)buffer,LEN_OUTPUT_PAD);
		    }

		} else if (record.ID==ID_END_PAGE) {
			PTAL_LOG_DEBUG("Reading end page record.\n");
			READ(&endPage,sizeof(endPage));
			blackRows=LEND_GET_LONG(endPage.black.numberOfRows);
			colorRows=LEND_GET_LONG(endPage.color.numberOfRows);
			if (!blackRows || blackRows!=colorRows) {
			    PTAL_LOG_DEBUG(
				"endPage.black.numberOfRows=%d, "
				"endPage.color.numberOfRows=%d!\n",
				blackRows,colorRows);
			}
			rcMfpdtf=blackRows;

		    if (compression==JPEG_COMPRESSION) {
			/* Pass MFPDTF rowCount to JPEG child process. */
			if (jpegAddHeader) {
				jpegEnd(fdJpeg,rcMfpdtf);
			}
#if 0
			/* For test purposes only. */
			write(fd,"\n",1);
#endif
		    } else {
			/* Go back and fill in image size fields in the
			 * header now that we know what they are. */
			if (pixelDataType==COLOR || pixelDataType==GRAY) {
				sprintf(buffer,"\n%d %d\n%d\n",
					LEND_GET_SHORT(
					  startPage.black.pixelsPerRow),
					rcMfpdtf,255);
			} else {
				sprintf(buffer,"\n%d %d\n",
					LEND_GET_SHORT(
					  startPage.black.pixelsPerRow),
					rcMfpdtf);
			}
			lseek(fd,LEN_OUTPUT_PAD-strlen((char *)buffer),
				SEEK_SET);
			write(fd,(char *)buffer,strlen((char *)buffer));
		    }

		} else {
			PTAL_LOG_ERROR("invalid record ID=%d!\n",record.ID);
			goto abort;
		}
	}

    } while (!pShortReadTimeout || !(fixedHeader.pageFlags&(
		FHFLAG_END_PAGE|FHFLAG_END_DOCUMENT|FHFLAG_END_STREAM)));

	/* Wait for JPEG child process to exit. */
	if (compression==JPEG_COMPRESSION && jpegAddHeader) {
		jpegDone(&fd,&fdJpeg);
	}

	/* Look for a confirmation that the scan finished. */
	if (prematureStateChange) {
		retcode=futureRetcode;
	} else {
		WAITOBJ_SCAN_UPLOAD(ACTIVE,DONE,NEWPAGE);
		if (uploadState==DONE) retcode=RETCODE_SUCCESS;
		else if (uploadState==NEWPAGE) retcode=RETCODE_NEW_PAGE;
	}

abort:
	/* Prevent endless loops in macros that jump to here. */
	if (abortState>=1) goto abort1;
	abortState=1;

	/* Check for upload error. */
	if (GETOBJ_OPTIONAL(objUploadError)!=PTAL_ERROR) {
		ptalPmlGetIntegerValue(objUploadError,0,&uploadError);
		if (uploadError==207) {
			PTAL_LOG_ERROR("Paper jam in scanner!\n");
		} else if (uploadError==208) {
			PTAL_LOG_ERROR("Scan channel was closed prematurely!\n");
		} else if (uploadError==209) {
			PTAL_LOG_ERROR("Scan was cancelled by the host!\n");
		} else if (uploadError==210) {
			PTAL_LOG_ERROR("Scan was cancelled from the device!\n");
		} else if (uploadError==211) {
			PTAL_LOG_ERROR("Document feeder is empty!\n");
			retcode=RETCODE_NO_PAGE;
		} else if (uploadError==213) {
			PTAL_LOG_ERROR("Cover is open!\n");
		} else if (uploadError==214) {
			PTAL_LOG_ERROR("Document not loaded!\n");
			retcode=RETCODE_NO_PAGE;
		} else if (uploadError) {
			PTAL_LOG_ERROR("Device reported upload error=%d!\n",
				uploadError);
		}
	}

abort1:
	/* Prevent endless loops in macros that jump to here. */
	if (abortState>=2) goto abort2;
	abortState=2;

	if (!multiPage || retcode!=RETCODE_NEW_PAGE || closeBetweenPages) {
		/* Reset the scanner. */
		SETOBJ_SCAN_UPLOAD(IDLE);
	}

abort2:
	/* Prevent endless loops in macros that jump to here. */
	/* if (abortState>=3) goto abort3; */
	abortState=3;

	if (!multiPage || retcode!=RETCODE_NEW_PAGE || closeBetweenPages) {
		/* Close scan channel. */
		ptalChannelClose(chan);
	}

	/* Close output file(s). */
	if (fd>=0) close(fd);
	if (retcode!=RETCODE_SUCCESS && retcode!=RETCODE_NEW_PAGE) {
		if (!keepOnError) unlink(outputFilename);
	}
	if (fdMfpdtf>=0) close(fdMfpdtf);

	if (retcode==RETCODE_SUCCESS || retcode==RETCODE_NEW_PAGE) {
		pageCount++;
		pageNumber++;
	}

	if (multiPage && retcode==RETCODE_NEW_PAGE) {
		goto nextPage;
	}

	/* TODO: Deallocate PML objects. */

	if (pageCount) {
		PTAL_LOG_ERROR("Successfully scanned %d page%s.\n",
			pageCount,pageCount==1?"":"s");
	}
	PTAL_LOG_DEBUG("handleScan returns %d.\n",retcode);
	return retcode;
}


struct {
	char *string;
	int (*function)(ptalDevice_t dev,int arg,int argc,char **argv);
	int arg;
} commandTable[]={
	{"device",handleDevice,0},
	{"display",handleDisplay,0},
	{"clock",handleClock,0},
	{"scan",handleScan,0},


	{0,0,0}
};

int main(int argc,char **argv) {
	char *argv0=*argv;
	ptalDevice_t dev=0;
	int i;

	if (ptalInit()==PTAL_ERROR) {
		PTAL_LOG_ERROR("ptal-pml: ptalInit() failed!\n");
		return RETCODE_DEVICE_ERROR;
	}

	argc--; argv++;
	if (argc<=0) {
syntaxError:
		PTAL_LOG_ERROR(
"Syntax: %s <dev> <command> [-help] [<params>...]\n"
"<dev> is the PTAL device name\n"
"Valid commands:\n"
			,argv0);
		for (i=0;commandTable[i].string;i++) {
			printf("\t%s\n",commandTable[i].string);
		}
		return RETCODE_SYNTAX_ERROR;
	}
	dev=ptalDeviceOpen(*argv);
	if (!dev) {
		PTAL_LOG_ERROR("ptal-hp: "
			"invalid device \"%s\"!\n",*argv);
		return RETCODE_DEVICE_ERROR;
	}
	if (ptalPmlOpen(dev)==PTAL_ERROR) {
		PTAL_LOG_ERROR("ptal-hp: error opening PML "
			"for device \"%s\"!\n",*argv);
		return RETCODE_DEVICE_ERROR;
	}

	argc--; argv++; if (argc<=0) goto syntaxError;
	for (i=0;commandTable[i].string;i++) {
		if (!strcmp(*argv,commandTable[i].string)) {
			argc--; argv++;
			return commandTable[i].function(
				dev,commandTable[i].arg,argc,argv);
		}
	}
	goto syntaxError;
}
