/* GNOME Transcript
 * Copyright (C) 1999-2000 the Free Software Foundation
 * Authors : Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <gnome.h>
#include <gtk/gtksignal.h>
#include "gtrans_db_editor.h"
#include <gtrans_ifase/gtrans_ifase.h>
#include "gtrans_button.h"


enum {
	NEW_TABLE,
	OPEN_TABLE,
	DEL_TABLE,
	REFRESH_TABLES,
	REFRESH,
	CLOSE,
	LAST_SIGNAL
};


static guint gtrans_db_editor_signals[LAST_SIGNAL] = { 0 };

static GtkWindow *parent_class = NULL;



/**
 * gtrans_db_editor_destroy:
 * @object : The GTransDbEditor
 *
 * Destroy the GTransDbEditor.
 */
static void
gtrans_db_editor_destroy(GtkObject *object)
{
	GTransDbEditor *gde;
	GnomeIconList  *gil;
	GtkWidget      *child;
	GList          *tmp;
	gchar          *data;
	gint            i;
	
	g_return_if_fail(object != NULL);
	g_return_if_fail(GTRANS_IS_DB_EDITOR(object));
	
	gde = GTRANS_DB_EDITOR(object);


	/*
	 * Destroy all the children
	 */
	for (tmp = gde->children ; tmp != NULL ; tmp = tmp->next){
		child = (GtkWidget *) tmp->data;
		gtk_widget_destroy(child);
	}
	g_list_free(gde->children);
	
	/*
	 * Free the icons attached data
	 */
	gil = GNOME_ICON_LIST(gde->table_ilist);
	for (i = 0 ; i < gil->icons ; i++){
		data = (gchar*) gnome_icon_list_get_icon_data(gil,i);
		if (data)
			g_free(data);
	}   
	
	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		(*GTK_OBJECT_CLASS(parent_class)->destroy) (object);
}




/**
 * gtrans_db_editor_class_init
 */
static void
gtrans_db_editor_class_init(GTransDbEditorClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;
	
	object_class = (GtkObjectClass *) class;
	widget_class = (GtkWidgetClass *) class;
	
	gtrans_db_editor_signals[NEW_TABLE] =
		gtk_signal_new ("new_table",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,new_table),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_db_editor_signals[OPEN_TABLE] =
		gtk_signal_new ("open_table",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,open_table),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);

	gtrans_db_editor_signals[DEL_TABLE] =
		gtk_signal_new ("del_table",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,del_table),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);

	gtrans_db_editor_signals[REFRESH_TABLES] =
		gtk_signal_new ("refresh_tables",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,refresh_tables),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);

	gtrans_db_editor_signals[REFRESH] =
		gtk_signal_new ("refresh",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,refresh),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtrans_db_editor_signals[CLOSE] =
		gtk_signal_new ("close",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (GTransDbEditorClass,close),
				gtk_signal_default_marshaller,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals(object_class, 
				     gtrans_db_editor_signals,
				     LAST_SIGNAL);
	
	parent_class = gtk_type_class(gtk_window_get_type());

	/* Destroy */
	object_class->destroy = gtrans_db_editor_destroy;
}




/*
 * gtrans_db_editor_init
 */
static void
gtrans_db_editor_init(GTransDbEditor *gde)
{
	gde->table_idata = NULL;
	gde->table_ipos  = -1;
	
	gde->children = NULL;
}




/*
 * gtrans_db_editor_get_type
 */
guint 
gtrans_db_editor_get_type()
{
	static guint gtrans_db_editor_type = 0;
	
	if(!gtrans_db_editor_type) {
		GtkTypeInfo gtrans_db_editor_info = {
			"GTransDbEditor",
			sizeof(GTransDbEditor),
			sizeof(GTransDbEditorClass),
			(GtkClassInitFunc) gtrans_db_editor_class_init,
			(GtkObjectInitFunc) gtrans_db_editor_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		
		gtrans_db_editor_type = 
			gtk_type_unique(gtk_window_get_type(),
					&gtrans_db_editor_info);
	}
	return gtrans_db_editor_type;
}



/**
 * SIGNAL EMITIONS ....
 */

/*
 * gtrans_db_editor_emit__new_table
 */
static void
gtrans_db_editor_emit__new_table(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[NEW_TABLE]);
}


/*
 * gtrans_db_editor_emit__open_table
 */
static void
gtrans_db_editor_emit__open_table(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[OPEN_TABLE]);
}


/*
 * gtrans_db_editor_emit__del_table
 */
static void
gtrans_db_editor_emit__del_table(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[DEL_TABLE]);
}



/*
 * gtrans_db_editor_emit_refresh_tables
 */
static void
gtrans_db_editor_emit_refresh__tables(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[REFRESH_TABLES]);
}


/*
 * gtrans_db_editor_emit__refresh
 */
static void
gtrans_db_editor_emit__refresh(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[REFRESH]);
}


/*
 * gtrans_db_editor_emit__close
 */
static void
gtrans_db_editor_emit__close(GtkWidget  *widget)
{
	gtk_signal_emit(GTK_OBJECT(widget),gtrans_db_editor_signals[CLOSE]);
}







/**
 * gtrans_db_editor_fill_ilist_with_tables
 * @gde : The Database Editor
 * @res : A GTransIFaseResult retriving the tables in a db.
 *
 * Fill the tabpe's page of the Database Editor with the
 * table icons.
 */
void
gtrans_db_editor_fill_with_tables(GTransDbEditor    *gde,
				  GTransIFaseResult *res)
{
	GnomeIconList *ilist;
	gchar         *pixfile;
	gint           i;
	
	g_return_if_fail(gde != NULL);
	g_return_if_fail(res != NULL);
	
	ilist = GNOME_ICON_LIST(gde->table_ilist);
	
	pixfile = g_strconcat(GNOME_TRANSCRIPT_PIXMAP_DIR,"/directory.png",NULL);
	
	for (i = 0 ; i < res->n_rows ; i++){
		gnome_icon_list_append(ilist,pixfile,
				       gtrans_ifase_result_get_item(res,i,0));
		
		gnome_icon_list_set_icon_data
			(ilist,i,g_strdup(gtrans_ifase_result_get_item(res,i,0)));
	}
	
	g_free(pixfile);
}





/**
 * gtrans_db_editor_refresh_tables
 * @gde : The Database Editor
 * @res : A GTransIFaseResult retriving the tables in a db.
 *
 * Refresh the table icons. This is, clear them all
 * and read them from the database.
 */
void
gtrans_db_editor_refresh_tables(GTransDbEditor    *gde,
				GTransIFaseResult *res)
{
	GnomeIconList *gil;
	gchar         *data;
	gint           i;
	
	g_return_if_fail(gde != NULL);
	g_return_if_fail(res != NULL);
	
	gde->table_idata = NULL;
	gde->table_ipos  = -1;

	gil = GNOME_ICON_LIST(gde->table_ilist);
	
	for (i = 0 ; i < gil->icons ; i++){
		data = (gchar*) gnome_icon_list_get_icon_data(gil,i);
		if (data)
			g_free(data);
	}
	
	gnome_icon_list_clear(gil);
	gtrans_db_editor_fill_with_tables(gde,res);
}




/**
 * gtrans_db_editor_refresh
 * @gde : The Database Editor
 * @res : A GTransIFaseResult retriving the tables in a db.
 *
 * Refresh all the icons. This is, clear them all
 * and read them from the database.
 */
void
gtrans_db_editor_refresh(GTransDbEditor    *gde,
			 GTransIFaseResult *res)
{
	g_return_if_fail(gde != NULL);
	g_return_if_fail(res != NULL);
	
	gtrans_db_editor_refresh_tables(gde,res);
}




/*
 * gtrans_db_editor_table_icon_selected
 */
static void
gtrans_db_editor_table_icon_selected(GnomeIconList  *gil,
				     gint            num,
				     GdkEvent       *event,
				     GTransDbEditor *gde)
{
	gde->table_idata = gnome_icon_list_get_icon_data(gil,num);
	gde->table_ipos  = num;
	
	if (event->type == GDK_2BUTTON_PRESS)
		gtrans_db_editor_emit__open_table(GTK_WIDGET(gde));
}



/*
 * gtrans_db_editor_table_icon_unselected
 */
static void
gtrans_db_editor_table_icon_unselected(GnomeIconList  *gil,
				       gint            num,
				       GdkEvent       *event,
				       GTransDbEditor *gde)
{
	gde->table_idata = NULL;
	gde->table_ipos  = -1;
}



/*
 * gtrans_db_editor_new
 */
GtkWidget *
gtrans_db_editor_new(GTransIFaseConn *conn)
{
	GTransDbEditor     *gde;
	GtkWidget          *vbox;
	GtkWidget          *hbox;
	
	GtkWidget          *table_box;
	GtkWidget          *table_frame;
	GtkWidget          *table_bbox;
	GtkWidget          *table_scrolled;
	GtkWidget          *new_t_button;
	GtkWidget          *del_t_button;
	GtkWidget          *open_t_button;
	
	GtkWidget          *close_button;
	GtkWidget          *refresh_button;
	GtkWidget          *sep;

	
	gde = gtk_type_new(gtrans_db_editor_get_type());
	gtk_window_set_title(GTK_WINDOW(gde),conn->db);
	gtk_container_set_border_width(GTK_CONTAINER(gde),8);
	
	vbox = gtk_vbox_new(FALSE,8);
	gtk_container_add(GTK_CONTAINER(gde),vbox);
	
	
	/*
	 * There starts the page for the table stuff 
	 */
	gde->notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox),gde->notebook,TRUE,TRUE,0);
	
	table_box = gtk_hbox_new(FALSE,8);
	gtk_container_set_border_width(GTK_CONTAINER(table_box),8);
	gtk_notebook_append_page(GTK_NOTEBOOK(gde->notebook),table_box,
				 gtk_label_new("Tables"));
	
	table_frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(table_frame),GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(table_box),table_frame,TRUE,TRUE,0);
	
	table_scrolled = gtk_scrolled_window_new(NULL,NULL);
	gtk_widget_set_usize(table_scrolled,390,160);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(table_scrolled),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(table_frame),table_scrolled);
	
	gde->table_ilist = gnome_icon_list_new(70,NULL,FALSE);
	gnome_icon_list_set_row_spacing(GNOME_ICON_LIST(gde->table_ilist),10);
	gtk_container_add(GTK_CONTAINER(table_scrolled),gde->table_ilist);
	gtk_signal_connect(GTK_OBJECT(gde->table_ilist),"select_icon",
			   GTK_SIGNAL_FUNC(gtrans_db_editor_table_icon_selected),
			   (gpointer) gde);
	gtk_signal_connect(GTK_OBJECT(gde->table_ilist),"unselect_icon",
			   GTK_SIGNAL_FUNC(gtrans_db_editor_table_icon_unselected),
			   (gpointer) gde);
	
	table_bbox = gtk_vbox_new(FALSE,5);
	gtk_box_pack_start(GTK_BOX(table_box),table_bbox,FALSE,FALSE,0);
	
	new_t_button = gtrans_button_with_stock_n_label2(GNOME_STOCK_PIXMAP_NEW,"New");
	gtk_box_pack_start(GTK_BOX(table_bbox),new_t_button,FALSE,FALSE,0);
	gtk_signal_connect_object(GTK_OBJECT(new_t_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__new_table),
				  GTK_OBJECT(gde));
	
	open_t_button = gtrans_button_with_stock_n_label2(GNOME_STOCK_PIXMAP_OPEN,"Open");
	gtk_box_pack_start(GTK_BOX(table_bbox),open_t_button,FALSE,FALSE,0);
	gtk_signal_connect_object(GTK_OBJECT(open_t_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__open_table),
				  GTK_OBJECT(gde));
	
	del_t_button = gtrans_button_with_stock_n_label2(GNOME_STOCK_PIXMAP_TRASH,"Drop");
	gtk_box_pack_start(GTK_BOX(table_bbox),del_t_button,FALSE,FALSE,0);
	gtk_signal_connect_object(GTK_OBJECT(del_t_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__del_table),
				  GTK_OBJECT(gde));
	
	/*
	 * Here ends the page for the table stuff
	 */
	
	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox),sep,FALSE,FALSE,0);
	
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox),GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox),0);
        gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,0);
	
	
	refresh_button = gtrans_button_with_stock_n_label(GNOME_STOCK_PIXMAP_REFRESH,"Refresh");
	GTK_WIDGET_SET_FLAGS(refresh_button,GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox),refresh_button,TRUE,TRUE,0);
	gtk_signal_connect_object(GTK_OBJECT(refresh_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__refresh),
				  GTK_OBJECT(gde));
	
	close_button = gtrans_button_with_stock_n_label(GNOME_STOCK_PIXMAP_CLOSE,"Close");
	GTK_WIDGET_SET_FLAGS(close_button,GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(hbox),close_button,TRUE,TRUE,0);
	gtk_widget_grab_default(close_button);
	gtk_signal_connect_object(GTK_OBJECT(close_button),"clicked",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__close),
				  GTK_OBJECT(gde));
	
	gtk_signal_connect_object(GTK_OBJECT(gde),"delete-event",
				  GTK_SIGNAL_FUNC(gtrans_db_editor_emit__close),
				  GTK_OBJECT(gde));
	
	/* Set the connection */
	gde->conn = conn;
	
	gtk_widget_show_all(vbox);
	
	return GTK_WIDGET(gde); 
}




/**
 * gtrans_db_editor_remove_table
 * @gde  : The GTransDbEditor
 * @pos  : The position of the GnomeIconListItem
 *
 * Remove the GnomeIconListItem at @pos, in the table's page. 
 */
void
gtrans_db_editor_remove_table(GTransDbEditor *gde,
			      gint            pos)
{
	gpointer data;
	
	g_return_if_fail(gde != NULL);
	g_return_if_fail(gde->table_ilist != NULL);
	g_return_if_fail(gde->table_idata != NULL);
	g_return_if_fail(gde->table_ipos != -1);
	
	data = gnome_icon_list_get_icon_data(GNOME_ICON_LIST(gde->table_ilist),pos);
	g_free(data);
	gnome_icon_list_remove(GNOME_ICON_LIST(gde->table_ilist),pos);
}





/**
 * gtrans_db_editor_add_child:
 * gde   : The GTransDbEditor
 * child : A widget
 *
 * Sets @gde to be the paren of @child, so when we destroy
 * the @gde @child will get destroyed.
 */
void
gtrans_db_editor_add_child(GTransDbEditor *gde,
			   GtkWidget      *child)
{
	
	g_return_if_fail(gde != NULL);
	g_return_if_fail(GTRANS_IS_DB_EDITOR(gde));
	
	gde->children = g_list_append(gde->children,child);
}





/**
 * gtrans_db_editor_remove_child:
 * gde   : The GTransDbEditor
 * child : A widget
 *
 * Removes @child from the  @gde children list.
 */
void
gtrans_db_editor_remove_child(GTransDbEditor *gde,
			      GtkWidget      *child)
{
	
	g_return_if_fail(gde != NULL);
	g_return_if_fail(GTRANS_IS_DB_EDITOR(gde));
	
	gde->children = g_list_remove(gde->children,child);
}

