# Description: third-party addon installation
# Author: Wijnand 'tehmaze' Modderman
# License: BSD

from gozerbot.config import config
from gozerbot.generic import geturl2, gozerpopen, rlog, touch
from gozerbot.fileutils import gunzip, bunzip2, tarextract
from gozerbot.myimport import my_import
from gozerbot.pgp import pgp, PgpNoPubkeyException, NoGPGException
import os
import sys
import md5
import shutil
import tempfile
import urllib2

# check if myplugs dir exists if not create it
if not os.path.isdir('myplugs'):
    os.mkdir('myplugs')
    touch(os.path.join('myplugs', '__init__.py')) 
if not os.path.isdir(os.path.join('myplugs', 'addons')):
    os.mkdir(os.path.join('myplugs', 'addons'))
    touch(os.path.join('myplugs', 'addons', '__init__.py')) 

addondir = os.path.abspath(os.path.join('myplugs', 'addons'))
if not addondir in sys.path:
    sys.path.append(addondir)

def allowed():
    return config.has_key('addonallow') and config['addonallow'] or 0

class Addon:
    def load(self, name, url=None, md5sig=None, pgpsig=None):
        '''
        Try to load addon ``name`` and install it if an ``url`` is specified and addon
        installation is allowed in the configuration.
        '''
        rlog(5, 'addon', 'load %s (%s)' % (name, str(url)))
        mod = None
        try:
            mod = my_import(name)
        except ImportError:
            pass
        if mod:
            return mod
        if allowed():
            if url:
                return self.install(name, url, md5sig=md5sig, pgpsig=pgpsig)
        else:
            rlog(10, 'addon', 'addon installation not allowed')
            raise ImportError('addon installation not allowed')

    def install(self, name, url, md5sig=None, pgpsig=None):
        '''
        Force installation of an addon with ``name`` from the given ``url``.
        '''
        assert md5sig or pgpsig, 'Need at least one type of signature'
        rlog(5, 'addon', 'installing %s from %s' % (name, url))
        tmpdir = tempfile.mkdtemp('addon')
        try:
            try:
                plugdata = geturl2(url)
            except urllib2.HTTPError, e:
                raise ImportError('Failed to install %s: %s' % (url, str(e)))
            if md5sig:
                md5sum = md5.md5(plugdata)
                if md5sum.hexdigest() != md5sig:
                    raise ImportError('Addon MD5 signature verification failed')
            if pgpsig:
                fingerprint = pgp.verify_signature(plugdata, pgpsig)
                if not fingerprint:
                    raise ImportError('Addon PGP signature verification failed')
            base = os.path.basename(url)
            # decompress if needed
            if base.endswith('.tgz'):
                base = base.replace('.tgz', '.tar.gz')
            if base.endswith('.tbz2'):
                base = base.replace('.tbz2', '.tar.bz2')
            if base.endswith('.gz'):
                plugdata = gunzip(plugdata)
                base = base.replace('.gz', '')
            elif base.endswith('.bz2'):
                plugdata = bunzip2(plugdata)
                base = base.replace('.bz2', '')
            # tarball installation
            if base.endswith('.tar'):
                files = tarextract(base.split('.')[0], plugdata, tmpdir, '')
                if not files:
                    raise ImportError('Invalid archive %s: no files found' % base)
                # try to find the most likely setup.py (the first child setup.py)
                setup = None
                lower = -1
                for item in files:
                    if os.path.basename(item) == 'setup.py' and (lower < 0 or len(item.split('/')) < lower):
                        setup = item
                if not setup:
                    raise ImportError('Invalid archive %s: no setup.py was found' % base)
                # run setup, if setup was ok, try the import
                if self.setup(name, setup):
                    return my_import('myplugs.addons.' + name)
                else:
                    raise ImportError('Setup of %s failed' % name)
            elif base.endswith('.py'):
                if os.path.isfile(os.path.join(addondir, base)):
                    os.rename(os.path.join(addondir, base), os.path.join(addondir, '%s~' % base))
                fh = open(os.path.join(addondir, base), 'w')
                fh.write(plugdata)
                fh.close()
                return my_import('myplugs.addons.' + name)
            raise ImportError('Can not install %s: don\'t know how' % base)
        finally:
            try:
                shutil.rmtree(tmpdir)
            except Exception, e:
                pass

    def remove(self, name):
        '''
        Remove an addon with ``name``, only names relative to ``addondir`` are accepted.
        '''
        fulladdondir = os.path.abspath(os.path.join(addondir, name))
        assert fulladdondir.startswith(addondir), 'Addon not in addondir'
        if os.path.isdir(fulladdondir):
            shutil.rmtree(fulladdondir)
            return True
        return False

    def setup(self, name, what):
        '''
        Run setup using ``what`` for addon ``name``.
        '''
        rlog(5, 'addon', 'setup for %s via %s' % (name, what))
        err = ""
        if not os.fork():
            pwd = os.getcwd()
            os.chdir(os.path.dirname(what))
            pkg = gozerpopen([sys.executable, what, 'build'])
            pkg.wait()
            err = pkg.errors.readlines()
            ret = pkg.close()
            if ret == 0:
                pkg = gozerpopen([sys.executable, what, 'install_lib', '--install-dir', addondir])
                pkg.wait()
                err = pkg.errors.readlines()
                ret = pkg.close()
            os._exit(ret)
            #raise ImportError('Could not install %s: %s' % (what, ' .. '.join(err))) 
        else:
            return os.wait()[0]

addon = Addon()
