/*
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* Create and delete interfaces and universes using the ioctl interface. */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include <dmx/dmxioctl.h>

#include <dmx/dmx.h>


static int create_interface (int dmx, const char *name, const char *options)
{
  struct dmx_createinterface_parm parm;

  strcpy (parm.name, name);
  strcpy (parm.properties, options ? options : "");

  if (ioctl(dmx, DMX_IOCTL_CREATE_INTERFACE, &parm) >= 0)
    {
      /* parm.id; return id of interface. */
      return 0;
    }
  return -1;
}

void display_interfaces_drivers (int dmx)
{
  struct dmx_info info;

  if (ioctl (dmx, DMX_IOCTL_GET_INFO, &info) >= 0 && info.num_entries > 0)
    {
      char *family_name = info.family_names;
      int i;

      printf ("dmx4linux version %d.%d\n", info.version_major, info.version_minor);
      printf ("  max-output-universes: %d\n",  info.max_out_universes);
      printf ("  max-input-universes: %d\n",   info.max_in_universes);
      printf ("  used-output-universes: %d\n", info.used_out_universes);
      printf ("  used-input-universes: %d\n",  info.used_in_universes);

      for (i=0; i<info.num_entries; i++, family_name += (strlen(family_name)+1))
	{
	  struct dmx_family_info dinfo;

	  strcpy (dinfo.familyname, family_name);
	  dinfo.offset = 0; /* start at offset 0 */

	  printf ("%s\n", family_name);

	  if (ioctl(dmx, DMX_IOCTL_GET_DRIVERS, &dinfo) >= 0 && dinfo.num_entries > 0)
	    {
	      char *driver_name = dinfo.driver_names;
	      int j;

	      for (j=0; j<dinfo.num_entries; j++, driver_name += (strlen(driver_name)+1))
		printf ("  %c- %s\n", (j<dinfo.num_entries-1)?'|':'\'', driver_name);
	    }
	}
    }
}


#if 0
get_universe_list
struct dmx_capabilities
{
    /* The following two are the primary key for pointing to a unique universe */
    int  universe;     /* index to the universe  */
    char direction;    /* DMX_IN, DMX_OUT or may be DMX_THRU */

    char family[20];   /* name of the driver that accesses the card */
    char driver[20];   /* name of the card that is used (detected)  */

    char connector[20]; /* name of the connector that the universe uses */
    int  conn_id;       /* interface wide id of the connector the universe uses */

    int  breaksize;    /* in uS */
    int  mabsize;      /* in uS */
    int  maxSlots;     /* in number of slots (channels) */
};
#endif


int main (int argc, const char **argv)
{
  const char *s=DMXdev(&argc, argv);
  int  dmx = open (s, O_RDONLY);
  if (dmx<0)
    {
      fprintf (stderr, "unable to open dmx-device\n");
      return 1;
    }

  if (argc > 1)
    {
      create_interface (dmx, argv[1], (argc > 2) ? argv[2] : NULL);
    }
  else
    {
      display_interfaces_drivers (dmx);

    }

  close (dmx);
  return 0;
}


