#! /usr/bin/perl -w

# debchange: update the debian changelog using your favorite visual editor
# Options:
# -i           generates a new changelog section Increasing the Debian
#              release number
# -a           Adds an entry to the current changelog section
# -v version   generates a new changelog section with given version number
# -p           preserve directory name
# Without any options, the package will look for an .upload file in the
# parent directory to determine whether or not the version number should
# be incremented or not.
#
# debchange -h prints a usage message.
#
# When creating a new changelog section, if either of the environment
# variables DEBEMAIL or EMAIL is set, debchange will use this as the
# uploader's email address (with the former taking precedence), and if
# DEBFULLNAME is set, it will use this as the uploader's full name.
# Otherwise, it will just copy the values from the previous changelog
# entry.
#
# Originally by Christoph Lameter <clameter@debian.org>
# Modified extensively by Julian Gilbey <jdg@debian.org>
#
# Copyright 1999 by Julian Gilbey 
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.003;  # We're using function prototypes
use Getopt::Long;
use File::Copy;
use Cwd;
use lib '@pkgdatadir@';
use checkgettext;

# Predeclare functions
sub fatal($);

setlocale(LC_MESSAGES(), "");
textdomain("devscripts");

my $progname;
($progname=$0) =~ s|.*/||;

sub usage () {
    printf gettext(<<'EOF'), $progname;
Usage: %s [options] [changelog entry]
Options:
  -i, --increment
         Increase the Debian release number, adding a new changelog entry
  -a, --append
         Append a new entry to the current changelog
  -v <version>, --newversion=<version>
         Add a new changelog entry with version number specified
  -p, --preserve
         Preserve the directory name
  -h, --help
         Display this help message and exit
  --version
         Display version information
  At most one of -a, -i and -v (or their long equivalents) may be used.
  With no options, one of -i or -a is chosen by looking for a .upload
  file in the parent directory and checking its contents.
EOF
}

sub version () {
    printf gettext(<<'EOF'), $progname, '@VERSION@';
This is %s, from the Debian devscripts package, version %s
This code is copyright 1999 by Julian Gilbey, all rights reserved.
Based on code by Christoph Lameter.
This program comes with ABSOLUTELY NO WARRANTY.
You are free to redistribute this code under the terms of the
GNU General Public License, version 2 or later.
EOF
}

# Look for the debian changelog
until (-e "debian/changelog")
{
    chdir ".." or fatal gettext("Can't chdir:") . " $!";
    fatal gettext("Cannot find debian/changelog anywhere!  " .
		  "Are you in the source code tree?")
	if cwd() eq "/";
}

# Clean up after old versions of debchange
if (-f "debian/RELEASED") {
    unlink("debian/RELEASED");
}

if ( -e "debian/changelog.dch" ) {
    fatal gettext("The backup file debian/changelog.dch already exists --\n" .
		  "please move it before trying again");
}

sub BEGIN {
    # Initialise the variable
    $tmpchk=0;
}

sub END {
    unlink "debian/changelog.dch" or
	warn gettext("Could not remove debian/changelog.dch")
	    if $tmpchk;
}

#####

# Find the current version number etc.
open PARSED, "dpkg-parsechangelog |"
    or fatal gettext("Cannot execute dpkg-parsechangelog:") . " $!";
while (<PARSED>) {
    chomp;
    if (/^(\S+):\s(.+?)\s*$/) { $changelog{$1}=$2; $last=$1; }
    elsif (/^(\S+):\s$/) { $changelog{$1}=''; $last=$1; }
    elsif (/^\s\.$/) { $changelog{$last}.="\n"; }
    elsif (/^\s(.+)$/) { $changelog{$last}.="$1\n"; }
    else {
	fatal gettext("Don't understand dpkg-parsechangelog output:") . " $_";
    }
}

close PARSED
    or fatal gettext("Problem executing dpkg-parsechangelog:") . " $!";
if ($?) { fatal gettext("dpkg-parsechangelog failed!") }

fatal gettext("No version number in changelog!")
    unless exists $changelog{'Version'};

# Is this a native Debian package, i.e., does it have a - in the
# version number?
$VERSION=$changelog{'Version'};
($SVERSION=$VERSION) =~ s/^\d+://;
$debian_native = ($VERSION =~ /-/) ? 0 : 1;

$PACKAGE=$changelog{'Source'};
($MAINTAINER,$EMAIL) = ($changelog{'Maintainer'} =~ /^([^<]+) <(.*)>/);

$MAINTAINER = $ENV{'DEBFULLNAME'} || $MAINTAINER;
$EMAIL = $ENV{'DEBEMAIL'} || $ENV{'EMAIL'} || $EMAIL;

#####

# We use config rather than the newer Configure so that this still works
# on slink machines, which have a pre-Configure Getopt::Long.
# We use bundling so that the short option behaviour is the same as
# with older debchange versions.
Getopt::Long::config('bundling');
GetOptions("h", "help" => \$opt_h,
	   "version" => \$opt_version,
	   "i", "increment" => \$opt_i,
	   "a", "append" => \$opt_a,
	   "v=s", "newversion=s" => \$opt_v,
	   "p", "preserve" => \$opt_p)
    or die sprintf
	gettext("Usage: %s [options] [changelog entry]\nRun %s --help for more details\n"),
	$progname, $progname;
if ($opt_h) { usage; exit 0; }
if ($opt_version) { version; exit 0; }

# Only allow at most one non-help option
fatal sprintf gettext(
	"Only one of -a, -i, -v is allowed; try %s -h for more help"),
	      $progname
    if ($opt_i?1:0) + ($opt_a?1:0) + ($opt_v?1:0) > 1;

# Get a possible changelog entry from the command line
$SAVETEXT=$TEXT="@ARGV";

# Get the date
chomp($DATE=`822-date`);

# Are we going to have to figure things out for ourselves?
if (! $opt_i && ! $opt_v && ! $opt_a) {
    # Yes, we are
    @UPFILES = glob("../$PACKAGE\_$SVERSION\_*.upload");
    if (@UPFILES > 1) {
	fatal gettext("Found more than one appropriate .upload file!\n" .
		      "Please use an explicit -a, -i or -v option instead.");
    }
    elsif (@UPFILES == 0) { $opt_a = 1 }
    else {
	open UPFILE, "<${UPFILES[0]}"
	    or fatal sprintf(gettext(
			"Couldn't open .upload file for reading: %s\n" .
			"Please use an explicit -a, -i or -v option instead."), $!);
	while (<UPFILE>) {
	    if (/^s $PACKAGE\_$SVERSION\_[\w\-]+\.changes /) {
		$opt_i = 1;
		last;
	    }
	}
	close UPFILE
	    or fatal sprintf(gettext(
			"Problems experienced reading .upload file: %s\n" .
			"Please use an explicit -a, -i or -v option instead."), $!);
	if (! $opt_i) {
	    warn gettext(
		"A successful upload of the current version was not logged\n" .
		"in the upload log file; adding log entry to current version.");
	    $opt_a = 1;
	}
    }
}


# Open in anticipation....
open S, "debian/changelog" or fatal gettext("Cannot open changelog:") . " $!";
open O, ">debian/changelog.dch"
    or fatal gettext("Cannot write to temporary file:") ." $!";
# Note that we now have to remove it
$tmpchk=1;

if ($opt_i || $opt_v) {
    # Check that a given explicit version number is sensible.
    if ($opt_v) {
	$NEW_VERSION=$opt_v;
	if(system("dpkg --compare-versions $VERSION lt $NEW_VERSION" .
		  " 2>/dev/null 1>&2")) {
	    fatal sprintf(gettext(
			"New version specified (%s) is less than\n" .
			"the current version number (%s)!"),
			$NEW_VERSION, $VERSION);
	}
    }

    # We use the following criteria for the version and release number:
    # the last component of the version number is used as the
    # release number.  If this is not a Debian native package, then the
    # upstream version number is everything up to the final '-', not
    # including epochs.

    if (! $NEW_VERSION) {
	if ($VERSION =~ /([a-z]*[a-y]|\d+)$/i) {
	    $RELEASE=$1; $RELEASE++;
	    $NEW_VERSION = $` . $RELEASE;
	} else {
	    fatal gettext ("Error parsing version number:") . $VERSION;
	}
    }

    print O "$changelog{'Source'} ($NEW_VERSION) $changelog{'Distribution'}; ",
	"urgency=$changelog{'Urgency'}\n\n";

    if ($TEXT) { write O } else { print O "  * \n"; $line=3; }
    print O "\n -- $MAINTAINER <$EMAIL>  $DATE\n\n";

    # Copy the old changelog file to the new one
    local $/ = undef;
    print O <S>;
}
else { # $opt_a = 1
    # This means we just have to generate a new * entry in changelog

    # The first lines are as we have already found
    print O $changelog{'Changes'};

    if ($TEXT) { write O; } else { print O "  * \n"; }
    print O "\n -- $MAINTAINER <$EMAIL>  $DATE\n";

    # Copy the rest of the changelog file to new one
    $line=-1;
    while (<S>) { $line++; last if /^ --/; }
    # Slurp the rest....
    local $/ = undef;
    print O <S>;
}

close S or fatal gettext("Error closing debian/changelog:") . " $!";
close O or fatal gettext("Error closing temporary changelog:") . " $!";

# Now Run the Editor
if (! $SAVETEXT) {  # $TEXT might have been modified by a write()
    system("sensible-editor +$line debian/changelog.dch") == 0 or
    fatal gettext("Error editing the changelog:") . " $!";
}

copy("debian/changelog.dch","debian/changelog") or
    fatal gettext("Couldn't replace changelog with new changelog:") . " $!";

if (($opt_i || $opt_v) && $debian_native && !$opt_p) {
    if (move(cwd(), "../$PACKAGE-$NEW_VERSION")) {
	print STDERR
	    gettext("Warning: your current directory has been renamed to:\n");
	print STDERR "../$PACKAGE-$NEW_VERSION\n";
    } else {
	warn gettext("Couldn't rename directory:") . " $!";
    }
}

# Format for standard Debian changelogs
format O =
  * ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    $TEXT
 ~~ ^<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
    $TEXT
.

sub fatal($) {
    my ($pack,$file,$line);
    ($pack,$file,$line) = caller();
    (my $msg = sprintf(gettext("%s: fatal error at line %d:\n"),
		       $progname, $line) . "@_\n") =~ tr/\0//d;
    $msg =~ s/\n\n$/\n/;
    die $msg;
}
