/* Copyright (C) 2004 - 2006 db4objects Inc. http://www.db4o.com */
using System;
using System.IO;
using com.db4o;
using com.db4o.query;

namespace com.db4odoc.f1.queries
{
	public class QueryExample
	{
		public readonly static string YapFileName = "formula1.yap";

		public static void Main(string[] args)
		{
			StorePilot();
			UpdatePilotWrong();
			UpdatePilot();
			DeletePilot();
			File.Delete(Util.YapFileName);
			ObjectContainer db = Db4o.OpenFile(YapFileName);
			try
			{
				RetrievePilotByName(db);
				RetrievePilotByExactPoints(db);
				RetrieveByNegation(db);
				RetrieveByConjunction(db);
				RetrieveByDisjunction(db);
				RetrieveByComparison(db);
				RetrieveByDefaultFieldValue(db);
				RetrieveSorted(db); 
			}
			finally
			{
				db.Close();
			}
		}
		// end Main
    
		public static void StorePilot()
		{
			File.Delete(Util.YapFileName);
			ObjectContainer db = Db4o.OpenFile(YapFileName);
			try
			{
				Pilot pilot = new Pilot("Michael Schumacher", 0);
				db.Set(pilot);
				// change pilot and resave updated
				pilot.AddPoints(10);
				db.Set(pilot);
				Console.WriteLine("Stored {0}", pilot);
			}
			finally 
			{
				db.Close();
			}
			RetrieveAllPilots();
		}
		// end StorePilot

		public static void UpdatePilotWrong()
		{
			StorePilot();
			ObjectContainer db = Db4o.OpenFile(YapFileName);
			try
			{
				// Even completely identical Pilot object
				// won't work for update of the saved pilot
				Pilot pilot = new Pilot("Michael Schumacher",0);
				pilot.AddPoints(11);
				db.Set(pilot);
				Console.WriteLine("Added 11 points to {0}", pilot);
			}
			finally 
			{
				db.Close();
			}
			RetrieveAllPilots();
		}
		// end UpdatePilotWrong

		public static void UpdatePilot()
		{
			StorePilot();
			ObjectContainer db = Db4o.OpenFile(YapFileName);
			try
			{
				// first retrieve the object from the database
				ObjectSet result = db.Get(new Pilot("Michael Schumacher",10));
				Pilot found=(Pilot)result.Next();
				found.AddPoints(11);
				db.Set(found);
				Console.WriteLine("Added 11 points to {0}", found);
			}
			finally 
			{
				db.Close();
			}
			RetrieveAllPilots();
		}
		// end UpdatePilot

		public static void DeletePilot() 
		{
			StorePilot();
			ObjectContainer db=Db4o.OpenFile(YapFileName);
			try 
			{
				// first retrieve the object from the database
				ObjectSet result=db.Get(new Pilot("Michael Schumacher",10));
				Pilot found=(Pilot)result.Next();
				db.Delete(found);
				System.Console.WriteLine("Deleted "+found);
			} 
			finally 
			{
				db.Close();
			}
			RetrieveAllPilots();
		}
		// end DeletePilot

		
		public static void RetrieveAllPilots()
		{
			ObjectContainer db = Db4o.OpenFile(YapFileName);
			try
			{
				Query query = db.Query();
				query.Constrain(typeof(Pilot));
				ObjectSet result = query.Execute();
				ListResult(result);
			}
			finally 
			{
				db.Close();
			}
		}
		// end RetrieveAllPilots
    
		public static void RetrievePilotByName(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_name").Constrain("Michael Schumacher");
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrievePilotByName
        
		public static void RetrievePilotByExactPoints(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_points").Constrain(100);
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrievePilotByExactPoints
    
		public static void RetrieveByNegation(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_name").Constrain("Michael Schumacher").Not();
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrieveByNegation
    
		public static void RetrieveByConjunction(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			Constraint constr = query.Descend("_name")
				.Constrain("Michael Schumacher");
			query.Descend("_points")
				.Constrain(99).And(constr);
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrieveByConjunction
    
		public static void RetrieveByDisjunction(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			Constraint constr = query.Descend("_name")
				.Constrain("Michael Schumacher");
			query.Descend("_points")
				.Constrain(99).Or(constr);
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrieveByDisjunction
    
		public static void RetrieveByComparison(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_points")
				.Constrain(99).Greater();
			ObjectSet result = query.Execute();
			ListResult(result);
		}
		// end RetrieveByComparison
    
		public static void RetrieveByDefaultFieldValue(ObjectContainer db)
		{
			Pilot somebody = new Pilot("Somebody else", 0);
			db.Set(somebody);
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_points").Constrain(0);
			ObjectSet result = query.Execute();
			ListResult(result);
			db.Delete(somebody);
		}
		// end RetrieveByDefaultFieldValue
        
		public static void RetrieveSorted(ObjectContainer db)
		{
			Query query = db.Query();
			query.Constrain(typeof(Pilot));
			query.Descend("_name").OrderAscending();
			ObjectSet result = query.Execute();
			ListResult(result);
			query.Descend("_name").OrderDescending();
			result = query.Execute();
			ListResult(result);
		}
		// end RetrieveSorted
    
		public static void ListResult(ObjectSet result)
		{
			Console.WriteLine(result.Size());
			while(result.HasNext()) 
			{
				Console.WriteLine(result.Next());
			}
		}
		// end ListResult
	}
}
