/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com

This file is part of the db4o open source object database.

db4o is free software; you can redistribute it and/or modify it under
the terms of version 2 of the GNU General Public License as published
by the Free Software Foundation and as clarified by db4objects' GPL 
interpretation policy, available at
http://www.db4o.com/about/company/legalpolicies/gplinterpretation/
Alternatively you can write to db4objects, Inc., 1900 S Norfolk Street,
Suite 350, San Mateo, CA 94403, USA.

db4o is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */
namespace Db4objects.Db4o.Internal.CS
{
	public sealed class ServerMessageDispatcher : Sharpen.Lang.Thread
	{
		private string i_clientName;

		private bool i_loggedin;

		private long i_lastClientMessage;

		private int i_pingAttempts = 0;

		private int i_nullMessages;

		private bool i_sendCloseMessage = true;

		private readonly Db4objects.Db4o.Internal.CS.ObjectServerImpl i_server;

		private Db4objects.Db4o.Foundation.Network.ISocket4 i_socket;

		private Db4objects.Db4o.Internal.CS.ClientTransactionHandle _transactionHandle;

		private Db4objects.Db4o.Foundation.Hashtable4 _queryResults;

		private Db4objects.Db4o.Internal.Config4Impl i_config;

		internal readonly int i_threadID;

		private readonly object _lock = new object();

		private readonly object _mainLock;

		internal ServerMessageDispatcher(Db4objects.Db4o.Internal.CS.ObjectServerImpl aServer
			, Db4objects.Db4o.Internal.CS.ClientTransactionHandle transactionHandle, Db4objects.Db4o.Foundation.Network.ISocket4
			 aSocket, int aThreadID, bool loggedIn, object mainLock)
		{
			_transactionHandle = transactionHandle;
			_mainLock = mainLock;
			SetDaemon(true);
			i_loggedin = loggedIn;
			i_lastClientMessage = Sharpen.Runtime.CurrentTimeMillis();
			i_server = aServer;
			i_config = (Db4objects.Db4o.Internal.Config4Impl)i_server.Configure();
			i_threadID = aThreadID;
			SetName("db4o message server " + aThreadID);
			try
			{
				i_socket = aSocket;
				i_socket.SetSoTimeout(((Db4objects.Db4o.Internal.Config4Impl)aServer.Configure())
					.TimeoutServerSocket());
			}
			catch (System.Exception e)
			{
				i_socket.Close();
				throw (e);
			}
		}

		public void Close()
		{
			lock (_mainLock)
			{
				lock (_lock)
				{
					if (IsClosed())
					{
						return;
					}
					_transactionHandle.ReleaseTransaction();
					SendCloseMessage();
					_transactionHandle.Close();
					CloseSocket();
					RemoveFromServer();
				}
			}
		}

		private void SendCloseMessage()
		{
			try
			{
				if (i_sendCloseMessage)
				{
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE);
				}
			}
			catch (System.Exception e)
			{
			}
		}

		private void RemoveFromServer()
		{
			try
			{
				i_server.RemoveThread(this);
			}
			catch (System.Exception e)
			{
			}
		}

		private void CloseSocket()
		{
			try
			{
				i_socket.Close();
			}
			catch (System.Exception e)
			{
			}
			i_socket = null;
		}

		private bool IsClosed()
		{
			return i_socket == null;
		}

		internal Db4objects.Db4o.Internal.Transaction GetTransaction()
		{
			return _transactionHandle.Transaction();
		}

		public override void Run()
		{
			while (i_socket != null)
			{
				try
				{
					if (!MessageProcessor())
					{
						break;
					}
				}
				catch (System.Exception e)
				{
					if (_transactionHandle.IsClosed())
					{
						break;
					}
					if (!i_socket.IsConnected())
					{
						break;
					}
					i_nullMessages++;
				}
				if (i_nullMessages > 20 || PingClientTimeoutReached())
				{
					if (i_pingAttempts > 5)
					{
						GetStream().LogMsg(33, i_clientName);
						break;
					}
					if (null == i_socket)
					{
						break;
					}
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.PING);
					i_pingAttempts++;
				}
			}
			Close();
		}

		private bool PingClientTimeoutReached()
		{
			return (Sharpen.Runtime.CurrentTimeMillis() - i_lastClientMessage > i_config.TimeoutPingClients
				());
		}

		private bool MessageProcessor()
		{
			Db4objects.Db4o.Internal.CS.Messages.Msg message = Db4objects.Db4o.Internal.CS.Messages.Msg
				.ReadMessage(GetTransaction(), i_socket);
			if (message == null)
			{
				i_nullMessages++;
				return true;
			}
			i_lastClientMessage = Sharpen.Runtime.CurrentTimeMillis();
			i_nullMessages = 0;
			i_pingAttempts = 0;
			if (!i_loggedin)
			{
				if (Db4objects.Db4o.Internal.CS.Messages.Msg.LOGIN.Equals(message))
				{
					string userName = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
						();
					string password = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
						();
					Db4objects.Db4o.User found = i_server.GetUser(userName);
					if (found != null)
					{
						if (found.password.Equals(password))
						{
							i_clientName = userName;
							Db4objects.Db4o.Internal.LocalObjectContainer mainStream = _transactionHandle.MainStream
								();
							mainStream.LogMsg(32, i_clientName);
							int blockSize = mainStream.BlockSize();
							int encrypt = mainStream.i_handlers.i_encrypt ? 1 : 0;
							Write(Db4objects.Db4o.Internal.CS.Messages.Msg.LOGIN_OK.GetWriterForInts(GetTransaction
								(), new int[] { blockSize, encrypt }));
							i_loggedin = true;
							SetName("db4o server socket for client " + i_clientName);
						}
						else
						{
							Write(Db4objects.Db4o.Internal.CS.Messages.Msg.FAILED);
							return false;
						}
					}
					else
					{
						Write(Db4objects.Db4o.Internal.CS.Messages.Msg.FAILED);
						return false;
					}
				}
				return true;
			}
			if (message.ProcessAtServer(this))
			{
				return true;
			}
			ProcessSpecialMsg(message);
			return true;
		}

		public bool ProcessSpecialMsg(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.PING.Equals(message))
			{
				WriteOK();
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.OBJECTSET_FINALIZED.Equals(message))
			{
				int queryResultID = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadInt(
					);
				QueryResultFinalized(queryResultID);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE.Equals(message))
			{
				Write(Db4objects.Db4o.Internal.CS.Messages.Msg.CLOSE);
				GetTransaction().Commit();
				i_sendCloseMessage = false;
				GetStream().LogMsg(34, i_clientName);
				return false;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.IDENTITY.Equals(message))
			{
				RespondInt((int)GetStream().GetID(GetStream().Identity()));
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.CURRENT_VERSION.Equals(message))
			{
				long ver = 0;
				lock (GetStream())
				{
					ver = GetStream().CurrentVersion();
				}
				Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ID_LIST.GetWriterForLong(GetTransaction
					(), ver));
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.RAISE_VERSION.Equals(message))
			{
				long minimumVersion = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadLong
					();
				Db4objects.Db4o.Internal.ObjectContainerBase stream = GetStream();
				lock (stream)
				{
					stream.RaiseVersion(minimumVersion);
				}
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.GET_THREAD_ID.Equals(message))
			{
				RespondInt(i_threadID);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.SWITCH_TO_FILE.Equals(message))
			{
				SwitchToFile(message);
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.SWITCH_TO_MAIN_FILE.Equals(message))
			{
				SwitchToMainFile();
				return true;
			}
			if (Db4objects.Db4o.Internal.CS.Messages.Msg.USE_TRANSACTION.Equals(message))
			{
				UseTransaction(message);
				return true;
			}
			return true;
		}

		private void WriteOK()
		{
			Write(Db4objects.Db4o.Internal.CS.Messages.Msg.OK);
		}

		private void QueryResultFinalized(int queryResultID)
		{
			_queryResults.Remove(queryResultID);
		}

		public void MapQueryResultToID(Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub
			 stub, int queryResultID)
		{
			if (_queryResults == null)
			{
				_queryResults = new Db4objects.Db4o.Foundation.Hashtable4();
			}
			_queryResults.Put(queryResultID, stub);
		}

		public Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub QueryResultForID(int queryResultID
			)
		{
			return (Db4objects.Db4o.Internal.CS.LazyClientObjectSetStub)_queryResults.Get(queryResultID
				);
		}

		private void SwitchToFile(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			lock (_mainLock)
			{
				string fileName = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadString
					();
				try
				{
					_transactionHandle.ReleaseTransaction();
					_transactionHandle.AcquireTransactionForFile(fileName);
					WriteOK();
				}
				catch (System.Exception e)
				{
					_transactionHandle.ReleaseTransaction();
					Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ERROR);
				}
			}
		}

		private void SwitchToMainFile()
		{
			lock (_mainLock)
			{
				_transactionHandle.ReleaseTransaction();
				WriteOK();
			}
		}

		private void UseTransaction(Db4objects.Db4o.Internal.CS.Messages.Msg message)
		{
			int threadID = ((Db4objects.Db4o.Internal.CS.Messages.MsgD)message).ReadInt();
			Db4objects.Db4o.Internal.Transaction transToUse = i_server.FindTransaction(threadID
				);
			_transactionHandle.Transaction(transToUse);
		}

		private void RespondInt(int response)
		{
			Write(Db4objects.Db4o.Internal.CS.Messages.Msg.ID_LIST.GetWriterForInt(GetTransaction
				(), response));
		}

		public void Write(Db4objects.Db4o.Internal.CS.Messages.Msg msg)
		{
			_transactionHandle.Write(msg, i_socket);
		}

		public Db4objects.Db4o.Foundation.Network.ISocket4 Socket()
		{
			return i_socket;
		}

		private Db4objects.Db4o.Internal.LocalObjectContainer GetStream()
		{
			return _transactionHandle.Stream();
		}
	}
}
