/* main.c */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <signal.h>
#include <ctype.h>
#include <getopt.h>
#include <errno.h>
#include <limits.h>

#include "config.h"
#include "main.h"
#include "bmark.h"
#include "list_load.h"
#include "process_load.h"
#include "ctar_load.h"
#include "xtar_load.h"
#include "read_load.h"
#include "io_load.h"
#include "dbench_load.h"
#include "mem_load.h"
#include "trivial.h"
#include "sysinfo.h"
#include "programs.h"

/* true if we are called from sighandler */
volatile sig_atomic_t exiting=0;
/*
 * to add a load, all you have to do is add an extra field
 * in the loads[] array. the first field must be the name of
 * the load, the second must be the address of a function that
 * does some kind of initialisation (or NULL if that is unnecessary).
 *
 * the third should be the address of the function that creates the
 * actual load. this function is expected to write a byte to TMP_FD
 * after each iteration. The size of the file referred to by TMP_FD
 * is used as the value for "Loads".
 * It executes as a separate process so it can call exec*().
 *
 * the fourth field should be a pointer to a cleanup function (or NULL)
 * the fifth field should be the name of the log file for this load.
 *
 * the other fields must be initialised in the same way as "no_load".
 */

struct load loads[] = {
	{"no_load", NULL, &do_no_load, NULL, "no_load.log", 0, 0, {0, 1}},
	{"cacherun", NULL, &do_no_load, NULL, "cacherun.log", 0, 0, {0, 1}},
	{"process_load", NULL, &do_process_load, NULL, "process_load.log", 0, 0, {0, 1}},
	{"ctar_load", &prep_ctar_load, &do_ctar_load, NULL, "ctar_load.log", 0, 0, {0, 1}},
	{"xtar_load", &prep_xtar_load, &do_xtar_load, NULL, "xtar_load.log", 0, 0, {0, 1}},
	{"io_load", NULL, &do_io_load, &cleanup_io_load, "io_load.log", 0, 0, {0, 1}},
	{"io_other", NULL, &do_io_other, &cleanup_io_other, "io_other.log", 0, 0, {0, 1}},
	{"read_load", &prep_read_load, &do_read_load, &cleanup_read_load, "read_load.log", 0, 0, {0, 1}},
	{"list_load", NULL, &do_list_load, NULL, "list_load.log", 0, 0, {0, 1}},
	{"mem_load", NULL, &do_mem_load, NULL, "mem_load.log", 0, 0, {0, 1}},
	{"dbench_load", NULL, &do_dbench_load, &cleanup_dbench_load, "dbench_load.log", 0, 0, {0, 1}},
	{NULL, NULL, NULL, NULL, NULL, 0, 0, {0, 0}},
};

/* used to cleanup on SIGTERM */
struct harvest deadbeef = {0, NULL};

char * opt_tmpf=DEF_TMP_FILE;
char * opt_kern_name=NULL;
char * opt_log_file=DEF_LOG_FILE;
char * opt_outfile=NULL;
char * opt_repfile=NULL;
char * opt_io_other_file=NULL;
int opt_debug=0;
int opt_cold=0;
int opt_cleanup=1;
int opt_nr_runs=0;
int opt_gen_rep=0;
int opt_prog_bar=0;
int opt_pl_nr_procs=0;
char * opt_pl_rec_size="8192";
char * opt_pl_inject="2";
char * opt_ml_pc_ram="0";	/* just a safe value */

int main(int argc, char *argv[])
{
	char c;
	int nr_runs;	/* for presentation */
	struct load *ld;
	struct load **lopt=NULL;
	struct load **opt_loads=NULL;
	struct bench_results res_kern, res_load;

	while((c=getopt(argc, argv, "bcrdpR:k:t:n:l:o:"))!=-1){
		switch(c){
			case 'b': opt_prog_bar=1;
				  break;
			case 'c': opt_cold=1;
				  break;
			case 'd': opt_debug=1;
				  break;
			case 'r': opt_gen_rep=1;
				  break;
			case 'R': opt_repfile=optarg;
				  break;
			case 'k': opt_kern_name=optarg;
				  break;
			case 'o': opt_io_other_file=optarg;
				  break;
			case 'p': opt_cleanup=0;
				  break;
			case 't': opt_tmpf=optarg;
				  break;
			case 'n': opt_nr_runs=strtol(optarg, NULL, 0);
				  if(!opt_nr_runs){
					  if(errno==EINVAL){
						  printe("could not parse "
							"number of runs\n");
					  } else {
						  /*
						   * we actually got zero, set
						   * opt_nr_runs invalid
						   */
						  opt_nr_runs--;
					  }
				  }
				  if((
					(opt_nr_runs==LONG_MIN)||
					(opt_nr_runs==LONG_MAX)
					)&&errno==ERANGE){
					  printe("number of runs is out of "
							 	"range\n");
				  }
				  break;
			default: usage();
		}
	}
	argc -= optind;
	argv += optind;

	printd("argc=%d\n", argc);
	if(!argc && !opt_gen_rep){
		int i;

		for(i=0; loads[i].load_name; i++);
		printd("i=%d\n", i);
		if(!(opt_loads=malloc((i+1)*sizeof(struct load *)))){
			printsys("out of memory\n");
		}
		for(i=0, lopt=opt_loads; loads[i].load_name; i++, lopt++){
			*lopt=&loads[i];
		}
		*lopt=NULL;
	} else if(argc) {
		if(!(opt_loads=malloc((argc+1)*sizeof(struct load *)))){
			printsys("out of memory\n");
		}
		lopt=opt_loads;
		for(; argc; argc--, argv++, lopt++){
			struct load *it;
			for(it=loads; it->load_name; it++){
				if(!strcmp(*argv, it->load_name)){
					*lopt=it;
					if(!strcmp(it->load_name, "io_other")&&
						!opt_io_other_file){
						printe("io_other requires -o\n");
						exit(1);
					}
					break;
				}
			}
			if(!it->load_name){
				printe("no such load: %s\n", *argv);
				exit(2);
			}
		}
		*lopt=NULL;
	}
	if(opt_nr_runs<0){
		printe("Invalid number of runs specified\n");
		exit(2);
	}
	if((!opt_nr_runs) && opt_loads){
		printw("Number of runs was not specified, using default (3)\n");
		opt_nr_runs=3;
	}
	if(!opt_kern_name){
		if(get_uname(&opt_kern_name)){
			opt_kern_name="def_kern_name";
			printw("error getting os version, using default: %s\n",
								opt_kern_name);
		}
	}
#ifndef DEBUG
	if(opt_debug){
		printw("contest was not compiled with support for debugging, "
						"option -d has no effect\n");
	}
#endif	/* !DEBUG */
	nr_runs=opt_nr_runs;
	init_signals();
	printd("pid=%d\n", getpid());
	while(opt_nr_runs--){
		printout("Run number %d of %d\n", nr_runs-opt_nr_runs, nr_runs);
		for(lopt=opt_loads; *lopt; lopt++){
			ld=*lopt;
			make_clean();
			if(ld->prep){
				ld->prep();
			}
			if(!strcmp("cacherun", ld->load_name)){
				printout("Skipping cache cleaning for "
								"cacherun\n");
			} else if(opt_cold){
				printw("Skipping cache cleaning, results will "
					"be useless unless you really know "
					"what you are doing!\n");
			} else {
				clean_mem();
			}
			printout("Starting %s\n", ld->load_name);
			if(!ld->do_load){
				printw("load %s not implemented\n",
							ld->load_name);
				break;
			}

			block_signals();
			bmark(ld);
			harvester_add_load(ld);
			unblock_signals();

			printout("Testing kernel compile time with %s\n",
								ld->load_name);
			res_kern=compile_kernel(ld);
			strncpy(res_kern.name, opt_kern_name, MAX_UNAME);
			printout("Finished compiling kernel: elapsed: %lu "
				"user: %lu system: %lu\n", res_kern.elapsed,
					res_kern.utime, res_kern.systime);

			kill_load(ld);

			printd("getting stats for load run...\n");
			res_load=load_get_stats(ld);
			printout("Finished %s: elapsed: %lu user: %lu system: "
					"%lu loads: %d\n", ld->load_name,
					res_load.elapsed, res_load.utime,
					res_load.systime, res_load.loadruns);
			strncpy(res_load.name, ld->load_name, MAX_UNAME);
			write_res(ld->logfile, res_kern, res_load);
			if(ld->cleanup && opt_cleanup){
				ld->cleanup();
			}
			if(opt_repfile){
				char *tmp=opt_outfile;

				if(unlink(opt_repfile) && (errno!=ENOENT)){
					printsys("could not unlink file "
						"\"%s\"\n", opt_repfile);
				}
				opt_outfile=opt_repfile;
				gen_report(loads);
				opt_outfile=tmp;
			}
		}
	}
	if(opt_gen_rep){
		return gen_report(loads);
	}
	printd("returning to the system\n");
	return 0;
}

void usage(void)
{
	struct load *ld;

	printf("Usage: contest [-cdr] [-k name] [-t file] [-o file2] [- n nrruns] "
								"[load...]\n");
	/*
	 * actually, this is just to speed up runs when the results
	 * are insignificant (e.g. during development :)
	 */
	printf("\t-b: print a progress bar according to estimated progress\n");
	printf("\t-c: assume a cold cache\n");
	printf("\t-d: print debugging messages if contest was compiled with\n"
		"\t    debugging on (this will flood the console)\n");
	printf("\t-p: don't cleanup after each load (don't use this on a system\n"
						"\t that's low on disk space)\n");
	printf("\t-r: generate report\n");
	printf("\tname: name to use for the current kernel\n");
	printf("\tfile: temporary file to use\n");
	printf("\tfile2: temporary file to use with io other\n");
	printf("\tnrruns: number of runs\n");
	printf("available loads are:");
	for(ld=loads; ld->load_name; ld++){
		printf(" %s", ld->load_name);
	}
	printf("\n");
	exit(2);
}

int clean_mem(void)
{
	unsigned long mem;
	struct load *ld;

	printout("Flushing memory and swap\n");
	if(get_ram(&mem)){
		printe("could not get info on ram size\n");
		exit(2);
	}
	mem/=1024;
	sync();
	sync();
	sync();
#ifdef __linux__
	
	printd("Turning off all swap devices... ");
	system("swapoff -a");
	printd("OK\n");
	
	printd("Reactivating swap devices... ");
	system("swapon -a");
	printd("OK\n");

#endif
	sync();
	sync();
	sync();
	for(ld=loads; ld->load_name; ld++){
		if(!strcmp("mem_load", ld->load_name))
			break;
	}
	if(!ld->load_name){
		printe("could not find mem_load (used to clean the cache), "
								"aborting\n");
		exit(2);
	}
	bmark(ld);
	sleep(20);
	kill_load(ld);
	load_get_stats(ld);
#ifdef __linux__
	
	printd("Turning off all swap devices... ");
	system("swapoff -a");
	printd("OK\n");
	
	printd("Reactivating swap devices... ");
	system("swapon -a");
	printd("OK\n");

#endif
	printd("done cleaning memory\n");
	return 0;
}

int write_res(char *fname, struct bench_results kern, struct bench_results load)
{
	FILE *fp;

	if(!fname){
		printe("internal error, missing name for logfile\n");
		exit(2);
	}
	if(!(fp=fopen(fname, "a"))){
		printsys("could not open file %s\n", fname);
	}
	printd("Writing results to log... ");
	fprintf(fp, "%s %lu %lu %lu %lu %lu %s %lu %lu %lu %lu %lu %lu\n",
		kern.name, kern.utime, kern.systime, kern.elapsed,
		kern.minor_faults, kern.major_faults,
		load.name, load.utime, load.systime, load.elapsed,
		load.minor_faults, load.major_faults, load.loadruns);
	fclose(fp);
	printd("OK\n");
	return 0;
}

int comp_fields(const void *c, const void *d)
{
	struct bench_results *a, *b;

	a=(struct bench_results *)c;
	b=(struct bench_results *)d;
	return strncmp(a->name, b->name, MAX_UNAME);
}

inline int same_res(struct bench_results *kern, int i)
{
	int backstep=2;

	if(i<1)
		return 0;
	/*
	 * if the previous element is unused, step further back.
	 * we are guaranteed that the first element of the array
	 * can be used (->name[0]!='\0')
	 */
	for(; (kern-backstep)->name[0] && (backstep/2)<i; backstep+=2);
	printd("same_res: using backstep=%d, i was %d\n", backstep, i);
	return !strcmp((kern-backstep)->name, kern->name);
}

/*
 * usually one of the structures pointed to by the arguments
 * is assigned to. this is terribly wasteful, but gives a
 * direct indication of which structure is modified
 */
struct bench_results add_res(struct bench_results *a, struct bench_results *b)
{
	struct bench_results res;

	/* we know that strcmp(a->name, b->name)==0 */
	strcpy(res.name, a->name);
	res.utime=a->utime+b->utime;
	res.systime=a->systime+b->systime;
	res.elapsed=a->elapsed+b->elapsed;
	res.minor_faults=a->minor_faults+b->minor_faults;
	res.major_faults=a->major_faults+b->major_faults;
	res.loadruns=a->loadruns+b->loadruns;
	return res;
}

/*
 * FIXME:
 * no need to return the struct here, it's obvious what the
 * function does with it
 */

struct bench_results div_res(struct bench_results *a, int num)
{
	struct bench_results res;

	/* we know that it's a '\0' terminated string */
	strcpy(res.name, a->name);
	res.utime=a->utime/num;
	res.systime=a->systime/num;
	res.elapsed=a->elapsed/num;
	res.minor_faults=a->minor_faults/num;
	res.major_faults=a->major_faults/num;
	res.loadruns=a->loadruns/num;
	return res;
}

int new_ref_stats(ref_stats_list *list, char *name, double time)
{
	struct ref_stats *it;

	printd("in, name=%s\n", name);
	if(!*list){
		printd("no *list\n");
		if(!(it=malloc(sizeof(struct ref_stats)))){
			return 1;
		}
		strncpy(it->kernel, name, MAX_UNAME);
		it->no_load_time=time;
		it->next=NULL;
		*list=it;
	} else {
		printd("appending to list\n");
		for(it=*list; it->next; it=it->next);
		if(!(it->next=malloc(sizeof(struct ref_stats)))){
			return 1;
		}
		strncpy(it->next->kernel, name, MAX_UNAME);
		it->next->no_load_time=time;
		it->next->next=NULL;
	}
	printd("returning\n");
	return 0;
}

int get_ref_stats(ref_stats_list list, char *name, double *time)
{
	printd("in, list=%p, name=%s\n", list, name);
	for(; list; list=list->next){
		if(!strncmp(name, list->kernel, MAX_UNAME)){
			*time=list->no_load_time;
			return 0;
		}
	};
	return 1;
}

void lose_ref_stats(ref_stats_list *list)
{
	struct ref_stats *it, *tmp;

	for(it=*list; it; it=tmp){
		tmp=it->next;
		free(it);
	}
	*list=NULL;
}

struct bench_results * parse_log(char *fname)
{
	int res;
	FILE *fp;
	unsigned long lines;
	char ch;
	struct bench_results *kern;

	printd("in parse_log(%s)\n", fname);
	if(!(fp=fopen(fname, "r"))){
		if(errno==ENOENT){	/* no log */
			return NULL;
		}
		printsys("error opening file stream for %s\n", fname);
	}
	lines=0;
	while((ch=fgetc(fp))!=EOF){
		if(ch=='\n')
			lines++;
	}
	if(!feof(fp)){
		printsys("error reading file %s\n", fname);
	}
	clearerr(fp);
	rewind(fp);
	printd("found %lu lines\n", lines);
	if(!(kern=malloc(2*(lines+1)*(sizeof(struct bench_results))))){
		printsys("Out of memory\n");
	}
	/* zero-terminate array */
	memset(kern+2*lines, '\0', 2*sizeof(struct bench_results));
	printd("memset ok\n");
	printd("going to read...\n");
	for(res=13; res==13; kern+=2){
		/* 200 is the value of MAX_UNAME, update as necessary */
		res=fscanf(fp, "%200s %lu %lu %lu %lu %lu %200s %lu %lu %lu %lu %lu %lu\n",
			kern->name, &kern->utime, &kern->systime, &kern->elapsed,
			&kern->minor_faults, &kern->major_faults,
			kern[1].name, &kern[1].utime, &kern[1].systime, &kern[1].elapsed,
			&kern[1].minor_faults, &kern[1].major_faults, &kern[1].loadruns);
		printd("parsed line, i=%d\n", res);
	}
	printd("got out\n");
	if(res!=EOF){
		printe("error parsing file %s\n", fname);
		exit(2);
	}
	kern=kern-2*(lines+1);
	fclose(fp);
	/*
	 * sorts an array of structure pairs -
	 * the first element of each pair contains results for the kernel,
	 * the second for the load
	 */
	qsort(kern, lines, 2*sizeof(struct bench_results), &comp_fields);
	return kern;
}

int get_max_width(struct load *all)
{
	int i, width=0;
	struct load *ld;
	struct bench_results *res, *it;

	for(ld=all; ld->load_name; ld++){
		if(!ld->logfile)
			continue;
		if(!(res=parse_log(ld->logfile)))
			continue;
		it=res;
		for(; it->name[0]; it+=2){
			i=strlen(it->name);
			if(i>width){
				width=i;
			}
		}
		free(res);
	}
	printd("width=%d\n", width);
	return width;
}

int gen_report(struct load *ld)
{
	struct bench_results *res, *res_save;
	int runs, width, i, tmp;
	double no_load_time=0;
	ref_stats_list st_l=INIT_REF_LIST;

	printd("in gen_report\n");
	width=get_max_width(loads);
	for(; ld->load_name; ld++){
		if(!ld->logfile)
			continue;
		if(!(res=parse_log(ld->logfile)))
			continue;
		res_save=res;
		printout("%s:\n", ld->load_name);
		printout("Kernel");
		/* but the output looks better */
		for(i=0; i<(width-5); i++){	/* 5==strlen("Kernel")-1; */
			printout(" ");
		}
		//printout("Kernel  [runs]\tTime\tCPU%%\tLoads\tLCPU%%\tRatio\n");
		printout("[runs]\tTime\tCPU%%\tLoads\tLCPU%%\tRatio\n");
		for(; res->name[0]; res+=2){
			/*
	 		* careful, this is really fragile
	 		*/
			for(runs=1; !strcmp((res+2)->name, res->name); res+=2, runs++){
				*(res+2)=add_res(res, res+2);
				*(res+3)=add_res(res+1, res+3);
			}
			printd("runs=%d!\n", runs);
			*res=div_res(res, runs);
			*(res+1)=div_res(res+1, runs);
			if(!strcmp(ld->load_name, "no_load")){
				if(new_ref_stats(&st_l, res->name, res[0].elapsed)){
					printe("could not add stats (out of memory?)\n");
					exit(2);
				}
			}
			if(get_ref_stats(st_l, res->name, &no_load_time)){
				printe("Time not available for no_load\n");
				exit(2);
			}
			printd("ratio is %lu/%f=%.1f\n", res[0].elapsed, no_load_time,
					res[0].elapsed/no_load_time);
			printout("%s", res[0].name);
			/*
			 * I *really* need to come up with a better way of
			 * keeping the sick fellows that use large kernel
			 * names happy.
			 */
			width -= (tmp=(strlen(res[0].name)-1));
			for(i=0; i < width; i++){
				printout(" ");
			}
			width += tmp;
			printout("%6d\t%lu\t%.1f\t%.1f\t%.1f\t%.2f\n",
				runs, res[0].elapsed,
				100*(double)(res[0].utime+res[0].systime)/
				res[0].elapsed,((double)res[1].loadruns)/100,
				100*((double)res[1].utime+
				res[1].systime)/res[1].elapsed,
				(double)res[0].elapsed/no_load_time);
#if 0
			printout("%s[%d]\t%lu\t%.1f\t%d\t%.1f\t%.2f\n",
				res[0].name, runs, res[0].elapsed,
				100*(double)(res[0].utime+res[0].systime)/
				res[0].elapsed,res[1].loadruns, 100*((double)res[1].utime+
				res[1].systime)/res[1].elapsed,
				(double)res[0].elapsed/no_load_time);
#endif	/* 0 */
		}
		free(res_save);
	}
	lose_ref_stats(&st_l);
	return 0;
}

int est_elapsed(char *log)
{
	struct bench_results *res;
	int i;
	unsigned long time=0;

	if(!(res=parse_log(log))){
		return 0;
	}
	for(i=0; res->name[0]; res+=2, i++){
		if(!strcmp(res->name, opt_kern_name)){
			time+=res->elapsed;
		}
	}
	if(!i){
		return 0;
	}
	time/=i;
	printd("time=%lu, i=%d\n", time, i);
	printd("estimated time for kernel compile from logfile %s: %lu\n", log, time);
	return time;
}

int do_no_load(void)
{
	printd("pid=%d\n", getpid());
	while(1){
		printd("once\n");
		sleep(10000);
	}
	return 0;
}

int do_preps(char **req_loads)
{
	int i;
	struct load *ld;
	for(i=0; i<MAX_LOADS && req_loads[i]; i++){
		for(ld=loads; ld->load_name; ld++){
			if(!strcmp(req_loads[i], ld->load_name)){
				if(ld->prep && !ld->prep_done){
					printd("doing prep for %s\n", ld->load_name);
					ld->prep();
					ld->prep_done++;
				}
			}
		}
	}
	return 0;
}

int kill_load(struct load *ld)
{
	if(!ld->child_pid){ /* can't happen, but we may use it in the future*/
		/* real error message is output by child */
		printe("we regret to inform you that a fatal error has "
				"occurred\n");
		exit(1);	/* it's a system error in the child */
	}
	printd("sending SIGUSR1 to pid %d\n", ld->child_pid);
	if(kill(ld->child_pid, SIGUSR1)){
		if(!exiting || (errno!=ESRCH)){
			printsys("error sending SIGUSR1 to pid %d\n", ld->child_pid);
		}
	}
	return 0;
}

struct bench_results load_get_stats(struct load *ld)
{
	struct bench_results res;
	printd("parent: trying to read data from fd=%d\n", ld->pipe[0]);
	if(read(ld->pipe[0], &res, sizeof(struct bench_results))!=sizeof(struct bench_results)){
		printsys("error getting stats from child with pid %lu\n", ld->child_pid);
	}
	printd("parent: read data from fd=%d\n", ld->pipe[0]);
	printd("parent: waiting for pid %d\n", ld->child_pid);
	if(wait4(ld->child_pid, NULL, 0, NULL)!=ld->child_pid){
		printsys("could not wait for %d\n", ld->child_pid);
	}
	printd("parent: waited for pid %d\n", ld->child_pid);
	if(close(ld->pipe[0])){
		printsys("could not close first pipe descriptor\n");
	}
	if(close(ld->pipe[1])){
		printsys("could not close second pipe descriptor\n");
	}
	return res;
}

int init_signals(void)
{
	struct sigaction sa;

	printd("in\n");
	sa.sa_handler=&harvester;
	sigfillset(&sa.sa_mask);
	sa.sa_flags=0;
	if(sigaction(SIGTERM, &sa, NULL)){
		printsys("could not set signal handler\n");
	}
	if(sigaction(SIGINT, &sa, NULL)){
		printsys("could not set signal handler\n");
	}
	printd("out\n");
	return 0;
}

int restore_signals(void)
{
	struct sigaction sa;

	sa.sa_handler=SIG_DFL;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags=0;
	if(sigaction(SIGTERM, &sa, NULL)){
		printsys("could not set signal handler\n");
	}
	if(sigaction(SIGINT, &sa, NULL)){
		printsys("could not set signal handler\n");
	}
	return 0;
}

void harvester(int err)
{
	char c='\n';

	printd("in\n");
	err=0;
	exiting=1;
	if(deadbeef.make){
		printd("killing make\n");
		if(kill(deadbeef.make, SIGTERM) && (errno!=ESRCH)){
				err++;
		} else if(wait4(deadbeef.make, NULL, 0, NULL)!=deadbeef.make){
			err++;
		}
	}
	if(deadbeef.load){
		printd("killing load %s\n", deadbeef.load->load_name);
		err += kill_load(deadbeef.load);
	}
	printd("out\n");
	write(STDOUT_FILENO, &c, 1);
	exit(err);
}

/* any similarities with minix naming are purely coincidental */
inline void block_signals(void)
{
	sigset_t sigset;

	sigfillset(&sigset);
	sigprocmask(SIG_SETMASK, &sigset, NULL);
}

inline void unblock_signals(void)
{
	sigset_t sigset;

	sigemptyset(&sigset);
	sigprocmask(SIG_SETMASK, &sigset, NULL);
}
