(in-package :maisql-sys)

;; Schema instantiation and versioning

(defun classes-for-schema (name)
  (gethash name *object-schemas*))

(defun describe-schema (name)
  (classes-for-schema name))


(defun %schema-layout-equal (sa sb)
  (and (equal (first sa) (first sb))
       (equal (second sa) (second sb))
       (equal (third sa) (third sb))
       (equal (or (and (fourth sa)
                       (symbol-name (fourth sa)))
                  (sql-escape (first sa)))
              (or (and (fourth sb)
                       (symbol-name (fourth sb)))
                  (sql-escape (first sb))))))


(defun validate-schema (name)
  (labels ((validate-object (schema-class)
             (let ((def (car (select [vers] [def] :from [usql_object_v]
                                     :where [= [name] (sql-escape schema-class)]))))
               (let ((installed-def (and def (read-from-string (cadr def))))
                     (compiled-def (maisql-sys::describe-db-layout (find-class schema-class))))
                 (mapcar #'dink-type installed-def)
                 (mapcar #'dink-type compiled-def)
                 (if (or (not (equal (length installed-def) (length compiled-def)))
                          (not (every #'identity (mapcar #'%schema-layout-equal installed-def compiled-def))))
                     (let ((int (intersection installed-def compiled-def :test #'%schema-layout-equal)))
                       (format t ";; Versions of ~s are incompatible~%" schema-class)
                       (format t ";; removed slots: ~s~%" (remove-if (lambda (x) (member x int :test #'%schema-layout-equal))
                                                                     installed-def))
                       (format t ";; added slots: ~s~%" (remove-if (lambda (x) (member x int :test #'%schema-layout-equal))
                                                                   compiled-def))
                       (let ((installed-version (if def (or (parse-integer-insensitively (car def)) 0) -1))
                             (compiled-version (or (car (maisql-sys::object-version (find-class schema-class)))
                                                   0)))
                         (if (>= installed-version compiled-version)
                             (format t ";; Versions claim to match: bump the version number in code, and write a patch function.~%")
                             (format t ";; ~s must be patched from version ~d to version ~d~%" schema-class installed-version compiled-version))
                         ))))))
           (dink-type (col)
             (if (eql (caddr col) :base)
                 (setf (caddr col) nil))))
    (mapc #'validate-object (classes-for-schema name))
    t))

(defun assert-schema-def (name)
  (assert-schema-for-classes (classes-for-schema name)))

(defun assert-schema-for-classes (classes)
  (flet ((assert-class-def (schema-class)
           (let* ((class (find-class schema-class))
                  (def (maisql-sys::describe-db-layout class)))
             (insert-records :into [usql_object_v]
                             :av-pairs `(([name] ,(sql-escape (class-name class)))
                                         ([vers] ,(or (ignore-errors (car (object-version class))) 0))
                                         ([def]  ,(prin1-to-string def)))))))
    (ensure-schema-version-table *default-database*)
    (delete-records :from [usql_object_v])
    (mapc #'assert-class-def classes)))

(defun ensure-schema (name &key (force nil))
  (declare (ignore force))
  (flet ((instantiate-table (schema-class)
           (unless (tablep (view-table (find-class schema-class)))
             (create-view-from-class schema-class)
             (create-sequence-from-class schema-class))))
    (mapc #'instantiate-table (flatten-schema name))
    t))

(defun install-schema (name &key (force nil))
  (declare (ignore force))
  (flet ((instantiate-table (schema-class)
           (when (tablep (view-table (find-class schema-class)))
             (drop-view-from-class schema-class)
             (ignore-errors
               (drop-sequence-from-class schema-class)))
           (create-view-from-class schema-class)
           (create-sequence-from-class schema-class)))
    (mapc #'instantiate-table (flatten-schema name))
    t))

(defun remove-schema (name)
  (declare (ignore name)))

(defun tail (list)
  (car (reverse list)))

(defun flatten-schema (name)
  (let* ((classes      (mapcar #'db-ancestors (classes-for-schema name)))
         (all-classes  (remove-duplicates (mapcar #'car classes))))
    (let ((names (mapcar #'class-name all-classes)))
      (sort names #'string< :key #'string))))

(defun db-ancestors (classname)
  (let ((class (find-class classname))
        (db-class (find-class 'standard-db-object)))
    (labels ((ancestors (class)
             (let ((scs (class-direct-superclasses class)))
               (if (member db-class scs)
                   class
                   (mapcar #'ancestors1 scs))))
             (ancestors1 (class)
               (let ((scs (class-direct-superclasses class)))
                 (if (member db-class scs)
                     nil
                     (mapcar #'ancestors scs)))))
      (cons class (ancestors1 class)))))

