;; -*- Mode: Lisp -*-
;; $Id: pools.lisp,v 1.5 2001/04/04 20:14:54 lyn Exp $

(in-package :maisql-sys)

;; maintain a pool of active connections to a database

(defstruct (connection-pool (:conc-name pool-))
  db-type
  db-dict
  connections
  (init-connection-function (lambda (connection)
                              (declare (ignore connection)) t))
  (max 16))

(defstruct connection
  database
  active)

(defun %create-connection (pool)
  "create a new connection, used when all existing ones are in use.
Connection is initialized with the pool's init-connection-function"
  (let* ((conn (sql:connect (pool-db-dict pool)
                            :database-type (pool-db-type pool)
                            :make-default nil
                            :if-exists :new))
         (connection (car (push (make-connection :database conn :active t)
                                (pool-connections pool)))))
    (funcall (pool-init-connection-function pool) connection)
    connection))

(defun obtain-connection (pool)
  "returns a quiescent connection. release it with
'release-connection' when done. will block until a free connection is
available."
  (loop
   (dolist (conn (pool-connections pool))
     (when (not (connection-active conn))
       (setf (connection-active conn) t)
       (return-from obtain-connection conn)))
   (when (not (= (length (pool-connections pool))
                 (pool-max pool)))
     (return-from obtain-connection (%create-connection pool)))
   #+cmu (mp:process-yield)))

(defun release-connection (conn)
  "releases a database connection"
  (setf (connection-active conn) nil))

(defmacro with-database-connection (pool &body body)
  `(let ((connection (obtain-connection ,pool))
         (results nil))
    (unwind-protect
         (with-database ((connection-database connection))
           (setq results (multiple-value-list (progn ,@body))))
      (release-connection connection))
    (values-list results)))

(defun pool-start-sql-recording (pool &key (types :command))
  "Start all stream in the pool recording actions of TYPES"
  (dolist (con (pool-connections pool))
    (start-sql-recording :type types
			 :database (connection-database con))))

(defun pool-stop-sql-recording (pool &key (types :command))
  "Start all stream in the pool recording actions of TYPES"
  (dolist (con (pool-connections pool))
    (stop-sql-recording :type types
			  :database (connection-database con))))
