;;;; MaiSQL --- Common Lisp Interface Layer to SQL Databases
;;;; This is copyrighted software.  See documentation for terms.
;;;; 
;;;; postgresql.cl --- FFI interface to PostgreSQL on Unix
;;;; 
;;;; Checkout Tag: $Name:  $
;;;; $Id: glue-cmucl.lisp,v 1.1 2001/01/29 19:22:43 jesse Exp $

(in-package :MAISQL-POSTGRESQL)

(defun %pg-database-connection (connection-spec)
  (check-connection-spec connection-spec :postgresql
			 (host db user password &optional port options tty))
  (destructuring-bind (host db user password &optional port options tty)
      connection-spec
    (let ((connection (pqsetdblogin host port options tty db user password)))
      (declare (type psql-conn-ptr connection))
      (when (not (eq (pqstatus connection) :connection-ok))
	;; Connect failed
	(error 'maisql-connect-error
	       :database-type :postgresql
	       :connection-spec connection-spec
	       :errno (pqstatus connection)
	       :error (pqerrormessage connection)))
      connection)))  

(defun %trim-crlf (string)
  (string-right-trim '(#\Return #\Newline) string))

(defun %pg-query (database query-expression)
  (let ((retry nil)
        (result nil)
        (conn-ptr (database-conn-ptr database)))
    (declare (type (alien psql-conn) conn-ptr))
    (tagbody
     execute
       (setq result (pqexec conn-ptr query-expression))
       (when (null-alien result)
         (when (null retry)
           (setq retry t)
           (database-reconnect database)
           (setq conn-ptr (database-conn-ptr database))
           (go execute))
         (error 'maisql-sql-error
                :database database
                :expression query-expression
                :errno nil
                :error (%trim-crlf (pqerrormessage conn-ptr)))))
    result))

(defun %pg-exec (database sql-expression)
  (let ((conn-ptr (database-conn-ptr database))
        (retry nil)
        (result nil))
    (declare (type (alien psql-conn) conn-ptr))
    (tagbody
     execute
       (setq result (pqexec conn-ptr sql-expression))
       (when (null-alien result)
         (when (null retry)
           (setq retry t)
           (database-reconnect database)
           (setq conn-ptr (database-conn-ptr database))
           (go execute))
         (error 'maisql-sql-error
                :database database
                :expression sql-expression
                :errno nil
                :error (%trim-crlf (pqerrormessage conn-ptr)))))
    result))

(defun %pg-query-result-set (database query-expression)
  (let ((conn-ptr (database-conn-ptr database)))
    (declare (type (alien psql-conn) conn-ptr))
    (let ((result (pqexec conn-ptr query-expression)))
      (when (null-alien result)
	(error 'maisql-sql-error
	       :database database
	       :expression query-expression
	       :errno nil
	       :error "PQexec failed to return result structure!"))
      result)))
