/* cp_io.c = input and output routines for circle packing */

#include "cp_head.h"
#include <xview/seln.h>
#include <xview/alert.h>
#include <xview/notice.h>

#define SHMODE 0666
#define MAXCMD 1000

static char *shmBlock=NULL;
static int *lockFlg=NULL;
static int *newFlag=NULL;
static int *vertNum=NULL;
static char *cmdStr=NULL;

extern void textsw_possibly_normalize(),textsw_set_selection();
extern Menu print_menu;
int dum_int;
FILE *fp,*sysfp;

int readcall(struct p_data *p,char *filename)
/* Open file ptr; assumed that overwrite has already been approved 
if p is nonempty; filename has complete path included. */
{
  int i;
  char packname[NAME_MAX];

  if ( (strlen(filename)>NAME_MAX) || ((fp=fopen(filename,"r"))==NULL) )
    goto FAILED;
  i=strlen(filename);
  while (i>0 && filename[i-1]!='/') i--;
  strcpy(packname,filename+i); /* cut off any directory names */
  if (!call_read_pack(fp,p,packname) ) goto FAILED;
  fclose(fp);
  return 1;

 FAILED:
  sprintf(msgbuf,"Read of %s has failed; check name or "
	  "path in Config window.",filename);
  emsg();
  if (fp) fclose(fp);
  return 0;
} /* readcall */

int call_read_pack(FILE *fp,struct p_data *p,char *packname)
/* assume fp open, call read_pack, update pack info. Return from 
read_pack has bits encoding what info was read:
Use this to know what to update.
    1: 0001     basic combinatorics (new pack)
    2: 0002     aims
    3: 0004     inv_dist
    4: 0010     radii
    5: 0020     centers
    6: 0040     angle sums
    7: 0100     vertex_map 
    8: 0200     lists of verts/faces/edges 
    9: 0400     colors (circle or face) */
{
  int i,val,pnum=pack_num(p);
  double halfwidth;
  struct K_data *pK_ptr;
  struct R_data *pR_ptr;
	
  if (!(val=readpack(fp,p)))
    {
      sprintf(msgbuf,"Read for pack %d has failed.",pnum);
      emsg();
      return 0;
    }
  pK_ptr=p->packK_ptr;
  pR_ptr=p->packR_ptr;

  /* depending on bit coding in val, update pack info */

  if (val & 0001) /* combinatoric info (ie, it's a new pack)
		   note: complex_count was done during reading. */
    {
      sprintf(msgbuf,"Read into p%d succeeded; %d circles.",
	      pnum,p->nodecount); 
      msg();
      choose_alpha(p);choose_beta(p);choose_gamma(p);
      facedraworder(p,0);
      p->active_node=p->alpha;
      if (!(val & 0010)) /* no radii read */
	for (i=1;i<=p->nodecount;i++) pR_ptr[i].rad=0.5;
      /* radii read but not centers */
      if ((val & 0010) && !(val & 0020)) 
	comp_pack_centers(p,0,0,2,okerr);
      if (!(val & 0040)) /* no angle sums read */
	fillcurves(p);

      /* fix up pack name and window label */
      if ((val & 0001) && p->file_name[0]=='\0')
	/* no pack name read, use call */
	strcpy(p->file_name,packname);
      sprintf(buf,"%s",p->file_name);
      if (p->hes<0) strcat(buf," (hyp)\0");
      else if (p->hes>0) strcat(buf," (sph)\0");
      else  strcat(buf," (eucl)\0");
      pmsg(pack_num(p),buf);

      /* reset screen data */
      p->screen->cflag_opt=1;
      p->screen->fflag_opt=0;
      p->screen->box=std_real_box;
      halfwidth=(p->screen->box.ry-p->screen->box.ly)
	*p->screen->pix_box.rx/p->screen->pix_box.ry/2.0;
      p->screen->box.lx=(-1)*halfwidth;
      p->screen->box.rx=halfwidth;
      /* must fix so it fits current canvas pixel size */
      if (p->hes>=0)
	{
	  p->screen->unitcircle=0;
	  set_u_cir_flag(pnum,FALSE);
	}
      else 
	{
	  p->screen->unitcircle=1;
	  set_u_cir_flag(pnum,TRUE);
	}
      set_cursor(p->screen,0);
      live_node[pnum]=live_face[pnum]=0;
    }
  else /* partial data only, not a new pack */
    {
      sprintf(msgbuf,"Read of data for p%d succeeded.",pnum); 
      msg();
    }

  return 1;
} /* call_read_pack */

int writecall(struct p_data *p,char *filename,int act)
/* calls routine for writing file from pack p. Assumed that 
overwrite has already been approved if filename	already exists. 
filename includes full path.*/
{
	if (p->status && open_and_write(p,filename,act))
	 {
		write_consolidate(p,filename);
		return 1;
	 }
	sprintf(msgbuf,"Write from %d has failed.",pack_num(p));
	emsg();
	return 0;
} /* writecall */

int write_consolidate(struct p_data *p,char *filename)
/* Update after successful write */
{
	int n,pnum=pack_num(p);
	char *name;

	sprintf(buf,"\nSENT pack %d to disc as %s.  ",pnum,filename);
	n=strlen(filename);
	while (filename[n-1]!='/' && n>0) n--;
	name=filename+n;  /* cut off any directory names */
	strcpy(p->file_name,name); /* update file_name */
		sprintf(buf,"%s",p->file_name);
		if (p->hes<0) strcat(buf," (hyp)\0");
		else if (p->hes==0) strcat(buf," (eucl)\0");
		else strcat(buf," (sph)\0");
		pmsg(pnum,buf);canvmsg(pnum);
	return 1;
} /* write_consolidate */

int open_and_write(struct p_data *p,char *filename,int act)
/* open input file, call write, return 1 if successful */
{
	int i;

	if ( (strlen(filename)>NAME_MAX) || ((fp=fopen(filename,"w"))==NULL) )
		return 0;
	i=writepack(fp,p,act,0);
	fclose(fp);
	return i;
} /* open_and_write */

int read_path_file(char *filename)
/* read a closed path from a file. */
{
	int count;

	if (!stat_file(filename))
	 {
	   sprintf(msgbuf,"The file `%s' cannot be found",filename);
	   emsg();
	   return 0;
	 }
	if ((fp=fopen(filename,"r"))!=NULL) 
	 {
	   count=readpath(fp,&pathlist,&pathlength);
	   fclose(fp);
	   return count;
	 }
	else return 0;
} /* read_path_file */

int write_path_file(char *filename)
/* write a closed path to a file.*/
{
	int count;

	if (pathlist!=NULL && (fp=fopen(filename,"w"))!=NULL
		&& strlen(filename)<=NAME_MAX)
	 {
		count=writepath(fp,pathlist);
		fclose(fp);
		return count;
	 }
	else return 0;
} /* write_path_file */

int copy_packcall(char *datastr)
/* copy p1 into p2; */
{
	int p1,p2,width,height;
	double midpt,halfwidth;
	Pixmap xpm_hold;
	XID xid_hold;

	if ( sscanf(datastr,"%d %d",&p1,&p2) ==2 && p1>=0 && p2!=p1
		&& p1<NUM_PACKS && p2>=0 && p2<NUM_PACKS &&
		packdata[p1].status && !packdata[p2].locks
		&& copy_pack(&packdata[p1],&packdata[p2]) )
	 {
	 		/* save pointers */
	        packdata[p2].screen=&screendata[p2];
	 	xid_hold=screendata[p2].xid;
	 	xpm_hold=screendata[p2].xpm;		
		screendata[p2]=screendata[p1];
		screendata[p2].xid=xid_hold;
		screendata[p2].xpm=xpm_hold;
			/* move screen parameters */
		if (!cmd_mode) screendata[p2].pix_box.rx=width=
			(int)xv_get(canvas_frame[p2],XV_WIDTH);
		if (!cmd_mode) screendata[p2].pix_box.ry=height=
			(int)xv_get(canvas_frame[p2],XV_HEIGHT);
		midpt=(screendata[p2].box.lx+screendata[p2].box.rx)/2.0;
		halfwidth=(screendata[p2].box.ry-screendata[p2].box.ly)
			*width/height/2.0;
		screendata[p2].box.lx=midpt-halfwidth;
		screendata[p2].box.rx=midpt+halfwidth;
		set_cursor(&screendata[p2],0);
		sprintf(buf,"%s",packdata[p2].file_name);
		if (packdata[p2].hes<0) strcat(buf," (hyp)\0");
		else if (packdata[p2].hes==0) strcat(buf," (eucl)\0");
		else strcat(buf," (sph)\0");
		pmsg(p2,buf);
		set_pack_labels();
		return 1;
	 }
	return 0;
} /* copy_packcall */

int seed_start_call(struct p_data *p,char *datastr)
/* bookkeeping */
{
  int draw_flag=1,n,pnum=pack_num(p);
  double halfwidth;

  stripsp(datastr);
  if (!strncmp(datastr,"-q",2)) /* -q flag for quiet, don't draw */
    {
      draw_flag=0;
      datastr[0]=datastr[1]=' ';
      stripsp(datastr);
    }
  if (!sscanf(datastr,"%d",&n) || n<3 || n>MAX_PETALS) n=6;
  if (!seed_pack(p,-1,n)) return 0; /* default geom is hyp */

/* set screen stuff */
  p->screen->box=std_real_box;
  halfwidth=(p->screen->box.ry-p->screen->box.ly)
    *p->screen->pix_box.rx/p->screen->pix_box.ry/2.0;
  p->screen->box.lx=(-1)*halfwidth;
  p->screen->box.rx=halfwidth;
  p->screen->cflag_opt=1;
  p->screen->fflag_opt=0;
  p->screen->unitcircle=1;
  set_cursor(p->screen,0);
  live_node[pnum]=live_face[pnum]=0;
  sprintf(buf,"Seed pack");
  strcpy(p->file_name,buf);
  sprintf(buf,"no header");
  strcpy(p->fhead,buf);
  set_u_cir_flag(pnum,TRUE);
  sprintf(buf,"Seed pack (hyp)");
  pmsg(pnum,buf);canvmsg(pnum); 
/* repack/fix/draw */
  sprintf(buf,"repack -p%d",pnum);
  handle_cmd(buf,&current_p);
  sprintf(buf,"fix -p%d -c",pnum);
  handle_cmd(buf,&current_p);
  if (draw_flag)
    {
      sprintf(buf,"disp -p%d -w -c",pnum);
      handle_cmd(buf,&current_p);
    }
  return 1;
} /* seed_start_call */

int empty_pack(struct p_data *p)
/* reset pack, default size =5000. */
{
	int pnum=pack_num(p);

	alloc_pack_space(p,5000,0);
	sprintf(packlabels[pnum],"Pack %d: empty             ",pnum);
	strcpy(p->file_name,"None\0");
	set_pack_labels();
	return 1;
} /* empty_pack */

int confirm_action(char *message)
/* alert package for confirming a read or write */
{
	int result;
	Event event;
	result=notice_prompt(base_frame,&event,
		NOTICE_MESSAGE_STRINGS,message,0,
		NOTICE_BUTTON_YES,"confirm",
		NOTICE_BUTTON_NO,"Cancel",
		0);
	switch (result)
	{
		case NOTICE_YES: return 1;
		case NOTICE_NO: return 0;
		case NOTICE_FAILED: 
		 {
			strcpy(msgbuf,"Alert failed for some reason");
			emsg();
			return 0;
		 }
	 }
	return 1;
} /* confirm_action */

int print_inq_data(struct p_data *p,int f,char *datastr)
     /* f is code for desired action. Comes either from menu
	choice or from cmd line; see cp_format/cp_action or cp_cmd. */
{
	int i,j,v,count=0,n1,n2,vv[3],length=0;
	int num,pnum=pack_num(p),hits,incompat_flag,*genlist;
	int max_ideg,min_ideg,max_bdeg,min_bdeg,digits=1;
	double dum,b_dist,nc,dumf,duml,tf,tl,tc;
	double b_vert_dist,datum,rad1,rad2,inv_d,act_inv_d,diff;
	double over_err,ic_err;
	/* double accum_ic_err=0.0,ic_mix_max=0.0,ic_large_max=0.0,
	   ic_small_max=0.0, worst_as_err=0.0,accum_as_err=0.0;*/
	complex ctr,e_cent;
	char buff[256],buff2[256],*endptr,format_buf[256];
	struct s_data *scr;
	struct Vertlist *vertlist,*trace;
	struct Edgelist *edgelist,*track;
	struct p_data *p1,*p2;
	struct K_data *pK_ptr;
	struct R_data *pR_ptr;

	pK_ptr=p->packK_ptr;pR_ptr=p->packR_ptr;
	while ((pow(10,(double)digits))*toler < 0.1 
	  && digits<MAX_ACCUR) digits++; /* for output accuracy */
	if (f==30) /* status of remote programs */
	  {
	    j=0;
	    for (i=0;i<NUM_PROC;i++) 
	      if (remote[i].pid)
		{
		  j++;
		  sprintf(msgbuf,"Remote[%d] is %s, moniker = %s.",
			  i,remote[i].program,remote[i].moniker);
		  msg();
		}
	    if (!j) 
	      {
		sprintf(msgbuf,"No remote processes active.");
		msg();
	      }
	    return 1;
	  }
	if (f==21) /* program parameters */
	  {
	    sprintf(msgbuf,"Param: Packing: accuracy=%e  "
		    "(okerr=%e);\n  Packing algorithm cycles=%d;  "
		    "iterations=%d; \n  Misc: post_size=%d; brush=%d;"
		    "act pack=%d; act node=%d",
		    toler,okerr, totalpasses,iterates,sc_print_size,
		    line_thick,pnum,p->active_node);
	    msg();
	    /* give remote routine status */
	    for (i=0;i<NUM_PROC;i++)
	      if (remote[i].pid)
		{
		  sprintf(msgbuf,"  Remote routine %d active: "
			  "pid=%d, program is '%s', moniker = %s.",
			  i,remote[i].pid,remote[i].program,remote[i].moniker);
		  msg();
		}
	    return 1;
	  }
	if (f==33) /* script loaded? */
	  {
	    buff[0]='\0';
	    if (!cmd_mode)
	      {
		if (textsw_append_file_name(script_sw,buff))
		  sprintf(msgbuf,"No script file loaded.");
		else sprintf(msgbuf,"Script filename: %s.",buff);
		msg();
		return 1;
	      }
	    if (!strlen(script_name))
	      sprintf(msgbuf,"No script file loaded.");
	    else sprintf(msgbuf,"Script filename: %s.",script_name);
	    msg();
	    return 1;
	  }
	if (!p->status)
	  {
	    sprintf(msgbuf,"Pack %d is empty.",pnum);
	    emsg();
	    return 0;
	  }
	scr=&screendata[pnum];
	switch (f)
	 {
case 1: /* pack data */
 {
	if (sscanf(datastr,"%d",&pnum)!=1 || pnum<0 || pnum>=NUM_PACKS
	   || !packdata[pnum].status)
		pnum=current_p;
	p1=&packdata[pnum];
	max_bdeg=0;
	for (v=1;v<=p1->nodecount;v++)
	  if (p->packK_ptr[v].bdry_flag && (p->packK_ptr[v].num+1)>max_bdeg)
	    max_bdeg=p->packK_ptr[v].num+1;
	min_bdeg=max_bdeg;
	for (v=1;v<=p1->nodecount;v++)
	  if (p->packK_ptr[v].bdry_flag && (p->packK_ptr[v].num+1)<min_bdeg)
	    min_bdeg=p->packK_ptr[v].num+1;
	min_ideg=max_ideg=p->packK_ptr[p->alpha].num;
	for (v=1;v<=p1->nodecount;v++)
	  if (!p->packK_ptr[v].bdry_flag)
	    {
	      i=p->packK_ptr[v].num;
	      if (i>max_ideg) max_ideg=i;
	      if (i<min_ideg) min_ideg=i;
	    }
	for (i=1;p1->edge_pair[i].edge;i++)
	  length=i;
	i=0;
	for (j=1;j<=length;j++) if (p1->edge_pair[j].mate) i++;
	sprintf(msgbuf,"PACK %d: nodecount=%d; (alp/gam)=(%d/%d); "
		"face#1=%d; red#1=%d\n Euler=%d; "
		"genus=%d; bdry segs: tot/paired=%d/%d; "
		"%d bdry_comp(s);\n",
		pnum,p1->nodecount,p1->alpha,p1->gamma,
		p1->first_face,p1->first_red_face,
		p1->euler,p1->genus,length,i,
		p1->num_bdry_comp);
	if (packdata[pnum].num_bdry_comp>0)
	  {
	    sprintf(msgbuf+strlen(msgbuf)," bdry_start(s): ");
	    for (i=1;i<=packdata[pnum].num_bdry_comp;i++)
	      sprintf(msgbuf+strlen(msgbuf),"%d ",
		      packdata[pnum].bdry_starts[i]);
	    sprintf(msgbuf+strlen(msgbuf),"\n");
	  }
	if (p->hes==0) 
	  {
	    dum=0.0;
	    for (i=1;i<=p->facecount;i++)
	      dum += e_area(pR_ptr[p->faces[i].vert[0]].rad,
			     pR_ptr[p->faces[i].vert[1]].rad,
			     pR_ptr[p->faces[i].vert[2]].rad);
	    sprintf(msgbuf+strlen(msgbuf)," eucl area=%f; ",dum);
	  }
	sprintf(msgbuf+strlen(msgbuf)," deg(max/min): int=(%d/%d), "
		"bdry=(%d/%d).",
		max_ideg,min_ideg,max_bdeg,min_bdeg);
	msg();
	return 1;
 }
case 2: /* vertex data */
  {
    if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
      {
	trace=vertlist;
	do 
	  {
	    v=trace->v;
	    sprintf(msgbuf,"\nData on vert# %d, p%d: rad=% .5e, "
		    "ang sum=% .5ePi, aim=% .5ePi, bdry?=%d, "
		    "#star=%d, plot_flag? %d.",
	      v,pnum,radius(p,v),
	      p->packR_ptr[v].curv/M_PI,p->packR_ptr[v].aim/M_PI,
	      p->packK_ptr[v].bdry_flag,p->packK_ptr[v].num,
	      p->packK_ptr[v].plot_flag);
	    msg();
	    count++;
	    trace=trace->next;
	  } while (trace!=NULL && count < 5);
	vert_free(&vertlist); /* free any excess */
	return count;
      }
    else v=p->active_node;
    sprintf(msgbuf,"\nData on vert# %d, p%d: rad=% .5e, "
	    "ang sum=% .5ePi, aim=% .5ePi, bdry?=%d, "
	    "#star=%d, plot_flag? %d.",
      v,pnum,radius(p,v),
      p->packR_ptr[v].curv/M_PI,p->packR_ptr[v].aim/M_PI,
      p->packK_ptr[v].bdry_flag,p->packK_ptr[v].num,
      p->packK_ptr[v].plot_flag);
    msg();
    return 1;
  }
case 7: /* flower */
case 8: /* selected radii */
case 9: /* selected angle sums */
case 10: /* selected aims */
case 15: /* selected model curvatures */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		if (f==7) sprintf(msgbuf,"Flowers:");
		else if (f==8) sprintf(msgbuf,"Radii:");
		else if (f==9) sprintf(msgbuf,"Angle sums/Pi:");
		else if (f==10) sprintf(msgbuf,"Aims/Pi:");
		else if (f==15) sprintf(msgbuf,"Kappa:");
		trace=vertlist;
		do 
		  {
		    if (f==8) datum=radius(p,trace->v);
		    else if (f==9) datum=p->packR_ptr[trace->v].curv/M_PI;
		    else if (f==10) datum=p->packR_ptr[trace->v].aim/M_PI;
		    else if (f==15)
		      {
			kappa_calc(p,trace->v,&datum);
			datum=datum*(fabs(datum));
		      }
		    else if (f==7)
		      {
			sprintf(buff,"\np%d,v%d,deg=%d,flower: ",
				pnum,trace->v,
				p->packK_ptr[trace->v].num);
			strcat(msgbuf,buff);
			for (i=0;i<=p->packK_ptr[trace->v].num;i++)
			  {
			    sprintf(buff,"%d  ",
				    p->packK_ptr[trace->v].flower[i]);
			    strcat(msgbuf,buff);
			  }
			*buff='\0';
		      }
		    if (f!=7) sprintf(buff,"%d % .5e; ",trace->v,datum);
		    strcat(msgbuf,buff);
		    count++;
		    trace=trace->next;
		  } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 16: /* antipodal point */
  {
    endptr=datastr;
    if (!(v=grab_one_vert(p,&endptr))) v=p->alpha;
    for (i=1;i<=p->nodecount;i++) p->packK_ptr[i].util_flag=0;
    p->packK_ptr[v].util_flag=1;
    if (!(genlist=label_generations(p,-1,&j,&i)))
      return 0;
    free(genlist);
    sprintf(msgbuf,"Antipodal (max comb distance) to %d is "
	       "vertex %d.",v,j);
    msg();
    return 1;
  }
case 11: /* data on selected faces */
  {
    if ( (vertlist=face_link_parse(p,datastr,&endptr,&hits)) != NULL)
      {
	trace=vertlist;
	do
	  {
	    incompat_flag=0.0;
	    for (j=0;j<3;j++) vv[j]=p->faces[trace->v].vert[j];
	    sprintf(msgbuf,"face %d data, p%d: verts (%d,%d,%d), "
		    "plot v%d, plot_flag? %d.\n",
	      trace->v,pnum,vv[0],vv[1],vv[2],
	      vv[p->faces[trace->v].index_flag],
	      p->faces[trace->v].plot_flag);
	    msg();
	    sprintf(msgbuf,"  Rad: (%f,%f,%f).\n  Angles: (%f,%f,%f).\n",
		    radius(p,vv[0]),radius(p,vv[1]),radius(p,vv[2]),
		    comp_single_angle(p,vv[0],vv[1],vv[2],&incompat_flag),
		    comp_single_angle(p,vv[1],vv[2],vv[0],&incompat_flag),
		    comp_single_angle(p,vv[2],vv[0],vv[1],&incompat_flag));
	    msg();
	    if (incompat_flag)
	      {
		sprintf(msgbuf," NOTE: radii/overlaps incompatible.\n");
		emsg();
	      }
	    sprintf(msgbuf,"  Overlaps (0->1->2->0): assigned=");
	    if (p->overlap_status)
	      sprintf(buff,"%f,%f,%f\n   errors "
		      "(assigned-actual)=%f,%f,%f.\n",
		    pK_ptr[vv[0]].overlaps[nghb(p,vv[0],vv[1])],
		    pK_ptr[vv[1]].overlaps[nghb(p,vv[1],vv[2])],
		    pK_ptr[vv[2]].overlaps[nghb(p,vv[2],vv[0])],
		    invdist_err(p,vv[0],vv[1]),
		    invdist_err(p,vv[1],vv[2]),
		    invdist_err(p,vv[2],vv[0]));
	    else sprintf(buff,"tangency\n   errors "
			 "(assigned-actual)=%f,%f,%f.\n",
		    invdist_err(p,vv[0],vv[1]),
		    invdist_err(p,vv[1],vv[2]),
		    invdist_err(p,vv[2],vv[0]));
	    strcat(msgbuf,buff);
	    msg();
	    count++;
	    trace=trace->next;
	  } while (trace!=NULL && count < 5);
	vert_free(&vertlist); /* free any excess */
      }
    return count;
  }
case 12: /* overlaps */
 {
	sprintf(msgbuf,"Overlaps, p%d:",pnum);
	if (!p->overlap_status)
	 {
		strcat(msgbuf," All set to 0.0, (default=tangency case).");
		msg();
		return count;
	 }
	if ((edgelist=node_pair_link(p,datastr,&endptr,&hits))!=NULL)
	 {
		track=edgelist;
		do {
		   inv_d=pK_ptr[track->v].
			overlaps[nghb(p,track->v,track->w)];
		   if (inv_d>=0 && inv_d<=1) /* want overlap */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d<=1)
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi); "
				  "actual=cos(% .5e Pi).\n",
				track->v,track->w,
				acos(inv_d)/M_PI,acos(act_inv_d)/M_PI);
			else
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi); "
				  "actual=*% .5e.\n",
				track->v,track->w,
				acos(inv_d)/M_PI,act_inv_d);
			strcat(msgbuf,buff);
			count++;
		    }
		   else if (inv_d>1) /* want separated */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d>1)
			  sprintf(buff," (%d,%d): aim=*% .5e; "
				  "actual=*% .5e.\n",
				track->v,track->w,inv_d,act_inv_d);
			else
			  sprintf(buff," (%d,%d): aim=*% .5e; "
				  "actual=cos(% .5e Pi).\n",
				track->v,track->w,inv_d,acos(act_inv_d)/M_PI);
			strcat(msgbuf,buff);
			count++;
		    }
		   else /* assigned overlaps exceeding Pi/2 */
		    {
			act_inv_d=comp_inv_dist(p,track->v,track->w);
			if (act_inv_d>1)
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi) "
				  "(NOTE: > Pi/2); actual=*% .5e.\n",
				track->v,track->w,acos(inv_d)/M_PI,act_inv_d);
			else
			  sprintf(buff," (%d,%d): aim=cos(% .5e Pi) "
				  "(NOTE: > Pi/2); actual=cos(% .5e Pi).\n",
				track->v,track->w,
				acos(inv_d)/M_PI,acos(act_inv_d)/M_PI);

			strcat(msgbuf,buff);
			count++;
		    }
		   track=track->next;
		 } while (track!=NULL);
		edge_free(&edgelist);
	 }
	msg();
	return count;
 } 
case 14: /* altn_rad alternative radii: eucl, hyp. */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		if (p->hes<0) sprintf(msgbuf,"Radii, p%d (v  eucl, hyp):",
			pnum);
		else if (p->hes==0) sprintf(msgbuf,"Eucl:");
		else sprintf(msgbuf,"Sph:");
		trace=vertlist;
		do {
			if (p->hes<0)		   
			   h_to_e_data(p->packR_ptr[trace->v].center,
				   p->packR_ptr[trace->v].rad,&e_cent,&datum);
			else datum=p->packR_ptr[trace->v].rad;
			sprintf(buff," %d  % .5e, ",trace->v,datum);
			strcat(msgbuf,buff);
			if (p->hes<0)
			 {
				datum=p->packR_ptr[trace->v].rad;
			   	if (datum<=0) sprintf(buff,"inf ");
			   	else {
					datum=(-log(datum));
					sprintf(buff,"% .5e ",datum);}
			   	strcat(msgbuf,buff);
			 }
			count++;
		   	trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 13: /* selected centers */
 {
   sprintf(msgbuf,"Centers: ");
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   ctr=p->packR_ptr[trace->v].center;
		   sprintf(buff,"%d (% .12e % .12e),\n |.|=% .12e ; ",
			   trace->v,ctr.re,ctr.im,cAbs(ctr));
		   strcat(msgbuf,buff);
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 3);
		vert_free(&vertlist); /* free any excess */
		msg();
	 }
	return count;
 }
case 20: /* ?screendata */
 {
	sprintf(msgbuf,
		"Screen %d: lower left=(%d, %d), upper right=(%d,%d).  ",
		pnum,(int)(scr->box.lx),(int)(scr->box.ly),
		(int)(scr->box.rx),(int)(scr->box.ry));
	msg();
	return 1;
 }
case 25: /* ?pk_status: curvature/overlap errors */
 {
/* Collect data, set vertex marks with codes: 
  -1 = not subject to change
  0 = packed within tolerance
  1 = incompat, too small in some faces
  2 = incompat, too large in some faces
  3 = incompat, mixed (too large in some faces, too small in others)
*/

   double accum_ic_err=0.0,ic_mix_max=0.0,ic_large_max=0.0;
   double ic_small_max=0.0,worst_as_err=0.0,accum_as_err=0.0;
   int ic_small_vert=0,ic_large_vert=0,ic_mix_vert=0;
   int free_count=0,worst_as_vert=0;

   for (i=1;i<=p->nodecount;i++)
     {
       if (p->packR_ptr[i].aim<0) p->packK_ptr[i].mark=-1;
       else 
	 {
	   free_count++;
	   if (p->hes<0)
	     h_as_overlap(p,i,p->packR_ptr[i].rad,
	       &p->packR_ptr[i].curv,&p->packK_ptr[i].mark,&ic_err);

	   if (!p->packK_ptr[i].mark)
	     {
	       diff=fabs(p->packR_ptr[i].curv-p->packR_ptr[i].aim);
	       if (diff > worst_as_err) 
		 {
		   worst_as_err=diff;
		   worst_as_vert=i;
		 }
	       accum_as_err += diff;
	     }
	   else if (p->packK_ptr[i].mark==1)
	     {
	       if (ic_err>ic_small_max) 
		 {
		   ic_small_max=ic_err;
		   ic_small_vert=i;
		 }
	       accum_ic_err += ic_err;
	     }
	   else if (p->packK_ptr[i].mark==2)
	     {
	       if (ic_err>ic_large_max)
		 {
		   ic_large_max=ic_err;
		   ic_large_vert=i;
		 }
	       accum_ic_err += ic_err;
	     }
	   else if (p->packK_ptr[i].mark==3)
	     {
	       if (ic_err>ic_mix_max)
		 {
		   ic_mix_max=ic_err;
		   ic_mix_vert=i;
		 }
	       accum_ic_err += ic_err;
	     }
	 }
     }
   over_err=0.0;
   if (p->overlap_status)
     for (i=1;i<p->nodecount;i++)
       for (j=0;j<p->packK_ptr[i].num;j++)
	 {
	   if (i<(v=p->packK_ptr[i].flower[j]))
	     {
	       diff=fabs(p->packK_ptr[i].overlaps[j]-
			 comp_inv_dist(p,i,v));
	       if (diff > okerr && diff > over_err)
		 {n1=i;n2=v;over_err=diff;} 
	       /* worst, so far */
	     }
	 }
 
 /* printing results */

   sprintf(msgbuf,"Pack status, p%d: %d verts subject to change.",
	   pack_num(p),free_count);
   msg();
   sprintf(format_buf,"   %% .%de = accum ang sum err; "
	   "worst = %% .%de at v%%d.",digits,digits);
   sprintf(msgbuf,format_buf,accum_as_err,worst_as_err,worst_as_vert);
   msg();
   if (p->overlap_status && accum_ic_err > 0.0)
     {
       sprintf(msgbuf,"   INCOMPATIBILITIES have occurred; verts "
	       "marked:\n          1 = too small,2 = too large, 3 = mixed.");
       msg();
       window_bell(msg_sw);
       sprintf(msgbuf,"   %f = accum incompatibility err",accum_ic_err);
       if (ic_small_vert)
	 {
	   sprintf(buff,"\n     worst too small = %f at v%d",
		   ic_small_max,ic_small_vert);
	   strcat(msgbuf,buff);
	 }
       if (ic_large_vert)
	 {
	   sprintf(buff,"\n     worst too large = %f at v%d",
		   ic_large_max,ic_large_vert);
	   strcat(msgbuf,buff);
	 }
       if (ic_mix_vert)
	 {
	   sprintf(buff,"\n     worst mixed = %f at v%d.",
		   ic_mix_max,ic_mix_vert);
	   strcat(msgbuf,buff);
	 }
       msg();
       if (over_err>0.0)
	 {
	   sprintf(msgbuf,"   OVERLAP ERRORS: worst is %f, edge (%d,%d).",
	       over_err,n1,n2);
	   msg();
	   window_bell(msg_sw);
	 }
       else
	 {
	   sprintf(msgbuf,"   Overlaps within tolerances.");
	   msg();
	 }
     }
   return 1;
 }
case 26: /* distance to bdry */
 {
	h_to_e_data(p->packR_ptr[p->alpha].center,
		p->packR_ptr[p->alpha].rad,&ctr,&dum);
	bdry_dist(p,ctr,&b_dist,&b_vert_dist);
	sprintf(msgbuf,"Euclidean distance from alpha to the "
		"boundary polygon is % .5e.",b_dist);
	msg();
	return 1;
 }
case 27: /* edge_pairing. Give mobius transforms */
  {
    n1=0;
    if (p->edge_pair[2].edge)
      {
	sprintf(msgbuf,"Edge pairings for p%d:\n",pnum);
	msg();
	for (i=1;p->edge_pair[i].edge;i++)
	  {
	    if (p->edge_pair[i].mate)
	      {
		for (j=1;j<i;j++) 
		  if (p->edge_pair[j].edge==p->edge_pair[i].mate
		      && p->edge_pair[j].edge_indx==p->edge_pair[i].mate_indx)
		    {
		      n1=j;
		      j=i;
		    }
		if (!n1) /* found mate, print mobius */
		  {
		    sprintf(msgbuf,"  Pairing %d <--> %d as 2x2 matrix:\n",
			    i,n1);
		    msg();
		    sprintf(msgbuf,"     [ %f + %f I, %f + %f I;\n",
			    p->edge_pair[i].mob.a.re,p->edge_pair[i].mob.a.im,
			    p->edge_pair[i].mob.b.re,p->edge_pair[i].mob.b.im);
		    msg();
		    sprintf(msgbuf,"      %f + %f I, %f + %f I ]\n",
			   p->edge_pair[i].mob.c.re,p->edge_pair[i].mob.c.im,
			   p->edge_pair[i].mob.d.re,p->edge_pair[i].mob.d.im);
		    msg();
		  }
	      }
	  }
      }
    else
      {
	sprintf(msgbuf,"No edge pairing for p%d.",pnum);
	msg();
      }
    return 1;
  }
case 28: /* ratio_ftn <p,q>. Gives (eucl radius of alpha vert of q)
divided by (eucl rad of alpha vert of p) */
 {
	if (sscanf(datastr,"%d %d",&n1,&n2)!=2 
		|| n1< 0 || n1>=NUM_PACKS || !packdata[n1].status 
		|| n2< 0 || n2>=NUM_PACKS || !packdata[n2].status)
	 {sprintf(msgbuf,"Specify two active packs.");emsg();return 0;}
	p1=&packdata[n1];p2=&packdata[n2];
	if (p1->hes>0 || p2->hes>0)
	 {sprintf(msgbuf,"One of packs is spherical.");emsg();return 0;}
	if (p1->hes<0) h_to_e_data(p1->packR_ptr[p1->alpha].center,
				   p1->packR_ptr[p1->alpha].rad,&e_cent,&rad1);
	else rad1=p1->packR_ptr[p1->alpha].rad;
	if (p2->hes<0) h_to_e_data(p2->packR_ptr[p2->alpha].center,
				   p2->packR_ptr[p2->alpha].rad,&e_cent,&rad2);
	else rad2=p2->packR_ptr[p2->alpha].rad;
	sprintf(msgbuf,"Radii comparisons (for alpha vert):\n  "
		"Eucl:  ratio p%d/p%d = % .5e;     "
		"difference p%d-p%d = % .5e.",
		n2,n1,rad2/rad1,n2,n1,rad2-rad1);
	msg();
	if (p->hes<0)
	 {
		rad2=-log(p2->packR_ptr[p2->alpha].rad);
		rad1=-log(p1->packR_ptr[p1->alpha].rad);
		sprintf(msgbuf,"  Hyp:   ratio p%d/p%d = % .5e;   "
			"difference p%d-p%d = % .5e.",
		n2,n1,rad2/rad1,n2,n1,rad2-rad1);
		msg();
	 }
	return 1;
 }
case 29: /* give conductances, transition probabilities for nodes. */
 {
	if (p->hes>=0) return 0;  /* Only for hyp case at this time. */
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
		trace=vertlist;
		do {
		   v=trace->v;
		   num=pK_ptr[v].num;
		   sprintf(msgbuf,"Random walk data, pack %d, "
			   "vert %d: neighbors = ",
		   	pnum,v);
		   for (i=0;i<=num;i++) 
		    {
			sprintf(buff," %d",pK_ptr[v].flower[i]);
			strcat(msgbuf,buff);
		    }
		   msg();
		   sprintf(msgbuf,"  Conductances:  total node = % .5e; "
			   "twin = % .5e",
			nc=node_conductance(p,v),tc=twin_conductance(p,v));
		   msg();
		   dumf=Cvw(pR_ptr[v].rad,0.0,
			pR_ptr[pK_ptr[v].flower[0]].rad,
			pR_ptr[pK_ptr[v].flower[1]].rad,1,p->hes);
		   tf=dumf/nc;
		   duml = Cvw(pR_ptr[v].rad,
			pR_ptr[pK_ptr[v].flower[num-1]].rad,
			pR_ptr[pK_ptr[v].flower[num]].rad,0.0,2,p->hes);
		   tl=duml/nc;
		   if (!pK_ptr[v].bdry_flag) {dumf += duml;tf += tl;}
		   sprintf(msgbuf,"  % .5e",dumf);
		   sprintf(buff2,"  % .5e",tf);
		   for (i=1;i<num;i++)
		    {
			dum=Cvw(pR_ptr[v].rad,
				pR_ptr[pK_ptr[v].flower[i+1]].rad,
				pR_ptr[pK_ptr[v].flower[i]].rad,
				pR_ptr[pK_ptr[v].flower[i-1]].rad,0,p->hes);
			tf=dum/nc;
			sprintf(buff,"  % .5e",dum);
			strcat(msgbuf,buff);
			sprintf(buff,"  % .5e",tf);
			strcat(buff2,buff);
		    }
		   if (pK_ptr[v].bdry_flag)
		    {
			sprintf(buff,"  % .5e",duml);
			strcat(msgbuf,buff);
			sprintf(buff,"  % .5e",tl);
			strcat(buff2,buff);
		    }
		   msg(); 	/* print conductances */
		   sprintf(msgbuf,"  Transition Probabilities:  "
			   "leakage = % .5e\n",
			tc/nc);
		   strcat(msgbuf,buff2);
		   msg(); 	/* print transition probabilities */
		   count++;
		   trace=trace->next;
		 } while (trace!=NULL && count < 5);
		vert_free(&vertlist); /* free any excess */
	 }
	return count;
 }
case 31: /* length of bdry */
 {
	if ( (vertlist=node_link_parse(p,datastr,&endptr,&hits)) != NULL)
	 {
	   sprintf(msgbuf,"Bdry counts:");
	   trace=vertlist;
	   do {
	     if ((hits=bdry_comp_count(p,trace->v)))
	       {
		 sprintf(buff," v%d --> count=%d, ",trace->v,hits);
		 strcat(msgbuf,buff);
		 count++;
	       }
	     trace=trace->next;
	   } while (trace!=NULL && count < 5);
	   vert_free(&vertlist);
	   msg();
	   return 1;
	 }
	sprintf(msgbuf,"Bdry length? Given vert doesn't seem to be "
		"in the boundary.");
	msg();
	return 0;
 }
	 } /* end of switch */
	return 0;
} /* print_inq_data */

int close_script_proc(Panel_item item,int value,Event *event)
{xv_set(script_frame,WIN_SHOW,FALSE,0);return 1;}

int reset_script_proc(Panel_item item,int value,Event *event)
{reset_script();return 1;}

int reset_script()
{
	Textsw_index first;

	first=(Textsw_index)0;	
	xv_set(script_sw,TEXTSW_INSERTION_POINT,first,0);
	textsw_possibly_normalize(script_sw,first);
	script_file_data_mark=script_file_cmd_mark
		=script_file_path_mark=(Textsw_mark)
		textsw_add_mark(script_sw,first,TEXTSW_MARK_DEFAULTS);
	return 1;
} /* reset_script */

int find_script_proc(Panel_item item,int value,Event *event)
{find_next_script_cmd(0,0);return 1;}

int next_script_proc(Panel_item item,int value,Event *event)
/* get next full line of commands from the script command file */
{
	if (find_next_script_cmd(0,1)==(-1)) return 1;
	sort_cmd(next_script_cmd,&current_p);
	free(next_script_cmd);next_script_cmd=NULL;
	return 1;
} /* next_script_proc */

int read_infile(struct p_data *p,char *datastr)
/* read pack data from script file */
{
  int Pid,flag,count=0;
  long save_pos,pos;
  char *databuf=NULL,*ptr;
  char tmpfile[NAME_MAX],filename[NAME_MAX];
  FILE *tmp_fp;

  filename[0]='\0';
  if (!cmd_mode)
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,1))==NULL ) 
	 {
		sprintf(msgbuf,"No pack data in script file with "
			"given label.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
		/* first have to put data in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of script file failed.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) ptr++;
	fclose(tmp_fp);
	if ( (tmp_fp=fopen(tmpfile,"r"))==NULL ) 
	 {
		if (databuf) free(databuf);
		return 0;
	 }
	if (!call_read_pack(tmp_fp,p,filename)) return 0;
	if (tmp_fp) fclose(tmp_fp);
	if (databuf) free(databuf);
	return 1;
    }
  else
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	    || (flag=find_name(script_fp,filename,&pos))<=0
	    || flag>2)
	 {
		sprintf(msgbuf,"No NODECOUNT or CHECKCOUNT "
			"data in script file with label %s.",
			filename);
		emsg();
		return 0;
	 }
	save_pos=ftell(script_fp);
	fseek(script_fp,pos,SEEK_SET);
	count=call_read_pack(script_fp,p,filename);
	fseek(script_fp,save_pos,SEEK_SET);
	return count;
    }
} /* read_infile */

int cmds_infile(char *datastr)
/* read string of cmds from script file */
{
  int Pid,flag,count=0;
  long save_pos,pos;
  char *databuf=NULL,*ptr;
  char tmpfile[NAME_MAX],filename[NAME_MAX],command[BUFSIZE];
  FILE *tmp_fp;

  filename[0]='\0';
  if (!cmd_mode)
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,3))==NULL ) 
	 {
		sprintf(msgbuf,"Didn't find labeled cmds stream "
			"in script file.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
		/* first have to put cmds in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of script data failed.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) ptr++;
	fclose(tmp_fp);
	fexec_call(tmpfile);
	if (databuf) free(databuf);
	return 1;
    }
  else
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	    || (flag=find_name(script_fp,filename,&pos))<=0
	    || flag!=4)
	 {
		sprintf(msgbuf,"No CMDS data in script file with "
			"label %s.",filename);
		emsg();
		return 0;
	 }
	save_pos=ftell(script_fp);
	fseek(script_fp,pos,SEEK_SET);
	while (fgets(command,BUFSIZE-1,script_fp) && command!='\0')
	 	count += handle_cmd(command,&current_p);
	fseek(script_fp,save_pos,SEEK_SET);
	return count;
    }
} /* cmds_infile */

int path_infile(char *datastr)
/* read path data from script file */
{
  int Pid,flag,count;
  long save_pos,pos;
  char *databuf=NULL,*ptr;
  char tmpfile[NAME_MAX],filename[NAME_MAX];
  FILE *tmp_fp;

  filename[0]='\0';
  if (!cmd_mode)
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	   || (databuf=ptr=find_script_data(filename,2))==NULL ) 
	 {
		sprintf(msgbuf,"No path data in script file with "
			"given label.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
		/* first have to put data in tmp file */
	Pid=(int)getpid();
	sprintf(tmpfile,"/tmp/%d%s",Pid,filename);
	if ( (tmp_fp=fopen(tmpfile,"w"))==NULL )
	 {
		sprintf(msgbuf,"Temporary store of path data failed.");
		emsg();
		if (databuf) free(databuf);
		return 0;
	 }
	while ( *ptr!='\0' && (fputc(*ptr,tmp_fp)!=EOF) ) 
		ptr++;
	fclose(tmp_fp);
/*	sysfp=popen("/bin/sh","w");
	fprintf(sysfp,"chmod 666 /tmp/%s",filename);
	fclose(sysfp);
*/
	if ( (tmp_fp=fopen(tmpfile,"r"))==NULL ) 
	 {
		if (databuf) free(databuf);
		return 0;
	 }
	readpath(tmp_fp,&pathlist,&pathlength); 
	fclose(tmp_fp);
	if (databuf) free(databuf);
	return 1;
    }
  else
    {
	if ((sscanf(datastr,"%s",filename)!=1) 
	    || (flag=find_name(script_fp,filename,&pos))<=0
	    || flag!=3)
	  {
	    sprintf(msgbuf,"No PATH data in script file with "
		    "label %s.",filename);
	    emsg();
	    return 0;
	  }
	save_pos=ftell(script_fp);
	fseek(script_fp,pos,SEEK_SET);
	count=readpath(script_fp,&pathlist,&pathlength);
	fseek(script_fp,save_pos,SEEK_SET);
	return count;
    }
} /* path_infile */

int fexec_call(char *datastr)
/* open file, execute commands */
{
	int count=0;
	char filename[256],command[BUFSIZE];
	FILE *fp;
	
	sscanf(datastr,"%s",filename);
	if ((fp=fopen(filename,"r"))==NULL)
	 {
	 	sprintf(msgbuf,"fexec error opening %s.",
	 		filename);
	 	emsg();
	 	return count;
	 }
	while (fgets(command,BUFSIZE-1,fp) && command!='\0')
	 	count += handle_cmd(command,&current_p);
	fclose(fp);
	return count;
} /* fexec_call */
	
/* ====================== shmem routines ======================
shared memory routines from Monica Hurdal and Kurt Shaper (Minn)
to pass commands between CirclePack and IDL (or other external
program) */

int c_initmem()
{
	int shmid;
	key_t key;
	int blockSize;
	char *keyName;

	/* create the key */
	keyName = (char *) getenv("SHMEMKEY");
	if (keyName == NULL)
		keyName = "shmem.key";
	key = ftok(keyName,'c');
	if ((int)key < 0) {
		perror("error creating key");
		return(1);
	}

	/* create share memory block and write id to interface file */
	blockSize = 3*sizeof(int) + MAXCMD*sizeof(char);
	shmid = shmget(key, blockSize, IPC_CREAT|SHMODE);
	if (shmid < 0) {
		perror("error getting shared memory block");
		return(2);
	}

	/* map the variable pointers to the shared memory block */
	shmBlock = (char *) shmat(shmid, 0, 0);
	if (!((long)shmBlock+1)) {
		perror("error attaching to shared memory block");
		return(3);
	}
	lockFlg = (int *) shmBlock;
	newFlag = (int *) (shmBlock+1*sizeof(int));
	vertNum = (int *) (shmBlock+2*sizeof(int));
	cmdStr = (char *) (shmBlock+3*sizeof(int));
	*lockFlg = 0;
	return(0);
} /* c_initmem */

int c_rdmem(int *vnum,char *vstr)
{
	int loopCount;

	/* if not initialized, error */
	if (lockFlg == NULL) {
		perror("shared memory not initialized");
		return(1);
	}

	/* if nothing new, just bail */
	if (*newFlag == 0) {
		*vnum = *vertNum;
		strncpy(vstr,cmdStr,MAXCMD);
		return(-1);
	}

	/* lock memory area */
	loopCount = 0;
	while (*lockFlg != 0) {
		usleep(100);
		loopCount++;
		if (loopCount > 100) {
			perror("error locking memory segment");
			return(2);
		}
	}

	/* ---------------------- critical section ------------------- */
	*lockFlg = 1;
	*vnum = *vertNum;
	strncpy(vstr,cmdStr,MAXCMD);
	*newFlag = 0;
	*lockFlg = 0;
	/* ---------------------- critical section ------------------- */
	return(0);
} /* c_rdmem */

int c_wrmem(int *vnum,char *vstr)
{
	int loopCount;

	/* if not initialized, error */
	if (lockFlg == NULL) {
		perror("shared memory not initialized");
		return(1);
	}

	/* lock memory area */
	loopCount = 0;
	while (*lockFlg != 0) {
		usleep(100);
		loopCount++;
		if (loopCount > 100) {
			perror("error locking memory segment");
			return(2);
                }
	}
	 
	/* ---------------------- critical section ------------------- */
	*lockFlg = 1;
	*vertNum = *vnum;
	strncpy(cmdStr,vstr,MAXCMD);
	*newFlag = 1;
	*lockFlg = 0;
	/* ---------------------- critical section ------------------- */
	return(0);
} /* c_wrmem */

int cmd_import(char *datastr)
/* import a string from shared memory */
{
	int stat;
	int vnum = 0;

        if (init_shmem == 0) {
           stat = c_initmem(); 
           if (stat > 0) {
              if (stat == 1) {
                 sprintf(msgbuf,"File shmem.key not found.");
                 emsg();
              }
              sprintf(msgbuf,"Not able to share memory with "
		      "external application.");
              emsg();
              return 0;
           }
        } else init_shmem = 1;
        stat = c_rdmem(&vnum,buf);
        if (stat > 0) {
           sprintf(msgbuf,"Error importing command from "
		   "external application.");
           emsg();
           return 0;
        } else if (stat < 0) {
           sprintf(msgbuf,"No change - nothing to do.");
           msg();
           return 1;
        } else {
           sprintf(msgbuf,"Importing %s",buf);
           msg();
           return(sort_cmd(buf,&current_p));
        }
} /* cmd_import */
         
int cmd_export(struct p_data *p,char *datastr)
/* export a string to shared memory */
{
        int stat,vnum = 0,hits;
	char *dpoint,cmd[128],outstr[128],*endptr;
	struct Vertlist *holdv;
 
        if (init_shmem == 0) {
           stat = c_initmem();
           if (stat > 0) {
              if (stat == 1) {
                 sprintf(msgbuf,"File shmem.key not found.");
                 emsg();
              }
              sprintf(msgbuf,"Not able to share memory with "
		      "external application.");
              emsg();
              return 0;
           }
        } else init_shmem = 1;
	dpoint=datastr;
	/* pre-processing datastr */
	if (!grab_next(&dpoint,cmd)) return 0; /* string was empty */
	switch(cmd[0])
	  {
	  case 'c':
	    {
	      if (!strncmp(cmd,"center_vert",11)) /* ======= center_vert */
		{
		  if ((holdv=node_link_parse(p,dpoint,&endptr,&hits))!=NULL)
		    {
		      sprintf(outstr,"center_vert %d",holdv->v);
		      vnum = holdv->v;
		      vert_free(&holdv);
		    }
		  else goto BAIL;
		  break;
		}
	      break;
	    }
	  }/* end of switch */
        if ((stat = c_wrmem(&vnum,outstr))>0) goto BAIL;
	sprintf(msgbuf,"Exporting %s",outstr);
	msg();
	return 1;	
 BAIL:
	sprintf(msgbuf,"Error exporting command to "
		"external application.");
	emsg();
	return 0;

} /* cmd_export */

/* ---------------------------------- some technical routines ---------- */
	
char *get_selection(int kill)
/* gets current primary selection; remove
highlighting in scratch subwindow if kill true. */
{
	static char datastr[BUFSIZE];
	Seln_holder holder;
	Seln_request *buffer;
	holder=seln_inquire(SELN_PRIMARY);
	buffer=seln_ask(&holder, SELN_REQ_CONTENTS_ASCII,0,0);
	strncpy(datastr,buffer->data+sizeof(Seln_attribute),BUFSIZE-10);
	if (kill) textsw_set_selection(scratch_sw,0,0,1);
	return (datastr);
} /* get_selection */

int stat_file(char *filename)
/* returns 1 if filename exists */
{
  struct stat statbuf;
  return( (strlen(filename)<=NAME_MAX) && (stat(filename, &statbuf)>=0) );
} /* stat_file */

int print_proc(Menu menu,Menu_item menuitem)
/* act on print_menu */
{
	int pnum,cflag,fflag,act = (int)xv_get(menuitem,MENU_VALUE)-1;
	char *datastr;
	struct p_data *p;
	char buff[128];

	p=&packdata[current_p];
	if (act>1 && act<11 && !custom_fp)
	 {
		strcpy(msgbuf,"The 'custom' postscript file has "
		       "not been opened.");
		emsg();
		return 1;
	 }
	switch (act)
	 {
case 1:  /* open default custom file (lose previous contents) */
  {
    if (!cmd_mode)
      strcpy(ps_file_name,(char *)xv_get(ps_file_item,PANEL_VALUE));
    open_psfile(p,1,"\0",&custom_fp,ps_file_name,user_name,
		CPVersion); return 1;
  }
case 3: sprintf(buff," -u "); break;	/* unit circle */ 
case 4: /* selected open circles */
 {
	if( strlen(datastr=get_selection(1)) ) 
		sprintf(buff," -c %s",datastr);
	else sprintf(buff," -c a");
	break;
 }
case 5:  /* selected filled circles */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buff," -cf %s",datastr);
	else sprintf(buff," -cf a");
	break;
 }
case 6: /* selected open faces */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buff," -f %s",datastr);
	else sprintf(buff," -f a");
	break;
 }
case 7: /* selected filled faces */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buff," -ff %s",datastr);
	else sprintf(buff," -ff a");
	break;
 }
case 8: /* selected edges */
 {
	if (!strlen(datastr=get_selection(1)))
	 {strcpy(msgbuf,"You must select vertices. ");emsg();break;}
	else sprintf(buff," -e %s",datastr);
	break;
 }	
case 9: /* selected vertex labels */
 {
	if ( strlen(datastr=get_selection(1)) )
		sprintf(buff," -nc %s",datastr);
	else sprintf(buff," -nc a");
	break;
 }
case 10: /* selected face labels */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buff," -nf %s",datastr);
	else sprintf(buff," -nf a");
	break;
 }
case 11: /* caption */
 {
	if ( strlen(datastr=get_selection(1)) ) 
		sprintf(buff," -t %s",datastr);
	else sprintf(buff," -t");
	break;
 } 
case 12: /* record display option objects from selected pack. */
 {
	if (!strlen(datastr=get_selection(1)) || sscanf(datastr,"%d",&pnum)!=1
		|| pnum<0 || pnum>NUM_PACKS || !packdata[pnum].status) 
	  return 1;
	cflag=screendata[pnum].cflag_opt;
	fflag=screendata[pnum].fflag_opt;
	if (cflag & 1) /* circle options */
	  {
	    strcpy(buff,"-c");
	    if (cflag & 8) strcat(buff,"c");
	    if (!(cflag & 16) && (cflag & 4)) strcat(buff,"bg");
	    else if (!(cflag & 16) && (cflag & 2)) strcat(buff,"fg");
	    else if (cflag & 2) strcat(buff,"f");
	    strcat(buff," a");
	    if (cflag & 32) strcat(buff,"-nc a");
	  }
	if (fflag & 1) /* face options */
	  {
	    strcat(buff," -f");
	    if (fflag & 8) strcat(buff,"c");
	    if (!(fflag & 16) && (fflag & 4)) strcat(buff,"bg");
	    else if (!(fflag & 16) && (fflag & 2)) strcat(buff,"fg");
	    else if (fflag & 2) strcat(buff,"f");
	    strcat(buff," a");
	    if (fflag & 32) strcat(buff,"-nf a");
	  }
	break;
 }
case 13: sprintf(buff," -g");break; 	/* print path */
case 14: /* wrap up the postscript file and close it */
 {
	close_psfile(&custom_fp);
	return 1;
 }
case 15: /* print the special postscript file */
 {
	print_psfile(&custom_fp);
	return 1;
 }
case 17: /* print the canvas postscript file. */
 {
	post_canvas(p);
	return 1; 
 }
	 } /* end of switch */
	if (!cmd_mode)
	  {
	    strcpy(ps_file_name,(char *)xv_get(print_file_item,PANEL_VALUE));
	    strcpy(print_cmd,(char *)get_print_cmd());
	  }
	print_call(&custom_fp,p,ps_file_name,
		   buff,user_name,CPVersion,print_cmd);
	return 1;
} /* print_proc */

int sel_print_proc(Panel_item item,Event *event)
/* bring up print_menu */
{if (event_action(event)==ACTION_MENU && event_is_down(event))
		menu_show(print_menu,panel,event,0);return 1; }

int post_canvas(struct p_data *p)
/* circles/triangles/edges/labels/etc using standard flags. */
{
	int cflag,fflag;
	char buff[NAME_MAX+128];

	if (!p->status) return 0;
	if (post_fp)
	  {
	    fprintf(post_fp,"\ngrestore\nend\nshowpage"); /* wrap up */
	    fclose(post_fp);
	    post_fp=NULL;
	  }
	switch (fill_mode)
	 {
		case 0: {graylevel=.5;break;/* grey */}
		case 1: {graylevel=1;break; /* white */}
		case 2: {graylevel=0;break; /* black */}
	 }
	ps_linewidth=(1+2*(line_thick-1))*PS_UNIT_LINEWIDTH;
	cflag=p->screen->cflag_opt;
	fflag=p->screen->fflag_opt;
	if (!cmd_mode)
	  sprintf(print_file_name,
		  (char *)get_print_file());
	sprintf(buff," -o %s ",print_file_name);
	if (p->screen->unitcircle) strcat(buff," -u ");
    
	if (cflag & 1) /* circle options */
	  {
	    strcat(buff," -c");
	    if (cflag & 8) strcat(buff,"c");
	    if (!(cflag & 16) && (cflag & 4)) strcat(buff,"bg");
	    else if (!(cflag & 16) && (cflag & 2)) strcat(buff,"fg");
	    else if (cflag & 2) strcat(buff,"f");
	    strcat(buff," a");
	    if (cflag & 32) strcat(buff," -nc a");
	  }
	if (fflag & 1) /* face options */
	  {
	    strcat(buff," -f");
	    if (fflag & 8) strcat(buff,"c");
	    if (!(fflag & 16) && (fflag & 4)) strcat(buff,"bg");
	    else if (!(fflag & 16) && (fflag & 2)) strcat(buff,"fg");
	    else if (fflag & 2) strcat(buff,"f");
	    strcat(buff," a");
	    if (fflag & 32) strcat(buff," -nf a");
	  }
	if (!cmd_mode)
	  strcpy(print_cmd,(char *)get_print_cmd());
	print_call(&post_fp,p,print_file_name,
		   buff,user_name,CPVersion,print_cmd);
	fprintf(post_fp,"\ngrestore\nend\nshowpage"); /* wrap up */
	fclose(post_fp);
	post_fp=NULL;
	/* to save paper, pop up ghostview image; user can then print */
	/* print_psfile(&post_fp); */
	if (!cmd_mode)
	  strcpy(print_file_name,(char *)get_print_file());
	strcpy(buff,print_file_name);
	sysfp=popen("/bin/sh","w");
	fprintf(sysfp," ghostview %s &\n",buff);
	pclose(sysfp);

	return 1;
} /* post_canvas */

int print_psfile(FILE **FP)
/* print canvas (flag=1) or custom (flag=2) file */
{
  char buff[64];
  FILE *sysfp;

  if (FP==&post_fp) 
    {
      if (!cmd_mode)
	strcpy(print_file_name,(char *)get_print_file());
      strcpy(buff,print_file_name);
    }
  else if (FP==&custom_fp)
    {
      if (custom_fp) close_psfile(&custom_fp);
      if (!cmd_mode)
	strcpy(ps_file_name,(char *)get_ps_file());
      strcpy(buff,ps_file_name);
    }
  else return 0;

  if (!cmd_mode)
    strcpy(print_cmd,(char *)get_print_cmd());
  strcpy(buf,print_cmd);
  strcat(buf," ");
  strcat(buf,buff);
  sysfp=popen("/bin/sh","w");
  fprintf(sysfp,"%s\n",buf);
  pclose(sysfp);
  sprintf(msgbuf,"Have sent `%s' to the printer.",buff);
  msg();
  return 1;
} /* print_psfile */
