// Copyright (c) 2012 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_BROWSER_UI_TABS_PINNED_TAB_SERVICE_H_
#define CHROME_BROWSER_UI_TABS_PINNED_TAB_SERVICE_H_

#include "base/compiler_specific.h"
#include "base/scoped_observer.h"
#include "chrome/browser/ui/browser_list_observer.h"
#include "chrome/browser/ui/tabs/tab_strip_model.h"
#include "chrome/browser/ui/tabs/tab_strip_model_observer.h"
#include "components/keyed_service/core/keyed_service.h"
#include "content/public/browser/notification_observer.h"
#include "content/public/browser/notification_registrar.h"

class Profile;

// PinnedTabService is responsible for updating preferences with the set of
// pinned tabs to restore at startup. PinnedTabService listens for the
// appropriate set of notifications to know it should update preferences.
class PinnedTabService : public content::NotificationObserver,
                         public BrowserListObserver,
                         public TabStripModelObserver,
                         public KeyedService {
 public:
  explicit PinnedTabService(Profile* profile);
  PinnedTabService(const PinnedTabService&) = delete;
  PinnedTabService& operator=(const PinnedTabService&) = delete;
  ~PinnedTabService() override;

 private:
  // content::NotificationObserver.
  void Observe(int type,
               const content::NotificationSource& source,
               const content::NotificationDetails& details) override;

  // BrowserListObserver:
  void OnBrowserAdded(Browser* browser) override;
  void OnBrowserClosing(Browser* browser) override;
  void OnBrowserRemoved(Browser* browser) override;

  // TabStripModelObserver:
  void OnTabStripModelChanged(
      TabStripModel* tab_strip_model,
      const TabStripModelChange& change,
      const TabStripSelectionChange& selection) override;

  // Writes the pinned tabs for |profile_|, but only if a new tab or browser
  // window has been added since the last time the method was called.
  void WritePinnedTabsIfNecessary();

  Profile* profile_;

  // True if we should save the pinned tabs when a browser window closes or the
  // user exits the application. This is set to false after writing pinned tabs,
  // and set back to true when new tabs or windows are added.
  bool need_to_write_pinned_tabs_ = true;

  content::NotificationRegistrar registrar_;
};

#endif  // CHROME_BROWSER_UI_TABS_PINNED_TAB_SERVICE_H_
