;*=====================================================================*/
;*    serrano/prgm/project/bdk/kbdb/src/Bee/etags.scm                  */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Thu Jul 29 09:18:33 1999                          */
;*    Last change :  Thu Feb  8 08:54:07 2001 (serrano)                */
;*    -------------------------------------------------------------    */
;*    The debugger tries to load the etags file for the project in     */
;*    order to be able to do some eager demangling. This is important  */
;*    in order to be able to set breakpoints using Bigloo symbolic     */
;*    names before the execution starts.                               */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    The module                                                       */
;*---------------------------------------------------------------------*/
(module bee_etags
   (import engine_param
	   tools_speek
	   tools_error
	   (console-prompt tools_io)
	   (bdb-repl-prompt engine_repl))
   (export (read-etags-file)
	   (bigloo-symbol-etags-source-location ::bstring)))

;*---------------------------------------------------------------------*/
;*    scm-info                                                         */
;*---------------------------------------------------------------------*/
(define-struct scm-info
   ;; scm-name
   name
   ;; source file
   file
   ;; line number
   line)

;*---------------------------------------------------------------------*/
;*    bigloo-symbol-etags-source-location ...                          */
;*    -------------------------------------------------------------    */
;*    If we have found source file information for a Bigloo symbols    */
;*    reading the etags file, return that information.                 */
;*---------------------------------------------------------------------*/
(define (bigloo-symbol-etags-source-location scm-name::bstring)
   (if (hashtable? *etags-table*)
       (let ((scm (hashtable-get *etags-table* scm-name)))
	  (if (scm-info? scm)
	      (string-append (scm-info-file scm)
			     ":"
			     (integer->string (scm-info-line scm)))
	      #f))
       #f))

;*---------------------------------------------------------------------*/
;*    *etags-table* ...                                                */
;*---------------------------------------------------------------------*/
(define *etags-table* #unspecified)
   
;*---------------------------------------------------------------------*/
;*    read-etags-file ...                                              */
;*---------------------------------------------------------------------*/
(define (read-etags-file)
   (let ((file (find-file/path *etags* (list "." *root-directory*))))
      (if (and (string? file) (file-exists? file))
	  ;; we really have to read the etags file
	  (begin
	     (verbose 4 "reading etags file \"" file "\"...")
	     (set! *etags-table* (make-hashtable))
	     (read-etags file)
	     (verbose 4 #"done.\n")))))

;*---------------------------------------------------------------------*/
;*    read-etags ...                                                   */
;*    -------------------------------------------------------------    */
;*    An etags file format is:                                         */
;*    ^L                                                               */
;*    <file-name>,<etags-len>                                          */
;*    <definitions>+                                                   */
;*    <definition> : <keyword> <ident>^?<line-start>,<line-stop>       */
;*    <keyword>    : (define                                           */
;*                 | (define-generic                                   */
;*                 | (define-method                                    */
;*                 | (class                                            */
;*---------------------------------------------------------------------*/
(define (read-etags file-name)
   (if (not (file-exists? file-name))
       (bdb-error "etags"
		  "Can't find etags file for input"
		  file-name)
       (let ((port (open-input-file file-name)))
          (if (not (input-port? port))
              (bdb-error "etags"
			 "Can't find etags file for input"
			 file-name)
              (unwind-protect
                 (begin
                    ;; we skip the ^L line
                    (read-line port)
                    ;; then we read all file entries
                    (let loop ((entry (read-etags-file-entry port)))
		       (if (not (eof-object? entry))
			   (loop (read-etags-file-entry port)))))
                 (close-input-port port))))))

;*---------------------------------------------------------------------*/
;*    read-etags-file-entry ...                                        */
;*    -------------------------------------------------------------    */
;*    Because etags file format are just brain damaged, we rely on     */
;*    the assumption that reading at the end-of-file returns the       */
;*    eof-object. That is we read twice the end of file. In            */
;*    consequence it is impossible to use that function to read        */
;*    in stdin.                                                        */
;*---------------------------------------------------------------------*/
(define (read-etags-file-entry port)
   (define (fetch-file-name str)
      ;; fetch the file name of an etags file-name entry
      (let ((port  (open-input-string str))
            (fetch (regular-grammar ()
                      ((+ (out #\,))
                       (the-string))
                      (else
                       (bdb-error "etags"
				  "Illegal etags file format"
				  str)))))
         (let ((name (read/rp fetch port)))
            (close-input-port port)
            name)))
   ;; we read the file name
   (let ((line (read-line port)))
      (cond
	 ((eof-object? line)
	  line)
	 ((string=? line "-,0")
	  ;; this is a meta file
	  (let loop ((line (read-line port)))
	     (if (not (or (eof-object? line) (string=? line "")))
		 (match-case (parse-etags-meta-entry-line line)
		    ;; a meta definition (i.e. a definition
		    ;; introduced by btags that tell us to add
		    ;; some new keyword recognition in the current
		    ;; etags parsing).
		    ((meta-define ?kind ?ident)
		     (let ((sym (string->symbol ident)))
			(if (not (getprop sym 'reserved))
			    (putprop! sym
				      'reserved
				      (string->symbol (string-upcase kind)))))
		     (loop (read-line port)))
		    (else
		     (bdb-error "read-etags-file-entry"
				"Illegal entry format"
				line)))))
	  #unspecified)
	 (else
          (let* ((file (fetch-file-name line))
		 (suf  (suffix file)))
	     (if (member suf *src-suffix*)
		 (let loop ((line (read-line port)))
		    (if (or (eof-object? line) (string=? line ""))
			line
			(begin
			   (match-case (parse-etags-entry-line line)
			      ((define (?name ?line))
			       ;; a function definition
			       (let ((name (string-upcase! name)))
				  (hashtable-put! *etags-table*
						  name
						  (scm-info name file line))))
			      ((define ?name ?line)
			       ;; a variable definition
			       (let ((name (string-upcase! name)))
				  (hashtable-put! *etags-table*
						  name
						  (scm-info name file line))))
			      ((define-generic (?name ?line))
			       ;; a generic function definition
			       (let ((name (string-upcase! name)))
				  (hashtable-put! *etags-table*
						  name
						  (scm-info name file line))))
			      ((extern ?name ?line)
			       ;; an extern definition
			       (let ((name (string-upcase! name)))
				  (hashtable-put! *etags-table*
						  name
						  (scm-info name file line)))))
			   (loop (read-line port)))))))))))

;*---------------------------------------------------------------------*/
;*    parse-etags-entry-line ...                                       */
;*    -------------------------------------------------------------    */
;*    I don't know if it is the best way to proceed but there is a     */
;*    strong separation between the parsing of the etags file and      */
;*    the insertion in the hash-table. That is the function            */
;*    PARSE-ETAGS-ENTRY-LINE decodes the etags file and put it in      */
;*    some kind of intermediate representation. That representation    */
;*    is scan afterward and during that late scan, the insertions      */
;*    are operated if necessary.                                       */
;*---------------------------------------------------------------------*/
(define (parse-etags-entry-line line)
   (let ((port  (open-input-string line))
	 (reg   (regular-grammar ((letter   (in ("azAZ") (#a128 #a255)))
				  (special  (in "!@~$%^&*></-_+\\|=?.:"))
				  (kspecial (in "!@~$%^&*></-_+\\|=?."))
				  (id       (: (* digit)
					       (or letter special)
					       (* (or letter
						      special
						      digit
						      (in ",'`"))))))
		   (blank
		    (ignore))
		   ("("
		    (list 'PAR-OPEN))
		   (")"
		    (list 'PAR-CLO))
		   ((+ digit)
		    (cons 'NUMBER (the-fixnum)))
		   (id
		    (let* ((string  (the-string))
			   (ustring (string-upcase string))
			   (symbol  (string->symbol string)) 
			   (kwd     (getprop symbol 'reserved)))
		       (if kwd
			   ;; this is a keyword
			   (cons kwd (string->symbol ustring))
			   ;; this is a regular identifier
			   (cons 'IDENT string))))
		   ((: #\" (* (out #\")) #\")
		    (list 'STRING))
		   (#a127
		    (list 'EOI))
		   (#\,
		    (ignore))
		   (else
		    (let ((c (the-failure)))
		       (if (eof-object? c)
			   c
			   (bdb-error "parse-etage-entry-line"
				      "Illegal char"
				      c))))))
	 (lalr  (lalr-grammar
		   (IDENT PAR-OPEN PAR-CLO EOI NUMBER
			  DEFINE INLINE DEFMACRO
			  GENERIC METHOD
			  TYPESEP
			  CLASS STRUCT EXTERN MACRO STRING INCLUDE TYPE
			  STATIC/EXPORT)
		   
		   ;; the line we are to parse
		   (line
		    ((function)
		     function)
		    ((variable)
		     variable)
		    ((genericdef)
		     genericdef)
		    ((methoddef)
		     methoddef)
		    ((classdef)
		     classdef)
		    ((structdef)
		     structdef)
		    ((externdef)
		     externdef)
		    ((macrodef)
		     macrodef))
		   
		   ;; function definitions
		   (function
		    ((PAR-OPEN DEFINE PAR-OPEN IDENT EOI NUMBER@line NUMBER)
		     `(define (,IDENT ,line)))
		    ((PAR-OPEN DEFINE PAR-OPEN TYPE EOI NUMBER@line NUMBER)
		     `(define (,TYPE ,line)))
		    ((PAR-OPEN INLINE PAR-OPEN IDENT EOI NUMBER@line NUMBER)
		     `(define (,IDENT ,line)))
		    ((PAR-OPEN INLINE PAR-OPEN TYPE EOI NUMBER@line NUMBER)
		     `(define (,TYPE ,line))))
		   
		   ;; variable definitions
		   (variable
		    ((PAR-OPEN DEFINE IDENT EOI NUMBER@line NUMBER)
		     `(define ,IDENT ,line))
		    ((PAR-OPEN DEFINE TYPE EOI NUMBER@line NUMBER)
		     `(define ,TYPE ,line)))
		   
		   ;; generic function definitions
		   (genericdef
		    ((PAR-OPEN GENERIC PAR-OPEN IDENT EOI NUMBER@line NUMBER)
		     `(define-generic (,IDENT ,line)))
		    ((PAR-OPEN GENERIC PAR-OPEN TYPE EOI NUMBER@line NUMBER)
		     `(define-generic (,TYPE ,line))))
		   
		   ;; method definitions
		   (methoddef
		    ((PAR-OPEN METHOD PAR-OPEN IDENT@mnane IDENT@aname dummys
			       EOI NUMBER@line NUMBER)
		     `(define-method (,mnane ,aname ,line)))
		    ((PAR-OPEN METHOD PAR-OPEN TYPE IDENT@aname dummys
			       EOI NUMBER@line NUMBER)
		     `(define-method (,TYPE ,aname ,line)))
		    ((PAR-OPEN METHOD PAR-OPEN TYPE@type1 TYPE@type2 dummys
			       EOI NUMBER@line NUMBER)
		     `(define-method (,type1 ,type2 ,line))))
		   
		   (dummys
		    (()
		     'dummy)
		    ((PAR-CLO dummys)
		     'dummy)
		    ((PAR-OPEN dummys)
		     'dummy)
		    ((IDENT dummys)
		     'dummy))
		   
		   ;; class definitions
		   (classdef
		    ((PAR-OPEN CLASS IDENT dummys
			       EOI NUMBER@line NUMBER)
		     `(,CLASS ,IDENT ,line))
		    ((PAR-OPEN CLASS TYPE dummys
			       EOI NUMBER@line NUMBER)
		     `(,CLASS ,TYPE ,line))
		    ((PAR-OPEN STATIC/EXPORT
			       PAR-OPEN CLASS TYPE dummys
			       EOI NUMBER@line NUMBER)
		     `(,CLASS ,TYPE ,line))
		    ((PAR-OPEN STATIC/EXPORT
			       PAR-OPEN CLASS IDENT dummys
			       EOI NUMBER@line NUMBER)
		     `(,CLASS ,IDENT ,line)))
		   
		   ;; struct definitions
		   (structdef
		    ((PAR-OPEN STRUCT IDENT EOI NUMBER@line NUMBER)
		     `(define-struct ,IDENT ,line))
		    ((PAR-OPEN STRUCT TYPE EOI NUMBER@line NUMBER)
		     `(define-struct ,TYPE ,line)))
		   
		   ;; extern definitions
		   (externdef
		    ((PAR-OPEN EXTERN extern-sans-clause)
		     extern-sans-clause)
		    ((extern-sans-clause)
		     extern-sans-clause))
		   
		   (extern-sans-clause
		    ((PAR-OPEN MACRO IDENT EOI NUMBER@line NUMBER)
		     `(extern ,IDENT ,line))
		    ((PAR-OPEN MACRO TYPE EOI NUMBER@line NUMBER)
		     `(extern ,TYPE ,line))
		    ((PAR-OPEN INCLUDE EOI NUMBER@line NUMBER)
		     `(ignore))
		    ((PAR-OPEN IDENT STRING EOI NUMBER@line NUMBER)
		     `(extern ,IDENT ,line))
		    ((PAR-OPEN TYPE STRING EOI NUMBER@line NUMBER)
		     `(extern ,TYPE ,line))
		    ((PAR-OPEN IDENT EOI NUMBER@line NUMBER)
		     `(extern ,IDENT ,line))
		    ((PAR-OPEN STATIC/EXPORT IDENT STRING
			       EOI NUMBER@line NUMBER)
		     '(ignore))
		    ((PAR-OPEN STATIC/EXPORT TYPE STRING
			       EOI NUMBER@line NUMBER)
		     '(ignore))
		    ((PAR-OPEN TYPE EOI NUMBER NUMBER)
		     '(ignore)))
		   
		   ;; macro definitions
		   (macrodef
		    ((PAR-OPEN DEFMACRO PAR-OPEN IDENT EOI NUMBER@line NUMBER)
		     `(define-macro (,IDENT ,line)))
		    ((PAR-OPEN DEFMACRO PAR-OPEN DEFINE EOI NUMBER@line NUMBER)
		     `(define-macro (,(symbol->string DEFINE) ,line)))
		    ((PAR-OPEN DEFMACRO PAR-OPEN TYPE EOI NUMBER@line NUMBER)
		     `(define-macro (,TYPE ,line)))))))
      (try (read/lalrp lalr reg port)
	   (lambda (escape obj proc msg)
	      (escape #f)))))

;*---------------------------------------------------------------------*/
;*    parse-etags-meta-entry-line ...                                  */
;*---------------------------------------------------------------------*/
(define (parse-etags-meta-entry-line line)
   (let ((port  (open-input-string line))
	 (reg   (regular-grammar ((letter  (in ("azAZ") (#a128 #a255)))
				  (special (in "!@~$%^&*></-_+\\|=?.:"))
				  (id      (: (* digit)
					      (or letter special)
					      (* (or letter
						     special
						     digit
						     (in ",'`"))))))
		   (blank
		    (ignore))
		   ("("
		    (list 'PAR-OPEN))
		   ((+ digit)
		    (cons 'NUMBER (the-fixnum)))
		   (id
		    (let* ((string  (the-string))
			   (ustring (string-upcase string))
			   (symbol  (string->symbol string)) 
			   (kwd     (getprop symbol 'meta-reserved)))
		       (if kwd
			   ;; this is a keyword
			   (cons kwd (string->symbol ustring))
			   ;; this is a regular identifier
			   (cons 'IDENT string))))
		   (#a127
		    (list 'EOI))
		   (#\,
		    (ignore))
		   (else
		    (let ((c (the-failure)))
		       (if (eof-object? c)
			   c
			   (bdb-error "parse-etage-meta-entry-line"
				      "Illegal char"
				      c))))))
	 (lalr  (lalr-grammar
		   (IDENT PAR-OPEN EOI NUMBER META-DEFINE)
		   
		   ;; variable keywords
		   (metadef
		    ((PAR-OPEN META-DEFINE IDENT@kd IDENT@id EOI NUMBER NUMBER)
		     `(meta-define ,kd ,id))))))
      (try (read/lalrp lalr reg port)
	   (lambda (escape obj proc msg)
	      (escape #f)))))

;*---------------------------------------------------------------------*/
;*    *keyword-list*                                                   */
;*---------------------------------------------------------------------*/
(define *keyword-list*
   '(("define" . DEFINE)
     ("define-inline" . INLINE)
     ("define-method" . METHOD)
     ("define-generic" . GENERIC)
     ("define-macro" . DEFMACRO)
     ("class" . CLASS) ("final-class" . CLASS) ("wide-class" . CLASS)
     ("extern" . EXTERN) ("macro" . MACRO)
     ("define-struct" . STRUCT)
     ("static" . STATIC/EXPORT) ("export" . STATIC/EXPORT)
     ("include" . INCLUDE)
     ("type" . TYPE)))

;*---------------------------------------------------------------------*/
;*    *meta-keyword-list* ... @label etags meta parsing@               */
;*    -------------------------------------------------------------    */
;*    The definition keywords list. This can be extended by means      */
;*    of a Kbrowse command line parameter.                             */
;*---------------------------------------------------------------------*/
(define *meta-keyword-list*
   '(("meta-define" . meta-define)))

;*---------------------------------------------------------------------*/
;*    The keyword initialization                                       */
;*---------------------------------------------------------------------*/
(for-each (lambda (word)
	     (putprop! (string->symbol (car word)) 'reserved (cdr word)))
	  *keyword-list*)
(for-each (lambda (word)
	     (putprop! (string->symbol (car word)) 'meta-reserved
		       (cdr word)))
	  *meta-keyword-list*)

