/* cat-archive-log.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "commands/cmdutils.h"
#include "commands/cmd.h"
#include "commands/cat-archive-log.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] revision");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_headers, 0, "headers", 0, \
      N_("show only log headers"))


t_uchar arch_cmd_cat_archive_log_help[] = N_("print the contents of an archived log entry\n"
                                           "Retrieve and print the log message for the indicated\n"
                                           "revision from its archive.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_cat_archive_log (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int headers;

  headers = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_cat_archive_log_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_headers:
          {
            headers = 1;
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  {
    t_uchar * revision_spec = NULL;
    t_uchar * revision = 0;
    struct arch_archive * arch = 0;
    t_uchar * log = 0;

    arch = arch_archive_connect_branch (argv[1], &revision_spec);
    if (!arch)
      {
	safe_printfmt (2, _("could not connect to archive '%s'\n"), argv[1]);
	exit (2);
      }

    if (!arch_valid_package_name (revision_spec, arch_maybe_archive, arch_req_patch_level, 0))
      {
        safe_printfmt (2, "%s: invalid revision name (%s)\n",
                       argv[0], revision_spec);
        exit (2);
      }

    revision = arch_parse_package_name (arch_ret_non_archive, 0, revision_spec);

    arch_check_for (arch, arch_req_patch_level, revision);
    log = arch_archive_log (arch, revision);

    if (!headers)
      safe_write_retry (1, log, str_length (log));
    else
      {
        t_uchar * headers_end;

        headers_end = log;
        while (1)
          {
            headers_end = str_chr_index (headers_end, '\n');
            if (!headers_end)
              {
                safe_write_retry (1, log, str_length (log));
                exit (0);
              }
            if (headers_end[1] == '\n')
              {
                safe_write_retry (1, log, headers_end + 1 - log);
                exit (0);
              }
            ++headers_end;
          }
      }
  }

  return 0;
}




/* tag: Tom Lord Tue May 20 16:05:52 2003 (cat-archive-log.c)
 */
