/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */
 
#include <qlist.h>
#include <qstring.h>
#include <qstringlist.h>

#ifndef HAS_METAQUALITY
#define HAS_METAQUALITY

/*
*	Quality
* @short Class providing the "Quality" concept (an ID (type) with some
* properties).
*/
class Quality
{
	QString 	*objectName;
	QStringList 	*objectContent;
	QString 	originalText;
	private:
		/**
		* This method is provided for parsing a name=properties type of string.
		* @param content the string containing the name & the properties.
		* @param list the list where all the properties are stored.
		*/
		void parseContent(QString content, QStringList &list);
		/**
		* This method is provided for parsing a name=properties type of string.
		* @param quality the string containing the properties.
		* @param list the list where all the properties are stored.
		*/
		void parseQuality(QString quality, QStringList &list);
		void init();
	public:
		/**
		* Default constructor. Generates a void object (empty name + empty property
		* list).
		*/
		Quality();
		/**
		* Constructor. Generates an object from a string
		* ("name=property,property...).
		* @param quality the string containing the name & the properties.
		*/
		Quality(const QString &quality);
		/**
		* Constructor. Generates an object from two strings: anme and properties.
		* @param name the string containing the name.
		* @param content the string containing the properties.
		*/
		Quality(QString name, const QString &content);
		/**
		* Copy constructor. 
		* @param _q the source object.
		*/
		Quality(const Quality &_q);
		~Quality();
		/**
		* This method is provided mainly for internal use. 
		* @return the name.
		*/
		QString name() const { return *objectName; }
		/**
		* This method is provided mainly for internal use. 
		* @return the content.
		*/
		QStringList content() const { return *objectContent; }
		/**
		* This method is provided mainly for internal use. 
		* @return the original text.
		*/
		QString original() const { return originalText; }
		/**
		* This method is provided mainly for internal use. 
		* @return the content as a string (old style).
		*/
		QString oldContent();
		/**
		* This method is provided for object regeneration. 
		* @return the metaquality as a string.
		*/
		QString recompose();
	// methods
		/**
		* This method is provided for object name refreshing.  
		* @param newName the new name to be assigned to the quality.
		*/
		int 	setName(QString newName);
		/**
		* This method is provided for object content refreshing.  
		* @param additionalContent the content to be appended to the quality.
		*/
		int 	appendContent(QString additionalContent);
		/**
		* This method is provided for object content refreshing.  
		* It clears the MetaQuality content.
		*/
		int 	clearContent();
		/**
		* This method is provided for object content refreshing.  
		* @param newContent the content to replace the quality's content.
		*/
		int 	replaceContent(QString newContent);
		/**
		* This method is provided for checking. 
		* @param probedContent the content to be cheched against.
		* @return TRUE if the quality contains the probedContent and FALSE 
		* otherwise.
		*/
		bool 	hasContent(QString probedContent);
		/**
		* This method is provided for checking. Is an enhanced version of the
		* previous one 
		* @param presentContent the content required in the qualities list.
		* @param excludedContent the content excluded (it has to be NOT PRESENT)
		* in the qualities list.
		* @return TRUE if the quality complies with the requirements.
		* NOTE: The checking IGNORES the empty strings. Therefore it is able to
		* check just if something is NOT included in the properties.
		*/
		bool	checkContents(QString presentContent, QString excludedContent);
		/**
		* This operator is provided for fast contents copying. 
		* @param q the original (source) object.
		*/
		Quality operator=(Quality &q);
		
		/**
		* This is a DEBUG function. 
		*/
		void	show();
};

typedef QList<Quality> QualityList;

/*
*	Quality
* @short Class providing the "Quality" concept (an ID (type) with some
* properties).
*/
class MetaQuality
{
  public:
		/**
		* Default constructor. Generates a void object.
		*/
	  MetaQuality() { }
		/**
		* Constructor. Generates an object from a string.
		* @param q the source string for the list of qualities.
		*/
	  MetaQuality(QString &q);
		/**
		* Constructor. Generates an object from a string.
		* @param q the source string for the list of qualities.
		*/
	  MetaQuality(const char *q);
		/**
		* Copy constructor.
		* @param mq the source object.
		*/
	  MetaQuality(MetaQuality &mq);
		/**
		* This operator is provided for easy (vector style) acces to the qualities
		* list.
		* @param index the index number of the quality.
		* @return the index-th quality in the list, or a default value if is not
		* found.
		*/
	  Quality &operator[](int index);
		/**
		* This operator is provided for easy (vector style) acces to the qualities
		* list.
		* @param name the name of the quality.
		* @return the iquality matching the given name in the list, or a default
		* value if is not found.
		*/
	  Quality &operator[](const char *name);
		/**
		* This operator is provided for easy (vector style) acces to the qualities
		* list.
		* @param name the name of the quality.
		* @return the iquality matching the given name in the list, or a default
		* value if is not found.
		*/
	  Quality &operator[](QString &name);
	  // method for accessing the list (copy constructor)
		/**
		* This method is needed for internal use.
		* @return a reference to the list of qualities.
		*/
	  QualityList getQualityList() { return qualityList;}
		/**
		* This operator is provided for easy content attribution/copying.
		* @param mq the source MetaQuality.
		* @return a reference "this".
		*/
	  MetaQuality &operator=(MetaQuality &mq);
		/**
		* This method is provided for cheching if the qualities are compliant with
		* some requirements.
		* @param name the name of the quality.
		* @param included the properties which HAVE to be contained by the quality.
		* @param excluded the properties which HAVE to be NOT contained by the
		* quality.
		* @return TRUE if the MetaQuality complies with the requirements, FALSE if
		* not.
		*/
		bool query(QString name="", QString included="", QString excluded="");
		/**
		* This method is provided for string reconstruction.
		* @return the string containing all the qualities.
		*/
		QString recompose();
		
		/**
		* This method is provided for quality refreshing.
		* @param name the name of the quality.
		* @param content the content to be appended.
		*/
		void appendQuality(QString name, QString content);
		/**
		* This method is provided for quality refreshing.
		* @param name the name of the quality.
		* @param content the content to replace the old one.
		*/
		void replaceQuality(QString name, QString content);
		/**
		* This method is provided for quality refreshing. It erases a quality 
		* -only- if it complies with the requirements.
		* @param name the name of the quality.
		* @param included the content which has to be contained by the quality.
		* @param excluded the content which has to be NOT contained by the quality.
		*/
		void eraseQuality(QString name="", 
				QString included="", QString excluded="");
		/**
		* This method is provided for internal use by MetaLine.
		* @return the number of elements in the quality list.
		*/
		unsigned count() { return qualityList.count(); }
	  void show();
  private:
	  QualityList qualityList;
		/**
		* This method is provided for internal use.
		* @return a pointer to a quality identified by "name".
		*/
		Quality *seekQuality(QString name);
		/**
		* This method generates a default value (void Quality object).
		*/
	  Quality &defaultValue() { static Quality defVal; return defVal; }
		/**
		* This method is used for string parsing.
		*/
	  void parse(QString q);
};

#endif








