/***************************************************************************
          imap4handler.cpp  -  the handler for the IMAPv4 protocol
                             -------------------
    begin                : Wed Feb  8 14:02:00 EET 2001
    copyright            : (C) 2001 by theKompany (www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#ifdef OS_LINUX
  #include <asm/errno.h>
#else
  #ifdef OS_FREEBSD
    #include <errno.h>
  #endif
#endif
#include <stdio.h>
#include <unistd.h>
#include <pthread.h>

#include <kconfig.h>

#include <qsocketnotifier.h>

#include <imap4handler.h>
#include <commandsprocessor.h>
#include <incomingauthstructure.h>
#include <accountmanager.h>
#include <accounts.h>
#include <uidjar.h>
#include <iosynchandler.h>
#include <servernotifier.h>
#include <remotemailfolder.h>
#include <orb.h>
#include <messagedevice.h>
#include <messagedescriptor.h>

extern KConfig *GlobalConfig;

/** The singleton IMAP handler instance.*/
IMAP4Handler *IMAP4Handler::thisInstance;

/** The pipe for receiving a message.*/
int IMAP4Handler::rcvSync[2];
/** The pipe for sending signals to the progress window.*/
int IMAP4Handler::progressSync[2];
/** The pipe for syncing the local with the remote folder.*/
int IMAP4Handler::commandSync[2];
/** The pipe for logout.*/
int IMAP4Handler::logoutSignal[2];
/** The pipe for updating the messages.*/
int IMAP4Handler::updateSync[2];
/** The pipe for subfolders syncronisation.*/
int IMAP4Handler::subfoldersSync[2];
/** The pipe for auth syncronisation.*/
int IMAP4Handler::authSync[2];
/** The pipe for creating the folders.*/
int IMAP4Handler::createSync[2];
/** The pipe for deleting the folders.*/
int IMAP4Handler::deleteSync[2];
/** The pipe for renaming the folders.*/
int IMAP4Handler::renameSync[2];
/** Check if the command list is empty.*/
int IMAP4Handler::checkList[2];

/** Lock the imap thread for having only one imap thread.*/
sem_t IMAP4Handler::imap4Lock;
/** Lock the thread for syncing with UI.*/
sem_t IMAP4Handler::threadLock;
/** Lock mechanism for getting an abort signal.*/
sem_t IMAP4Handler::stopThread;
/** Lock mechanism for sending commands.*/
sem_t IMAP4Handler::imap4Command;

/** An info data for logining to the server.*/
IncomingAuthStructure *IMAP4Handler::incomingConnectionData;
/** A buffer for reciving the messages.*/
IMAP_MESSAGE IMAP4Handler::incomingMessage;
/** A buffer for reciving the progress signals.*/
string IMAP4Handler::progressData;
/** A buffer for sending the commands.*/
COMMAND_STRUCT IMAP4Handler::commandData;
/** The subfolders answer list.*/
MboxList IMAP4Handler::subfoldersList;
/** The command list status.*/
bool IMAP4Handler::emptyList;

IMAP4Handler::IMAP4Handler():QObject()
{
	incomingConnectionData=0;
	commandsList.clear();
	
	timer=new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(__timerAck()));
	
	sem_init(&imap4Lock, 0, 1);
	sem_init(&threadLock, 0, 0);
	sem_init(&stopThread, 0, 0);
	sem_init(&imap4Command, 0, 0);
	
	emptyList=true;

	pipe(rcvSync);
	QSocketNotifier *receiveNotifier=new QSocketNotifier(rcvSync[0], QSocketNotifier::Read, this);
	connect(receiveNotifier, SIGNAL(activated(int)), this, SLOT(__rcvAck(int)));
	
	pipe(progressSync);
	QSocketNotifier *progressNotifier=new QSocketNotifier(progressSync[0], QSocketNotifier::Read, this);
	connect(progressNotifier, SIGNAL(activated(int)), this, SLOT(__progressAck(int)));

	pipe(commandSync);
	QSocketNotifier *commandNotifier=new QSocketNotifier(commandSync[0], QSocketNotifier::Read, this);
	connect(commandNotifier, SIGNAL(activated(int)), this, SLOT(__commandAck(int)));
	
	pipe(updateSync);
	QSocketNotifier *updateNotifier=new QSocketNotifier(updateSync[0], QSocketNotifier::Read, this);
	connect(updateNotifier, SIGNAL(activated(int)), this, SLOT(__updateAck(int)));
	
	pipe(logoutSignal);
	QSocketNotifier *logoutNotifier=new QSocketNotifier(logoutSignal[0], QSocketNotifier::Read, this);
	connect(logoutNotifier, SIGNAL(activated(int)), this, SLOT(__logout(int)));
	
	pipe(subfoldersSync);
	QSocketNotifier *subfoldersNotifier=new QSocketNotifier(subfoldersSync[0], QSocketNotifier::Read, this);
	connect(subfoldersNotifier, SIGNAL(activated(int)), this, SLOT(__subfolders(int)));
	
	pipe(authSync);
	QSocketNotifier *authNotifier=new QSocketNotifier(authSync[0], QSocketNotifier::Read, this);
	connect(authNotifier, SIGNAL(activated(int)), this, SLOT(__authSync(int)));
	
	pipe(createSync);
	QSocketNotifier *createNotifier=new QSocketNotifier(createSync[0], QSocketNotifier::Read, this);
	connect(createNotifier, SIGNAL(activated(int)), this, SLOT(__createFolder(int)));
	
	pipe(deleteSync);
	QSocketNotifier *deleteNotifier=new QSocketNotifier(deleteSync[0], QSocketNotifier::Read, this);
	connect(deleteNotifier, SIGNAL(activated(int)), this, SLOT(__deleteFolder(int)));
	
	pipe(renameSync);
	QSocketNotifier *renameNotifier=new QSocketNotifier(renameSync[0], QSocketNotifier::Read, this);
	connect(renameNotifier, SIGNAL(activated(int)), this, SLOT(__renameFolder(int)));
	
	pipe(checkList);
	QSocketNotifier *checkListNotifier=new QSocketNotifier(checkList[0], QSocketNotifier::Read, this);
	connect(checkListNotifier, SIGNAL(activated(int)), this, SLOT(__checkList(int)));
	
	readConfiguration();
}

IMAP4Handler *IMAP4Handler::ref()
{
	return thisInstance?thisInstance:(thisInstance=new IMAP4Handler());
}

void IMAP4Handler::readConfiguration()
{
  // TODO: read the timer config
}

void IMAP4Handler::startThread()
{
	if( !sem_trywait(&imap4Lock) )
	{
    //Init the semaphores
  	sem_init(&imap4Lock, 0, 0);
  	sem_init(&threadLock, 0, 0);
  	sem_init(&stopThread, 0, 0);
  	sem_init(&imap4Command, 0, 1);

  	// start thread
  	pthread_t thread;
  	
    pthread_attr_t attr;
    pthread_attr_init(&attr);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_create(&thread, &attr, imap4_thread, 0);
	}
}

bool IMAP4Handler::mount(QString url, bool bSync)
{
  printf("\nMount folder=%s\n", url.latin1());
  fflush(stdout);

  RemoteMailFolder *folder=(RemoteMailFolder*)ObjectRequestBroker::thisInstance()->mailfolderReference(url);
  if( !folder )
  {
		printf("imap4handler: mount: wrong folder url\n");
    return false;
  }

  // Set the command buffer.
  commandsList.insertCommand(IMAP4Handler::MOUNT, folder->account().latin1(), folder->mailbox().latin1(), url.latin1());
  if( bSync )
    commandsList.insertCommand(IMAP4Handler::SYNC, folder->account().latin1(), folder->mailbox().latin1(), url.latin1());

  sem_trywait(&imap4Command);
  if( !sem_post(&imap4Command) )
    return true;
  return false;
}

bool IMAP4Handler::login(QString account)
{
  printf("\nLogin account %s\n", account.latin1());
  fflush(stdout);

  // Set the command buffer.
  commandsList.insertCommand(IMAP4Handler::LOGIN, account.latin1());

  sem_trywait(&imap4Command);
  if( !sem_post(&imap4Command) )
    return true;
  return false;
}

bool IMAP4Handler::sync(QString url)
{
  printf("\nSync %s\n", url.latin1());
  fflush(stdout);

  RemoteMailFolder *folder=(RemoteMailFolder*)ObjectRequestBroker::thisInstance()->mailfolderReference(url);
  if( !folder )
  {
		printf("imap4handler: sync: wrong folder url\n");
    return false;
  }

  // Set the command buffer.
  commandsList.insertCommand(IMAP4Handler::SYNC, folder->account().latin1(), folder->mailbox().latin1(), url.latin1());

  sem_trywait(&imap4Command);
  if( !sem_post(&imap4Command) )
    return true;
  return false;
}

bool IMAP4Handler::unmount(QString url)
{
  printf("\nUnMount folder=%s\n", url.latin1());
  fflush(stdout);

  RemoteMailFolder *folder=(RemoteMailFolder*)ObjectRequestBroker::thisInstance()->mailfolderReference(url);
  if( !folder )
  {
		printf("imap4handler: mount: wrong folder url\n");
    return false;
  }

  // Set the command buffer.
  commandsList.insertCommand(IMAP4Handler::UNMOUNT, folder->account().latin1(), folder->mailbox().latin1(), url.latin1());

  sem_trywait(&imap4Command);
  if( !sem_post(&imap4Command) )
    return true;
  return false;
}

bool IMAP4Handler::logout(QString account)
{
  printf("\nLogout %s\n", account.latin1());
  fflush(stdout);

  // Set the command buffer.
  commandsList.insertCommand(IMAP4Handler::LOGOUT, account.latin1());

  sem_trywait(&imap4Command);
  if( !sem_post(&imap4Command) )
    return true;
  return false;
}

bool IMAP4Handler::stopDelivery()
{
	// check if another session is in progress
	int i=0;
	sem_getvalue(&imap4Lock, &i);
	if( i )
	{
		printf("imap4handler: no imap thread available\n");
		return false;
	}
	
  // Send the abort signal.
	if( !sem_post(&stopThread) )
	  return true;
	return false;
}

bool IMAP4Handler::__timerAck()
{
	// check if another session is in progress
	int i=0;
	sem_getvalue(&imap4Lock, &i);
	if( i )
	{
		printf("imap4handler: no imap thread available\n");
		return false;
	}
	
//  insertCommand(IMAP4Handler::NOOP);

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  return true;
}

void IMAP4Handler::__logout(int)
{
  // save the folder status
  imapFolder()->saveStatus();
  // save the folder messages UID
  imapFolder()->uidJar->saveLists();
}

void IMAP4Handler::__rcvAck(int)
{
	// debug
	printf("imap4handler: message received, saving...\n");
	
	// empty the rcv sync pipe
	static char buf;
	::read(rcvSync[0], &buf, 1);
	
	// debug
	printf("imap4handler: sending message to the iosynchandler...\n");
	
	// get message and pass it to the IOSyncHandler
	if( commandData.type == IMAP4Handler::ADD_MESSAGE )
	  IOSyncHandler::ref()->dispatchMessage(IMAP4Handler::incomingMessage, false);
	else
	  IOSyncHandler::ref()->dispatchMessage(IMAP4Handler::incomingMessage, true);
	
	// unlock thread for getting the next message
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

void IMAP4Handler::__progressAck(int)
{
	// empty the progress sync pipe
	static char buf;
	::read(progressSync[0], &buf, 1);
	
	// sync the messages
	if( !progressData.compare("imap4:completed") ||
	    !progressData.compare("imap4:sync") )
	  IOSyncHandler::ref()->imap4Sync();
	
	// send the progress info to the notifier
//	ServerNotifier::thisInstance()->ioProgress(QString(progressData.c_str()));
	
	// unlock thread
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

void IMAP4Handler::__commandAck(int)
{
	char buf;
	::read(commandSync[0], &buf, 1);
	
  if( !commandsList.isEmpty() )
  {
    // Set the command buffer with the first command
    commandsList.popCommand(commandData);
    // Unlock the IMAP thread for getting the command.
	  sem_init(&imap4Command, 0, 1);
  }
	// unlock thread
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

bool IMAP4Handler::updateMessage(IndexClass &index)
{
  // get the parent folder
  RemoteMailFolder *folder=(RemoteMailFolder*)index.getParentFolder();

	// load descriptor
	MessageDevice *dev=new MessageDevice(&index);
	dev->loadDescriptor();
	MessageDescriptor &descriptor=dev->getDescriptor();
	
  // call the update command
  commandsList.insertCommand(IMAP4Handler::UPDATE_MESSAGE, folder->account().latin1(),
    folder->mailbox().latin1(), folder->name().latin1(), (string)descriptor.uid.latin1());

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  // deallocate the device
  delete dev;

  return true;
}

bool IMAP4Handler::deleteMessage(IndexClass &index)
{
  // get the parent folder
  RemoteMailFolder *folder=(RemoteMailFolder*)index.getParentFolder();

	// load descriptor
	MessageDevice *dev=new MessageDevice(&index);
	dev->loadDescriptor();
	MessageDescriptor &descriptor=dev->getDescriptor();
  	
  commandsList.insertCommand(IMAP4Handler::DELETE_MESSAGE, folder->account().latin1(),
    folder->mailbox().latin1(),  folder->name().latin1(), (string)descriptor.uid.latin1());

	// Unlock the command
  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  // deallocate the device
  delete dev;

  return true;
}

bool IMAP4Handler::getMail(Account*)
{
  printf("\nIMAP4Handler::getMail() is not available yet!");
  printf("\nPlease select the IMAP folder if you want to get the mails\n");

  // TODO: check the INBOX folder

  return true;
}

bool IMAP4Handler::addMessage(QString url, IndexClass *index)
{
  // get the parent folder
  RemoteMailFolder *folder=(RemoteMailFolder*)ObjectRequestBroker::thisInstance()->mailfolderReference(url);
  if( !folder )
  {
		printf("imap4handler: addMessage: wrong folder url\n");
    return false;
  }

	// load descriptor
	MessageDevice *dev=new MessageDevice(index);
	string message=dev->rfc822Message().data();
	
  commandsList.insertCommand(IMAP4Handler::ADD_MESSAGE, folder->account().latin1(),
    folder->mailbox().latin1(), url.latin1(), message);

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  // deallocate the device
  delete dev;

  return true;
}

bool IMAP4Handler::noop(QString account)
{
  // call the NOOP command
  commandsList.insertCommand(IMAP4Handler::NOOP, account.latin1());
  sem_post(&imap4Command);
  return true;
}

void IMAP4Handler::__updateAck(int)
{
	static char buf;
	::read(updateSync[0], &buf, 1);
	
	IOSyncHandler::ref()->updateMessage(IMAP4Handler::incomingMessage);
	
  // Unlock the thread
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

bool IMAP4Handler::createFolder(QString account, QString mailbox, QString url)
{
  commandsList.insertCommand(IMAP4Handler::CREATE_FOLDER, account.latin1() ,mailbox.latin1(),
    url.latin1(), mailbox.latin1());

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  return true;
}

bool IMAP4Handler::deleteFolder(QString account, QString mailbox, QString url)
{
  commandsList.insertCommand(IMAP4Handler::DELETE_FOLDER, account.latin1() ,mailbox.latin1(),
    url.latin1(), mailbox.latin1());

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  return true;
}

bool IMAP4Handler::subfolders(QString account, QString mailbox, QString other)
{
  commandsList.insertCommand(IMAP4Handler::SUBFOLDERS, account.latin1() ,mailbox.latin1(),
    other.latin1(), mailbox.latin1());

  sem_trywait(&imap4Command);
  sem_post(&imap4Command);
  return true;
}

void IMAP4Handler::__subfolders(int)
{
	static char buf;
	::read(subfoldersSync[0], &buf, 1);
	
	for(MboxListIterator it=subfoldersList.begin(); it!=subfoldersList.end(); ++it)
	{
	  string type="imap";
	  string viewType="mail";
	  string name=(*it).name;
	
	  if( ((*it).state & NetIMAP4::NoSelect) == NetIMAP4::NoSelect )
	    viewType="genmail";
	
    IOSyncHandler::ref()->createFolder(commandData.account, name, type, viewType);
	}
	
  // Unlock the thread
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}


void IMAP4Handler::__authSync(int)
{
	static char buf;
	::read(authSync[0], &buf, 1);
	
	Account* acc=AccountManager::ref()->getAccount( commandData.account.c_str() );
	if(!acc)
	{
		printf("imap4handler: mount: cannot proceed, wrong account (%s)\n", imapFolder()->account().latin1());
		incomingConnectionData=0;
		return;
	}
	// return if get temporary disabled or the type differs from IMAP4
	if(acc->getdisabled || acc->type!=Account::IMAP4)
	{
		printf("imap4handler: mount: cannot proceed, account (%s) temporarely disabled or of wrong type\n", (const char *)acc->name);
		incomingConnectionData=0;
		return;
	}
	
	// get the incoming connection data
	if(incomingConnectionData) delete incomingConnectionData;
	incomingConnectionData=new IncomingAuthStructure(acc);
	
  // Unlock the thread
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

RemoteMailFolder* IMAP4Handler::imapFolder()
{
  return (RemoteMailFolder*)ObjectRequestBroker::thisInstance()->mailfolderReference(commandData.other.c_str());
}

void IMAP4Handler::cleanCommandData()
{
  commandData.type=0;
  commandData.account="";
  commandData.mailbox="";
  commandData.other="";
  commandData.parameters="";
}

void IMAP4Handler::__createFolder(int)
{
	static char buf;
	::read(createSync[0], &buf, 1);
	
	// debug
	printf("\nimap4handler: create folder ...\n");
	fflush(stdout);
	
	// folder URL,  name, view type
	string type="imap";
	
	// create the Aethera folder
	IOSyncHandler::ref()->createFolder(commandData.account, commandData.mailbox, type, commandData.other);
	
	// unlock thread for getting the next message
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

void IMAP4Handler::__deleteFolder(int)
{
	static char buf;
	::read(deleteSync[0], &buf, 1);
	
	// debug
	printf("\nimap4handler: delete folder ...\n");
	fflush(stdout);
	
	// delete the Aethera folder
	string type="imap";
	IOSyncHandler::ref()->deleteFolder(commandData.account, commandData.mailbox, type, commandData.other);
	
	// unlock thread for getting the next message
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

void IMAP4Handler::__renameFolder(int)
{
	static char buf;
	::read(renameSync[0], &buf, 1);
	
	// debug
	printf("\nimap4handler: rename folder ...\n");
	fflush(stdout);
	
	// unlock thread for getting the next message
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}

void IMAP4Handler::__checkList(int)
{
	static char buf;
	::read(checkList[0], &buf, 1);
	
	// debug
	printf("\nimap4handler: check list ...\n");
	fflush(stdout);
	
	emptyList=commandsList.isEmpty();
	
	// unlock thread for getting the next message
  sem_trywait(&threadLock);
	sem_post(&threadLock);
}













