#include <textprocessor.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qvaluelist.h>
#include <stdio.h>

QString TextProcessor::quoteText(QString &text, const QString &quoteSeq)
{
	QString outstr;
	QTextStream lines(&text, IO_ReadOnly);
	
	while(!lines.atEnd())
		outstr+=quoteSeq+lines.readLine()+'\n';
	
	return outstr;
}

QString TextProcessor::unquoteText(QString &text, const QStringList &quoteSeqList)
{
	QString outstr;
	QTextStream lines(&text, IO_ReadOnly);
	
	while(!lines.atEnd())
	{
		QString line=lines.readLine();
		bool unquoted=false;
		for(int i=0;i<quoteSeqList.count() && !unquoted; i++)
		{
			QRegExp rexp(quoteSeqList[i]);
			int len;
			if(!rexp.match(line, 0, &len))
			{
				line.remove(0, len);
				unquoted=true;
			}
		}
		outstr+=line+'\n';
	}
	
	return outstr;
}

QString TextProcessor::indentText(QString &text, int spaces)
{
	QString strspaces;
	strspaces.fill(' ', spaces);
	return TextProcessor::quoteText(text, strspaces);
}

QString TextProcessor::unindentText(QString &text, int spaces)
{
	QString strspaces="^";
	for(int i=0;i<spaces;i++) strspaces+="\\s";
	return TextProcessor::unquoteText(text, strspaces);
}

QString TextProcessor::indentTabText(QString &text, int tabs)
{
	QString strspaces;
	strspaces.fill('\t', tabs);
	return TextProcessor::quoteText(text, strspaces);
}

QString TextProcessor::unindentTabText(QString &text, int tabs)
{
	QString strspaces="^";
	for(int i=0;i<tabs;i++) strspaces+="\t";
	return TextProcessor::unquoteText(text, strspaces);
}

QString TextProcessor::metaFormat(QString text, const MetaFormatControl &controlData)
{
	if(text.isEmpty()) return text;

	// any sequences
	if(controlData.enableAnySequence)
	{
		QStringList textLines=QStringList::split("\n", text, true);
		bool rebuild=false;
		for(int i=0;i<textLines.count();i++)
		{
			// try all patterns
			for(int j=0;j<controlData.anyData.count();j++)
			{
				if(controlData.anyData[j].strip)
				{
					bool retry=true;
					QRegExp br(controlData.anyData[j].beginPattern), tr(controlData.anyData[j].textPattern), er(controlData.anyData[j].endPattern);
					int blen, tlen, elen;
					while(retry)
					{
						retry=false;
						int bpos, tpos=-1, epos=-1;
						bpos=br.match(textLines[i], 0, &blen);
						if(bpos!=-1) epos=er.match(textLines[i], bpos+blen, &elen);
						if(epos!=-1) tpos=tr.match(textLines[i].mid(bpos+blen, epos-bpos-blen), 0, &tlen);
						if(bpos!=-1 && tpos!=-1 && epos!=-1)
						{
							// got a complete match
							QString mText="<"+controlData.anyData[j].tag+">"+textLines[i].mid(bpos+blen+tpos, tlen)+"</"+controlData.anyData[j].tag+">";
							printf("replacement text: %s\n", (const char *)mText);
							textLines[i].remove(bpos, blen+tlen+elen);
							textLines[i].insert(bpos, mText);
							retry=true;
							rebuild=true;
						}
					}
				}
				else
				{
					bool retry=true;
					QRegExp regexp(controlData.anyData[j].textPattern);
					int len, pos=0;
					while(retry)
					{
						retry=false;
						pos=regexp.match(textLines[i], pos, &len);
						if(pos!=-1)
						{
							textLines[i].insert(pos+len, "</"+controlData.anyData[j].tag+">");
							textLines[i].insert(pos, "<"+controlData.anyData[j].tag+">");
							pos+=controlData.anyData[j].tag.length()+2;
							retry=true;
							rebuild=true;
						}
					}
				}
			}
		}
		if(rebuild)
		{
			text="";
			for(int i=0;i<textLines.count()-1;i++) text+=textLines[i]+"\n";
		}
	}
		
	// split into chunks, according to the reply quote depth
	QStringList replyChunks;
	QValueList<int> replyChunkDepths;
	
	if(controlData.enableReplyTags)
	{
		QTextStream lines(&text, IO_ReadOnly);
		QString currentChunk=lines.readLine();
		QStringList seq=TextProcessor::depth(currentChunk, QStringList::fromStrList(controlData.replyData));
		int currentDepth=seq.count();
		currentChunk=TextProcessor::strip(currentChunk, seq);
		currentChunk+="\n";
				
		while(!lines.atEnd())
		{
			QString line=lines.readLine();
			seq=TextProcessor::depth(line, QStringList::fromStrList(controlData.replyData));
			int newDepth=seq.count();
			line=TextProcessor::strip(line, seq);
			line+="\n";
			
			if(newDepth==currentDepth)
			{
				currentChunk+=line;
			}
			else
			{
				replyChunks.append(currentChunk);
				replyChunkDepths.append(currentDepth);
				currentChunk=line;
				currentDepth=newDepth;
			}
		}
		
		replyChunks.append(currentChunk);
		replyChunkDepths.append(currentDepth);
	}
	else
	{
		replyChunks.append(text);
		replyChunkDepths.append(0);
	}
	
	// check next-line folding sequences
	if(controlData.enableNextLineFolding)
	{
		int i=1;
		while(i<replyChunks.count())
		{
			if(replyChunkDepths[i-1]!=replyChunkDepths[i] && !replyChunkDepths[i]) // got reply depth difference, second is not indented
			{
				QRegExp regexp("^\\s*[^A-Z]"); // if it begins with capital letter don't process, it might be "Stephan Kulow-style" reply formatting
				int pos=regexp.match(replyChunks[i]), crpos;
				
				// make sure it's on the first line
				if(pos!=-1 && pos<(crpos=replyChunks[i].find('\n'))) // (...remember that we always end chunks with <cr>)
				{
					// ...and that the next line is empty (or does not exist)
					int nextcrpos=replyChunks[i].find('\n', crpos+1);
					if(nextcrpos==-1 || replyChunks[i].mid(crpos, nextcrpos-crpos-1).stripWhiteSpace().isEmpty())
					{
						// all conditions are met, wrap line into previous chunk
						replyChunks[i-1]+=replyChunks[i].left(crpos+1); // +1 adds the last <cr>
						
						// dissolve next chunk if empty
						if(nextcrpos==-1)
						{
							replyChunks.remove(replyChunks.at(i));
							replyChunkDepths.remove(replyChunkDepths.at(i));
						}
						else
						{
							// ...or else just remove the line we've wrapped
							replyChunks[i].remove(0, crpos+1);
							
							i++;
						}
					}
					else
					{
						i++;
					}
				}
				else
				{
					i++;
				}
			}
			else
			{
				// if we've dissolved one chunk between two chunks with equal depth, collapse chunks to get optimal output
				if(replyChunkDepths[i-1]==replyChunkDepths[i])
				{
					replyChunks[i-1]+=replyChunks[i];
					replyChunks.remove(replyChunks.at(i));
					replyChunkDepths.remove(replyChunkDepths.at(i));
				}
				else
				{
					i++;
				}
			}
		}
	}
	
	// check wrote sequences
	if(controlData.enableWroteSequence)
	{
		for(int i=0;i<replyChunks.count();i++)
		{
			QStringList chunkLines=QStringList::split("\n", replyChunks[i], true);
			bool rebuild=false;
			for(int j=0;j<chunkLines.count();j++)
			{
				bool found=false;
				for(QStrListIterator it(controlData.wroteData); it.current() && !found; ++it)
				{
					QRegExp regexp((*it));
					if(regexp.match(chunkLines[j])!=-1)
					{
						chunkLines[j]="<wrote>"+chunkLines[j]+"</wrote>";
						found=true;
						rebuild=true;
					}
				}
			}
			if(rebuild)
			{
				replyChunks[i]="";
				for(int j=0;j<chunkLines.count()-1;j++) replyChunks[i]+=chunkLines[j]+"\n";
			}
		}
	}
	
	// add reply tags
	if(controlData.enableReplyTags)
	{
		QString outtext;
		if(controlData.enableBlockReplyTags)
		{
			for(int i=0;i<replyChunks.count();i++)
				if(replyChunkDepths[i])
					outtext+="<reply "+QString::number(replyChunkDepths[i])+">"+replyChunks[i]+"</reply>";
				else
					outtext+="<normal>"+replyChunks[i]+"</normal>";
		}
		else
		{
			for(int i=0;i<replyChunks.count();i++)
			{
				if(replyChunkDepths[i])
				{
					QStringList chunkLines=QStringList::split('\n', replyChunks[i], true);
					for(int j=0;j<chunkLines.count()-1;j++)
						outtext+="<reply "+QString::number(replyChunkDepths[i])+">"+chunkLines[j]+"</reply>";
				}
				else
				{
					outtext+="<normal>"+replyChunks[i]+"</normal>";
				}
			}
		}
		return outtext;
	}
}

QStringList TextProcessor::depth(QString text, const QStringList &quoteList)
{
	QStringList maxseq;
	
	for(int i=0;i<quoteList.count();i++)
	{
		QRegExp regexp(quoteList[i]);
		int len;
		
		if(!regexp.match(text, 0, &len))
		{
			QString nexttext(text.mid(len));
			QStringList nextseq(quoteList[i]);
			nextseq+=depth(nexttext, quoteList);
			if(nextseq.count()>maxseq.count()) maxseq=nextseq;
		}
	}
	
	return maxseq;
}

QString TextProcessor::strip(QString text, const QStringList &quoteList)
{
	QRegExp regexp(quoteList.join(""));
	int len;
	if(!regexp.match(text, 0, &len)) text.remove(0, len);
	return text;
}
