/**************************************************************
*   
*   Creation Date: <1999-12-28 14:03:18 samuel>
*   Time-stamp: <2001/04/15 13:45:09 samuel>
*   
*	<mmu_fb.c>
*	
*	Offscreen framebuffer acceleration (indented for X-video)
*   
*   Copyright (C) 1999, 2000, 2001 Samuel Rydh
*   
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*   
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <asm/uaccess.h>

#include "mmu.h"
#include "asmfuncs.h"

typedef struct line_entry {
	short	y1, y2;
	ulong	*pte;
	ulong	word1;		/* word1 of PTE */
	ulong	ea;
} line_entry_t;

typedef struct fb_data {
	line_entry_t 	*line_table;
	int 		nrec;
	char		*lv_base;		/* linux virtual of first entry in table */
} fb_data_t;


#define MMU		(kv->mmu)
#define DECLARE_FB	fb_data_t *fb = MMU.fb_data

/**************************************************************
*  		Implementation
**************************************************************/

int
init_mmu_fb( kernel_vars_t *kv )
{
	/* setup_fb_acceleration does the initialization */
	return 0;
}

void
cleanup_mmu_fb( kernel_vars_t *kv )
{
	DECLARE_FB;
	if( !fb )
		return;
	if( fb->line_table )
		vfree( fb->line_table );

	kfree( fb );
	MMU.fb_data = NULL;
}

void
video_pte_inserted( kernel_vars_t *kv, char *lvptr, ulong *pte_slot, ulong word1, ulong ea )
{
	DECLARE_FB;
	int 		i;

	/* printk("video_pte_inserted %p, slot %08lX\n", lvptr, pte_slot );*/

	if( !fb )
		return;

	i = ((ulong)lvptr - (ulong)fb->lv_base) >> 12;
	if( i>=0 && i< fb->nrec ) {
#if 0
		if( line_table[i].pte )
			printk("video pte overwritten, y=%d\n", line_table[i].y1 );
		else
			printk("video pte inserted, y=%d\n", line_table[i].y1 );
#endif
		fb->line_table[i].word1 = word1;
		fb->line_table[i].pte = pte_slot;
		fb->line_table[i].ea = ea;
	} else {
		printk("Warning: video_page outside range, %p %p\n", lvptr, fb->lv_base );
	}
}

/* setup/remove framebuffer acceleration */
void
setup_fb_acceleration( kernel_vars_t *kv, char *lvbase, int bytes_per_row, int height )
{
	DECLARE_FB;
	int i, offs = (ulong)lvbase & 0xfff;
	line_entry_t *p;
	
	if( fb )
		cleanup_mmu_fb( kv );
	if( !lvbase )
		return;
	if( !(fb=kmalloc( sizeof(fb_data_t), GFP_KERNEL )) )
		return;
	memset( fb, 0, sizeof(fb_data_t) );
	MMU.fb_data = fb;

	fb->nrec = (bytes_per_row * height + offs + 0xfff) >> 12;
	if( !(p = (line_entry_t*) vmalloc( sizeof(line_entry_t) * fb->nrec )) ) {
		cleanup_mmu_fb( kv );
		return;
	}
	fb->line_table = p;

	fb->lv_base = (char*)((ulong)lvbase & ~0xfff);
	for(i=0; i<fb->nrec; i++, p++ ){
		p->y1 = (0x1000*i - offs) / bytes_per_row;
		p->y2 = (0x1000*(i+1)-1 -offs) / bytes_per_row;
		if( p->y1 < 0 )
			p->y1 = 0;
		if( p->y2 >= height )
			p->y2 = height-1;

		/* we should make sure the page is really unmapped here! */
		p->pte = NULL;
	}
}

/* return format is {startline,endline} pairs */
int
get_dirty_fb_lines( kernel_vars_t *kv, short *retbuf, int num_bytes )
{
	DECLARE_FB;
	int i, n, s;
	short y1=-1,y2=0;
	line_entry_t *p;
	int is603 = cpu_is_603();

	if( !fb )
		return -1;

	p = fb->line_table;
	
	if( verify_area(VERIFY_WRITE,retbuf,num_bytes) ){
		printk("get_dirty_fb_lines: Bad area!\n");
		return -1;
	}

	s = num_bytes/(2*sizeof(short));
	for( n=0, i=0; i<fb->nrec && n<s; i++, p++ ){
		int dirty=0;

		/* PTE dirty? */
		if( p->pte ) {
			if( *p->pte != p->word1 ) {
				p->pte = NULL;
				dirty = 1;
				/* printk("word changed y=%d\n", p->y1 ); */
			} else if( *(p->pte+1) & BIT(24) ) {  /* C-BIT */
				dirty = 1;
				if( is603 ) {
					/* the 603 handler in the kernel does not update R/C bits */
					*(p->pte) = 0;
					p->pte = NULL;
				} else {
					*(p->pte+1) &= ~BIT(24);
				}
				/* ...might this modify the R/C bits? */
				_tlbie( p->ea );
			}
		}
		/* collect lines */
		if( dirty ){
			if( y1>=0 && y2+1 < p->y1 ) {
				*retbuf++ = y1;
				*retbuf++ = y2;
				n++;
				y1 = p->y1;
			} else if( y1 < 0 )
				y1=p->y1;
			y2 = p->y2;
		}
	}
	if( n<s && y1>=0 ){
		*retbuf++ = y1;
		*retbuf++ = y2;
		n++;
	}
	return n;
}
