/* 
 *   Creation Date: <1999/12/21 20:40:23 samuel>
 *   Time-stamp: <2001/09/30 16:52:37 samuel>
 *   
 *	<xvideo.c>
 *	
 *	X-windows video driver
 *	
 *	Derived from the Basillisk II X-driver, 
 *	(C) 1997-1999 Christian Bauer
 *   
 *   Adapted for MOL by Samuel Rydh, 1999 (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/XShm.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/mman.h>
#include <pthread.h>
#include <errno.h>
#include <driver_mgr.h>

#include "verbose.h"
#include "debugger.h"
#include "res_manager.h"
#include "memory.h"
#include "mouse_sh.h"
#include "wrapper.h"
#include "thread.h"
#include "video_module.h"
#include "keycodes.h"
#include "booter.h"
#include "video.h"
#include "molcpu.h"

SET_VERBOSE_NAME("xvideo");

/* #define USE_ROOT_WINDOW */

#define ADBKeyDown( ch ) PE_adb_key_event( ((ch) & ~0x80) );
#define ADBKeyUp( ch ) PE_adb_key_event( ((ch) | 0x80) );

// umm... better to include a header with these 
typedef unsigned char 	uint8;
typedef int 		bool;
#define false	0
#define true	1

// Global variables
static volatile bool redraw_thread_active = false;	// Flag: Redraw thread installed
static volatile bool redraw_thread_idle = true;		// Flag: Redraw thread idle
static volatile bool redraw_thread_cancel = false;	// Flag: Cancel Redraw thread
static pthread_t redraw_thread;				// Redraw thread

// keyboard handling
static bool cmd_down = false;				// Flag: Ctrl key pressed
static bool caps_on = false;				// Flag: Caps Lock on

// X11 variables
static Display  *x_display;
static int 	screen;					// Screen number
static int 	xdepth;					// Depth of X screen
static Window 	rootwin, the_win;			// Root window and our window
static XVisualInfo visualInfo;
static Visual 	*vis;
static Colormap cmap;					// Colormaps for 8-bit mode
static XColor 	black, white;
static unsigned long black_pixel, white_pixel;
static int 	eventmask;
static const int win_eventmask = KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask 
				| PointerMotionMask | EnterWindowMask | ExposureMask | StructureNotifyMask;

static XColor 	palette[256];				// Color palette for 8-bit mode
static volatile bool palette_changed = false;		// Flag: Palette changed, redraw thread must set new colors

static int byte_order = MSBFirst; 			// this is Mac (sensible) byte ordering
static int pixel_format = 555;				// Mac 16 bit colour is 555
							// Intel PCs are usually 565

// Variables for window mode
static GC 	the_gc;
static XImage 	*img = NULL;
static XShmSegmentInfo shminfo;
#if 0
static XImage 	*cursor_image, *cursor_mask_image;
static Pixmap 	cursor_map, cursor_mask_map;
static Cursor 	mac_cursor;
static GC 	cursor_gc, cursor_mask_gc;
static uint8 	the_cursor[64];				// Cursor image data
#endif
static bool	is_mapped = false;

Cursor no_x_cursor;
Pixmap no_cursor_bitmap;
static unsigned char no_cursor_data[] = { 0, 0, 0, 0, 0, 0, 0, 0 };

static int use_x_cursor;

// Variables only valid when is_open is true
static volatile bool	is_open = false;
static uint8 	*the_buffer;				// Mac frame buffer
static bool 	have_shm = false;			// Flag: SHM extensions available

// X-locking
static pthread_mutex_t x_lock = PTHREAD_MUTEX_INITIALIZER; 
#define LOCK pthread_mutex_lock( &x_lock );
#define UNLOCK pthread_mutex_unlock( &x_lock );

// Prototypes
static void 	redraw_func(void *arg);
static void 	update_display(void);
static bool 	init_window(int width, int height);

static int 	event2keycode(XKeyEvent *ev);
static void 	handle_events(void);
static void 	handle_events_closed(void);

static int 	xvideo_init( video_module_t *mod );
static void 	xvideo_cleanup( video_module_t *mod );
static void 	setcmap(char *pal);
static int	vopen( video_desc_t *mode );
static void	vclose( void );
static void	vrefresh( void );
static int 	error_handler(Display *d, XErrorEvent *e);
static void	set_window_title( Window win, char *title );

static void	depth_blit_8_to_32( int y, int height );
static void	depth_blit_8_to_16( int y, int height );

static void	setup_keymapping( void );


// Video module interface
video_module_t xvideo_module  = {
	"xvideo",
	xvideo_init,
	xvideo_cleanup,
	vopen,
	vclose,
	vrefresh,
	setcmap,

	NULL,			// video modes
	NULL			// next
};

static video_desc_t vmode;		// virtual video mode (might a different depth)
static video_desc_t mac_vmode;		// video mode the mac expects

static bool	depth_emulation = false;	// in depth emulation mode
static void	(*depth_blit_proc)( int y, int height ) = NULL;
static char	*offscreen_buf = NULL;
static ulong	*depth_blit_palette = NULL;   	// 256 byte table (8 -> 24/32 pixel value conversion)
static bool	force_redraw = false;

/************************************************************************/
/*	Init / Cleanup							*/
/************************************************************************/

int 
xvideo_init( video_module_t *m )
{
	int n,i,color_class;
	int width = 640, height = 480;
	char *str;
	video_desc_t *vm;

	if( !get_bool_res("enable_xvideo") )
		return 1;
	
	VPRINT("xvideo_init\n"); 

	use_x_cursor = get_bool_res("use_x_cursor")? 1: 0;

	// Open X-display (if str==NULL, then $DISPLAY is used)
	str = get_str_res("xdisplay");
	x_display = XOpenDisplay( str );
	if( x_display == NULL ){
		LOG("Could not connect to X server %s\n", XDisplayName(str));
		return 1;
	}
	XSetErrorHandler(error_handler);

//	XSynchronize( x_display, True );  	/* DEBUG DEBUG DEBUG */

	byte_order = XImageByteOrder(x_display);

	if (byte_order != MSBFirst) {
		LOG("X display is Intel byte ordered. Your colours will be funny unless you switch your Mac"
		    " into 8 bit colour\n");	
	}

	// Find screen and root window
	screen = XDefaultScreen(x_display);
	rootwin = XRootWindow(x_display, screen);

	// Get screen depth
	xdepth = DefaultDepth(x_display, screen);

	// Find black and white colors
	XParseColor(x_display, DefaultColormap(x_display, screen), "rgb:00/00/00", &black);
	XAllocColor(x_display, DefaultColormap(x_display, screen), &black);
	XParseColor(x_display, DefaultColormap(x_display, screen), "rgb:ff/ff/ff", &white);
	XAllocColor(x_display, DefaultColormap(x_display, screen), &white);
	black_pixel = BlackPixel(x_display, screen);
	white_pixel = WhitePixel(x_display, screen);

	// Get appropriate visual
	switch (xdepth) {
		case 1:
			color_class = StaticGray;
			break;
		case 8:
			color_class = PseudoColor;
			break;
		case 15:
		case 16:
		case 24:
		case 32:
			color_class = TrueColor;
			break;
		default:
			LOG("Unsupported depth\n");
			return 1;
	}
	if (!XMatchVisualInfo(x_display, screen, xdepth, color_class, &visualInfo)) {
		LOG("No XVisual Error\n");
		return 1;
	}
	if (visualInfo.depth != xdepth) {
		LOG("No XVisual Error\n");
		return 1;
	}
	vis = visualInfo.visual;

	if ((xdepth == 16) && (visualInfo.green_mask == 0x7e0)) {
		// the X server is running RGB565
		pixel_format = 565;	
	}
	
	// Create color maps for 8 bit mode
	if (xdepth == 8) {
		cmap = XCreateColormap(x_display, rootwin, vis, AllocAll);
		/* this might cause miscolorization of the console if X 
		 * is running on an inactive VT.  */
		XInstallColormap(x_display, cmap);
	}

	// Initialize according to display type
	if (!init_window(width, height))
		return 1;

	// Start redraw/input thread
	XSync(x_display, false);
	redraw_thread_active = (redraw_thread = create_thread( redraw_func, NULL, "X-thread")) != 0;

	if (!redraw_thread_active) {
		LOG("FATAL: cannot create redraw thread\n");
		exit(1);
	}

	// Fill in supported video modes (emulate 8 bit)
	n = (xdepth > 8) ? 2 : 1;
	vm = calloc( n, sizeof( video_desc_t ));
	xvideo_module.modes = vm;
	for(i=0; i<n; i++ ){
		vm[i].offs = -1;
		vm[i].rowbytes = -1;
		vm[i].h = -1;
		vm[i].w = -1;
		vm[i].depth = (i==0) ? xdepth : 8;
		vm[i].next = (i+1<n)? &vm[i+1] : NULL;
	}

	setup_keymapping();
	return 0;
}

void 
xvideo_cleanup( video_module_t *m )
{
	int i;
	VPRINT("cleanup\n");

	// Stop redraw thread
	redraw_thread_cancel = true;
	for(i=0; i<10 && redraw_thread_active; i++ )
		usleep(100000);

	if( redraw_thread_active ) {
		printm("Redraw thread still running!\n");
		kill_thread( redraw_thread );
	}

	// Close window and server connection
	if (x_display != NULL) {
		XSync(x_display, false);

		if( is_open )
			vclose();

		XFlush(x_display);
		XSync(x_display, false);
		if (xdepth == 8)
			XFreeColormap(x_display, cmap);

		// XXX: close server conection here!
	}

	free( xvideo_module.modes );
}

static bool 
init_window(int width, int height)
{
	XSetWindowAttributes wattr;
	XTextProperty winName, iconName;
	XWMHints *wm_hints;
	XClassHint *class_hints;
	XSizeHints *size_hints;
	char *window_name = "Mac-on-Linux";
	char *icon_name = "MOL";

	// Create window
	CLEAR( wattr );
	memset( &wattr, 0, sizeof( XSetWindowAttributes ));
	wattr.event_mask = eventmask = win_eventmask;
	wattr.background_pixel = black_pixel;
	wattr.border_pixel = black_pixel;
	wattr.backing_store = NotUseful;
	if( get_bool_res("use_backing_store")==1 ) {
		printm("Using backing store for X-window\n");
		wattr.backing_store = WhenMapped; 	/* better than Always, I think */
	}
	wattr.backing_planes = xdepth;
#ifdef USE_ROOT_WINDOW
	wattr.override_redirect = True;
#else
	wattr.override_redirect = False;
#endif
	XSync(x_display, false);
	the_win = XCreateWindow(x_display, rootwin, 0, 0, width, height, 0, xdepth,
				InputOutput, vis, CWEventMask | CWBackPixel | CWBorderPixel 
				/*| CWOverrideRedirect*/ | CWBackingStore | CWBackingPlanes, &wattr);

	// Setup properties
	wm_hints = XAllocWMHints();
	class_hints = XAllocClassHint();
	size_hints = XAllocSizeHints();
	XStringListToTextProperty( &window_name, 1, &winName );
	XStringListToTextProperty( &icon_name, 1, &iconName );
	wm_hints->initial_state = NormalState;
	wm_hints->input = True;
/*	wm_hints->icon_pixmap = mol_icon; */
	wm_hints->flags = StateHint | InputHint /* | IconPixmapHint */;
	class_hints->res_name = "mol";
	class_hints->res_class = "Mol";
	size_hints->flags = PPosition | PSize | PMinSize;
	size_hints->min_width = 100;
	size_hints->min_height = 100;
	XSetWMProperties( x_display, the_win, &winName, &iconName, NULL, 0,
			  size_hints, wm_hints, class_hints );
	XFree( winName.value );
	XFree( iconName.value );
	XFree( wm_hints );
	XFree( class_hints );
	XFree( size_hints );

/*	XStoreName(x_display, the_win, "Mac-on-Linux" ); */

	// Set colormap
	if (xdepth == 8) {
		XSetWindowColormap(x_display, the_win, cmap);
		XSetWMColormapWindows(x_display, the_win, &the_win, 1);
	}

	// Create GC
	the_gc = XCreateGC(x_display, the_win, 0, 0);
	XSetState(x_display, the_gc, black_pixel, white_pixel, GXcopy, AllPlanes);

#if 0
	// Create cursor
	cursor_image = XCreateImage(x_display, vis, 1, XYPixmap, 0, (char *)the_cursor, 16, 16, 16, 2);
	cursor_image->byte_order = MSBFirst;
	cursor_image->bitmap_bit_order = MSBFirst;
	cursor_mask_image = XCreateImage(x_display, vis, 1, XYPixmap, 0, (char *)the_cursor+32, 16, 16, 16, 2);
	cursor_mask_image->byte_order = MSBFirst;
	cursor_mask_image->bitmap_bit_order = MSBFirst;
	cursor_map = XCreatePixmap(x_display, the_win, 16, 16, 1);
	cursor_mask_map = XCreatePixmap(x_display, the_win, 16, 16, 1);
	cursor_gc = XCreateGC(x_display, cursor_map, 0, 0);
	cursor_mask_gc = XCreateGC(x_display, cursor_mask_map, 0, 0);
	mac_cursor = XCreatePixmapCursor(x_display, cursor_map, cursor_mask_map, &black, &white, 0, 0);
#endif

	if( !use_x_cursor ){
		no_cursor_bitmap = XCreateBitmapFromData(x_display, the_win, no_cursor_data, 8,8);
		if( no_cursor_bitmap != None ){
			no_x_cursor = XCreatePixmapCursor(x_display, no_cursor_bitmap, no_cursor_bitmap, &black, &black, 0, 0);
			XDefineCursor(x_display, the_win, no_x_cursor);
			XFreePixmap( x_display, no_cursor_bitmap );
		}
	}

	// Why not wait until we have something to display?
	XMapRaised(x_display, the_win);
	XSync( x_display, false );
	return true;
}

static int 
error_handler(Display *d, XErrorEvent *e)
{
	char buf[80];
	XGetErrorText( x_display, e->error_code, buf, 80 );

	/* This function isn't really called asynchronously, is it? */
	/* If so, we should use async_logprint instead */
	printm("---> X Error: %s\n", buf );

	/* XXX: for now simply disregard the error... */
	return 0;
}


/**************************************************************
*  	vopen / vclose
**************************************************************/

// Trap SHM errors
static bool shm_error = false;
static int (*old_error_handler)(Display *, XErrorEvent *);

static int 
shm_error_handler(Display *d, XErrorEvent *e)
{
	if (e->error_code == BadAccess) {
		shm_error = true;
		return 0;
	} else
		return old_error_handler(d, e);
}

/* this is called to set video mode and prepare buffers. Supposedly,
 * any offset or row_bytes value should work  */
static int
vopen( video_desc_t *org_vm )
{
	XSizeHints *hints;

	if( is_open ){
		LOG("Open called twice!\n");
		return 1;
	}

	LOCK;
	XSync( x_display, false );
	vmode = *org_vm;

	depth_emulation = false;
	depth_blit_proc = NULL;
	if( std_depth(xdepth) != std_depth(vmode.depth) ) {
		// Use a X depth emulation mode...
		if( (std_depth(xdepth) != std_depth(15) && std_depth(xdepth) != std_depth(32)) 
		    || (vmode.depth != 8))
		{
			LOG("Bad vmode depth %d (xdepth = %d)!\n", vmode.depth, xdepth );
			UNLOCK;
			return 1;
		}
		/* LOG ("WARNING: X blitting necessary due to depth mismatch (slower video)\n"); */
		if( std_depth(xdepth) == std_depth(32) )
			depth_blit_proc = depth_blit_8_to_32;
		else
			depth_blit_proc = depth_blit_8_to_16;
		
		vmode.rowbytes = vmode.w;
		switch( xdepth ){
		case 1:
			vmode.rowbytes /= 8; break;
		case 15: 
		case 16:
			vmode.rowbytes *= 2; break;
		case 24:
		case 32:
			vmode.rowbytes *= 4; break;
		}
		vmode.depth = xdepth;
		vmode.offs = 0;
		depth_emulation = true;
		if( offscreen_buf || depth_blit_palette )
			printm("Internal error in xvideo.c (unreleased resource)\n");
		offscreen_buf = map_zero( NULL, FBBUF_SIZE(org_vm));

		depth_blit_palette = calloc( 256, sizeof(ulong) );
	}

	// Try to create and attach SHM image
	have_shm = false;
	if ( vmode.depth != 1 && XShmQueryExtension(x_display)) {

		// Manipulating the bytes_per_line field in the XImage structure has no effect. Thus
		// we need to set the width such that the correct row_bytes value is obtained.
		int fwidth = vmode.rowbytes;
		switch( xdepth ){
		case 1: 		
			fwidth *= 8; break;
		case 15: 
		case 16: 	
			fwidth /= 2; break;
		case 24: 
		case 32:
			fwidth /= 4; break;
		}
		// Create SHM image ("height + 2" for safety)
		img = XShmCreateImage(x_display, vis, xdepth, xdepth == 1 ? XYBitmap : ZPixmap, 
				      0, &shminfo, fwidth, vmode.h);
		if( img->bytes_per_line != vmode.rowbytes ){
			LOG("row_bytes mismatch, %d != %d.\n", vmode.rowbytes, img->bytes_per_line );
			XDestroyImage(img);
		} else {
			shminfo.shmid = shmget(IPC_PRIVATE, FBBUF_SIZE(&vmode), IPC_CREAT | 0777);
			the_buffer = (uint8 *)shmat(shminfo.shmid, 0, 0);
			shminfo.shmaddr = (char *)the_buffer;
			img->data = (char*)the_buffer + vmode.offs;
			shminfo.readOnly = True /*False*/;


			// Try to attach SHM image, catching errors
			shm_error = false;
			old_error_handler = XSetErrorHandler(shm_error_handler);
			XShmAttach(x_display, &shminfo);
			XSync(x_display, false);
			XSetErrorHandler(old_error_handler);

			if (shm_error) {
				shmdt(shminfo.shmaddr);
				XDestroyImage(img);
				shminfo.shmid = -1;
			} else {
				have_shm = true;
				/* printm("Using XShm extension\n"); */
				shmctl(shminfo.shmid, IPC_RMID, 0);
			}
		}
	}

	// Create normal X image if SHM doesn't work ("height + 2" for safety)
	if (!have_shm) {
		the_buffer = (uint8 *)map_zero( NULL, FBBUF_SIZE(&vmode) );
		img = XCreateImage(x_display, vis, xdepth, xdepth == 1 ? XYBitmap : ZPixmap, 
				   0, (char *)the_buffer + vmode.offs, vmode.w, vmode.h, 32, vmode.rowbytes );
	}

	// 1-Bit mode is big-endian
	if (xdepth == 1) {
		img->byte_order = MSBFirst;
		img->bitmap_bit_order = MSBFirst;
	}

	// Make window unresizable
	if ((hints = XAllocSizeHints()) != NULL) {
		hints->min_width = 100;
		hints->max_width = vmode.w;
		hints->min_height = 100;
		hints->max_height = vmode.h;
		hints->flags = PMinSize | PMaxSize;
		XSetWMNormalHints(x_display, the_win, hints);
		XFree((char *)hints);
	}
	XResizeWindow( x_display, the_win, vmode.w, vmode.h );
#ifdef USE_ROOT_WINDOW
	XLowerWindow( x_display, the_win );
	XMapWindow( x_display, the_win );
#endif
	is_open = true;

	// Hide Mac-cursor
	use_hw_cursor(use_x_cursor);

	// Fill in the fields video.c expects
	org_vm->mmu_flags = MAPPING_FB_ACCEL | MAPPING_FORCE_CACHE;
	org_vm->map_base = 0;
	if( !depth_emulation ) {
		vmode.lvbase = org_vm->lvbase = the_buffer;
	} else {
		vmode.lvbase = the_buffer;
		org_vm->lvbase = offscreen_buf;
	}
	mac_vmode = *org_vm;

	// Set MMU acceleration parameters
	_setup_fb_accel( mac_vmode.lvbase+mac_vmode.offs, mac_vmode.rowbytes, mac_vmode.h );

	// Change window name to reflect depth emulation
	set_window_title( the_win, depth_emulation ? "Mac-on-Linux (slow video)" : "Mac-on-Linux" );
	
	// Refresh and sync
	force_redraw = true;
	XSync( x_display, false );
	UNLOCK;
	VPRINT("vopen - OK\n");
	return 0;
}


static void
vclose( void )
{
	int i;
	if( !is_open )
		return;       
	is_open = false;

	// Wait until the redraw thread is idle
	for(i=0; i<10 && !redraw_thread_idle; i++)
		usleep( 100000 );
	if( !redraw_thread_idle )
		printm("WARNING: redraw_thread_not idle\n");
	LOCK;
	XSync(x_display, false);

	depth_blit_proc = NULL;

	_setup_fb_accel( NULL, 0, 0 );

	XSync(x_display, false);
	if( have_shm ) {
		/* is this the correct sequence? */
		XShmDetach( x_display, &shminfo );
		XDestroyImage(img);
		shmdt( shminfo.shmaddr );
	} else {
		munmap( the_buffer, FBBUF_SIZE(&vmode) );
		img->data = NULL;
		XDestroyImage(img);
	}

	// free depth blitting tables
	if( offscreen_buf ){
		munmap( offscreen_buf, FBBUF_SIZE( &mac_vmode ));
		offscreen_buf = NULL;
	}
	if( depth_blit_palette ) {
		free( depth_blit_palette );
		depth_blit_palette = NULL;
	}

	// And force a screen refresh
	XClearArea( x_display, the_win, 0,0,0,0, True );
	XSync(x_display, false);
	UNLOCK;
}

static void
vrefresh( void )
{
	LOCK;
	if( !is_open ) {
		UNLOCK;
		return;
	}
	force_redraw = true;
	/* XClearArea( x_display, the_win, 0,0,0,0, True ); */
	UNLOCK;
}


static void 
setcmap(char *pal)
{
	int i;

	if( !is_open )
		return;
	LOCK;
	// Convert colors to XColor array
	for ( i=0; i<256; i++) {
		palette[i].pixel = i;
		palette[i].red = pal[i*3] * 0x0101;
		palette[i].green = pal[i*3+1] * 0x0101;
		palette[i].blue = pal[i*3+2] * 0x0101;
		palette[i].flags = DoRed | DoGreen | DoBlue;
	}
	// Setup depth emulation tables
	if( depth_blit_palette ){
		char *p = pal;
		for(i=0; i<256; i++, p+=3 ) {
			if( std_depth(xdepth) == std_depth(32) ) {
				if (byte_order == MSBFirst)
					depth_blit_palette[i] = ((ulong)*p<<16) | ((ulong)*(p+1)<<8) | *(p+2);
				else
					depth_blit_palette[i] = ((ulong)*(p+2)<<16) | ((ulong)*(p+1)<<8) | *p;
			} else {
				if (pixel_format == 555) {
					depth_blit_palette[i] = (((ulong)*p<<(10-3)) & 0x7c00)
						| (((ulong)*(p+1)<<(5-3)) & 0x3e0)
						| (((ulong)*(p+2)>>3) & 0x1f);
				} else {
					// 565 pixels
					depth_blit_palette[i] = (((ulong)*p<<(11-3)) & 0xf800)
						| (((ulong)*(p+1)<<(5-2)) & 0x7e0)
						| (((ulong)*(p+2)>>3) & 0x1f);
				}
				
				if (byte_order == LSBFirst)
					depth_blit_palette[i] =	((depth_blit_palette[i] & 0xff) << 8) |
								((depth_blit_palette[i] & 0xff00) >> 8);
			}
		}
		if( depth_blit_proc )
			depth_blit_proc( 0, mac_vmode.h );
		force_redraw = true;
	}

	// Tell redraw thread to change palette
	palette_changed = true;
	UNLOCK;
}

static void
set_window_title( Window w, char *title )
{
	XTextProperty tp;

	if( !XStringListToTextProperty( &title, 1, &tp ) )
		return;
	XSetWMName( x_display, w, &tp );
	XFree( tp.value );
}


/************************************************************************/
/*	Thread for screen refresh, event handling etc.			*/
/************************************************************************/

static void 
redraw_func(void *arg)
{
	struct timespec req = {0, 16666667}; /* 60 Hz */
	struct timespec close_req = {0, 16666667*1}; /* 6 Hz */

	while (!redraw_thread_cancel) {
		if( !is_open ) {
			redraw_thread_idle = true;
			nanosleep(&close_req, NULL);
			LOCK;
			handle_events_closed();
			UNLOCK;
			continue;
		}
		redraw_thread_idle = false;
		nanosleep( &req, NULL );

		LOCK;
		if( !is_open ) {
			UNLOCK;
			continue;
		}

		// Handle palette changes
		if (palette_changed) {
			palette_changed = false;
			if (xdepth == 8)
				XStoreColors(x_display, cmap, palette, 256);
		}

		// In window mode, update display and mouse pointer
		update_display();
		handle_events();

		UNLOCK;
	}
	redraw_thread_active = false;
}


static void
update_display( void )
{
	short buf[80];
	int i, n, y=-1, hight;

	n = _get_dirty_fb_lines( buf, sizeof(buf) );

	if( force_redraw ){
		n=1;
		buf[0]=0;
		buf[1]=vmode.h-1;
		force_redraw = false;
	}

	for(i=0; i<n; i++ ){
		y = buf[i*2];
		hight = buf[i*2+1] - buf[i*2] +1;

		if( depth_blit_proc != NULL )
			depth_blit_proc( y, hight );
	       	if (have_shm)
			XShmPutImage(x_display, the_win, the_gc, img, 0, y, 0, y, 
				     vmode.w, hight, 0);
		else
			XPutImage(x_display, the_win, the_gc, img, 0, y, 0, y, 
				  vmode.w, hight);
		/* printm("Updating %d [%d]\n", y, hight ); */
	}
}

static void
depth_blit_8_to_32( int y, int height )
{
	char *dest = vmode.lvbase + vmode.offs + vmode.rowbytes * y;
	char *s, *src = mac_vmode.lvbase + mac_vmode.offs + mac_vmode.rowbytes * y;
	ulong *d;
	int i;

	/* printm("Depth blit %d %d\n", y, height ); */
	for( ; height-- > 0 ; dest += vmode.rowbytes, src+= mac_vmode.rowbytes ){
		d = (ulong*) dest;
		s = src;
		for( i=0; i<vmode.w; i++ )
			*d++ = depth_blit_palette[(int)*s++];
	}
}
static void
depth_blit_8_to_16( int y, int height )
{
	char *dest = vmode.lvbase + vmode.offs + vmode.rowbytes * y;
	char *s, *src = mac_vmode.lvbase + mac_vmode.offs + mac_vmode.rowbytes * y;
	ushort *d;
	int i;

	/* printm("Depth blit %d %d\n", y, height ); */
	for( ; height-- > 0 ; dest += vmode.rowbytes, src+= mac_vmode.rowbytes ){
		d = (ushort*) dest;
		s = src;
		for( i=0; i<vmode.w; i++ )
			*d++ = depth_blit_palette[(int)*s++];
	}
}

static void
handle_events_closed(void )
{
	char failed_text[]="This video mode might be unsupported by the X-driver. "
			  "Press space to switch to console mode."; 

	XEvent event;
	
	for(;;){
		if (!XCheckMaskEvent(x_display, eventmask, &event))
			break;
		switch( event.type ){
		case KeyPress: {
			KeySym k = XLookupKeysym( (XKeyEvent*)&event, 0 );
			if( k == XK_space || k == XK_Tab )
				switch_to_console_video();
			break;
		}
		case Expose:
			if( !is_mapped )
				break;
			#define ev ((XExposeEvent*)&event)
			if( ev->window != the_win )
				continue;
			XClearArea( x_display, the_win, ev->x, ev->y, ev->width, ev->height, False );
			XDrawImageString( x_display, the_win, the_gc, 10,10,
				 failed_text, strlen(failed_text) );
			#undef ev
			break;
		case MapNotify:
			is_mapped = true;
			break;
		case UnmapNotify:
			is_mapped = false;
			break;
		}
	}
}

static void 
handle_events(void)
{
	XEvent event;

	for (;;) {
		if (!XCheckMaskEvent(x_display, eventmask, &event))
			break;
		switch (event.type) {
		// Mouse button
		case ButtonPress: {
			unsigned int button = ((XButtonEvent *)&event)->button;
			if (button < 4)	/* 1..3 */
				mouse_but_event( kMouseEvent_Down1 << (button-1) );
			break;
		}
		case ButtonRelease: {
			unsigned int button = ((XButtonEvent *)&event)->button;
			if (button < 4)
				mouse_but_event( kMouseEvent_Up1 << (button-1) );
			break;
		}

		// Mouse moved
		case EnterNotify:
		case MotionNotify:
			mouse_move_to( ((XMotionEvent *)&event)->x, ((XMotionEvent *)&event)->y );
			break;

		// Keyboard
		case KeyPress: {
			KeySym k = XLookupKeysym( (XKeyEvent*)&event, 0 );
			int code = event2keycode((XKeyEvent*)&event);
			if( k == XK_F12 ) {
				save_session();
				break;
			}
			if (code != -1) {
				if (code == 0x39) {	// Caps Lock pressed
					if (caps_on) {
						ADBKeyUp(code);
						caps_on = false;
					} else {
						ADBKeyDown(code);
						caps_on = true;
					}
				} else {
					if( code == 0x37 /* cmd */ )
						cmd_down = true;
					if( cmd_down && code == 0x30 /*TAB*/ ) {
						ADBKeyUp( 0x37 /*cmd*/ );
						cmd_down = false;
						switch_to_console_video();
					} else
						ADBKeyDown(code);
				}
			}
			break;
		}
		case KeyRelease: {
			int code = event2keycode((XKeyEvent *)&event);
			if (code != -1 && code != 0x39) {	// Don't propagate Caps Lock releases
				ADBKeyUp(code);
				if( code == 0x37 /*cmd*/ )
					cmd_down = false;
			}
			break;
		}

		// Hidden parts exposed, force complete refresh of window
		case Expose:
			#define ev ((XExposeEvent*)&event)
			// XXX: We should only update the affected rectangle
			if( ev->window != the_win )
				break;
			if( ev->x < 0 || ev->y < 0 || ev->x + ev->width > vmode.w || ev->y + ev->height > vmode.h ){
				printm("Bad Expose event, %d %d %d %d\n", ev->x, ev->y, ev->width, ev->height );
				ev->x = ev->y = 0;
				ev->width = vmode.w;
				ev->height = vmode.h;
			}
			if (have_shm)
				XShmPutImage(x_display, the_win, the_gc, img, ev->x, ev->y, ev->x, ev->y, 
					     ev->width, ev->height, 0);
			else
				XPutImage(x_display, the_win, the_gc, img, ev->x, ev->y, ev->x, ev->y, ev->width,
					  ev->height);
			break;
			#undef ev
		case MapNotify:
			is_mapped = true;
			break;
		case UnmapNotify:
			is_mapped = false;
			break;
		}
	}
}

static int 
event2keycode(XKeyEvent *ev)
{
	/* This seem to do the trick. Here, we always want the key-down keycodes
	 * since the up/down bit is set by ADBKeyDown / ADBKeyUp.
	 */
	return keycode_to_adb( kRawXKeytable, ev->keycode, 1 );
}


/************************************************************************/
/*	U N U S E D   S T U F F   B E L O W				*/
/************************************************************************/

#if 0
static void
cursor_setter( void )
{
	// Has the Mac started? (cursor data is not valid otherwise)
	if ( /*HasMacStarted()*/ 1) {
		// Set new cursor image if it was changed
		if (memcmp(the_cursor, EA_TO_LVPTR(0x844), 64)) {
			memcpy(the_cursor, EA_TO_LVPTR(0x844), 64);
			memcpy(cursor_image->data, the_cursor, 32);
			memcpy(cursor_mask_image->data, the_cursor+32, 32);
			XFreeCursor(x_display, mac_cursor);
			XPutImage(x_display, cursor_map, cursor_gc, cursor_image, 0, 0, 0, 0, 16, 16);
			XPutImage(x_display, cursor_mask_map, cursor_mask_gc, cursor_mask_image, 0, 0, 0, 0, 16, 16);
			mac_cursor = XCreatePixmapCursor(x_display, cursor_map, cursor_mask_map, &black, &white, 
							*EA_TO_LVPTR(0x885), *EA_TO_LVPTR(0x887) );
			XDefineCursor(x_display, the_win, mac_cursor);
		}
	}
}
#endif

/************************************************************************/
/*	Keycode stuff							*/
/************************************************************************/

typedef struct {
	KeySym	ksym;
	int	adb_code;
} ksym_tran_t;
typedef struct {
	KeySym	ksym1, ksym2;
	int	adb_code;
} ksym_tran2_t;

static ksym_tran_t ttab[] = {
	{ XK_Left,		0x3b	},
	{ XK_Right,		0x3c	},
	{ XK_Up,		0x3e	},
	{ XK_Down,		0x3d	},

	{ XK_Shift_L,		0x38	},
	{ XK_Shift_R,		0x38	},
	{ XK_Control_L,		0x36	},
	{ XK_Control_R,		0x36	},
	{ XK_Caps_Lock,		0x39	},
	{ XK_Mode_switch,	0x3a	},
	{ XK_space,		0x31	},
	{ XK_Return,		0x24	},
	{ XK_Tab,		0x30	},
	{ XK_Escape,		0x35	},

	{ XK_KP_1,		0x53	},
	{ XK_KP_2,		0x54	},
	{ XK_KP_3,		0x55	},
	{ XK_KP_4,		0x56	},
	{ XK_KP_5,		0x57	},
	{ XK_KP_6,		0x58	},
	{ XK_KP_7,		0x59	},
	{ XK_KP_8,		0x5b	},
	{ XK_KP_9,		0x5c	},
	{ XK_KP_0,		0x52	},
	{ XK_KP_Decimal,	0x41	},
	{ XK_KP_Separator,	0x41	},
	{ XK_KP_Enter,		0x4c	},	/* ibook enter = 0x34 */

	{ XK_KP_Up,		0x5b	},	/* num-lock mode translates */
	{ XK_KP_Down,		0x54	},	/* to keypad keys */
	{ XK_KP_Begin,		0x57	},
	{ XK_KP_Home,		0x59	},
	{ XK_KP_End,		0x53	},
	{ XK_KP_Prior,		0x5c	},
	{ XK_KP_Next,		0x55	},
	{ XK_KP_Left,		0x56	},
	{ XK_KP_Right,		0x58	},
	{ XK_KP_Insert,		0x52	},
	{ XK_KP_Delete,		0x41	},
	
	{ XK_Num_Lock,		0x47	},
	{ XK_KP_Equal,		0x51	},
	{ XK_KP_Divide,		0x4b	},
	{ XK_KP_Multiply,	0x43	},
	{ XK_KP_Add,		0x45	},
	{ XK_KP_Subtract,	0x4e	},

	{ XK_Help,		0x72	},
	{ XK_Insert,		0x72	},

	{ XK_Begin,		0x73	},
	{ XK_Home,		0x73	},
	{ XK_End,		0x77	},
	{ XK_Prior,		0x74	},
	{ XK_Page_Up,		0x74	},
	{ XK_Next,		0x79	},
	{ XK_Page_Down,		0x79	},

	{ XK_paragraph,		0x0a	},
	{ XK_1,			0x12    },
	{ XK_2,			0x13    },
	{ XK_3,			0x14    },
	{ XK_4,			0x15    },
	{ XK_5,			0x17    },
	{ XK_6,			0x16    },
	{ XK_7,			0x1a    },
	{ XK_8,			0x1c    },
	{ XK_9,			0x19    },
	{ XK_0,			0x1d    },
	{ XK_minus,		0x1b	},
	{ XK_equal,		0x18	},

	{ XK_q,			0x0c    },
	{ XK_w,			0x0d    },
	{ XK_e,			0x0e    },
	{ XK_r,			0x0f    },
	{ XK_t,			0x11    },
	{ XK_y,			0x10    },
	{ XK_u,			0x20    },
	{ XK_i,			0x22    },
	{ XK_o,			0x1f    },
	{ XK_p,			0x23    },
	{ XK_bracketleft,	0x21	},
	{ XK_bracketright,	0x1e	},

	{ XK_a,			0x00    },
	{ XK_s,			0x01    },
	{ XK_d,			0x02    },
	{ XK_f,			0x03    },
	{ XK_g,			0x05    },
	{ XK_h,			0x04    },
	{ XK_j,			0x26    },
	{ XK_k,			0x28    },
	{ XK_l,			0x25    },
	{ XK_semicolon,		0x29	},
	{ XK_apostrophe,	0x27	},
	{ XK_backslash,		0x2a	},
	
	{ XK_grave,		0x32	},
	{ XK_z,			0x06    },
	{ XK_x,			0x07    },
	{ XK_c,			0x08    },
	{ XK_v,			0x09    },
	{ XK_b,			0x0b    },
	{ XK_n,			0x2d    },
	{ XK_m,			0x2e    },
	{ XK_comma,		0x2b	},
	{ XK_period,		0x2f    },
	{ XK_slash,		0x2c	},

	{ XK_F1,		0x7a	},
	{ XK_F2,		0x78	},
	{ XK_F3,		0x63	},
	{ XK_F4,		0x76	},
	{ XK_F5,		0x60	},
	{ XK_F6,		0x61	},
	{ XK_F7,		0x62	},
	{ XK_F8,		0x64	},
	{ XK_F9,		0x65	},
	{ XK_F10,		0x6d	},
	{ XK_F11,		0x67	},
	{ XK_F12,		0x6f	},
	{ XK_F13,		0x69	},
	{ XK_F14,		0x6b	},
	{ XK_F15,		0x71	},

	/* International keyboards */
	{ XK_less,		0x32	},
#if 1
	/* These depends on the distribution... */
	{ XK_BackSpace,		0x33    },	/* Real backspace */
	{ XK_Delete,		0x75    },
	{ XK_Alt_L,		0x37	},	
	{ XK_Alt_R,		0x37	},	/* 0x3a = alt, 0x37 = command */
	{ XK_Meta_L,		0x37	},
	{ XK_Meta_R,		0x37	},
#endif

	/* Swedish / Finnish */
	{ XK_odiaeresis,	0x29	},
	{ XK_adiaeresis,	0x27	},
	{ XK_aring,		0x21	},
	{ XK_dead_diaeresis,	0x1e	},
	{ XK_dead_grave,	0x18	},

	/* Norwegian */
	{ XK_ae,		0x27	},
	{ XK_oslash,		0x29	}
};

/* These fixes are not perfect (but reduces the need for manual tuning) */
static ksym_tran2_t ttab2[] = {
	/* For myself */
	{ XK_plus,		XK_grave,	0x18	},

	/* U.S. */
	{ XK_grave,		XK_asciitilde,	0x0a	},

	/* fixes for Swedish (and others) */
	{ XK_apostrophe,	XK_asterisk,	0x2a	},
	{ XK_dead_grave,	XK_dead_acute,	0x18	},
	{ XK_grave,		XK_acute,	0x18	},

	/* Norwegian */
	{ XK_apostrophe,	XK_paragraph,	0x0a	},

	/* German */
	{ XK_asciicircum,	XK_degree,	0x0a	},
	{ XK_plus,		XK_asterisk,	0x1e	},

};

static ksym_tran2_t ttab_key[] = {
	/* Matching is performed against the first entry. If there is a 
	 * binding maches, all the following bindings are used too.
	 */

	/* German */
	{ XK_udiaeresis,	XK_Udiaeresis,	0x21	},	/* Signature */
	{ XK_z,			XK_Z,		0x10	},
	{ XK_y,			XK_Y,		0x06	},
 	{ XK_minus,		XK_underscore,	0x2c	},
	{ 0,0,0 },

	/* Scandinavian languages */
	{ XK_aring,		XK_Aring,	0x21	},	/* Signature */
 	{ XK_minus,		XK_underscore,	0x2c	},
	{ 0,0,0 },

	/* French */
	{ XK_ampersand,		XK_1, 		0x12	},	/* Signature */
	{ XK_at,		XK_numbersign,	0x0a	},
	{ XK_eacute,		XK_2, 		0x13	},
	{ XK_egrave,		XK_2, 		0x13	},
	{ XK_quotedbl,		XK_3, 		0x14	},
	{ XK_apostrophe,	XK_4, 		0x15	},
	{ XK_parenleft,		XK_5, 		0x17	},
	{ XK_minus,		XK_6,		0x16	},
	{ XK_paragraph,		XK_6, 		0x16	},
	{ XK_egrave,		XK_7, 		0x1a	},
	{ XK_eacute,		XK_7, 		0x1a	},
	{ XK_underscore,	XK_8, 		0x1c	},
	{ XK_exclam,		XK_8, 		0x1c	},
	{ XK_ccedilla,		XK_9, 		0x19	},
	{ XK_aacute,		XK_0, 		0x1d	},
	{ XK_agrave,		XK_0, 		0x1d	},
	{ XK_parenright,	XK_degree,	0x1b	},
 	/* { XK_minus,		XK_underscore,	0x51	},*/
	{ XK_dollar,		XK_asterisk,	0x1e	},
	{ XK_dollar,		XK_sterling,	0x1e	},
	{ XK_a,			XK_A,		0x0c    },
	{ XK_z,			XK_Z,		0x0d    },
	{ XK_e,			XK_E,		0x0e    },
	{ XK_q,			XK_Q,		0x00    },
	{ XK_m,			XK_M,		0x29	},
	{ XK_w,			XK_W,		0x06	},
	{ XK_ugrave,		XK_percent,	0x27	},
	{ XK_grave,		XK_sterling,	0x2a	},
	{ XK_comma,		XK_question,	0x2e	},
	{ XK_semicolon,		XK_period, 	0x2b	},
	{ XK_colon,		XK_slash,	0x2f	},
	{ XK_equal,		XK_plus,	0x18	},
	{ XK_exclam,		XK_paragraph, 	0x2c	},
	{ 0,0,0 },
};


static void
setup_keymapping( void )
{
	KeySym *list, *p;
	int i, j, min, max, ksyms_per_keycode, match;

	XDisplayKeycodes( x_display, &min, &max );
	register_key_table( kRawXKeytable, min, max );
	
	if( get_bool_res("disable_xkey_remap") != 0 ) {
		list = XGetKeyboardMapping( x_display, min, max-min+1, &ksyms_per_keycode );

		for( i=min,p=list; i<=max; i++, p+=ksyms_per_keycode ) {
			for(j=0; j<sizeof(ttab)/sizeof(ksym_tran_t); j++ )
			if( ttab[j].ksym == *p )
				set_keycode( kRawXKeytable, i, ttab[j].adb_code );
			if( ksyms_per_keycode < 2 )
				continue;
			for( j=0; j<sizeof(ttab2)/sizeof(ksym_tran2_t); j++ )
				if( ttab2[j].ksym1 == p[0] && ttab2[j].ksym2 == p[1] )
					set_keycode( kRawXKeytable, i, ttab2[j].adb_code );
		}

		match=0;
		for(j=0; j<sizeof(ttab_key)/sizeof(ksym_tran2_t); j++) {
			if( !ttab_key[j].ksym1 ) {
				match=0;
				continue;
			}
			for( i=min,p=list; match>=0 && i<=max; i++, p+=ksyms_per_keycode ) {
				if( ttab_key[j].ksym1 == p[0] && ttab_key[j].ksym2 == p[1] ) {
					set_keycode( kRawXKeytable, i, ttab_key[j].adb_code );
					match=1;
				}
			}
			if( !match )
				match--;
		}
		XFree(list);
	}

	user_kbd_customize( kRawXKeytable );
}

