/* 
 *   Creation Date: <1999/11/16 00:49:26 samuel>
 *   Time-stamp: <2001/06/24 17:55:09 samuel>
 *   
 *	<init.c>
 *	
 *	Newworld booter (Mac OS 8.6 and later)
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel & David Rydh 
 #      (samuel@ibrium.se, dary@lindesign.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include <sys/param.h>
#include <linux/pci.h>

/* #define VERBOSE */
#include "verbose.h"
#include "booter.h"

#include "ofmem.h"
#include "res_manager.h"
#include "debugger.h"
#include "memory.h"
#include "mac_registers.h"
#include "os_interface.h"
#include "promif.h"
#include "rtas.h"
#include "io.h"
#include "session.h"
#include "elfload.h"
#include "fs.h"
#include "newworld.h"
#include "mol_assert.h"

#include "../drivers/include/pci.h"

#include "phandles.h"

extern ulong of_entry_start[1];
extern ulong of_entry_end[1];

SET_VERBOSE_NAME( "newworld" );

/* where we decide to put things */
#define ROM_BASE		0x1100000
#define OF_GLUE_BASE		0xe00000

static void fix_phandles( void );
static void ihandles_init( void );
static void ihandles_cleanup( void );
static void setup_nw_registers( ulong entry );
static void patch_elf_rom( char *start, size_t size );
static void load_elf_rom( ulong *entry );
static void newworld_startup( void );
static void newworld_cleanup( void );

static fs_ops_t *s_rom_fs = NULL;
static file_desc_t s_rom_fd = 0;

/************************************************************************/
/*	INIT / CLEANUP							*/
/************************************************************************/

void
newworld_booter_init( void )
{
	char *str;

       	gPE.booter_startup = newworld_startup;
	gPE.booter_cleanup = newworld_cleanup;
	
	if( (str=get_str_res("newworld_rom")) && (s_rom_fs=fs_native_open()) ) {
		if( !(s_rom_fd = fs_open_path( s_rom_fs, str )) ) {
			fs_close( s_rom_fs );
			s_rom_fs = NULL;
			perrorm("---> Failed opening the newworld ROM '%s'", str );
			sleep(2);
		}
	}
}

static void
newworld_startup( void )
{
	ulong entry;
	
	ofmem_init();
	rtas_init();

	fix_phandles();
	ihandles_init();

	if( !loading_session() ) {
		load_elf_rom( &entry );
		setup_nw_registers( entry );
	}

	os_interface_add_proc( OSI_OF_INTERFACE, osip_of_interface );

	/* XXX: Fixme - session support for the PCI bus is incomplete! */ 
	pci_assign_addresses();
}


static void
newworld_cleanup( void )
{
	ihandles_cleanup();

	rtas_cleanup();
	ofmem_cleanup();

	if( s_rom_fs ){
		s_rom_fs->close( s_rom_fd );
		fs_close( s_rom_fs );
		s_rom_fs = NULL;
	}
}


/************************************************************************/
/*	ROM loading							*/
/************************************************************************/

int
newworld_rom_found( void )
{
	return s_rom_fs != NULL;
}

int
load_newworld_rom( int fs_type, char *name, ullong offs )
{
	fs_ops_t 	*fs;
	file_desc_t	fd=0;
	char 		*s;
	int		ind, ret=1, searched=0;
	
	assert( is_newworld_boot() );
	
	if( !(fs = fs_open( fs_type, name, offs )) ){
		printm("%s: No HFS[+] filesystem found\n", name );
		return 1;
	}
	for( ind=0 ; !fd && (s=get_str_res_ind("macos_rompath", ind++, 0)) ; )
		fd = fs_open_path( fs, s );
	for( ind=0 ; !fd && (s=get_str_res_ind("macos_rompath_", ind++, 0)) ; )
		fd = fs_open_path( fs, s );
	if( !fd ) {
		printm("Searching %s for a 'Mac OS ROM' file\n", name );
		fd=fs_search_rom(fs);
		searched++;
	}
	if( !fd ) {
		printm("%s: No 'Mac OS ROM' file found\n", name );
	} else {
		if( searched ) {
			char buf[200];
			fs->get_path( fd, buf, sizeof(buf) );
			printm("\n**** HINT ***********************************************\n");
			printm("*  The ROM search can be speeded up by adding the line\n");
			printm("*    macos_rompath: '%s'\n", buf );
			printm("*  to the /etc/molrc file.\n");
			printm("*********************************************************\n");
		}
		/* The actually loading of the ELF image is postponed */
		s_rom_fs = fs;
		s_rom_fd = fd;
		return 0;
	}
	printm("\n");
	fs_close( fs );
	return ret;
}


static void
load_elf_rom( ulong *entry )
{
	Elf32_Ehdr		ehdr;
	Elf32_Phdr		*phdr;
	char 			buf[200];
	int			i;
	int			lszz_offs;		/* start of compressed data */
	int			lszz_size;
	mol_device_node_t	*dn;
	int			elfoffs;
	file_desc_t		fd = s_rom_fd;
	fs_ops_t		*fs = s_rom_fs;
	size_t			s;
	char			*addr;

	if( !newworld_rom_found() ) {
		printm("No newworld ROM was found\n");
		exit(1);
	}
	fs->get_path( fd, buf, sizeof(buf) );
	printm("Loading '%s' %s%s\n\n", buf,
	       fs->dev_name? "from " : "", fs->dev_name? fs->dev_name : "" );

	/* The ELF-image (usually) starts at offset 0x4000 */
	if( (elfoffs=find_elf32(fs, fd)) < 0 ) {
		LOG("----> %s is not an ELF image\n", buf );
		exit(1);
	}
	if( !(phdr = elf32_readhdrs(fs, fd, elfoffs, &ehdr)) ) {
		LOG("elf32_readhdrs failed\n");
		exit(1);
	}
	*entry = ehdr.e_entry;
	
	/* Load segments. Compressed ROM-image assumed to be located immediately after the last segment */
	lszz_offs = elfoffs;
	for( i=0; i<ehdr.e_phnum; i++ ) {
		/* p_memsz, p_flags */
		s = MIN( phdr[i].p_filesz, phdr[i].p_memsz );
		fs->lseek( fd, elfoffs + phdr[i].p_offset, SEEK_SET );

		/* printm("filesz: %08lX memsz: %08lX p_offset: %08lX p_vaddr %08lX\n", 
		   phdr[i].p_filesz, phdr[i].p_memsz, phdr[i].p_offset, phdr[i].p_vaddr ); */

		if( phdr[i].p_vaddr != phdr[i].p_paddr )
			printm("WARNING: ELF segment virtual addr != physical addr\n");
		lszz_offs = MAX( lszz_offs, elfoffs + phdr[i].p_offset + phdr[i].p_filesz );
		if( !s )
			continue;
		if( ofmem_claim( phdr[i].p_vaddr, phdr[i].p_memsz, 0 ) == -1 ) {
			LOG("Claim failed!\n");
			exit(1);
		}
		addr = ram.lvbase + phdr[i].p_vaddr;
		if( fs->read( fd, addr, s) != s ) {
			LOG_ERR("read failed\n");
			exit(1);
		}

		/* patch CODE segment */
		if( *entry >= phdr[i].p_vaddr && *entry < phdr[i].p_vaddr + s )
			patch_elf_rom( ram.lvbase+phdr[i].p_vaddr, s );

		flush_icache_range( addr, addr+s );

		VPRINT("ELF ROM-section loaded at %08lX (size %08lX)\n", 
		       (ulong)phdr[i].p_vaddr, (ulong)phdr[i].p_memsz );
	}
	lszz_size = fs->lseek( fd, 0, SEEK_END ) - lszz_offs;
	VPRINT("Compressed ROM image: offset %08X, size %08X loaded at %08x\n", 
	       lszz_offs, lszz_size, ROM_BASE );

	fs->lseek( fd, lszz_offs, SEEK_SET );
	if( ofmem_claim( ROM_BASE, lszz_size, 0 ) == -1 ) {
		LOG("Claim failure (lszz)!\n");
		exit(1);
	}
	fs->read( fd, ram.lvbase+ROM_BASE, lszz_size );

	/* copy OF glue code */
	if( ofmem_claim( OF_GLUE_BASE, (ulong)of_entry_end-(ulong)of_entry_start, 0 ) == -1 ) {
		LOG("Claim failure (OF_GLUE)!\n");
		exit(1);
	}	
	s = (ulong)of_entry_end - (ulong)of_entry_start;
	addr = ram.lvbase + OF_GLUE_BASE;
	memcpy( addr, of_entry_start, s );
	flush_icache_range( addr, addr + s );
	fs->close( fd );
	fs_close( fs );
	s_rom_fs = NULL;
	
	/* Fix the /rom/macos/AAPL,toolbox-image,lzss property (phys, size) */
	if( (dn = prom_create_node( "/rom/macos/" )) != NULL ) {
		ulong data[2] = { ROM_BASE, lszz_size };
		prom_add_property( dn, "AAPL,toolbox-image,lzss", (char*)data, sizeof(data) );

		/* The 7.8 rom (MacOS 9.2) uses AAPL,toolbox-parcels instead of 
		 * AAPL,toolbox-image,lzss. It probably doesn't hurt to have it
		 * always present (we don't have an easy way to determine ROM version...)
		 */
		prom_add_property( dn, "AAPL,toolbox-parcels", (char*)data, sizeof(data) );
	}
	free( phdr );
}


/* Fix bug present in the 2.4 and the 3.0 Apple ROM */
static void 
patch_elf_rom( char *start, size_t size )
{
	int s;
	ulong mark[] = { 0x7c7d1b78, 		/* mr r29,r3 */
			 0x7c9c2378,		/* mr r28,r4 */
			 0x7cc33378,		/* mr r3,r6 */
			 0x7c864214,		/* add r4,r6,r8   <------ BUG -- */
			 0x80b10000,		/* lwz r5,0(r17) */
			 0x38a500e8 };		/* addi r5,r5,232 */

	/* Correcting add r4,r6,r8  ---->  addi r4,r6,8 */
	for( s=0; s<size-sizeof(mark); s+=4 ){
		if( memcmp( start+s, mark, sizeof(mark)) == 0 ) {
			printm("FIXING ROM BUG @ %X!\n", s+12);
			((ulong*)(start+s))[3] = 0x38860008;	/* addi r4,r6,8 */
		}
	}
}


static void 
setup_nw_registers( ulong entry )
{
	/* According to IEEE 1275, PPC bindings:
	 *	
	 * 	MSR = FP, ME + (DR|IR)
	 *	r1 = stack (32 K + 32 bytes link area above)
	 *	r5 = clint interface handler
	 *	r6 = address of client program arguments (unused)
	 *	r7 = length of client program arguments (unsed)
	 */

	/* provided stack is not used by MacOS */
	mregs->nip = entry; 
	mregs->msr = MSR_FP | MSR_ME | MSR_DR | MSR_IR;
	mregs->gpr[5] = OF_GLUE_BASE;
	mregs->gpr[6] = 0;
	mregs->gpr[7] = 0;

	mregs->spr[ S_SPRG0 ] = 0;

	/* allocate stack (currently unused) */
	mregs->gpr[1] = ofmem_claim( 0, 0x8000, 0x1000 );
	mregs->gpr[1] += 0x8000 - 32;

	/* virt-base, virt-size, real-base, real-size properties
	 * tells us where we may allocate memory (-1 by default, so we don't care)
	 */
}


/************************************************************************/
/*	phandles 							*/
/************************************************************************/

typedef struct {
	char *property;
	char *real_path;
} phfix_t;

/* /node_path/prop contains a phandle to /real_path */
static phfix_t phfixtab[] = {
	{ "/chosen/interrupt-controller", "/pci/pci-bridge/mac-io/interrupt-controller" },
	{ NULL, NULL }
};

/*
 * Fix the phandle identifier for some nodes (the original phandle id is not 
 * normally available, thus we must do this manually).
 */
static void
fix_phandles( void )
{
	mol_device_node_t *dn;
	phfix_t *p;
	ulong *phptr;
	int len;

	for( p=phfixtab; p->property; p++ ){
		dn = NULL;
		if( (phptr = (ulong*)prom_get_prop_by_path( p->property, &len )) && len == 4 )
			if( (dn = prom_find_dev_by_path( p->real_path ) ) != NULL )
				dn->node = (void*) *phptr;
		if( !dn )
			printm("Property %s not found or invalid\n", p->property );
	}
}


/************************************************************************/
/*	ihandles							*/
/************************************************************************/

typedef struct
{
	char 	*property;
	int     molih;		/* MOL ihandle */
	char	*real_path;	/* corresponding node */
} ih_sentry_t;

static ih_sentry_t ih_list[] = {
	{ "/chosen/memory", 	molih_memory, 	"/memory" },
	{ "/chosen/mmu", 	molih_mmu, 	"/cpus/@0" },
	{ "/chosen/stdin",   	molih_stdin, 	NULL /* ? */ },
	{ "/chosen/stdout",  	molih_stdout, 	NULL /* ? */ },
	{ "/chosen/nvram",   	molih_nvram, 	"/pci/pci-bridge/mac-io/nvram" },
	{ (char*)-1,		molih_rtas,	"/rtas"	},
	{ NULL, 0, NULL }
};

typedef struct 
{
	ulong	molih;		/* MOL ihandle identifier */
	ulong	ihandle;	/* ihandle, as seen by OF */

	ulong	phandle;	/* corresponding phandle or 0 */
} ih_entry_t;

#define MAX_NUM_IHANDLES	16

static ih_entry_t 	ihtab[ MAX_NUM_IHANDLES ];
static int 		num_ihandles=0;

static ulong		s_next_ihandle;
static int		s_next_molih;


static void
ihandles_init( void )
{
	mol_device_node_t *dn;
	ih_sentry_t *p;
	int len;
	ulong *ihptr;
	ulong phandle;

	num_ihandles = 0;
	s_next_ihandle = 1;
	s_next_molih = molih__next_free;

	for( p=ih_list; p->property; p++ ){
		dn = NULL;
		ihptr = NULL;
		if( p->property != (char*)-1 ) {
			if( !(ihptr = (ulong*)prom_get_prop_by_path( p->property, &len )) || len != 4 ) {
				printm("Property %s not found or invalid\n", p->property );
				continue;
			}
		}
		phandle = 0;
		if( p->real_path )
			phandle = prom_dn_to_phandle( prom_find_dev_by_path( p->real_path ) );
		if( !phandle && p->real_path )
			printm("Phandle for node %s could not be constructed\n", p->real_path );

		add_ihandle( p->molih, ihptr ? *ihptr : -1, phandle );
	}
}

static void
ihandles_cleanup( void )
{
	num_ihandles = 0;
}

/* ihandle = 0 allocates new ihandle, ihandle = -1 adds phandle -> molih rule */
ulong
add_ihandle( int molih, ulong ihandle, ulong phandle )
{
	ih_entry_t *ihp = &ihtab[ num_ihandles ];

	if( num_ihandles >= MAX_NUM_IHANDLES ) {
		printm("MAX_NUM_IHANDLES exceeded!\n");
		return 0;
	}
	
	if( !ihandle )
		ihandle = s_next_ihandle++;
	if( ihandle == -1 )
		ihandle = 0;
	if( !molih ) {
		molih = phandle_to_molih( phandle );
		if( !molih )
			molih = s_next_molih++;
	}

	num_ihandles++;
	ihp->molih = molih;
	ihp->ihandle = ihandle;
	ihp->phandle = phandle;
/*	printm("MOLIH: %08X IHANDLE: %08lX PHANDLE: %08lX\n", molih, ihandle, phandle );*/

	return ihandle;
}

ulong
ihandle_to_phandle( ulong ihandle )
{
	int i;
	for(i=0; i<num_ihandles; i++ )
		if( ihtab[i].ihandle == ihandle )
			return ihtab[i].phandle;
	return 0;
}

ulong
molih_to_ihandle( int molih )
{
	int i;
	for(i=0; i<num_ihandles; i++ )
		if( ihtab[i].molih == molih )
			return ihtab[i].ihandle;
	return 0;
}

int
ihandle_to_molih( ulong ihandle )
{
	int i;
	for(i=0; i<num_ihandles; i++ )
		if( ihtab[i].ihandle == ihandle )
			return ihtab[i].molih;
	return 0;
}

int
phandle_to_molih( ulong phandle )
{
	int i;
	for(i=0; i<num_ihandles; i++ )
		if( ihtab[i].phandle == phandle )
			return ihtab[i].molih;
	return 0;
}

ulong
phandle_to_ihandle( ulong phandle )
{
	int i;
	for(i=0; i<num_ihandles; i++ )
		if( ihtab[i].phandle == phandle )
			return ihtab[i].ihandle;	/* might be zero */

	/* printm("----> **** phandle_to_ihandle failed (%08lX) *****\n", phandle ); */
	return 0;
}
