// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: aquire.h,v 1.12 1998/05/16 04:57:06 jgg Exp $
/* ######################################################################

   This is the Archive Aquire engine.
   
   URIs (probably from a source list) are passed to this class which builds
   up a list of things to do and their status. It then begins forking off
   new processes to handle the requests and monitors the output from the
   processes.
   
   It needs an external select loop (hooked via RegisterWorker) to work
   most effeciently, on it's own it will be able to spawn only a single 
   worker at a time. With a select loop it is capable of fully concurrant
   operation, including MD5 checking and .gz decompression.
   
   The Item structure contains information about each URI to get including
   status and error information from the method itself.
   
   A directory must be given for the files to be placed into, a partial 
   sub directory must exist for temporary work.
   
   Several different things can be done based on the contents of the 
   Item structure. If MD5Sum is set then an MD5 check will be performed
   on the file after transfer. If the ending suffix of the URI is 
   different from the ending suffix of the output name then automatic
   decompression will occure. If FinalPath is zero then all the files will
   be copied even if they are available in the local namespace, if it is
   not zero then the path relative to the output directory will be stored
   there and the file will not be copied if it already exists.
   
   ##################################################################### */
									/*}}}*/
// Header section: pkglib
#ifndef PKGLIB_AQUIRE_H
#define PKGLIB_AQUIRE_H

#include <string>
#include <vector>
#include <iostream.h>
#include <sys/types.h>
#include <utime.h>

class pkgAquire
{
   protected:
   
   // Single URI to get
   struct Item
   {
      string URI;
      string MD5Sum;
      string OutputName;
      string CompressSuffix;
      string GetInfo;
      enum {Error,Waiting,Downloading,Decompressing,Partial,Complete} Status;
      unsigned long ExpectedSize;
      unsigned long EstimatedSize;
      string *FinalPath;
      string ErrorText;
      string CompFinalPath;
      struct utimbuf Time;
      	 
      inline bool operator <(const Item &rhs) const {return URI < rhs.URI;};
      inline bool operator >(const Item &rhs) const {return rhs.URI < URI;};
      inline bool operator ==(const Item &rhs) const {return URI == rhs.URI;};
      
      Item();
   };
 
   // Worker process
   class Worker
   {
      public:
      
      // ID of the worker
      pid_t Pid;
      int Fd;
      pkgAquire &Owner;
      Item *Cur;
      Item **From;
      Item **To;
      string Info;

      // IO buffer
      string Buf;
      string OutputFile;
      
      void ItemDone();
      void Line(char Mode,string Line);
      bool Read();
      bool Error();
      
      Worker(pkgAquire &Owner,pid_t Pid,int Fd);
      ~Worker();
   };
   friend Worker;

   vector<Item *> List;
   string StartDir;
   string iOutputDir;
   int Lock;
   unsigned long iFetchBytes;
   
   // Helpers for internal methods
   bool RunDecompressor(string Program,string Args,string File,Item *For);
   bool SchemeFile(string Loc,Item &Itm);
   bool CdPartial(string &Error);
   bool DoClean(string Dir);
   void SetCloseExec(int Fd,bool Close = false);
   
   enum LogTypes {StatError,StatInfo,StatNewFile,StatReference,StatDoneFile};
   
   // Hook for a select loop
   virtual bool RegisterWorker(Worker *Work);
   virtual bool Log(LogTypes /*Type*/,Item * /*Itm*/,string /*Text*/) {return true;};
   
   public:

   // Add URIS
   bool OutputDir(string Out);
   bool Get(string URI,string Output,string GetInfo,unsigned long Size = 0,
	    string MD5Sum = "",string *FinalPath = 0);

   bool Clean();
   bool Run();
   inline unsigned long FetchBytes() {return iFetchBytes;};
   
   pkgAquire() : Lock(-1), iFetchBytes(0) {};
   virtual ~pkgAquire() {};
};
ostream &operator <<(ostream &O,pkgAquire::Item &Itm);

#endif
