// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: tree.cc,v 1.23 1998/06/21 03:25:10 jgg Exp $
/* ######################################################################

   Tree - Text Tree Widget
   
   This class handles the usual Expandable tree. It also doubles as a
   list class (just have items with no children)
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#include <deity/tree.h>
									/*}}}*/

void *Tree::Type = (void *)&Tree::Type;

// Tree::Tree - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Tree::Tree(Widget *Parent) : BasicWidget(Parent), iRoot(0), Top(0), 
                             Current(0), ItemHeight(0)
{
   if (TextGC::GC != 0)
      iBackground = Wc_LightGray;
   else
      iBackground = Wc_White;
   iActiveColor = Wc_Blue;
   iActiveTextColor = Wc_White;
   Flag(Focusable);
}
									/*}}}*/
// Tree::~Tree - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* */
Tree::~Tree()
{
   delete iRoot;
}
									/*}}}*/
// Tree::QuickRedraw - Draw the tree 					/*{{{*/
// ---------------------------------------------------------------------
/* If A == B == 0 then a normal full render is done. Otherwise only A
   and B are effected.*/
void Tree::QuickRedraw(CombinedGC &GC,Tree::Item *A,Tree::Item *B)
{
   if (A == 0 && B == 0)
      BasicRender(GC,false);
   if (iRoot == 0)
   {
      BasicRender(GC,true);
      return;
   }
   
   if (ItemHeight == 0)
      ItemHeight = iRoot->Height();
   
   unsigned long Items = CanvasSize().y/ItemHeight;
   unsigned long Depth = 0;
   
   // Determine the depth of the top item
   for (Item *I = Top; I != 0; I = I->Parent)
      Depth++;
   Depth--;
   long Y = BorderY;
   for (Item *I = Top; I != 0 && Items != 0;Items--,Y += ItemHeight)
   {
      // Check if this item should be drawn
      if (A == 0 && B == 0 || A == I || B == I)
      {
	 /* We mark it selected for the render. This is for future use
	    in a muli-select tree */
	 if (Current == I)
	    I->Flag(Item::Selected);
	 
	 // Draw it
	 I->Render(GC,Depth,Rect(BorderX,Y,CanvasSize().x,ItemHeight),this);
	 I->Flag(0,Item::Selected);

	 // Rotate B->A
	 if (B == I)
	    B = 0;
	 
	 if (A == I)
	 {
	    if (B == 0)
	       return;
	    A = B;
	 }
      }
      
      // Check if we should descend to the child list
      if (I->Child != 0 && I->IsFlag(Item::Expanded))
      {
	 I = I->Child;
	 Depth++;
	 continue;
      }
      
      // Ascend
      if (I->Next == 0)
      {	 
	 Item *J = I->Parent;
	 Depth--;
	 for (; J != 0 && J->Next == 0; J = J->Parent)
	    Depth--;
	 
	 if (J != 0)
	    I = J->Next;
	 else
	    I = 0;
      }
      else
	 if (I != 0)
	    I = I->Next;
   }

   // Fill the empty section of the list.
   GC->Background(iBackground);
   GC->BFill(AbsRect(BorderX,Y,Pos.w - BorderX,Pos.h - BorderY));
}
									/*}}}*/
// Tree::Root - Set the root item					/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Root(Item *R)
{
   if (iRoot == 0)
   {
      Current = R;
      Top = R;
      iRoot = R;
      Trigger(Nt_SelectChange,0);
   }
   
   iRoot = R;
}
									/*}}}*/
// Tree::Key - Handle keystrokes					/*{{{*/
// ---------------------------------------------------------------------
/* This handles all of the keyboard interaction from the list. The usual
   keys, Up/Down Home/End are supported. */
bool Tree::Key(const KeyEvent &Key,Widget *)
{
   if (iRoot == 0)
      return true;

   Item *OldCur = Current;
   switch (Key.Extended)
   {
      // Move up an item
      case KeyEvent::Up:
      if (Current != 0)
	 Current = Current->Up();
      Center(Current,1,false);
      break;
   
      // Move down an item
      case KeyEvent::Down:
      if (Current != 0)
	 Current = Current->Down();
      Center(Current,CanvasSize().y/ItemHeight,false);
      break;
      
      // Home
      case KeyEvent::Home:
      Current = iRoot;
      Center(Current,1,false);
      break;
      
      // End
      case KeyEvent::End:
      {
	 // Jump to the last shown item in the tree
	 Current = iRoot;
	 do
	 {
	    for(;Current != 0 && Current->Next != 0; Current = Current->Next);
	    if (Current->Child != 0 && Current->IsFlag(Item::Expanded))
	       Current = Current->Child;
	 }
	 while ((Current->Child != 0 && Current->IsFlag(Item::Expanded)) || 
		Current->Next != 0);
	 
	 Center(Current,CanvasSize().y/ItemHeight,false);
	 break;
      }
      
      // PgUp
      case KeyEvent::PgUp:
      {
	 if (Current == 0)
	    break;
	 
	 // Make current equal to top and move up N items
	 Current = Top;
	 for (int I = CanvasSize().y/ItemHeight; I != 0; I--)
	    Current = Current->Up();
	 Center(Current,1,false);
	 break;
      }
      
      // PgDown
      case KeyEvent::PgDown:
      {
	 if (Current == 0)
	    break;
	 
	 // Make current equal to top and move up N items
	 Current = Top;
	 for (int I = CanvasSize().y/ItemHeight; I != 0; I--)
	    Current = Current->Down();
	 Center(Current,1,false);
	 break;
      }
      
      default:
      break;
   }

   switch (Key.Key)
   {
      // Branch expansion
      case '+':
      Current->ExpandChildren(this);
      break;

      // Branch contraction
      case '-':
      Current->ContractChildren(this);
      break;      
   }

   if (OldCur != Current)
      Trigger(Nt_SelectChange,0);
   
   // Quick redraw
   if (OldCur != Current && IsDamaged() == false)
   {
      CombinedGC GC;
      if (ConnectGC(GC) == false)
	 return true;
      QuickRedraw(GC,OldCur,Current);
   }
   
   return true;
}
									/*}}}*/
// Tree::Center - Center an item in the window				/*{{{*/
// ---------------------------------------------------------------------
/* Pos selects the index from the top that the item should be positioned
   0 is center, 1 is the first position. If force is false then no change
   will be made if the current item is already visible. */
void Tree::Center(Item *Target,unsigned long Pos,bool Force = true)
{
   // Check if the item is visible
   if (Force == false)
   {
      Item *I = Top;
      unsigned long Count = CanvasSize().y/ItemHeight - 1;
      for (; I != 0 && Count != 0 && I != Target; I = I->Down(), Count--);
      if (I == Target)
	 return;
   }
   
   // Compute the number of items from the top
   if (Pos == 0)
      Pos = CanvasSize().y/ItemHeight/2;
   else
      Pos--;
   
   // Go up that many items
   Item *I = Target;
   for (; I != 0 && Pos != 0; I = I->Up(), Pos--);
   if (Top != I)
      Damage();
   Top = I;
}
									/*}}}*/
// Tree::ItemFromTop - Return the item N items from the top		/*{{{*/
// ---------------------------------------------------------------------
/* This is usefull for mouse routines */
Tree::Item *Tree::ItemFromTop(unsigned long Count)
{
   Item *I = Top;
   for (; I != 0 && Count != 0; Count--)
   {
      Item *Tmp = I->Down();
      if (Tmp == I)
	 break;
      I = Tmp;
   }
   
   if (Count == 0)
      return I;
   return 0;
}
									/*}}}*/
// Tree::Mouse - Mouse event handler for the list			/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Mouse(const MouseEvent &Event)
{
   if (iRoot == 0)
      return;

   // Outside of the widgets canvas?
   if (Event.Pos.y < 0 || Event.Pos.x < 0 ||
       Event.Pos.y >= (Pos.h - 2*BorderY) || 
       (Event.Pos.x >= Pos.w - 2*BorderX))
      return;
   
   // Locate the item the mouse is under
   Item *I = ItemFromTop((Event.Pos.y)/ItemHeight);
   if (I == 0)
      return;
   
   // Call the items mouse handler
   I->Mouse(this,Event);
}
									/*}}}*/
// Tree::EraseAll - Remove all items from the tree			/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::EraseAll()
{
   Top = 0;
   Current = 0;
   ItemHeight = 0;
   
   // We have to erase across the root node 
   for (;iRoot != 0;)
   {
      Item *Tmp = iRoot;
      iRoot = Tmp->Next;
      delete Tmp;
   }
   
   Damage();
}
									/*}}}*/
// Tree::CurrentItem - Change the current item				/*{{{*/
// ---------------------------------------------------------------------
/* It is up to the caller to recenter the item if it is off the screen */
void Tree::CurrentItem(Item *I)
{
   if (Current == I)
      return;
   
   CombinedGC GC;
   if (ConnectGC(GC) == false)
      return;
   Item *OldCur = Current;
   Current = I;
   Trigger(Nt_SelectChange,0);
   QuickRedraw(GC,OldCur,Current);
}
									/*}}}*/

// Item::Item - Constructor						/*{{{*/
// ---------------------------------------------------------------------
/* Since it is considerably more effecient to use AfterStep to add 
   new children this does not add this to Parents child list. */
Tree::Item::Item(Item *Parent) : Flags(0), Parent(Parent), Child(0), 
                                    Next(0), Last(0)
                                    
{
}
									/*}}}*/
// Item::~Item - Destructor						/*{{{*/
// ---------------------------------------------------------------------
/* This should really unlink from the parent too..  */
Tree::Item::~Item()
{
   for (Item *I = Child; I != 0;)
   {
      Item *Next = I->Next;
      I->Parent = 0;
      I->Last = 0;
      I->Next = 0;
      delete I;
      I = Next;
   }
}
									/*}}}*/
// Item::After - Add a new item after this item				/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Item::After(Item *Previous)
{
   Next = Previous->Next;
   Last = Previous;
   Previous->Next = this;
   if (Next != 0)
      Next->Last = this;
   Parent = Previous->Parent;   
}
									/*}}}*/
// Item::AfterStep - Add a new item after this item and then inc arg	/*{{{*/
// ---------------------------------------------------------------------
/* This is usefull for building trees, it acepts a reference to a pointer
   to the current endpoint of the insertion. If the pointer is not 0 this
   chains to be after Previous. In all cases Previous will point to this. */
void Tree::Item::AfterStep(Item *&Root,Item *&Previous)
{
   if (Root == 0)
      Root = this;
   
   if (Previous == 0)
   {
      Previous = this;
      return;
   }
   
   Next = Previous->Next;
   Last = Previous;
   Previous->Next = this;
   if (Next != 0)
      Next->Last = this;
   Parent = Previous->Parent;   
   Previous = this;
}
									/*}}}*/
// Item::Down - Step down one item					/*{{{*/
// ---------------------------------------------------------------------
/* This linearly traces down the tree to the next logical item below
   this item (as seen visually) */
Tree::Item *Tree::Item::Down()
{
   // Descend to a child item if one exists
   if (Child != 0 && IsFlag(Item::Expanded))
      return Child;
   else
   {
      // Move across a branch
      if (Next != 0)
	 return Next;
      else
      {
	 Item *I = Parent;
	 for (; I != 0 && I->Next == 0; I = I->Parent);
	 if (I != 0)
	    return I->Next;
      }      
   }
   return this;
}
									/*}}}*/
// Item::Up - Step up one item						/*{{{*/
// ---------------------------------------------------------------------
/* Inverse of Down */
Tree::Item *Tree::Item::Up()
{
   // Check if we are at the end of a branch
   if (Last != 0)
   {
      /* See if the new item has any children which are shown
       in the tree */
      Item *Cur = Last;
      while (Cur != 0)
      {
	 if (Cur->Child != 0 && Last->IsFlag(Item::Expanded))
	 {
	    Cur = Cur->Child;
	    for (; Cur->Next != 0; Cur = Cur->Next);
	 }
	 else 
	    break;
      }      
      return Cur;
   }      
   else
      if (Parent != 0)
	 return Parent;
   return this;
}
									/*}}}*/
// Item::DrawLines - Draw tree lines					/*{{{*/
// ---------------------------------------------------------------------
/* This will draw the tree lines in both text and graphics mode. It takes
   care of the more complex aspects of attempting to determine where
   the various breaking items should occur .*/
void Tree::Item::DrawLines(CombinedGC &GC,long Space,unsigned long Depth,
			   Rect Pos)
{
   // Background fill
   if (GC.IsGraphic() == true && Depth > 0)
      GC->BFill(Rect(Pos.x,0,Space*Depth,Pos.h));
   
   // Vert lines
   Item *Cur = this;
   Item *Last = this;
   long LineWidth = Pos.x + Space*(Depth) - Space/2;
   for (unsigned int I = Depth; I >= 1 ; I--)
   {
      long P = Pos.x + Space*(I-1);
      
      // Figure out of this should be a half height line
      if ((Next == 0 && IsFlag(Expanded) == false) &&
	  Cur->Next == 0 && Last->Next == 0)
      {
	 LineWidth = P + Space/2;
	 if (GC.IsGraphic() == true)
	    GC.gGC->Line(Point(LineWidth,0),Point(LineWidth,Pos.h/2));
	 else
	 {
	    /* We draw a bottom tee if the next left line is also going
	       to end. */
	    if (I != 1 && Cur->Parent->Next == 0 && Cur->Next == 0)
	       GC.tGC->DrawLineChar(Point(Pos.x + I - 1,0),
				    TextGC::BottomTeeChar);
	    else
	       GC.tGC->DrawLineChar(Point(Pos.x + I - 1,0),
				    TextGC::LlCornerChar);
	 }	    
      }
      else
      {
	 if (GC.IsGraphic() == true)
	    GC.gGC->Line(Point(P+Space/2,0),Point(P+Space/2,Pos.h));
	 else
	 {
	    /* The far right line has little lines sticking out to connect
	       to the items */
	    if (I == Depth)
	       GC.tGC->DrawLineChar(Point(Pos.x + I - 1,0),TextGC::LeftTeeChar);
	    else
	       GC.tGC->DrawLineChar(Point(Pos.x + I - 1,0),TextGC::VLineChar);
	 }	    
      }
      
      Last = Cur;
      Cur = Cur->Parent;
   }
   
   // Little horizontal side line
   if (GC.IsGraphic() == true && Depth > 0)
      GC.gGC->Line(Point(LineWidth,Pos.h/2),
		   Point(Pos.x + Space*Depth,Pos.h/2));
}
									/*}}}*/
// Item::SetColors - Set the drawing colours				/*{{{*/
// ---------------------------------------------------------------------
/* Set the drawing colours and return the forground colour */
Color Tree::Item::SetColors(CombinedGC &GC,Tree *Owner,bool Highlight)
{
   // Choose the proper colours
   Color CurColor;
   if (Highlight == true)
   {
      GC->SetColor(CurColor = Owner->ActiveTextColor());
      GC->Background(Owner->ActiveColor());
   }
   else
   {
      GC->SetColor(CurColor = Owner->Foreground());
      GC->Background(Owner->Background());
   }
   return CurColor;
}
									/*}}}*/
// Item::SetColors - Set the drawing colours				/*{{{*/
// ---------------------------------------------------------------------
/* */
Color Tree::Item::SetColors(CombinedGC &GC,Tree *Owner,bool Highlight,
			    Color Clr)
{   
   if (Clr == Wc_None)
      return SetColors(GC,Owner,Highlight);
   
   // Choose the proper colours
   Color CurColor;
   if (Highlight == true)
   {
      GC->SetColor(CurColor = Owner->ActiveTextColor());
      GC->Background(Clr);
   }
   else
   {
      GC->SetColor(CurColor = Clr);
      GC->Background(Owner->Background());
   }
   return CurColor;
}
									/*}}}*/
// Item::Mouse - Mouse handler for the item				/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Item::Mouse(Tree *Owner,const MouseEvent &Event)
{
   // Select button is down
   if (Event.IsButtonDn(MouseEvent::Select) == true)
      Owner->CurrentItem(this);
}
									/*}}}*/
// Item::ExpandChildren - Expand the children list			/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Item::ExpandChildren(Tree *Owner)
{
   if (IsExpandable() == true && IsFlag(Expanded) == false)
   {
      if (Expand(Owner) == true && Child != 0)
	 Flag(Expanded);
      Owner->Damage();
   }
}
									/*}}}*/
// Item::ContractChildren - Contract the children list			/*{{{*/
// ---------------------------------------------------------------------
/* */
void Tree::Item::ContractChildren(Tree *Owner)
{
   if (IsExpandable() == true && IsFlag(Expanded) == true)
   {
      Flag(0,Expanded);
      Contract(Owner);
      Owner->Damage();
   }
}
									/*}}}*/

// TreeString::TreeString - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
TreeString::TreeString(string Text, Tree::Item *Parent) : 
                      Tree::Item(Parent), S(Text) 
{
}
									/*}}}*/
// TreeString::Render - Render a text string item			/*{{{*/
// ---------------------------------------------------------------------
/* */
void TreeString::Render(CombinedGC &GC,unsigned long Depth,Rect Pos, 
			Tree *Owner)
{
   GC->SetFont(Owner->Font());

   if (IsFlag(Selected) == true)
   {
      GC->SetColor(Owner->ActiveColor());
      GC->Fill(Pos);
      GC->SetColor(Owner->ActiveTextColor());
      
      if (GC.IsText() == true)
	 GC.tGC->Background(Owner->ActiveColor());
   }
   else
   {
      GC->SetColor(Owner->Foreground());
      if (GC.IsText() == true)
	 GC.tGC->Background(Owner->Background());
   }
  
   if (GC.IsGraphic() == true)
      GC->DrawString(Point(Pos.x + Depth,Pos.y),S);
   else
      GC->DrawString(Point(Pos.x + 4*Depth,Pos.y),S);
}
									/*}}}*/
// TreeString::Height - Compute the height of the item			/*{{{*/
// ---------------------------------------------------------------------
/* */
long TreeString::Height()
{
   return GenGC::GC->ExtentText("").h;
}
									/*}}}*/
// TreeString::InButton - True if the X coord is within the +/- button	/*{{{*/
// ---------------------------------------------------------------------
/* */
bool TreeString::InButton(unsigned long Depth,long X)
{
   if (Child != 0)
      if (X == (signed)(Depth*2 - 2))
	 return true;
   return false;
}
									/*}}}*/
