# include <stdio.h>
# include <stdlib.h>
# include <time.h>
# include <ctype.h>
# include <fcntl.h>

# include "hfs.h"
# include "hcwd.h"
# include "hfsutil.h"
# include "glob.h"
# include "hls.h"

/*
 * NAME:	timestr()
 * DESCRIPTION:	convert seconds to printable string for directory listings
 */
static
char *timestr(time_t secs)
{
  static char str[18];
  struct tm *t;
  static char *months[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

  /* MMM DD HH:MM YYYY */

  t = localtime(&secs);

  sprintf(str, "%s %2d %02d:%02d %4d",
	  months[t->tm_mon], t->tm_mday,
	  t->tm_hour, t->tm_min, 1900 + t->tm_year);

  return str;
}

/*
 * NAME:	cleanstr()
 * DESCRIPTION:	sanitize a string for printable characters
 */
static
void cleanstr(char *dest, char *str)
{
  for ( ; *str; ++str)
    *dest++ = isprint(*str) ? *str : '?';

  *dest = 0;
}

/*
 * NAME:	hls->main()
 * DESCRIPTION:	implement hls command
 */
int hls_main(int argc, char *argv[])
{
  hfsvol *vol;
  hfsdir *dir;
  hfsdirent ent;
  char *path;
  int fargc;
  char **fargv = 0;
  int result = 0;

  if (argc > 2)
    {
      fprintf(stderr, "Usage: %s [hfs-path]\n", argv[0]);
      return 1;
    }

  vol = hfs_remount(hcwd_getvol(-1), O_RDONLY);
  if (vol == 0)
    return 1;

  if (argc == 2)
    {
      fargv = hfs_glob(vol, 1, &argv[1], &fargc);
      if (fargv == 0)
	{
	  fprintf(stderr, "%s: out of memory\n", argv[0]);
	  result = 1;
	}
      else if (fargc != 1)
	{
	  fprintf(stderr, "%s: %s: ambiguous path\n", argv[0], argv[1]);
	  result = 1;
	}
      else
	path = fargv[0];
    }
  else
    path = ":";

  if (result == 0)
    {
      dir = hfs_opendir(vol, path);
      if (dir == 0)
	{
	  hfs_perrorp(path);
	  result = 1;
	}
    }

  if (result == 0)
    {
      while (hfs_readdir(dir, &ent) >= 0)
	{
	  char name[32];

	  if (ent.flags & HFS_ISDIR)
	    {
	      cleanstr(name, ent.name);

	      printf("d%c %9lu item%c               %s %s\n",
		     ent.fdflags & HFS_FNDR_ISINVISIBLE ? 'i' : ' ',
		     ent.dsize, ent.dsize == 1 ? ' ' : 's',
		     timestr(ent.mddate), name);
	    }
	  else
	    {
	      char type[5], creator[5];

	      cleanstr(type,    ent.type);
	      cleanstr(creator, ent.creator);
	      cleanstr(name,    ent.name);

	      printf("%c%c %4s/%4s %9lu %9lu %s %s\n",
		     ent.flags & HFS_ISLOCKED ? 'F' : 'f',
		     ent.fdflags & HFS_FNDR_ISINVISIBLE ? 'i' : ' ',
		     type, creator, ent.rsize, ent.dsize,
		     timestr(ent.mddate), name);
	    }
	}

      if (hfs_closedir(dir) < 0)
	{
	  hfs_perror("Error closing HFS directory");
	  result = 1;
	}
    }

  if (hfs_umount(vol) < 0 && result == 0)
    {
      hfs_perror("Error closing HFS volume");
      result = 1;
    }

  if (fargv)
    free(fargv);

  return result;
}
