
/* Used by both Maelstrom and the sound server for Maelstrom */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/file.h>			/* For FASYNC on Solaris */
#include <sys/ioctl.h>			/* For FIOASYNC on HPUX */

#ifdef TESTING
#define LIBDIR	"."
#else
#ifndef LIBDIR
#define LIBDIR	"/usr/local/lib/Maelstrom"
#endif
#endif /* TESTING */

char *file2libpath(char *filename)
{
	char *libdir;
	static char fullpath[256];

	if ( (libdir=getenv("MAELSTROM_LIB")) == NULL )
		libdir=LIBDIR;
	sprintf(fullpath, "%s/%s", libdir, filename);
	return(fullpath);
}

#ifdef _SGI_SOURCE
#include <limits.h>
void usleep(unsigned long usec)
{
	sginap((CLK_TCK*usec)/1000000);
}
#else
#if defined(_INCLUDE_HPUX_SOURCE) || defined(__mips)
#if 0

/* Sort of a kludgy usleep().  It assumes that there will be no SIGALRM
   pending besides ours.  Support is there for better SIGALRM handling.
*/
#include <unistd.h>
#include <time.h>
#include <signal.h>

#ifndef __harg
#define __harg int
#endif
static void (*old_alarmhandler)(__harg);
static struct itimerval old_itimer;
static int alarmed;
static void new_alarmhandler(__harg unused)
{
	alarmed=1;
}
void usleep(unsigned long usec)
{
	struct itimerval usleep_itimer;

	old_alarmhandler=signal(SIGALRM, new_alarmhandler);
	usleep_itimer.it_interval.tv_sec = 0;
	usleep_itimer.it_interval.tv_usec = 0;
	usleep_itimer.it_value.tv_sec = (usec/1000000);
	usleep_itimer.it_value.tv_usec = (usec%1000000);
	setitimer(ITIMER_REAL, &usleep_itimer, &old_itimer);
	alarmed=0;
	while ( ! alarmed )
		pause();
	signal(SIGALRM, old_alarmhandler);
}
#else
#include "usleep.c"
#endif /* 0 */
#endif /* HPUX or DEC */
#endif /* SGI */

/* Routine to set up asynchronous I/O on a file descriptor */

int Set_AsyncIO(int fd, void (*handler)(int sig), long handler_flags)
{
        struct sigaction action;
        long  flags;

	/* Set up the I/O handler */
	action.sa_handler = handler;
	sigemptyset(&action.sa_mask);
#ifdef _INCLUDE_HPUX_SOURCE
        action.sa_flags   = 0;
#else
	action.sa_flags   = handler_flags;
#endif
	sigaction(SIGIO, &action, NULL);
#ifdef _INCLUDE_HPUX_SOURCE
	flags = 1;
	if ( ioctl(fd, FIOASYNC, &flags) < 0 ) {
		perror(
                "Set_AsyncIO: Can't set asynchronous I/O on socket");
		exit(255);
	}
	flags = getpid();
	if ( ioctl(fd, SIOCSPGRP, &flags) < 0 ) {
		perror(
                "Set_AsyncIO: Can't set process group for socket");
		exit(255);
	}
#else /* linux, SGI, sparc, etc */
	flags = fcntl(fd, F_GETFL, 0);
	flags |= FASYNC;
	if ( fcntl(fd, F_SETFL, flags) < 0 ) {
		perror(
                "Set_AsyncIO: Can't set asynchronous I/O on socket");
		exit(255);
	}
	if ( fcntl(fd, F_SETOWN, getpid()) < 0 ) {
		perror(
                "Set_AsyncIO: Can't set process group for socket");
		exit(255);
	}
#endif
}
