;;; po-mode.el -- for helping GNU gettext lovers to edit PO files.
;;; Copyright (C) 1995, 1996 Free Software Foundation, Inc.
;;; Franois Pinard <pinard@iro.umontreal.ca>, 1995.
;;; Helped by Greg McGary <gkm@magilla.cichlid.com>.

;; This file is part of GNU gettext.

;; GNU gettext is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU gettext is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, 59 Temple Place - Suite 330, Boston,
;; MA 02111-1307, USA.

;;; This package provides the tools meant to help editing PO files,
;;; as documented in the GNU gettext user's manual.  See this manual
;;; for user documentation, which is not repeated here.

;;; To install, merely put this file somewhere GNU Emacs will find it,
;;; then add the following lines to your .emacs file:
;;;
;;;   (setq auto-mode-alist
;;;         (cons (cons "\\.po[tx]?\\'\\|\\.po\\." 'po-mode) auto-mode-alist))
;;;   (autoload 'po-mode "po-mode")
;;;
;;; You may also adjust some customizations variables, below, by defining
;;; them in your .emacs file.

(defvar po-auto-edit-with-msgid nil
  "*Automatically use msgid when editing untranslated entries.")

(defvar po-auto-fuzzy-on-edit nil
  "*Automatically mark entries fuzzy when being edited.")

(defvar po-auto-select-on-unfuzzy nil
  "*Automatically select some new entry while make an entry not fuzzy.")

(defvar po-auto-replace-revision-date 'ask
  "*Automatically revise date in headers.  Value is nil, t, or ask.")

(defvar po-default-file-header "\
# SOME DESCRIPTIVE TITLE.
# Copyright (C) YEAR Free Software Foundation, Inc.
# FIRST AUTHOR <EMAIL@ADDRESS>, YEAR.
#
#, fuzzy
msgid \"\"
msgstr \"\"
\"Project-Id-Version: PACKAGE VERSION\\n\"
\"PO-Revision-Date: YEAR-MO-DA HO:MI +ZONE\\n\"
\"Last-Translator: FULL NAME <EMAIL@ADDRESS>\\n\"
\"Language-Team: LANGUAGE <LL@li.org>\\n\"
\"MIME-Version: 1.0\\n\"
\"Content-Type: text/plain; charset=CHARSET\\n\"
\"Content-Transfer-Encoding: ENCODING\\n\"
"
  "*Default PO file header.")

;;;; The following block of declarations has the main purpose of avoiding
;;;; byte compiler warnings.  It also introduces some documentation for
;;;; each of these variables, all meant to be local to PO mode buffers.

;; Flag telling that MODE-LINE-STRING should be displayed.  See `Window'
;; page below.  Exceptionally, this variable is local to *all* buffers.

(defvar po-mode-flag)

;; The current entry extends from START-OF-ENTRY to END-OF-ENTRY, and
;; the line containing the msgstr keyword line starts at MIDDLE-OF-ENTRY.
;; ENTRY-TYPE classifies the entry.

(defvar po-start-of-entry)
(defvar po-middle-of-entry)
(defvar po-end-of-entry)
(defvar po-entry-type)

;; A few counters are usefully shown in the Emacs mode line.

(defvar po-translated-counter)
(defvar po-fuzzy-counter)
(defvar po-untranslated-counter)
(defvar po-obsolete-counter)
(defvar po-mode-line-string)

;; A WORK-BUFFER is associated with this PO file, for edition and other
;; various tasks.  WORK-BUFFER-LOCK indicates that the work buffer
;; is already in use, most probably editing some string through Emacs
;; recursive edit.  In this case, one cannot modify the buffer.

(defvar po-work-buffer)
(defvar po-work-buffer-lock)

;; We maintain a set of movable pointers for returning to entries.

(defvar po-marker-stack)

;; SEARCH path contains a list of directories where files may be found,
;; in a format suitable for read completion.  Each directory includes
;; its trailing slash.  PO mode starts with "./" and "../".

(defvar po-search-path)

;; The following variables are meaningful only when REFERENCE-CHECK
;; is identical to START-OF-ENTRY, else they should be recomputed.
;; REFERENCE-ALIST contains all known references for the current
;; entry, each list element is (PROMPT FILE LINE), where PROMPT may
;; be used for completing read, FILE is a string and LINE is a number.
;; REFERENCE-CURSOR is a cycling cursor into REFERENCE-ALIST.

(defvar po-reference-alist)
(defvar po-reference-cursor)
(defvar po-reference-check)

;; The following variables are for marking translatable strings in
;; program sources.  KEYWORDS is the list of keywords for marking
;; translatable strings, kept in a format suitable for reading with
;; completion.  NEXT-FILE-LIST is the list of source files to visit,
;; gotten from the tags table.  STRING-START is the position for
;; the beginning of the last found string, or nil if the string is
;; invalidated.  STRING-END is the position for the end of the string
;; and indicates where the search should be resumed, or nil for the
;; beginning of the current file.

(defvar po-keywords)
(defvar po-next-file-list)
(defvar po-string-start)
(defvar po-string-end)

;;; PO mode variables and constants (usually not to customize).

(eval-and-compile
  (or (fboundp 'gettext) (defsubst gettext (string) string)))
(defsubst _ (string) (gettext string))
(defsubst N_ (string) string)

(defun po-mode-version ()
  "Show Emacs PO mode version."
  (interactive)
  (message (_"Emacs PO mode, version %s") (substring "$Revision: 1.36 $" 11 -2)))

(defvar po-help-display-string
  (_"\
  [PO Mode Summary]       Next Previous            Miscellaneous
                          n    p    Any type       .     Redisplay
Moving around             /t   /M-t Translated     /v    Version info
<    First if any         f    M-f  Fuzzy          ?, h  This help
>    Last if any          o    M-o  Obsolete       =     Current index
/SPC Auto select          u    M-u  Untranslated   O     Other window
                                                   V     Validate
                        Msgstr Comments            Q     Quit
Modifying entries         RET  #    Call editor    U     Undo
TAB   Remove fuzzy mark   k    K    Kill to        E    Edit out full
/DEL  Fuzzy or fade out   w    W    Copy to
LFD   Init with msgid     y    Y    Yank from        [*: Later, /: Docum]

gettext Keyword Marking                            Position Stack
,    Find next string     Compendiums              m  Mark and push current
M-,  Mark translatable    *c    To compendium      r  Pop and return
M-.  Change mark, mark    *M-C  Select, save       x  Exchange current/top

Program Sources           Auxiliary Files          Lexicography
s    Cycle reference      *a    Cycle file         *l    Lookup translation
M-s  Select reference     *M-a  Select file        *M-l  Add/edit translation
S    Consider path        *A    Consider file      *L    Consider lexicon
M-S  Ignore path          *M-A  Ignore file        *M-L  Ignore lexicon
"))

(defvar po-any-msgid-regexp
  "^\\(#~?[ \t]*\\)?msgid.*\n\\(\\(#~?[ \t]*\\)?\".*\n\\)*"
  "Regexp matching a whole msgid field, whether obsolete or not.")

(defvar po-any-msgstr-regexp
  "^\\(#~?[ \t]*\\)?msgstr.*\n\\(\\(#~?[ \t]*\\)?\".*\n\\)*"
  "Regexp matching a whole msgstr field, whether obsolete or not.")

(defvar po-msgfmt-program "msgfmt"
  "Path to msgfmt program from GNU gettext package.")

;; Font lock based highlighting code.
(defconst po-font-lock-keywords
  '(
    ("^\\(msgid \\|msgstr \\)?\"\\|\"$" . font-lock-keyword-face)
    ("[%\\\\]." . font-lock-variable-name-face)
    ("^# .*\\|^#[:,]?" . font-lock-comment-face)
    ("^#:\\(.*\\)" 1 font-lock-reference-face)
    ;; The following line does not work, and I wonder why.
    ;;("^#,\\(.*\\)" 1 font-function-name-reference-face)
    )
  "Additional expressions to highlight in PO mode.")

;; Old activator for `font lock'.  Is it still useful?  I don't think so.
;;
;;(if (boundp 'font-lock-keywords)
;;    (put 'po-mode 'font-lock-keywords 'po-font-lock-keywords))

;; `hilit19' based hilighting code has been disabled, as most probably
;; nobody really needs it (it also generates ugly byte-compiler warnings).
;;
;;(if (fboundp 'hilit-set-mode-patterns)
;;    (hilit-set-mode-patterns 'po-mode
;;			     '(("^# .*\\|^#$" nil comment)
;;			       ("^#[.,:].*" nil include)
;;			       ("^\\(msgid\\|msgstr\\) *\"" nil keyword)
;;			       ("^\"\\|\"$" nil keyword))))

;;; Mode activation.

(defvar po-mode-map nil
  "Keymap for PO mode.")
(if po-mode-map
    ()
  ;; The following line because (make-sparse-keymap) does not work on Demacs.
  (setq po-mode-map (make-keymap))
  (suppress-keymap po-mode-map)
  (define-key po-mode-map "\C-i" 'po-unfuzzy)
  (define-key po-mode-map "\C-j" 'po-msgid-to-msgstr)
  (define-key po-mode-map "\C-m" 'po-edit-msgstr)
  (define-key po-mode-map " " 'po-auto-select-entry)
  (define-key po-mode-map "?" 'po-help)
  (define-key po-mode-map "#" 'po-edit-comment)
  (define-key po-mode-map "," 'po-tags-search)
  (define-key po-mode-map "." 'po-current-entry)
  (define-key po-mode-map "<" 'po-first-entry)
  (define-key po-mode-map "=" 'po-statistics)
  (define-key po-mode-map ">" 'po-last-entry)
;;;;  (define-key po-mode-map "a" 'po-cycle-auxiliary)
;;;;  (define-key po-mode-map "c" 'po-save-entry)
  (define-key po-mode-map "f" 'po-next-fuzzy-entry)
  (define-key po-mode-map "h" 'po-help)
  (define-key po-mode-map "k" 'po-kill-msgstr)
;;;;  (define-key po-mode-map "l" 'po-lookup-lexicons)
  (define-key po-mode-map "m" 'po-push-location)
  (define-key po-mode-map "n" 'po-next-entry)
  (define-key po-mode-map "o" 'po-next-obsolete-entry)
  (define-key po-mode-map "p" 'po-previous-entry)
  (define-key po-mode-map "q" 'po-confirm-and-quit)
  (define-key po-mode-map "r" 'po-pop-location)
  (define-key po-mode-map "s" 'po-cycle-source-reference)
  (define-key po-mode-map "t" 'po-next-translated-entry)
  (define-key po-mode-map "u" 'po-next-untranslated-entry)
  (define-key po-mode-map "v" 'po-mode-version)
  (define-key po-mode-map "w" 'po-kill-ring-save-msgstr)
  (define-key po-mode-map "x" 'po-exchange-location)
  (define-key po-mode-map "y" 'po-yank-msgstr)
;;;;  (define-key po-mode-map "A" 'po-consider-auxiliary-file)
  (define-key po-mode-map "E" 'po-edit-out-full)
  (define-key po-mode-map "K" 'po-kill-comment)
;;;;  (define-key po-mode-map "L" 'po-consider-lexicon-file)
  (define-key po-mode-map "O" 'po-other-window)
  (define-key po-mode-map "Q" 'po-quit)
  (define-key po-mode-map "S" 'po-consider-source-path)
  (define-key po-mode-map "U" 'po-undo)
  (define-key po-mode-map "V" 'po-validate)
  (define-key po-mode-map "W" 'po-kill-ring-save-comment)
  (define-key po-mode-map "Y" 'po-yank-comment)
  (define-key po-mode-map "\177" 'po-fade-out-entry)
  (define-key po-mode-map "\M-," 'po-mark-translatable)
  (define-key po-mode-map "\M-." 'po-select-mark-and-mark)
;;;;  (define-key po-mode-map "\M-a" 'po-select-auxiliary)
;;;;  (define-key po-mode-map "\M-c" 'po-select-and-save-entry)
  (define-key po-mode-map "\M-f" 'po-previous-fuzzy-entry)
;;;;  (define-key po-mode-map "\M-l" 'po-edit-lexicon-entry)
  (define-key po-mode-map "\M-o" 'po-previous-obsolete-entry)
  (define-key po-mode-map "\M-t" 'po-previous-translated-entry)
  (define-key po-mode-map "\M-u" 'po-previous-untranslated-entry)
  (define-key po-mode-map "\M-s" 'po-select-source-reference)
;;;;  (define-key po-mode-map "\M-A" 'po-ignore-auxiliary-file)
;;;;  (define-key po-mode-map "\M-L" 'po-ignore-lexicon-file)
  (define-key po-mode-map "\M-S" 'po-ignore-source-path)
  )

(defvar po-subedit-mode-map nil
  "Keymap while editing a PO mode entry (or the full PO file).")
(if po-subedit-mode-map
    ()
  (setq po-subedit-mode-map (make-sparse-keymap))
  (define-key po-subedit-mode-map "\C-c\C-c" 'exit-recursive-edit))

(defun po-mode ()
  "Major mode for translators when they edit PO files.
Special commands:\\{po-mode-map}
Turning on PO mode calls the value of the variable `po-mode-hook',
if that value is non-nil.  Behaviour may be adjusted with
variables po-auto-edit-with-msgid, po-auto-fuzzy-on-edit,
po-auto-select-on-unfuzzy and po-auto-replace-revision-date."

  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'po-mode)
  (setq mode-name "PO")
  (use-local-map po-mode-map)
  (setq buffer-read-only t)

  (make-local-variable 'font-lock-defaults)
  (setq font-lock-defaults '(po-font-lock-keywords t))

  (make-local-variable 'po-start-of-entry)
  (make-local-variable 'po-middle-of-entry)
  (make-local-variable 'po-end-of-entry)
  (make-local-variable 'po-entry-type)

  (make-local-variable 'po-translated-counter)
  (make-local-variable 'po-fuzzy-counter)
  (make-local-variable 'po-untranslated-counter)
  (make-local-variable 'po-obsolete-counter)
  (make-local-variable 'po-mode-line-string)

  (setq po-mode-flag t)

  (po-check-file-header)
  (po-compute-counters nil)

  (make-local-variable 'po-work-buffer)
  (make-local-variable 'po-work-buffer-lock)
  (setq po-work-buffer
	(generate-new-buffer (concat "*Edit " (buffer-name nil) "*")))
  (setq po-work-buffer-lock nil)

  (make-local-variable 'po-marker-stack)
  (setq po-marker-stack nil)

  (make-local-variable 'po-search-path)
  (setq po-search-path '(("./") ("../")))

  (make-local-variable 'po-reference-alist)
  (make-local-variable 'po-reference-cursor)
  (make-local-variable 'po-reference-check)
  (setq po-reference-alist nil)
  (setq po-reference-cursor nil)
  (setq po-reference-check 0)

  (make-local-variable 'po-keywords)
  (make-local-variable 'po-next-file-list)
  (make-local-variable 'po-string-start)
  (make-local-variable 'po-string-end)
  (setq po-keywords '(("gettext") ("gettext_noop") ("_") ("N_")))
  (setq po-next-file-list nil)
  (setq po-string-start nil)
  (setq po-string-end nil)

  (message (_"You may type `h' or `?' for a short PO mode reminder."))
  (run-hooks 'po-mode-hook))

(defun po-mode-destroy-cleanup ()
  "When destroying a PO mode buffer, kill editing buffer as well."
  (and (string= mode-name "PO")
       (bufferp po-work-buffer)
       (kill-buffer po-work-buffer)))

(if (boundp 'kill-buffer-hook)
    (add-hook 'kill-buffer-hook 'po-mode-destroy-cleanup))

;;; Window management.

(make-variable-buffer-local 'po-mode-flag)

(defvar po-mode-line-entry '(po-mode-flag ("  " po-mode-line-string))
  "Mode line format entry displaying MODE-LINE-STRING.")

;; Insert MODE-LINE-ENTRY in mode line, but on first load only.
(or (member po-mode-line-entry mode-line-format)
    (let ((entry (member 'global-mode-string mode-line-format)))
      (setcdr entry (cons po-mode-line-entry (cdr entry)))))

(defun po-update-mode-line-string ()
  "Compute a new statistics string to display in mode line."
  (setq po-mode-line-string
	(concat (format "%dt" po-translated-counter)
		(if (> po-fuzzy-counter 0)
		    (format "+%df" po-fuzzy-counter))
		(if (> po-untranslated-counter 0)
		    (format "+%du" po-untranslated-counter))
		(if (> po-obsolete-counter 0)
		    (format "+%do" po-obsolete-counter))))
  (set-buffer-modified-p (buffer-modified-p)))

(defun po-type-counter ()
  "Return the symbol name of the counter appropriate for the current entry."
  (cond ((eq po-entry-type 'obsolete) 'po-obsolete-counter)
	((eq po-entry-type 'fuzzy) 'po-fuzzy-counter)
	((eq po-entry-type 'translated) 'po-translated-counter)
	((eq po-entry-type 'untranslated) 'po-untranslated-counter)
	(t (error (_"Unknown entry type")))))

(defun po-decrease-type-counter ()
  "Decrease the counter corresponding to the nature of the current entry."
  (let ((counter (po-type-counter)))
    (set counter (1- (eval counter)))))

(defun po-increase-type-counter ()
  "Increase the counter corresponding to the nature of the current entry.
Then, update the mode line counters."
  (let ((counter (po-type-counter)))
    (set counter (1+ (eval counter))))
  (po-update-mode-line-string))

;; Avoid byte compiler warnings.
(defvar po-fuzzy-regexp)
(defvar po-untranslated-regexp)

(defun po-compute-counters (flag)
  "Prepare counters for mode line display.  If FLAG, also echo entry position."
  (and flag (po-find-span-of-entry))
  (setq po-translated-counter 0)
  (setq po-fuzzy-counter 0)
  (setq po-untranslated-counter 0)
  (setq po-obsolete-counter 0)
  (let ((position 0) (total 0) here)
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward po-any-msgstr-regexp nil t)
	(and (= (% total 20) 0)
	     (if flag
		 (message (_"Position %d/%d") position total)
	       (message (_"Position %d") total)))
	(setq here (point))
	(goto-char (match-beginning 0))
	(setq total (1+ total))
	(and flag (eq (point) po-middle-of-entry) (setq position total))
	(cond ((eq (following-char) ?#)
	       (setq po-obsolete-counter (1+ po-obsolete-counter)))
	      ((looking-at po-untranslated-regexp)
	       (setq po-untranslated-counter (1+ po-untranslated-counter)))
	      (t (setq po-translated-counter (1+ po-translated-counter))))
	(goto-char here))

      ;; Make another pass just for the fuzzy entries, kind of kludgey.
      ;; Counts will be wrong if untranslated entries are fuzzy, yet this
      ;; should not normally happen.
      (goto-char (point-min))
      (while (re-search-forward po-fuzzy-regexp nil t)
	(setq po-fuzzy-counter (1+ po-fuzzy-counter)))
      (setq po-translated-counter (- po-translated-counter po-fuzzy-counter)))

    ;; Push the results out.
    (if flag
	(message (_"\
Position %d/%d; %d translated, %d fuzzy, %d untranslated, %d obsolete")
		 position total po-translated-counter po-fuzzy-counter
		 po-untranslated-counter po-obsolete-counter)
      (message "")))
  (po-update-mode-line-string))

(defun po-redisplay ()
  "Redisplay the current entry."
  (goto-char po-middle-of-entry))

(defun po-other-window ()
  "Get the cursor into another window, out of PO mode."
  (interactive)
  (if (one-window-p t)
      (progn
	(split-window)
	(switch-to-buffer (other-buffer)))
    (other-window 1)))

(defun po-check-lock ()
  "Ensure that GNU Emacs is not currently in recursive edit for PO mode."
  (if po-work-buffer-lock
      (progn
	(pop-to-buffer po-work-buffer)
	(if (y-or-n-p (_"Here is your current edit.  Do you wish to abort it? "))
	    (abort-recursive-edit)
	  (error (_"Type `C-c C-c' once done"))))))

(defun po-edit-out-full ()
  "Go into recursive edit for editing the PO file in fundamental mode."
  (interactive)
  (if (yes-or-no-p (_"Should I let you edit the whole PO file? "))
      (progn
	(message (_"Type `C-c C-c' once done"))
	(let ((buffer-read-only nil))
	  (fundamental-mode)
	  (use-local-map po-subedit-mode-map)
	  (recursive-edit)
	  (use-local-map nil)
	  (po-mode)))))

;;; Processing the PO file header entry.

(defun po-check-file-header ()
  "Create a missing PO mode file header, or replace an oldish one."
  (save-excursion
    (let ((buffer-read-only nil)
	  insert-flag end-of-header)
      (goto-char (point-min))
      (if (re-search-forward po-any-msgstr-regexp nil t)
	  (progn

	    ;; There is at least one entry.
	    (goto-char (match-beginning 0))
	    (previous-line 1)
	    (setq end-of-header (match-end 0))
	    (if (looking-at "msgid \"\"\n")

		;; There is indeed a PO file header.
		(if (re-search-forward "\n\"PO-Revision-Date: "
				       end-of-header t)
		    nil

		  ;; This is an oldish header.  Replace it all.
		  (goto-char end-of-header)
		  (while (> (point) (point-min))
		    (previous-line 1)
		    (insert "#~ ")
		    (beginning-of-line))
		  (beginning-of-line)
		  (setq insert-flag t))

	      ;; The first entry is not a PO file header, insert one.
	      (setq insert-flag t)))

	;; Not a single entry found.
	(setq insert-flag t))

      (goto-char (point-min))
      (and insert-flag (insert po-default-file-header "\n")))))

(defun po-replace-revision-date ()
  "Replace the revision date by current time in the PO file header."
  (if (fboundp 'format-time-string)
      (if (or (eq po-auto-replace-revision-date t)
	      (and (eq po-auto-replace-revision-date 'ask)
		   (y-or-n-p (_"May I set PO-Revision-Date? "))))
	  (save-excursion
	    (goto-char (point-min))
	    (if (re-search-forward "^\"PO-Revision-Date:.*" nil t)
		(let ((buffer-read-only nil))
		  (replace-match (concat "\"PO-Revision-Date: "
					 (format-time-string
					  "%Y-%m-%d %H:%M %Z" (current-time))
					 "\\n\"")
				 t t))))
	(message ""))
    (message (_"PO-Revision-Date should be adjusted..."))))

;;; Handling span of entry, entry type and entry attributes.

(defun po-find-span-of-entry ()
  "Find the extent of the PO file entry where the cursor is.
Set variables PO-START-OF-ENTRY, PO-MIDDLE-OF-ENTRY, PO-END-OF-ENTRY
and PO-ENTRY-TYPE to meaningful values.  Decreasing priority of type
interpretation is: obsolete, fuzzy, untranslated or translated."
  (let ((here (point)))
    (if (re-search-backward po-any-msgstr-regexp nil t)
	(progn

	  ;; After a backward match, (match-end 0) will not extend
	  ;; beyond point, in case point was *inside* the regexp.  We
	  ;; need a dependable (match-end 0), so we redo the match in
	  ;; the forward direction.
	  (re-search-forward po-any-msgstr-regexp)
	  (if (<= (match-end 0) here)
	      (progn

		;; We most probably found the msgstr of the previous
		;; entry.  The current entry then starts just after
		;; its end, save this information just in case.
		(setq po-start-of-entry (match-end 0))

		;; However, it is also possible that we are located in
		;; the crumb after the last entry in the file.  If
		;; yes, we know the middle and end of last PO entry.
		(setq po-middle-of-entry (match-beginning 0))
		(setq po-end-of-entry (match-end 0))

		(if (re-search-forward po-any-msgstr-regexp nil t)
		    (progn

		      ;; We definitely were not in the crumb.
		      (setq po-middle-of-entry (match-beginning 0))
		      (setq po-end-of-entry (match-end 0)))

		  ;; We were in the crumb.  The start of the last PO
		  ;; file entry is the end of the previous msgstr if
		  ;; any, or else, the beginning of the file.
		  (goto-char po-middle-of-entry)
		  (setq po-start-of-entry
			(if (re-search-backward po-any-msgstr-regexp nil t)
			    (match-end 0)
			  (point-min)))))

	    ;; The cursor was inside msgstr of the current entry.
	    (setq po-middle-of-entry (match-beginning 0))
	    (setq po-end-of-entry (match-end 0))
	    ;; The start of this entry is the end of the previous
	    ;; msgstr if any, or else, the beginning of the file.
	    (goto-char po-middle-of-entry)
	    (setq po-start-of-entry
		  (if (re-search-backward po-any-msgstr-regexp nil t)
		      (match-end 0)
		    (point-min)))))

      ;; The cursor was before msgstr in the first entry in the file.
      (setq po-start-of-entry (point-min))
      (goto-char po-start-of-entry)
      ;; There is at least the PO file header, so this should match.
      (re-search-forward po-any-msgstr-regexp)
      (setq po-middle-of-entry (match-beginning 0))
      (setq po-end-of-entry (match-end 0)))

    ;; Classify the entry.
    (goto-char po-middle-of-entry)
    (setq po-entry-type
	  (if (eq (following-char) ?#)
	      'obsolete
	    (goto-char po-start-of-entry)
	    (if (re-search-forward po-fuzzy-regexp po-middle-of-entry t)
		'fuzzy
	      (goto-char po-middle-of-entry)
	      (if (looking-at po-untranslated-regexp)
		  'untranslated
		'translated))))

    ;; Put the cursor back where it was.
    (goto-char here)))

(defun po-add-attribute (name)
  "Add attribute NAME to the current entry, unless it is already there."
  (save-excursion
    (let ((buffer-read-only nil))
      (goto-char po-start-of-entry)
      (if (re-search-forward "\n#[,!] .*" po-middle-of-entry t)
	  (save-restriction
	    (narrow-to-region (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (if (re-search-forward (concat "\\b" name "\\b") nil t)
		nil
	      (goto-char (point-max))
	      (insert ", " name)))
	(skip-chars-forward "\n")
	(while (eq (following-char) ?#)
	  (next-line 1))
	(insert "#, " name "\n")))))

(defun po-delete-attribute (name)
  "Delete attribute NAME from the current entry, if any."
  (save-excursion
    (let ((buffer-read-only nil))
      (goto-char po-start-of-entry)
      (if (re-search-forward "\n#[,!] .*" po-middle-of-entry t)
	  (save-restriction
	    (narrow-to-region (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (if (re-search-forward
		 (concat "\\(\n#[,!] " name "$\\|, " name "$\\| " name ",\\)")
		 nil t)
		(replace-match "" t t)))))))

;;; Entry positionning.

(defun po-say-location-depth ()
  "Tell how many entries in the entry location stack."
  (let ((depth (length po-marker-stack)))
    (cond ((= depth 0) (message (_"Empty location stack")))
	  ((= depth 1) (message (_"One entry in location stack")))
	  (t (message (_"%d entries in location stack") depth)))))

(defun po-push-location ()
  "Stack the location of the current entry, for later return."
  (interactive)
  (po-find-span-of-entry)
  (save-excursion
    (goto-char po-middle-of-entry)
    (setq po-marker-stack (cons (point-marker) po-marker-stack)))
  (po-say-location-depth))

(defun po-pop-location ()
  "Unstack a saved location, and return to the corresponding entry."
  (interactive)
  (if po-marker-stack
      (progn
	(goto-char (car po-marker-stack))
	(setq po-marker-stack (cdr po-marker-stack))
	(po-current-entry)
	(po-say-location-depth))
    (error (_"The entry location stack is empty"))))

(defun po-exchange-location ()
  "Exchange the location of the current entry with the top of stack."
  (interactive)
  (if po-marker-stack
      (progn
	(po-find-span-of-entry)
	(goto-char po-middle-of-entry)
	(let ((location (point-marker)))
	  (goto-char (car po-marker-stack))
	  (setq po-marker-stack (cons location (cdr po-marker-stack))))
	(po-current-entry)
	(po-say-location-depth))
    (error (_"The entry location stack is empty"))))

(defun po-current-entry ()
  "Display the current entry."
  (interactive)
  (po-find-span-of-entry)
  (po-redisplay))

(defun po-first-entry-with-regexp (regexp)
  "Display the first entry in the file which msgstr matches REGEXP."
  (let ((here (point)))
    (goto-char (point-min))
    (if (re-search-forward regexp nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (po-current-entry))
      (goto-char here)
      (error (_"There is no such entry")))))

(defun po-last-entry-with-regexp (regexp)
  "Display the last entry in the file which msgstr matches REGEXP."
  (let ((here (point)))
    (goto-char (point-max))
    (if (re-search-backward regexp nil t)
	(po-current-entry)
      (goto-char here)
      (error (_"There is no such entry")))))

(defun po-next-entry-with-regexp (regexp wrap)
  "Display the entry following the current entry which msgstr matches REGEXP.
If WRAP is not nil, the search may wrap around the buffer."
  (po-find-span-of-entry)
  (let ((here (point)))
    (goto-char po-end-of-entry)
    (if (re-search-forward regexp nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (po-current-entry))
      (if (and wrap
	       (progn
		 (goto-char (point-min))
		 (re-search-forward regexp po-start-of-entry t)))
	  (progn
	    (goto-char (match-beginning 0))
	    (po-current-entry)
	    (message (_"Wrapping around the buffer")))
	(goto-char here)
	(error (_"There is no such entry"))))))

(defun po-previous-entry-with-regexp (regexp wrap)
  "Redisplay the entry preceding the current entry which msgstr matches REGEXP.
If WRAP is not nil, the search may wrap around the buffer."
  (po-find-span-of-entry)
  (let ((here (point)))
    (goto-char po-start-of-entry)
    (if (re-search-backward regexp nil t)
	(po-current-entry)
      (if (and wrap
	       (progn
		 (goto-char (point-max))
		 (re-search-backward regexp po-end-of-entry t)))
	  (progn
	    (po-current-entry)
	    (message (_"Wrapping around the buffer")))
	(goto-char here)
	(error (_"There is no such entry"))))))

;; Any entries.

(defun po-first-entry ()
  "Display the first entry."
  (interactive)
  (po-first-entry-with-regexp po-any-msgstr-regexp))

(defun po-last-entry ()
  "Display the last entry."
  (interactive)
  (po-last-entry-with-regexp po-any-msgstr-regexp))

(defun po-next-entry ()
  "Display the entry following the current entry."
  (interactive)
  (po-next-entry-with-regexp po-any-msgstr-regexp nil))

(defun po-previous-entry ()
  "Display the entry preceding the current entry."
  (interactive)
  (po-previous-entry-with-regexp po-any-msgstr-regexp nil))

;; Untranslated entries.

(defvar po-after-entry-regexp
  "\\(\\'\\|\\(#[ \t]*\\)?[^\"]\\)"
  "Regexp which should be true after a full msgstr string matched.")

(defvar po-untranslated-regexp
  (concat "^msgstr[ \t]*\"\"\n" po-after-entry-regexp)
  "Regexp matching a whole msgstr field, but only if active and empty.")

(defun po-next-untranslated-entry ()
  "Find the next untranslated entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-untranslated-regexp t))

(defun po-previous-untranslated-entry ()
  "Find the previous untranslated entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-untranslated-regexp t))

(defun po-msgid-to-msgstr ()
  "Use another window to edit msgstr reinitialized with msgid."
  (interactive)
  (po-find-span-of-entry)
  (if (or (eq po-entry-type 'untranslated)
	  (eq po-entry-type 'obsolete)
	  (y-or-n-p (_"Really loose previous translation? ")))
      (progn
	(po-decrease-type-counter)
	(po-set-field nil (po-get-field t nil))
	(po-current-entry)
	(po-increase-type-counter)))
  (message ""))

;; Obsolete entries.

(defvar po-obsolete-msgstr-regexp
  "^#~?[ \t]*msgstr.*\n\\(#~?[ \t]*\".*\n\\)*"
  "Regexp matching a whole msgstr field of an obsolete entry.")

(defun po-next-obsolete-entry ()
  "Find the next obsolete entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-obsolete-msgstr-regexp t))

(defun po-previous-obsolete-entry ()
  "Find the previous obsolete entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-obsolete-msgstr-regexp t))

;; Fuzzy entries.

(defvar po-fuzzy-regexp "^#[,!] .*fuzzy"
  "Regexp matching the string inserted by msgmerge for translations
which does not match exactly.")

(defun po-next-fuzzy-entry ()
  "Find the next fuzzy entry, wrapping around if necessary."
  (interactive)
  (po-next-entry-with-regexp po-fuzzy-regexp t))

(defun po-previous-fuzzy-entry ()
  "Find the next fuzzy entry, wrapping around if necessary."
  (interactive)
  (po-previous-entry-with-regexp po-fuzzy-regexp t))

(defun po-unfuzzy ()
  "Remove the fuzzy attribute for the current entry."
  (interactive)
  (po-find-span-of-entry)
  (cond ((eq po-entry-type 'fuzzy)
	 (po-decrease-type-counter)
	 (po-delete-attribute "fuzzy")
	 (po-current-entry)
	 (po-increase-type-counter)))
  (if po-auto-select-on-unfuzzy
      (po-auto-select-entry))
  (po-update-mode-line-string))

;; Translated entries.

(defun po-next-translated-entry ()
  "Find the next untranslated entry, wrapping around if necessary."
  (interactive)
  (if (= po-translated-counter 0)
      (error (_"There is no such entry"))
    (po-next-entry-with-regexp po-untranslated-regexp t)
    (po-find-span-of-entry)
    (while (not (eq po-entry-type 'translated))
      (po-next-entry-with-regexp po-any-msgstr-regexp t)
      (po-find-span-of-entry))))

(defun po-previous-translated-entry ()
  "Find the previous untranslated entry, wrapping around if necessary."
  (interactive)
  (if (= po-translated-counter 0)
      (error (_"There is no such entry"))
    (po-previous-entry-with-regexp po-any-msgstr-regexp t)
    (po-find-span-of-entry)
    (while (not (eq po-entry-type 'translated))
      (po-previous-entry-with-regexp po-untranslated-regexp t)
    (po-find-span-of-entry))))

;; Auto-selection feature.

(defun po-auto-select-entry ()
  "Select the next entry having the same type as the current one.
If none, wrap from the beginning of the buffer with another type,
going from untranslated to fuzzy, and from fuzzy to obsolete.
Plain translated entries are always disregarded unless there are
no entries of the other types."
  (interactive)
  (po-find-span-of-entry)
  (goto-char po-end-of-entry)
  (if (and (= po-untranslated-counter 0)
	   (= po-fuzzy-counter 0)
	   (= po-obsolete-counter 0))

      ;; All entries are plain translated.  Next entry will do, or
      ;; wrap around if there is none.
      (if (re-search-forward po-any-msgstr-regexp nil t)
	  (goto-char (match-beginning 0))
	(goto-char (point-min)))

    ;; If over a translated entry, look for an untranslated one first.
    ;; Else, look for an entry of the same type first.
    (let ((goal (if (eq po-entry-type 'translated)
		    'untranslated
		  po-entry-type)))
      (while goal

	;; Find an untranslated entry, or wrap up for a fuzzy entry.
	(if (eq goal 'untranslated)
	    (if (and (> po-untranslated-counter 0)
		     (re-search-forward po-untranslated-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'fuzzy)))

	;; Find a fuzzy entry, or wrap up for an obsolete entry.
	(if (eq goal 'fuzzy)
	    (if (and (> po-fuzzy-counter 0)
		     (re-search-forward po-fuzzy-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'obsolete)))

	;; Find an obsolete entry, or wrap up for an untranslated entry.
	(if (eq goal 'obsolete)
	    (if (and (> po-obsolete-counter 0)
		     (re-search-forward po-obsolete-msgstr-regexp nil t))
		(progn
		  (goto-char (match-beginning 0))
		  (setq goal nil))
	      (goto-char (point-min))
	      (setq goal 'untranslated))))))

  ;; Display this entry nicely.
  (po-current-entry))

;;; Killing and yanking fields.

(eval-and-compile
  (if (fboundp 'kill-new)

      (fset 'po-kill-new (symbol-function 'kill-new))

    (defun po-kill-new (string)
      "Push STRING onto the kill ring, for Emacs 18 where kill-new is missing."
      (po-check-lock)
      (save-excursion
	(set-buffer po-work-buffer)
	(erase-buffer)
	(insert string)
	(kill-region (point-min) (point-max))))))

(defun po-extract-unquoted (buffer start end)
  "Extract and return the unquoted string in BUFFER going from START to END.
Crumb preceding or following the quoted string is ignored."
  (po-check-lock)
  (save-excursion
    (set-buffer po-work-buffer)
    (erase-buffer)
    (insert-buffer-substring buffer start end)
    (goto-char (point-min))
    (search-forward "\"")
    (delete-region (point-min) (point))
    (goto-char (point-max))
    (search-backward "\"")
    (delete-region (point) (point-max))
    (goto-char (point-min))
    ;; It seems that "\"[ \t]*\\\\?\n\(#~?\)?[ \t]*\"" would not work
    ;; below, for some unknown reason.  Could it be a regexp bug?
    (while (re-search-forward "\"[ \t]*\\\\?\n#?~?[ \t]*\"" nil t)
      (replace-match "" t t))
    (goto-char (point-min))
    (while (re-search-forward "\\\\[\\\"abfnt\\\\0-7]" nil t)
      (cond ((eq (preceding-char) ?\") (replace-match "\"" t t))
	    ((eq (preceding-char) ?a) (replace-match "\a" t t))
	    ((eq (preceding-char) ?b) (replace-match "\b" t t))
	    ((eq (preceding-char) ?f) (replace-match "\f" t t))
	    ((eq (preceding-char) ?n) (replace-match "\n" t t))
	    ((eq (preceding-char) ?t) (replace-match "\t" t t))
	    ((eq (preceding-char) ?\\) (replace-match "\\" t t))
	    (t (let ((value (- (preceding-char) ?0)))
		 (replace-match "" t t)
		 (while (looking-at "[0-7]")
		   (setq value (+ (* 8 value) (- (following-char) ?0)))
		   (replace-match "" t t))
		 (insert value)))))
    (buffer-string)))

(defun po-eval-requoted (form prefix obsolete)
  "Eval FORM, which inserts a string, and return the string fully requoted.
If PREFIX, precede the result with its contents.  If OBSOLETE, comment all
generated lines in the returned string.  Evaluating FORM should insert the
wanted string in the buffer which is current at the time of evaluation.
If FORM is itself a string, then this string is used for insertion."
  (po-check-lock)
  (save-excursion
    (set-buffer po-work-buffer)
    (erase-buffer)
    (if (stringp form)
	(insert form)
      (push-mark)
      (eval form))
    (goto-char (point-min))
    (let ((multi-line (re-search-forward "[^\n]\n+[^\n]" nil t)))
      (goto-char (point-min))
      (while (re-search-forward "[\\\"\a\b\f\n\t\\\\]" nil t)
	(cond ((eq (preceding-char) ?\") (replace-match "\\\"" t t))
	      ((eq (preceding-char) ?\a) (replace-match "\\a" t t))
	      ((eq (preceding-char) ?\b) (replace-match "\\b" t t))
	      ((eq (preceding-char) ?\f) (replace-match "\\f" t t))
	      ((eq (preceding-char) ?\n)
	       (replace-match (if (or (not multi-line) (eobp))
				  "\\n"
				"\\n\"\n\"")
			      t t))
	      ((eq (preceding-char) ?\t) (replace-match "\\t" t t))
	      ((eq (preceding-char) ?\\) (replace-match "\\\\" t t))))
      (goto-char (point-min))
      (if prefix (insert prefix " "))
      (insert (if multi-line "\"\"\n\"" "\""))
      (goto-char (point-max))
      (insert "\"")
      (if prefix (insert "\n"))
      (if obsolete
	  (progn
	    (goto-char (point-min))
	    (while (not (eobp))
	      (or (eq (following-char) ?\n) (insert "#~ "))
	      (search-forward "\n"))))
      (buffer-string))))

(defun po-get-field (msgid kill)
  "Extract and return the unquoted msgstr string, unless MSGID selects msgid.
If KILL, then add the unquoted string to the kill ring."
  (let ((string (if msgid
		    (progn
		      (save-excursion
			(goto-char po-start-of-entry)
			(re-search-forward po-any-msgid-regexp
					   po-end-of-entry t))
		      (po-extract-unquoted (current-buffer)
					   (match-beginning 0) (match-end 0)))
		  (po-extract-unquoted (current-buffer)
				       po-middle-of-entry po-end-of-entry))))
    (if kill (po-kill-new string))
    string))

(defun po-set-field (msgid form)
  "Replace the current msgstr, unless MSGID, using FORM to get a string.
If MSGID is true, replace the current msgid instead.  In either case,
evaluating FORM should insert the wanted string in the current buffer.
If FORM is itself a string, then this string is used for insertion.
The string is properly requoted before the replacement occurs."
  (or msgid (po-decrease-type-counter))
  (let ((string (po-eval-requoted form (if msgid "msgid" "msgstr")
				  (eq po-entry-type 'obsolete))))
    (save-excursion
      (goto-char po-start-of-entry)
      (re-search-forward (if msgid po-any-msgid-regexp po-any-msgstr-regexp)
			 po-end-of-entry)
      (if (not (string-equal (buffer-substring (match-beginning 0)
					       (match-end 0))
			     string))
	  (let ((buffer-read-only nil))
	    (replace-match string t t)))
      (if msgid
	  (progn
	    (re-search-forward po-any-msgstr-regexp)
	    (setq po-middle-of-entry (match-beginning 0))
	    (setq po-end-of-entry (match-end 0)))
	(setq po-end-of-entry (point)))))
  (or msgid (po-increase-type-counter))
  (po-redisplay))

(defun po-kill-ring-save-msgstr ()
  "Push the msgstr string from current entry on the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-get-field nil t))

(defun po-kill-msgstr ()
  "Empty the msgstr string from current entry, pushing it on the kill ring."
  (interactive)
  (po-kill-ring-save-msgstr)
  (po-decrease-type-counter)
  (po-set-field nil "")
  (po-current-entry)
  (po-increase-type-counter))

(defun po-yank-msgstr ()
  "Replace the current msgstr string by the top of the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-decrease-type-counter)
  (po-set-field nil (if (eq last-command 'yank) '(yank-pop 1) '(yank)))
  (po-current-entry)
  (po-increase-type-counter)
  (setq this-command 'yank))

(defun po-fade-out-entry ()
  "Mark an active entry as fuzzy; obsolete a fuzzy or untranslated entry;
or completely delete an obsolete entry, saving its msgstr on the kill ring."
  (interactive)
  (po-check-lock)
  (po-find-span-of-entry)

  (cond ((eq po-entry-type 'translated)
	 (po-decrease-type-counter)
	 (po-add-attribute "fuzzy")
	 (po-current-entry)
	 (po-increase-type-counter))

	((or (eq po-entry-type 'fuzzy)
	     (eq po-entry-type 'untranslated))
	 (if (yes-or-no-p (_"Should I really obsolete this entry? "))
	     (progn
	       (po-decrease-type-counter)
	       (save-excursion
		 (save-restriction
		   (narrow-to-region po-start-of-entry po-end-of-entry)
		   (let ((buffer-read-only nil))
		     (goto-char (point-min))
		     (while (not (eobp))
		       (or (eq (following-char) ?\n) (insert "#~ "))
		       (search-forward "\n")))))
	       (po-current-entry)
	       (po-increase-type-counter)))
	 (message ""))

	((eq po-entry-type 'obsolete)
	 (po-decrease-type-counter)
	 (po-update-mode-line-string)
	 (po-get-field nil t)
	 (let ((buffer-read-only nil))
	   (delete-region po-start-of-entry po-end-of-entry))
	 (goto-char po-start-of-entry)
	 (if (re-search-forward po-any-msgstr-regexp nil t)
	     (goto-char (match-beginning 0))
	   (re-search-backward po-any-msgstr-regexp nil t))
	 (po-current-entry)
	 (message ""))))

;;; Killing and yanking comments.

(defvar po-active-comment-regexp
  "^\\(#\n\\|# .*\n\\)+"
  "Regexp matching the whole editable comment part of an active entry.")

(defvar po-obsolete-comment-regexp
  "^\\(#~? #\n\\|#~? # .*\n\\)+"
  "Regexp matching the whole editable comment part of an obsolete entry.")

(defun po-get-comment (kill-flag)
  "Extract and return the editable comment string, uncommented.
If KILL-FLAG, then add the unquoted comment to the kill ring."
  (po-check-lock)
  (let ((buffer (current-buffer))
	(obsolete (eq po-entry-type 'obsolete)))
    (save-excursion
      (goto-char po-start-of-entry)
      (if (re-search-forward (if obsolete po-obsolete-comment-regexp
			         po-active-comment-regexp)
			     po-end-of-entry t)
	  (progn
	    (set-buffer po-work-buffer)
	    (erase-buffer)
	    (insert-buffer-substring buffer (match-beginning 0) (match-end 0))
	    (goto-char (point-min))
	    (while (not (eobp))
	      (if (looking-at (if obsolete "#~? # ?" "#~? ?"))
		  (replace-match "" t t))
	      (forward-line 1))
	    (and kill-flag (copy-region-as-kill (point-min) (point-max)))
	    (buffer-string))
	""))))

(defun po-set-comment (form)
  "Using FORM to get a string, replace the current editable comment.
Evaluating FORM should insert the wanted string in the current buffer.
If FORM is itself a string, then this string is used for insertion.
The string is properly recommented before the replacement occurs."
  (po-check-lock)
  (let ((buffer (current-buffer))
	(obsolete (eq po-entry-type 'obsolete))
	string)
    (save-excursion
      (set-buffer po-work-buffer)
      (erase-buffer)
      (if (stringp form)
	  (insert form)
	(push-mark)
	(eval form))
      (if (not (or (bobp) (= (preceding-char) ?\n)))
	  (insert "\n"))
      (goto-char (point-min))
      (while (not (eobp))
	(insert (if (= (following-char) ?\n)
		    (if obsolete "#~ #" "#")
		  (if obsolete "#~ # " "# ")))
	(search-forward "\n"))
      (setq string (buffer-string)))
    (goto-char po-start-of-entry)
    (if (and (re-search-forward
	      (if obsolete po-obsolete-comment-regexp po-active-comment-regexp)
	      po-end-of-entry t)
	     (not (string-equal
		   (buffer-substring (match-beginning 0) (match-end 0))
		   string)))
	(let ((buffer-read-only nil))
	  (replace-match string t t))
      (skip-chars-forward " \t\n")
      (let ((buffer-read-only nil))
	(insert string))))
  (re-search-forward po-any-msgstr-regexp)
  (setq po-middle-of-entry (match-beginning 0))
  (setq po-end-of-entry (match-end 0))
  (po-redisplay))

(defun po-kill-ring-save-comment ()
  "Push the msgstr string from current entry on the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-get-comment t))

(defun po-kill-comment ()
  "Empty the msgstr string from current entry, pushing it on the kill ring."
  (interactive)
  (po-kill-ring-save-comment)
  (po-set-comment "")
  (po-redisplay))

(defun po-yank-comment ()
  "Replace the current comment string by the top of the kill ring."
  (interactive)
  (po-find-span-of-entry)
  (po-set-comment (if (eq last-command 'yank) '(yank-pop 1) '(yank)))
  (setq this-command 'yank)
  (po-redisplay))

;;; Editing translations.

(defun po-edit-string (string)
  "Edit STRING recursively in a pop-up buffer, return the edited string.
Run po-subedit-mode-hook first prior to beginning edition.  If recursive edit
is aborted, return nil instead."
  (po-check-lock)
  (let ((po-work-buffer-lock t)
	(start po-start-of-entry)
	(middle po-middle-of-entry)
	(end po-end-of-entry)
	(obsolete (eq po-entry-type 'obsolete)))
    (prog1
	(save-window-excursion
	  (pop-to-buffer po-work-buffer)
	  (erase-buffer)
	  (run-hooks 'po-subedit-mode-hook)
	  (insert string "<")
	  (goto-char (point-min))
	  (condition-case nil
	      (progn
		(use-local-map po-subedit-mode-map)
		(message (_"Type `C-c C-c' once done"))
		(recursive-edit)
		(goto-char (point-max))
		(skip-chars-backward " \t\n")
		(if (eq (preceding-char) ?<)
		    (delete-region (1- (point)) (point-max)))
		(buffer-string))
	    (quit nil)))
      (bury-buffer po-work-buffer))))

(defun po-edit-comment ()
  "Use another window to edit the current translator comment."
  (interactive)
  (po-find-span-of-entry)
;  ;; Try showing all of msgid in the upper window while editing.
;  (goto-char po-start-of-entry)
;  (re-search-forward po-any-msgid-regexp)
;  (backward-char 1)
;  (recenter -1)
  (let ((string (po-edit-string (po-get-comment nil))))
    (and string (po-set-comment string))
    (po-redisplay)))

(defun po-edit-msgstr ()
  "Use another window to edit the current msgstr."
  (interactive)
  (po-find-span-of-entry)
  (and po-auto-edit-with-msgid
       (eq po-entry-type 'untranslated)
       (po-msgid-to-msgstr))
  ;; ;; Try showing all of msgid in the upper window while editing.
  ;; (goto-char po-start-of-entry)
  ;; (re-search-forward po-any-msgid-regexp)
  ;; (backward-char 1)
  ;; (recenter -1)
  (let ((string (po-edit-string (po-get-field nil nil))))
    (if string
	(progn
	  (po-decrease-type-counter)
	  (po-set-field nil string)
	  (po-current-entry)
	  (if (and po-auto-fuzzy-on-edit
		   (eq po-entry-type 'translated))
	      (progn
		(po-add-attribute "fuzzy")
		(po-current-entry)))
	  (po-increase-type-counter)))))

;;; String normalization and searching.

(defun po-normalize-old-style (explain)
  "Normalize old gettext style fields using K&R C multiline string syntax.
To minibuffer messages sent while normalizing, add the EXPLAIN string."
  (let ((here (point-marker))
	(counter 0)
	(buffer-read-only nil))
    (goto-char (point-min))
    (message (_"Normalizing %d, %s") counter explain)
    (while (re-search-forward
	    "\\(^#?[ \t]*msg\\(id\\|str\\)[ \t]*\"\\|[^\" \t][ \t]*\\)\\\\\n"
	    nil t)
      (if (= (% counter 10) 0)
	  (message (_"Normalizing %d, %s") counter explain))
      (replace-match "\\1\"\n\"" t nil)
      (setq counter (1+ counter)))
    (goto-char here)
    (message (_"Normalizing %d...done") counter)))

(defun po-normalize-field (msgid explain)
  "Normalize all msgstr's, or msgid's if MSGID.
To minibuffer messages sent while normalizing, add the EXPLAIN string."
  (let ((here (point-marker))
	(counter 0))
    (goto-char (point-min))
    (while (re-search-forward po-any-msgstr-regexp nil t)
      (if (= (% counter 10) 0)
	  (message (_"Normalizing %d, %s") counter explain))
      (goto-char (match-beginning 0))
      (po-find-span-of-entry)
      (po-set-field msgid (po-get-field msgid nil))
      (goto-char po-end-of-entry)
      (setq counter (1+ counter)))
    (goto-char here)
    (message (_"Normalizing %d...done") counter)))

(defun po-normalize ()
  "Normalize all entries in the PO file."
  (interactive)
  (po-normalize-old-style (_"pass 1/3"))
  (po-normalize-field t (_"pass 2/3"))
  (po-normalize-field nil (_"pass 3/3"))
  ;; The last PO file entry has just been processed.
  (if (not (= po-end-of-entry (point-max)))
      (let ((buffer-read-only nil))
	(kill-region po-end-of-entry (point-max))))
  ;; A bizarre format might have fooled the counters, so recompute
  ;; them to make sure their value is dependable.
  (po-compute-counters nil))

;;; Original program sources as context.

(defun po-show-source-path ()
  "Echo the current source search path in the message area."
  (let ((path po-search-path)
	(string (_"Path is:")))
    (while path
      (setq string (concat string " " (car (car path))))
      (setq path (cdr path)))
    (message string)))

(defun po-consider-source-path (directory)
  "Add a given DIRECTORY, requested interactively, to the source search path."
  (interactive "DDirectory for search path: ")
  (setq po-search-path (cons (list directory) po-search-path))
  (setq po-reference-check 0)
  (po-show-source-path))

(defun po-ignore-source-path ()
  "Delete a directory, selected with completion, from the source search path."
  (interactive)
  (setq po-search-path
	(delete (list (completing-read (_"Directory to remove? ")
				       po-search-path nil t))
		po-search-path))
  (setq po-reference-check 0)
  (po-show-source-path))

(defun po-ensure-source-references ()
  "Extract all references into a list, with paths resolved, if necessary."
  (po-find-span-of-entry)
  (if (= po-start-of-entry po-reference-check)
      ()
    (setq po-reference-alist nil)
    (save-excursion
      (goto-char po-start-of-entry)
      (if (re-search-forward "^#:" po-middle-of-entry t)
	  (while (looking-at "\\(\n#:\\)? *\\([^: ]+\\):\\([0-9]+\\)")
	    (goto-char (match-end 0))
	    (let* ((name (buffer-substring (match-beginning 2) (match-end 2)))
		   (line (buffer-substring (match-beginning 3) (match-end 3)))
		   (path po-search-path)
		   file)
	      (while (and (progn (setq file (concat (car (car path)) name))
				 (not (file-exists-p file)))
			  path)
		(setq path (cdr path)))
	      (if path
		  (setq po-reference-alist
			(cons (list (concat file ":" line)
				    file
				    (string-to-int line))
			      po-reference-alist)))))))
    (setq po-reference-alist (nreverse po-reference-alist))
    (setq po-reference-cursor po-reference-alist)
    (setq po-reference-check po-start-of-entry)))

(defun po-show-source-context (triplet)
  "Show the source context given a TRIPLET which is (PROMPT FILE LINE)."
  (find-file-other-window (car (cdr triplet)))
  (goto-line (car (cdr (cdr triplet))))
  (other-window 1)
  (let ((maximum 0)
	position
	(cursor po-reference-alist))
    (while (not (eq triplet (car cursor)))
      (setq maximum (1+ maximum))
      (setq cursor (cdr cursor)))
    (setq position (1+ maximum))
    (setq po-reference-cursor cursor)
    (while cursor
      (setq maximum (1+ maximum))
      (setq cursor (cdr cursor)))
    (message (_"Displaying %d/%d: \"%s\"") position maximum (car triplet))))

(defun po-cycle-source-reference ()
  "Display some source context for the current entry.
If the command is repeated many times in a row, cycle through contexts."
  (interactive)
  (po-ensure-source-references)
  (if po-reference-cursor
      (po-show-source-context
       (car (if (eq last-command 'po-cycle-source-reference)
		(or (cdr po-reference-cursor) po-reference-alist)
	      po-reference-cursor)))
    (error (_"No resolved source references"))))

(defun po-select-source-reference ()
  "Select one of the available source contexts for the current entry."
  (interactive)
  (po-ensure-source-references)
  (if po-reference-alist
      (po-show-source-context
       (assoc
	(completing-read (_"Which source context? ") po-reference-alist nil t)
	po-reference-alist))
    (error (_"No resolved source references"))))

;;; Program sources strings though tags table.

;;; C mode.

(defun po-find-c-string (keywords)
  "Find the next C string, excluding those marked by any of KEYWORDS.
Returns (START . END) for the found string, or (nil . nil) if none found."
  (let (start end)
    (while (and (not start)
		(re-search-forward "\\([\"']\\|/\\*\\)" nil t))
      (cond ((= (preceding-char) ?*)
	     ;; Disregard comments.
	     (search-forward "*/"))

	    ((= (preceding-char) ?\')
	     ;; Disregard character constants.
	     (forward-char (if (= (following-char) ?\\) 3 2)))

	    ((save-excursion
	       (beginning-of-line)
	       (looking-at "^# *\\(include\\|line\\)"))
	     ;; Disregard lines being #include or #line directives.
	     (end-of-line))

	    ;; Else, find the end of the string.
	    (t (setq start (1- (point)))
	       (while (not (= (following-char) ?\"))
		 (skip-chars-forward "^\"\\\\")
		 (if (= (following-char) ?\\) (forward-char 2)))
	       (forward-char 1)
	       (setq end (point))

	       ;; Check before string for keyword and opening parenthesis.
	       (goto-char start)
	       (skip-chars-backward " \n\t")
	       (if (= (preceding-char) ?\()
		   (progn
		     (backward-char 1)
		     (skip-chars-backward " \n\t")
		     (let ((end-keyword (point)))
		       (skip-chars-backward "_A-Za-z0-9")
		       (if (member (list (buffer-substring (point) end-keyword))
				   keywords)

			   ;; Disregard already marked strings.
			   (progn
			     (goto-char end)
			     (setq start nil)
			     (setq end nil)))))))))

    ;; Return the found string, if any.
    (cons start end)))

(defun po-mark-c-string (start end keyword)
  "Mark the C string, from START to END, with KEYWORD.
Return the adjusted value for END."
  (goto-char end)
  (insert ")")
  (goto-char start)
  (insert keyword)
  (if (not (string-equal keyword "_"))
      (progn (insert " ") (setq end (1+ end))))
  (insert "(")
  (+ end 2 (length keyword)))

;;; Emacs LISP mode.

(defun po-find-emacs-lisp-string (keywords)
  "Find the next Emacs LISP string, excluding those marked by any of KEYWORDS.
Returns (START . END) for the found string, or (nil . nil) if none found."
  (let (start end)
    (while (and (not start)
		(re-search-forward "[;\"?]" nil t))

      (cond ((= (preceding-char) ?\;)
	     ;; Disregard comments.
	     (search-forward "\n"))

	    ((= (preceding-char) ?\?)
	     ;; Disregard character constants.
	     (forward-char (if (= (following-char) ?\\) 2 1)))

	    ;; Else, find the end of the string.
	    (t (setq start (1- (point)))
	       (while (not (= (following-char) ?\"))
		 (skip-chars-forward "^\"\\\\")
		 (if (= (following-char) ?\\) (forward-char 2)))
	       (forward-char 1)
	       (setq end (point))

	       ;; Check before string for keyword and opening parenthesis.
	       (goto-char start)
	       (skip-chars-backward " \n\t")
	       (let ((end-keyword (point)))
		 (skip-chars-backward "-_A-Za-z0-9")
		 (if (and (= (preceding-char) ?\()
			  (member (list (buffer-substring (point) end-keyword))
				  keywords))

		     ;; Disregard already marked strings.
		     (progn
		       (goto-char end)
		       (setq start nil)
		       (setq end nil)))))))

    ;; Return the found string, if any.
    (cons start end)))

(defun po-mark-emacs-lisp-string (start end keyword)
  "Mark the Emacs LISP string, from START to END, with KEYWORD.
Return the adjusted value for END."
  (goto-char end)
  (insert ")")
  (goto-char start)
  (insert "(" keyword)
  (if (not (string-equal keyword "_"))
      (progn (insert " ") (setq end (1+ end))))
  (+ end 2 (length keyword)))

;;; Processing generic to all programming modes.

(eval-and-compile
  (autoload 'visit-tags-table-buffer "etags"))

(defun po-tags-search (restart)
  (interactive "P")
  "Find an unmarked translatable string through all files in tags table.
Disregard some simple strings which are most probably non-translatable.
With prefix argument, restart search at first file."

  ;; Take care of restarting the search if necessary.
  (if restart (setq po-next-file-list nil))

  ;; Loop doing things until an interesting string is found.
  (let ((keywords po-keywords)
	find-string found buffer start
	(end po-string-end))
    (while (not found)

      ;; Reinitialize the source file list if necessary.
      (if (not po-next-file-list)
	  (progn
	    (setq po-next-file-list
		  (save-excursion
		    (visit-tags-table-buffer)
		    (copy-sequence (tags-table-files))))
	    (or po-next-file-list (error (_"No files to process")))
	    (setq end nil)))

      ;; Try finding a string after resuming the search position.
      (message (_"Scanning %s...") (car po-next-file-list))
      (save-excursion
	(setq buffer (find-file-noselect (car po-next-file-list)))
	(set-buffer buffer)
	(goto-char (or end (point-min)))

	(cond ((string= mode-name "C")
	       (let ((pair (po-find-c-string keywords)))
		 (setq start (car pair))
		 (setq end (cdr pair))))
	      ((string= mode-name "Emacs-Lisp")
	       (let ((pair (po-find-emacs-lisp-string keywords)))
		 (setq start (car pair))
		 (setq end (cdr pair))))
	      (t (message (_"Unknown source mode for PO mode, skipping..."))
		 (setq start nil)
		 (setq end nil))))

      ;; Advance to next file if no string was found.
      (if (not start)
	  (progn
	    (setq po-next-file-list (cdr po-next-file-list))
	    (if (not po-next-file-list) (error (_"All files processed")))
	    (setq end nil))

	;; Push the string just found string into the work buffer for study.
	(po-extract-unquoted buffer start end)
	(save-excursion
	  (set-buffer po-work-buffer)
	  (goto-char (point-min))

	  ;; Do not disregard if at least three letters in a row.
	  (if (re-search-forward "[A-Za-z][A-Za-z][A-Za-z]" nil t)
	      (setq found t)

	    ;; Disregard if two letters, and more punctuations than letters.
	    (if (re-search-forward "[A-Za-z][A-Za-z]" nil t)
		(let ((total (buffer-size)))
		  (goto-char (point-min))
		  (while (re-search-forward "[A-Za-z]+" nil t)
		    (replace-match "" t t))
		  (if (< (* 2 (buffer-size)) total)
		      (setq found t))))

	    ;; Disregard if single letters or no letters at all.
	    ))))

    ;; Ensure the string is being displayed.

    (if (one-window-p t) (split-window) (other-window 1))
    (switch-to-buffer buffer)
    (goto-char start)
    (recenter 1)
    (if (pos-visible-in-window-p end)
	(goto-char end)
      (goto-char end)
      (recenter -1))
    (other-window 1)

    ;; Save the string for later commands.
    (message (_"Scanning %s...done") (car po-next-file-list))
    (setq po-string-start start)
    (setq po-string-end end)))

(defun po-mark-found-string (keyword)
  "Mark last found string in program sources as translatable, using KEYWORD."
  (let ((buffer (find-file-noselect (car po-next-file-list)))
	(start po-string-start)
	(end po-string-end)
	line string)

    ;; Mark string in program sources.
    (setq string (po-extract-unquoted buffer start end))
    (save-excursion
      (set-buffer buffer)
      (setq line (count-lines (point-min) start))
      (setq end (cond ((string= mode-name "C")
		       (po-mark-c-string start end keyword))
		      ((string= mode-name "Emacs-Lisp")
		       (po-mark-emacs-lisp-string start end keyword))
		      (t (error (_"Cannot mark in unknown source mode"))))))
    (setq po-string-end end)

    ;; Add PO file entry.
    (let ((buffer-read-only nil))
      (goto-char (point-max))
      (insert "\n"
	      (format "#: %s:%d\n" (car po-next-file-list) line)
	      (po-eval-requoted string "msgid" nil)
	      "msgstr \"\"\n")
      (previous-line 1)
      (setq po-untranslated-counter (1+ po-untranslated-counter))
      (po-update-mode-line-string))))

(defun po-mark-translatable ()
  (interactive)
  "Mark last found string in program sources as translatable, using `_'."
  (if (and po-string-start po-string-end)
      (progn
	(po-mark-found-string "_")
	(setq po-string-start nil))
    (error (_"No such string"))))

(defun po-select-mark-and-mark (arg)
  (interactive "P")
  "Mark last found string in program sources as translatable, ask for keywoard,
using completion.  With prefix argument, just ask the name of a preferred
keyword for subsequent commands, also added to possible completions."
  (if arg
      (let ((keyword (list (read-from-minibuffer (_"Keyword: ")))))
	(setq po-keywords (cons keyword (delete keyword po-keywords))))
    (if (and po-string-start po-string-end)
	(let* ((default (car (car po-keywords)))
	       (keyword (completing-read (format (_"Mark with keywoard? [%s] ")
						 default)
					 po-keywords nil t )))
	  (if (string-equal keyword "") (setq keyword default))
	  (po-mark-found-string keyword)
	  (setq po-string-start nil))
      (error (_"No such string")))))

;;; Miscellaneous features.

(defun po-help ()
  "Provide an help window for PO mode."
  (interactive)
  (po-check-lock)
  (save-window-excursion
    (switch-to-buffer po-work-buffer)
    (erase-buffer)
    (insert po-help-display-string)
    (delete-other-windows)
    (goto-char (point-min))
    (message (_"Type any character to continue"))
    (read-char))
  (bury-buffer po-work-buffer))

(defun po-undo ()
  "Undo the last change to the PO file."
  (interactive)
  (let ((buffer-read-only nil))
    (undo))
  (po-compute-counters nil))

(defun po-statistics ()
  "Say how many entries in each category, and the current position."
  (interactive)
  (po-compute-counters t))

(defun po-validate ()
  "Use `msgfmt' for validating the current PO file contents."
  (interactive)

  ;; If modifications were done already, change the last revision date.
  (if (buffer-modified-p)
      (po-replace-revision-date))

  ;; This `let' is for protecting the previous value of compile-command.
  (let ((compile-command (concat po-msgfmt-program
				 " --statistics -c -v -o /dev/null "
				 buffer-file-name)))
    (compile compile-command)))

(defun po-confirm-and-quit ()
  "Confirm if quit should be attempted and then, do it.
This is a failsafe.  Confirmation is asked if only the real quit would not."
  (interactive)
  (if (or (buffer-modified-p)
	  (> po-untranslated-counter 0)
	  (> po-fuzzy-counter 0)
	  (> po-obsolete-counter 0)
	  (y-or-n-p (_"Really quit editing this PO file? ")))
      (po-quit))
  (message ""))

(defun po-quit ()
  "Save the PO file and kill buffer.  However, offer validation if
appropriate and ask confirmation if untranslated strings remain."
  (interactive)
  (let ((quit t))

    ;; Offer validation of newly modified entries.
    (if (and (buffer-modified-p)
	     (not (y-or-n-p (_"File was modified; skip validation step? "))))
	(progn
	  (message "")
	  (setq quit nil)
	  (po-validate)))

    ;; Offer to work on untranslated entries.
    (if (and quit
	     (or (> po-untranslated-counter 0)
		 (> po-fuzzy-counter 0)
		 (> po-obsolete-counter 0))
	     (not (y-or-n-p (_"Unprocessed entries remain; quit anyway? "))))
	(progn
	  (setq quit nil)
	  (po-auto-select-entry)))

    ;; Clear message area.
    (message "")

    ;; Or else, kill buffers and quit for true.
    (if quit
	(progn
	  (if (buffer-modified-p)
	      (po-replace-revision-date))
	  (save-buffer)
	  ;; kill-buffer-hook is used if it exists: avoid overkilling! :-)
	  (or (boundp 'kill-buffer-hook) (kill-buffer po-work-buffer))
	  (kill-buffer (current-buffer))))))

;;; po-mode.el ends here
