/*
 * dfile.c - SunOS (Solaris 1.x and 2.x) file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.11 99/10/22 08:23:39 abe Exp $";
#endif


#include "lsof.h"


/*
 * Local structures
 */

struct hsfile {
	struct sfile *s;		/* the Sfile table address */
	struct hsfile *next;		/* the next hash bucket entry */
};


/*
 * Local static variables
 */

static struct hsfile *HbyCd =		/* hash by clone buckets */
	(struct hsfile *)NULL;
static int HbyCdCt = 0;			/* HbyCd entry count */
static struct hsfile *HbyFdi =		/* hash by file buckets */
	(struct hsfile *)NULL;
static int HbyFdiCt = 0;		/* HbyFdi entry count */
static struct hsfile *HbyFsd =		/* hash by file system buckets */
	(struct hsfile *)NULL;
static int HbyFsdCt = 0;		/* HbyFsd entry count */
static struct hsfile *HbyNm =		/* hash by name buckets */
	(struct hsfile *)NULL;
static int HbyNmCt = 0;			/* HbyNm entry count */


/*
 * Local definitions
 */

#define	SFCDHASH	1024		/* Sfile hash by clone device */
#define	SFDIHASH	4094		/* Sfile hash by (device,inode) number
					 * pair bucket count (power of 2!) */
#define	SFFSHASH	128		/* Sfile hash by file system device
					 * number bucket count (power of 2!) */
#define SFHASHDEVINO(maj, min, ino, mod) ((int)(((int)((((int)(maj+1))*((int)((min+1))))+ino)*31415)&(mod-1)))
					/* hash for Sfile by major device,
					 * minor device, and inode, modulo m
					 * (m must be a power of 2) */
#define	SFNMHASH	4096		/* Sfile hash by name bucket count
					   (power of 2!) */


#if	defined(solaris) && solaris<20500
/*
 * get_max_fd() - get maximum file descriptor plus one
 */

int
get_max_fd()
{
	struct rlimit r;

	if (getrlimit(RLIMIT_NOFILE, &r))
	    return(-1);
	return(r.rlim_cur);
}
#endif	/* defined(solaris) && solaris<20500 */


/*
 * hashSfile() - hash Sfile entries for use in is_file_named() searches
 */

void
hashSfile()
{
	int cmaj, hvc, i;
	static int hs = 0;
	struct sfile *s;
	struct hsfile *sh, *sn;
/*
 * Do nothing if there are no file search arguments cached or if the
 * hashes have already been constructed.
 */
	if (!Sfile || hs)
	    return;
/*
 * Preset the clone major device for Solaris of SunOS.
 */

#if	defined(solaris)
	if (HaveCloneMaj) {
	    cmaj = CloneMaj;
	    hvc = 1;
	} else
	    hvc = 0;
#else	/* !defined(SOLARIS) */
	cmaj = CLONEMAJ;
	hvc = 1;
#endif	/* defined(solaris) */

/*
 * Allocate hash buckets by clone device, (device,inode), file system device,
 * and file name.
 */
	if (hvc) {
	    if (!(HbyCd = (struct hsfile *)calloc((MALLOC_S)SFCDHASH,
						  sizeof(struct hsfile))))
	    {
		(void) fprintf(stderr,
		    "%s: can't allocate space for %d clone hash buckets\n",
		    Pn, SFCDHASH);
		Exit(1);
	    }
	}
	if (!(HbyFdi = (struct hsfile *)calloc((MALLOC_S)SFDIHASH,
					       sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d (dev,ino) hash buckets\n",
		Pn, SFDIHASH);
	    Exit(1);
	}
	if (!(HbyFsd = (struct hsfile *)calloc((MALLOC_S)SFFSHASH,
					       sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d file sys hash buckets\n",
		Pn, SFFSHASH);
	    Exit(1);
	}
	if (!(HbyNm = (struct hsfile *)calloc((MALLOC_S)SFNMHASH,
					      sizeof(struct hsfile))))
	{
	    (void) fprintf(stderr,
		"%s: can't allocate space for %d name hash buckets\n",
		Pn, SFNMHASH);
	    Exit(1);
	}
	hs++;
/*
 * Scan the Sfile chain, building file, file system, and file name hash
 * bucket chains.
 */
	for (s = Sfile; s; s = s->next) {
	    for (i = 0; i < 3; i++) {
		if (i == 0) {
		    if (!s->aname)
			continue;
		    sh = &HbyNm[hashbyname(s->aname, SFNMHASH)];
		    HbyNmCt++;
		} else if (i == 1) {
		    if (s->type) {
			sh = &HbyFdi[SFHASHDEVINO(major(s->dev),
						  minor(s->dev),
						  s->i,
						  SFDIHASH)];
			HbyFdiCt++;
		    } else {
			sh = &HbyFsd[SFHASHDEVINO(major(s->dev),
						  minor(s->dev),
						  0,
						  SFFSHASH)];
			HbyFsdCt++;
		    }
		} else {
		    if (!hvc || (major(s->dev) != cmaj))
			continue;
		    sh = &HbyCd[SFHASHDEVINO(0, minor(s->dev), 0, SFCDHASH)];
		    HbyCdCt++;
		}
		if (!sh->s) {
		    sh->s = s;
		    sh->next = (struct hsfile *)NULL;
		    continue;
		} else {
		    if (!(sn = (struct hsfile *)malloc(
				(MALLOC_S)sizeof(struct hsfile))))
		    {
			(void) fprintf(stderr,
			    "%s: can't allocate hsfile bucket for: %s\n",
			    Pn, s->aname);
			Exit(1);
		    }
		    sn->s = s;
		    sn->next = sh->next;
		    sh->next = sn;
		}
	    }
	}
}


/*
 * is_file_named() - is this file named?
 */

int
is_file_named(p, nt, vt, ps)
	char *p;			/* path name; NULL = search by device
					 * and inode (from *Lf) */
	int nt;				/* node type -- e.g., N_* */
	enum vtype vt;			/* vnode type */
	int ps;				/* print status: 0 = don't copy name
					 * to Namech */
{
	int f = 0;
	struct sfile *s;
	struct hsfile *sh;
/*
 * Check for a path name match, as requested.
 */
	if (p && HbyNmCt) {
	    for (sh = &HbyNm[hashbyname(p, SFNMHASH)]; sh; sh = sh->next) {
		if ((s = sh->s) && strcmp(p, s->aname) == 0) {
		    f = 2;
		    break;
		}
	    }
	}
/*
 * Check for a Sun clone file.
 */
	if (!f && HbyCdCt && nt == N_STREAM && Lf->dev_def) {
	    for (sh = &HbyCd[SFHASHDEVINO(0, major(Lf->dev), 0, SFCDHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s) && (major(Lf->dev) == minor(s->dev))) {
		    f = 1;
		    break;
		}
	    }
	}
/*
 * Check for a regular file.
 */
	if (!f && HbyFdiCt && Lf->dev_def
	&&  (Lf->inp_ty == 1 || Lf->inp_ty == 3))
	{
	    for (sh = &HbyFdi[SFHASHDEVINO(major(Lf->dev),
					   minor(Lf->dev),
					   Lf->inode,
					   SFDIHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s) && (Lf->dev == s->dev)
		&&  ((ino_t)Lf->inode == s->i)) {
		    f = 1;
		    break;
		}
	    }
	}
/*
 * Check for a file system.
 */
	if (!f && HbyFsdCt && Lf->dev_def) {
	    for (sh = &HbyFsd[SFHASHDEVINO(major(Lf->dev), minor(Lf->dev), 0,
					   SFFSHASH)];
		 sh;
		 sh = sh->next)
	    {
		if ((s = sh->s) && Lf->dev == s->dev) {
		    if (!(vt == VCHR && s->mode != S_IFCHR)) {
			f = 1;
			break;
		    }
		}
	    }
	}
/*
 * Convert the name if a match occurred.
 */
	if (f) {
	    if (f == 2) {
		if (ps)
		    (void) strcpy(Namech, p);
	    } else {
		if (ps && s->type) {

		/*
		 * If the search argument isn't a file system, propagate it
		 * to Namech[]; otherwise, let printname() compose the name.
		 */
		    (void) strcpy(Namech, s->name);
		    if (s->devnm)
			(void) sprintf(endnm(), " (%s)", s->devnm);
		}
	    }
	    s->f = 1;
	    return(1);
	}
	return(0);
}


#if	defined(solaris)
/*
 * print_dev() - print device
 */

char *
print_dev(lf)
	struct lfile *lf;		/* file whose device is to be printed */
{
	static char buf[128];
/*
 * Avoid the Solaris major() and minor() functions from makedev(3C) to get
 * printable major/minor numbers.
 *
 * We would like to use the L_MAXMAJ definition from <sys/sysmacros.h> all
 * the time, but it's not always correct in all versions of Solaris.
 */
	(void) sprintf(buf, "%d,%d", (int)((lf->dev >> L_BITSMINOR) &

#if	solaris>=20501
	    L_MAXMAJ
#else	/* solaris<20501 */
	    0x3fff
#endif	/* solaris>=20501 */

	    ), (int)(lf->dev & L_MAXMIN));
	return(buf);
}
#else	/* !defined(solaris) */


/*
 * print_ino() - print inode
 */

char *
print_ino(lf)
	struct lfile *lf;		/* file whose device is to be printed */
{
	static char buf[128];

	if ((long)lf->inode < 0L)
	    (void) sprintf(buf, " %ld", (long)lf->inode);
	else
	    (void) sprintf(buf, " %lu", (unsigned long)lf->inode);
	return(buf);
}
#endif	/* defined(slaris) */


/*
 * process_file() - process file
 */

void
process_file(fp)
	KA_T fp;		/* kernel file structure address */
{
	struct file f;
	int flag;

#if	defined(FILEPTR)
	FILEPTR = &f;
#endif	/* defined(FILEPTR) */

	if (kread(fp, (char *)&f, sizeof(f))) {
	    (void) sprintf(Namech, "can't read file struct from %s",
		print_kptr(fp, (char *)NULL));
	    enter_nm(Namech);
	    return;
	}
	Lf->off = (SZOFFTYPE)f.f_offset;

	if (f.f_count) {

	/*
	 * Construct access code.
	 */
	    if ((flag = (f.f_flag & (FREAD | FWRITE))) == FREAD)
		Lf->access = 'r';
	    else if (flag == FWRITE)
		Lf->access = 'w';
	    else if (flag == (FREAD | FWRITE))
		Lf->access = 'u';

#if	defined(HASFSTRUCT)
	/*
	 * Save file structure values.
	 */
	    if (Fsv & FSV_CT) {
		Lf->fct = (long)f.f_count;
		Lf->fsv |= FSV_CT;
	    }
	    if (Fsv & FSV_FA) {
		Lf->fsa = fp;
		Lf->fsv |= FSV_FA;
	    }
	    if (Fsv & FSV_FG) {
		Lf->ffg = (long)f.f_flag;
		Lf->fsv |= FSV_FG;
	    }
	    if (Fsv & FSV_NI) {

# if	defined(solaris)
		Lf->fna = (KA_T)f.f_vnode;
# else	/* !defined(solaris) */
		Lf->fna = (KA_T)f.f_data;
# endif	/* defined(solaris) */

		Lf->fsv |= FSV_NI;
	    }
#endif	/* defined(HASFSTRUCT) */


#if	defined(solaris)
	/*
	 * Solaris file structures contain a vnode pointer.  Process it.
	 */
	    process_node((KA_T)f.f_vnode);
	    return;
#else	/* !solaris */

	/*
	 * Process non-Solaris file structure by its type.
	 */
	    switch (f.f_type) {

	    case DTYPE_VNODE:
		if (!Selinet)
		    process_node((KA_T)f.f_data);
		return;
	    case DTYPE_SOCKET:
		process_socket((KA_T)f.f_data);
		return;
	    default:
		if (f.f_type != 0 || f.f_ops) {
		    (void) sprintf(Namech, "%s file struct, ty=%#x, op=%#x",
			print_kptr(fp, (char *)NULL), f.f_type, f.f_ops);
		    enter_nm(Namech);
		    return;
		}
	    }
#endif	/* solaris */

	}
	enter_nm("no more information"); }


#if	defined(HASIPv6)
/*
 * gethostbyname2() -- an RFC2133-compatible get-host-by-name-two function
 *                     to get AF_INET and AF_INET6 addresses from host names,
 *                     using the RFC2553-compatible getipnodebyname() function
 */

extern struct hostent *
gethostbyname2(nm, prot)
	const char *nm; 		/* host name */
	int prot;			/* protocol -- AF_INET or AF_INET6 */
{
	int err;
	static struct hostent *hep = (struct hostent *)NULL;

	if (hep)
	    (void) freehostent(hep);
	return((hep = getipnodebyname(nm, prot, 0, &err)));
}
#endif	/* defined(HASIPv6) */
