/*
 * Copyright (c) 1999 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software and design ideas developed by the Java 
 *    Apache Project (http://java.apache.org/)."
 *
 * 4. The names "Cocoon", "Cocoon Servlet" and "Java Apache Project" must 
 *    not be used to endorse or promote products derived from this software 
 *    without prior written permission.
 *
 * 5. Products derived from this software may not be called "Cocoon"
 *    nor may "Cocoon" and "Java Apache Project" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software and design ideas developed by the Java 
 *    Apache Project (http://java.apache.org/)."
 *           
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *           
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Project. For more information
 * on the Java Apache Project please see <http://java.apache.org/>.
 */
 
package org.apache.cocoon.framework;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * This class watches over the changes of indicated resources.
 *
 * @author <a href="mailto:stefano@apache.org">Stefano Mazzocchi</a>
 * @version $Revision: 1.1 $ $Date: 1999/09/13 00:24:43 $
 */

public class Monitor {
    
    private Hashtable table;
    
    class Container {
        public Object resource;
        public long timestamp;
        
        public Container(Object resource, long timestamp) {
            this.resource = resource;
            this.timestamp = timestamp;
        }
    }
    
    public Monitor(int capacity) {
        this.table = new Hashtable(capacity);
    }
    
    /**
     * Tells the monitor to watch the given resource, timestamps it
     * and associate it to the given key.
     */
    public void watch(Object key, Object resource) {
        this.table.put(key, new Container(resource, timestamp(resource)));
    }
    
    /**
     * Queries the monitor for changes. For maximum reliability, this
     * method is synchronous, but less reliable for faster asynchronous
     * versions could be implemented. Returns true if the resource
     * associated to the given key has changed since the last call
     * to watch.
     *
     * WARNING: due to a stupid bug in "FileURLConnection", the
     * class that implements the "file:" protocol for the java.net.URL
     * framework, the getLastModified() method always returns 0.
     * For this reason, the use of the File resource is strongly
     * suggested over the "file:" type URL.
     * NOTE: this may not be (and should not be) the case in other
     * virtual machine implementations or if we rewrite the URL
     * handler ourselves (which I don't care to do at this point).
     */
    public boolean hasChanged(Object key) {
        Object o = this.table.get(key);
        if (o != null) {
            Container c = (Container) o;
            return c.timestamp != timestamp(c.resource);
        } else {
            return true;
        }
    }
    
    /**
     * Create a timestamp indicating the last modified time
     * of the given resource.
     */
    private long timestamp(Object resource) {
        long timestamp;
        if (resource instanceof File) {
            timestamp = ((File) resource).lastModified();
        } else if (resource instanceof URL) {
            try {
                timestamp = ((URL) resource).openConnection().getLastModified();
            } catch (IOException e) {
                timestamp = 0;
            }
        } else {
            throw new IllegalArgumentException("Resource not monitorizable.");
        }
        return timestamp;
    }
}