/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "fltk_LcdWindow.h"

#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Image.H>
#include <FL/Fl_Window.H>
#include <FL/math.h>

#include "CPU_MT.h"				// CPUStopper
#include "CPU_REG.h"			// Screen::InvalidateAll, Hardware::PenEvent, etc.
#include "fltk_MainWindow.h"
#include "JPEGGlue.h"			// JpegToDib
#include "PreferenceMgr.h"
#include "Skins.h"
#include "SocketMessaging.h"	// CSocket::IdleAll


const float REDRAW_FREQ = .1;	// = 10x / second

Bool gNeedWindowReset;
Bool gNeedWindowInvalidate;

// Add some more constants, naming the same way FLTK
// names some key constants.  These values may be
// specific to certain operating systems; I don't know.
// They were determined empirically by pressing keys
// and seeing what showed up in handle().

const int FL_F1			= 65470;
const int FL_F2			= 65471;
const int FL_F3			= 65472;
const int FL_F4			= 65473;
const int FL_F9			= 65478;
const int FL_F10		= 65479;
const int FL_PageUp		= 65365;
const int FL_PageDown	= 65366;
const int FL_LeftArrow	= 65361;
const int FL_RightArrow	= 65363;
const int FL_UpArrow	= 65362;
const int FL_DownArrow	= 65364;


// -----------------------------------------------------------------------------
// file static methods
// -----------------------------------------------------------------------------

struct RGBQUAD
{
	uae_u8	rgbBlue;
	uae_u8	rgbGreen;
	uae_u8	rgbRed;
	uae_u8	rgbReserved;
};


struct BITMAPINFOHEADER
{
	uae_u32		biSize;
	uae_s32		biWidth;
	uae_s32		biHeight;
	uae_u16		biPlanes;
	uae_u16		biBitCount;
	uae_u32		biCompression;
	uae_u32		biSizeImage;
	uae_s32		biXPelsPerMeter;
	uae_s32		biYPelsPerMeter;
	uae_u32		biClrUsed;
	uae_u32		biClrImportant;
};


struct BITMAPINFO
{
	BITMAPINFOHEADER	bmiHeader;
	RGBQUAD				bmiColors[1];
};

// -----------------------------------------------------------------------------
// Fl_Image helper class
// -----------------------------------------------------------------------------

// An Fl_Image subclass which owns dynamicly allocated image data
class Dynamic_Image : public Fl_Image
{
public:
	Dynamic_Image(const void* data, int width, int height, int rowBytes) :
		Fl_Image ((const uchar*) data, width, height, 3, rowBytes),
		_width (width),
		_height (height)
	{
	}

	virtual ~Dynamic_Image(void)
	{
		Platform::DisposeMemory (_data);
	}

	virtual void measure (int& w, int& h);

private:
	char* _data;
	int _width;
	int _height;
};

void Dynamic_Image::measure (int& w, int& h)
{
	w = _width;
	h = _height;
}


// -----------------------------------------------------------------------------
// handle periodic tasks:
// redraw the LCD every REDRAW_FREQ seconds
// check for pending data on the debug socket

static void handlePeriodic (void* data)
{
	LCD_window* lcd = (LCD_window*) data;

	if (lcd != NULL)
	{
		Document* doc = lcd->getDocument();
		if (doc != NULL)
		{
			CSocket::IdleAll ();
		}

		if (gNeedWindowReset)
		{
			gNeedWindowReset = false;
			gNeedWindowInvalidate = false;
			lcd->ResetSkin();
		}

		if (gNeedWindowInvalidate)
		{
			gNeedWindowInvalidate = false;
			lcd->damageLcdArea ();
		}

		if (lcd->isOn())
		{
			lcd->damageLcdArea ();
		}
	}

	Fl::add_timeout (REDRAW_FREQ, handlePeriodic, lcd);
}

/***********************************************************************
 *
 * FUNCTION:    PrvPrefsChanged
 *
 * DESCRIPTION: If any preference relevent to our display has changed,
 *				set a flag remembering it.  We'll respond to that flag
 *				later in some idle-time function.  We don't respond
 *				to it now in case several preferences change all at
 *				once; we don't want to respond to each one individually.
 *
 * PARAMETERS:  key - the key of the preference that changed.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

static void PrvPrefsChanged (PrefKeyType key)
{
	if ((strcmp (key, kPrefKeyScale) == 0) || (strcmp (key, kPrefKeySkins) == 0))
	{
		gNeedWindowReset = true;
	}
	else if (strcmp (key, kPrefKeyBackgroundColor) == 0)
	{
		gNeedWindowInvalidate = true;
	}
}


// -----------------------------------------------------------------------------
// constructor / destructor
// -----------------------------------------------------------------------------

LCD_window::LCD_window (MainWindow* parent) :
	Fl_Double_Window (0, 0, 600, 600, ""),
	_case (NULL),
	_buffer (NULL),
	_penTracking (false),
	_currentButton (kElement_None),
	_parent (parent),
	_doc (NULL),
	_lcdOn (true)
{
	resetToDefault();
	initialize();
	
	Fl::add_timeout (REDRAW_FREQ, handlePeriodic, this);

	gPrefs->AddNotification (PrvPrefsChanged, kPrefKeySkins);
	gPrefs->AddNotification (PrvPrefsChanged, kPrefKeyScale);
	gPrefs->AddNotification (PrvPrefsChanged, kPrefKeyBackgroundColor);
}

LCD_window::~LCD_window (void)
{
	// No longer need notification.

	gPrefs->RemoveNotification (PrvPrefsChanged);

	delete _buffer;
}

// -----------------------------------------------------------------------------
// public methods
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// Fl_Double_Window overrides
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// Handle the GUI event. This routine translates the FLTK GUI events into
// the corresponding emulator actions.

int LCD_window::handle (int event)
{
	PointType			where = {Fl::event_x(), Fl::event_y()};
	SkinElementType		what = ::SkinTestPoint (where);

	switch (event)
	{
		case FL_PUSH:
		case FL_DRAG:
			if (Fl::event_button () == 3)
			{
				_parent->popupMenu ();
				return 1;
			}

			if (_doc != NULL)
			{
				if (_penTracking)
				{
					PointType	whereLCD = ::SkinWindowToTouchscreen (where);
					_doc->pen (true, whereLCD.x, whereLCD.y);
				}
				else if (_currentButton != kElement_None)
				{
					// Nothing to do...
				}
				else
				{
					if (what == kElement_Touchscreen)
					{
						PointType		whereLCD = ::SkinWindowToTouchscreen (where);
						_doc->pen (true, whereLCD.x, whereLCD.y);
						_penTracking = true;
					}
					else
					{
						_doc->button (true, what);
						_currentButton = what;
					}
				}
			}

			return 1;

		case FL_RELEASE:
			if (_doc != NULL)
			{
				if (_penTracking)
				{
					PointType	whereLCD = ::SkinWindowToTouchscreen (where);
					_doc->pen (false, whereLCD.x, whereLCD.y);
					_penTracking = false;
				}
				else if (_currentButton != kElement_None)
				{
					_doc->button (false, _currentButton);
					_currentButton = kElement_None;
				}
			}
			return 1;

		case FL_FOCUS:
			// always accept the focus
			return 1;

		case FL_KEYBOARD:
			if (Fl::event_state (FL_ALT|FL_META))
			{
				// reserved for shortcuts
				return 0;
			}

			if (_doc != NULL)
			{
				int c = Fl::event_key();

				// handle printable characters
				if (strlen (Fl::event_text()) > 0)
				{
					_doc->putkey (Fl::event_text()[0]);
					return 1;
				}

				// handle all other characters
				SkinElementType	whichButton = kElement_None;
				int				whichKey	= 0;
				switch (c)
				{
					case FL_Enter:
					case FL_KP_Enter:
						whichKey = '\n';
						break;

					case FL_F1:
						whichButton = kElement_App1Button;
						break;

					case FL_F2:
						whichButton = kElement_App2Button;
						break;

					case FL_F3:
						whichButton = kElement_App3Button;
						break;

					case FL_F4:
						whichButton = kElement_App4Button;
						break;

					case FL_F9:
						whichButton = kElement_PowerButton;
						break;

					case FL_F10:
						_parent->popupMenu();
						return 1;

					case FL_PageUp:
						whichButton=kElement_UpButton;
						break;

					case FL_PageDown:
						whichButton=kElement_DownButton;
						break;

					case FL_LeftArrow:
						whichKey = leftArrowChr;
						break;

					case FL_RightArrow:
						whichKey = rightArrowChr;
						break;

					case FL_UpArrow:
						whichKey = upArrowChr;
						break;

					case FL_DownArrow:
						whichKey = downArrowChr;
						break;

					default:
						return 0;
				}

				if (whichButton != kElement_None)
				{
					_doc->button (true, whichButton);
					_doc->button (false, whichButton);
				}

				if (whichKey != 0)
				{
					_doc->putkey (whichKey);
				}
			}

			return 1;

		default:
			return Fl_Double_Window::handle (event);
	}
}

// -----------------------------------------------------------------------------
// draw the case and the inner LCD screen
void LCD_window::draw()
{
	// redraw everything if anything other than just the lcd damage bit is on
	bool drawAll = (damage() & ~LCD_DAMAGE_BIT) != 0;

	fl_color (FL_WHITE);
	if (drawAll)
	{
		_case->draw (0,0);
	}

	// only draw the LCD if we have a document & the LCD is on
	if (_lcdOn && (_doc != NULL))
	{
		CPUStopper stopper (kStopOnADime);
		if (!stopper.Stopped())
			return;

		if (updateBuffer (drawAll))
		{
			RectangleType lcdRect = ::SkinGetElementRect (kElement_LCD);

			int x = lcdRect.topLeft.x;
			int y = lcdRect.topLeft.y;
			int w = lcdRect.extent.x;
			int h = lcdRect.extent.y;

			fl_draw_image (_buffer, x, y, w, h);
		}
	}
}

// -----------------------------------------------------------------------------
// turn the lcd off or on
void LCD_window::setOn( bool on )
{
	if (!_lcdOn && on)
	{
		// turn on lcd
		initialize();
		_lcdOn = true;
		Fl::add_timeout (REDRAW_FREQ, handlePeriodic, this);
		redraw();
	}
	else if (_lcdOn && !on)
	{
		// turn lcd off
		_lcdOn = false;

		delete _buffer;
		_buffer = NULL;

		redraw();
	}
}

// -----------------------------------------------------------------------------
// reset configurable parameters to default values.  called during construction.

void LCD_window::resetToDefault()
{
	_currentButton = kElement_None;
	_penTracking = false;

	this->ResetSkin ();
}

// -----------------------------------------------------------------------------
// initialize data & presentation.  called during construction and when
// the display is toggled on.

void LCD_window::initialize()
{
	delete _buffer;
	_buffer = NULL;

	RectangleType bounds = ::SkinGetElementRect(kElement_LCD);
	bounds = ::SkinScaleUp (bounds);

	_buffer = new unsigned char [bounds.extent.x * bounds.extent.y * 3];
}

// -----------------------------------------------------------------------------
// load the graphics for the case.

void LCD_window::ResetSkin (void)
{
	// Initialize the Skin engine with the preference settings.
	::SkinSetSkin ();

	delete _case;
	_case = NULL;

	const void*	jpegData;
	long		jpegLength;
	::SkinGetImageAsJPEG (jpegData, jpegLength);

	const void*	bmpData;
	int			width;
	int			height;
	int			rowBytes;
	this->ConvertJpegToBmp (jpegData, jpegLength, bmpData, width, height, rowBytes);

	_case = new Dynamic_Image (bmpData, width, height, rowBytes);

	::SkinDoneWithImage (jpegData);

	// resize this window to the size of the surrounding case
	int w, h;
	_case->measure (w, h);

	int was_visible = _parent->visible();

	if (was_visible)
		_parent->hide ();

	this->initialize();
	this->size (w, h);
	_parent->size (w, h);

	if (was_visible)
		_parent->show();
}

// -----------------------------------------------------------------------------
// invalidate the entire lcd area.

void LCD_window::damageLcdArea ()
{
	RectangleType bounds = ::SkinGetElementRect (kElement_LCD);
	this->damage (LCD_DAMAGE_BIT,
				  bounds.topLeft.x, bounds.topLeft.y,
				  bounds.extent.x, bounds.extent.y);
}


// -----------------------------------------------------------------------------
// private methods
// -----------------------------------------------------------------------------

static int PrvGetNumColors (const BITMAPINFO* dib)
{
	int	numColors;

	UInt16	biBitCount = dib->bmiHeader.biBitCount;
	UInt32	biClrUsed = dib->bmiHeader.biClrUsed;

	if (biBitCount <= 8)
		numColors = (1 << biBitCount);
	else
		numColors = 0;  // No palette needed for 24 BPP DIB

	if (biClrUsed > 0)
		numColors = biClrUsed;  // Use biClrUsed

	return numColors;
}


static uae_u8* PrvGetBits (const BITMAPINFO* dib)
{
	UInt32	biSize = dib->bmiHeader.biSize;

	return ((uae_u8*) dib) + biSize + ::PrvGetNumColors (dib) * sizeof(RGBQUAD);
}


static long PrvGetRowBytes (int biWidth, int biBitCount)
{
	return ((biWidth * biBitCount + 31) & ~31) / 8;
}


void LCD_window::ConvertJpegToBmp (const void* jpegData,
								   long jpegLength,
								   const void*& bmpData,
								   int& width,
								   int& height,
								   int& rowBytes)
{
	BITMAPINFO*	bmp = (BITMAPINFO*) ::JpegToDib (jpegData, jpegLength);

#if !BYTESWAP
	Byteswap (bmp->bmiHeader.biWidth);
	Byteswap (bmp->bmiHeader.biHeight);
	Byteswap (bmp->bmiHeader.biBitCount);
	Byteswap (bmp->bmiHeader.biSize);
	Byteswap (bmp->bmiHeader.biClrUsed);
#endif

	width = bmp->bmiHeader.biWidth;
	height = bmp->bmiHeader.biHeight;
	rowBytes = ::PrvGetRowBytes (width, bmp->bmiHeader.biBitCount);

	bmpData = (const void*) Platform::AllocateMemory (height * rowBytes);

	char* src = (char*) ::PrvGetBits (bmp);
	char* dest = (char*) bmpData;
	for (int yy = 0; yy < height; ++yy)
	{
		memcpy (dest + yy * rowBytes, src + (height - yy - 1) * rowBytes, rowBytes);
	}

	Platform::DisposeMemory (bmp);
}

// -----------------------------------------------------------------------------
// update the offscreen lcd buffer from the emulator's screen RAM. Returns true
// if the offscreen buffer data has been changed, false otherwise. Note that
// you must own the buffer lock to call this method.

int LCD_window::updateBuffer (int always)
{
	// Get information about the screen.  Return immediately if nothing's changed.
	Screen::BufferInfo	info;
	if (!Screen::GetBits (info, !always))
		return false;

	RectangleType		bounds = ::SkinGetElementRect (kElement_LCD);
	RectangleType		scaledBounds = ::SkinScaleUp (bounds);

	int scale = scaledBounds.extent.x / bounds.extent.x;

	// Get the bounds of the area we'll be blitting from.
	int x = info.visibleBounds.left;
	int y = info.firstLine;
	int width = info.visibleBounds.right - x;
	int height = info.lastLine - y;

	// There is a moment during boot-up when the LCD width is 1024.
	// We don't really want that as the width (we only allow for 160
	// pixels in our window), so clip to 160.
	if (width > bounds.extent.x)
	{
		width = bounds.extent.x;
	}

	if (info.myBuffer == NULL)
	{
		// during a reset, screen buffer may be empty
		return false;
	}

	// buffer access macros
#define SRCPTR(r, c) ((unsigned char*) info.myBuffer) + (r) * info.myRowBytes + (c)
#define DESTPTR(r, c, n) _buffer + \
		(((r) * scale + (n)) * bounds.extent.x * 3) + \
		((c) * scale * 3)
	
// CLUT access macros
#define CLUT_R(x) info.myCLUT[(x)].fRed
#define CLUT_G(x) info.myCLUT[(x)].fGreen
#define CLUT_B(x) info.myCLUT[(x)].fBlue

	// convert 8bpp Screen RAM data to 24bpp buffer data
	switch (scale)
	{
		default:
		case 1:
		{
			for (int ly = y; ly < y + height; ++ly)
			{
				for (int lx = x; lx < x + width; ++lx)
				{
					unsigned char pix = *(SRCPTR(ly, lx));
					unsigned char* p = DESTPTR(ly, lx, 0);

					*p++ = CLUT_R(pix);
					*p++ = CLUT_G(pix);
					*p++ = CLUT_B(pix);
				}
			}
			break;
		}
		case 2:
		{
			for (int ly = y; ly < y + height; ++ly)
			{
				for (int lx = x; lx < x + width; ++lx)
				{
					unsigned char pix = *(SRCPTR(ly, lx));
					unsigned char* p1 = DESTPTR(ly, lx, 0);
					unsigned char* p2 = DESTPTR(ly, lx, 1);

					*p1++ = *p2++ = CLUT_R(pix);
					*p1++ = *p2++ = CLUT_G(pix);
					*p1++ = *p2++ = CLUT_B(pix);

					*p1++ = *p2++ = CLUT_R(pix);
					*p1++ = *p2++ = CLUT_G(pix);
					*p1++ = *p2++ = CLUT_B(pix);
				}
			}
			break;
		}
	}

	return true;
}
