/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef __DOCUMENT_H__
#define __DOCUMENT_H__

#include "EmulatorTypes.h"		// Configuration
#include "Platform.h"			// Platform::AutoSaveState
#include "Skins.h"				// SkinElementType


// -----------------------------------------------------------------------------
// class Document
//
// The class encapsulates all of the DATA pertinent to an emulation session.
// Some of this data is persistent, and this class provides methods for saving
// that data.
// In the future, it would be nice if each of the data member of this class
// (in particular the CPU) would be thread safe, so that I could have several
// Documents open at once.
// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
// local types
// -----------------------------------------------------------------------------

typedef void (*PropertyChangedHandler)( int property, void *param );

struct PropertyChangedHandlerSpec
{
	PropertyChangedHandler	func;
	void*					param;
};

class Document
{
public:

	// -----------------------------------------------------------------------------
	// public constants
	// -----------------------------------------------------------------------------

	enum
	{
		unspecifiedChanged = 0,
		documentOpening,
		documentClosing,
		cpuStatusChanged
	};

	// -----------------------------------------------------------------------------
	// constructor / destructor
	// -----------------------------------------------------------------------------

	Document( const Configuration& cfg );
	virtual ~Document( void );

	// -----------------------------------------------------------------------------
	// public methods
	// -----------------------------------------------------------------------------

	ErrCode open( FileReference& f );

	void emulatorStart( void );
	void emulatorStop( void );
	void emulatorReset( void );
	void loadApplication( const FileReference& appFile );
	void exportDatabase (const DatabaseInfo&, const FileReference&);
	void saveMemory();
	void saveMemoryAs( FileReference& dest );
	void saveScreen( FileReference& dest );

	void pollDebugPort();

	void hotsync();
	void pen( bool down, int x, int y );
	void button( bool down, SkinElementType button );
	void putkey( unsigned char key );

	FileReference getCurrentRAMRef() { return _currentRAMRef; }

	void addPropertyChangedHandler( PropertyChangedHandler callback, void *param );
	void removePropertyChangedHandler( PropertyChangedHandler callback );

	bool LcdIsOn();

	bool initializedOk() { return _cpu != NULL; }
	Configuration& getCurrentConfiguration() { return _currentConfiguration; }

	void gremlinsNew();
	void gremlinsStep();
	void gremlinsResume();
	void gremlinsStop();

	CPU* getCPU() { return _cpu; } // Called by Platform::GetCPU.

private:

	// -----------------------------------------------------------------------------
	// private methods
	// -----------------------------------------------------------------------------

	void propertyChanged( int property = unspecifiedChanged );

	FileReference findPairedROMFile( FileHandle& docFile );
	void saveRAMImage( const FileReference&, Bool forAutoSave );

	// -----------------------------------------------------------------------------
	// private friends
	// -----------------------------------------------------------------------------

	friend void _hotSyncUp( void *data );
	friend void Platform::AutoSaveState (const FileReference&);	// saveRAMImage

	// -----------------------------------------------------------------------------
	// private data members
	// -----------------------------------------------------------------------------

	CPU*					_cpu;
	Configuration			_currentConfiguration;
	FileReference			_currentRAMRef;
	list<PropertyChangedHandlerSpec>	_handlers;
};


#endif /* __DOCUMENT_H__ */
