/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "Skins.h"

#include "PreferenceMgr.h"		// Preference
#include "SkinData.h"			// kPil500015Image, etc.

#include <algorithm>			// find()

struct ButtonBounds
{
	SkinElementType	fButton;
	RectangleType	fBounds;
};
typedef vector<ButtonBounds>	ButtonBoundsList;
 
struct Skinfo
{
	Skinfo () :
		fName (),
		fImageName1x (),
		fImageName2x (),
		fDevices (),
		fButtons ()
		{}

	SkinName			fName;
	string				fImageName1x;
	string				fImageName2x;
	RGBType				fBackgroundColor;
	RGBType				fHighlightColor;
	DeviceList			fDevices;
	ButtonBoundsList	fButtons;
};


typedef vector<Skinfo>	SkinList;

static Skinfo			gCurrentSkin;
static ScaleType		gCurrentScale;

static void				PrvBuildSkinList	(SkinList&);
static void				PrvGetSkins			(DeviceType type, SkinList& results);
static Bool				PrvGetNamedSkin		(DeviceType type, const SkinName& name, Skinfo& result);
static void				PrvGetDefaultSkin	(DeviceType type, Skinfo& skin);
static Bool				PrvValidSkin		(DeviceType type, const SkinName& skinName);
static void				PrvSetSkin			(const Skinfo&, ScaleType scale);
static SkinElementType	PrvTestPoint		(const PointType&, int outset);
static Bool				PrvPtInRect			(const PointType&, const AbsRectType&);


struct ResourceMap
{
	const char*	name;
	const void*	image;
	size_t		length;
};

ResourceMap kSkinResources[] =
{
	{ "Pil500016",	kPil500016Image,	kPil500016ImageSize },
	{ "Pil500032",	kPil500032Image,	kPil500032ImageSize },
	{ "PilPro16",	kPilPro16Image,		kPilPro16ImageSize },
	{ "PilPro32",	kPilPro32Image,		kPilPro32ImageSize },
	{ "Palm316",	kPalm316Image,		kPalm316ImageSize },
	{ "Palm332",	kPalm332Image,		kPalm332ImageSize },
	{ "Palm3e16",	kPalm3e16Image,		kPalm3e16ImageSize },
	{ "Palm3e32",	kPalm3e32Image,		kPalm3e32ImageSize },
	{ "Palm3x16",	kPalm3x16Image,		kPalm3x16ImageSize },
	{ "Palm3x32",	kPalm3x32Image,		kPalm3x32ImageSize },
	{ "IBMWP16",	kIBMWP16Image,		kIBMWP16ImageSize },
	{ "IBMWP32",	kIBMWP32Image,		kIBMWP32ImageSize },
	{ "Palm516",	kPalm516Image,		kPalm516ImageSize },
	{ "Palm532",	kPalm532Image,		kPalm532ImageSize },
	{ "IBMWPc316",	kIBMWPc316Image,	kIBMWPc316ImageSize },
	{ "IBMWPc332",	kIBMWPc332Image,	kIBMWPc332ImageSize },
	{ "Palm716",	kPalm716Image,		kPalm716ImageSize },
	{ "Palm732",	kPalm732Image,		kPalm732ImageSize },
	{ "Sym150016",	kSym150016Image,	kSym150016ImageSize },
	{ "Sym150032",	kSym150032Image,	kSym150032ImageSize },
	{ "Sym174016",	kSym174016Image,	kSym174016ImageSize },
	{ "Sym174032",	kSym174032Image,	kSym174032ImageSize },
	{ "ColorDev16",	kEnglishAustinSmallImage,	kEnglishAustinSmallImageSize },
	{ "ColorDev32",	kEnglishAustinLargeImage,	kEnglishAustinLargeImageSize }
};

const long kNumSkinResources = countof(kSkinResources);


/***********************************************************************
 *
 * FUNCTION:    SkinGetSkinName
 *
 * DESCRIPTION: Get the name of the user-chosen skin for the given
 *				device type.
 *
 * PARAMETERS:  type - the device type for which we need the name
 *					of the skin to use.
 *
 * RETURNED:    The skin name.  At the very least, this will be the
 *				name of some default skin if the user hasn't made a
 *				choice or if the chosen skin is invalid.
 *
 ***********************************************************************/

SkinName SkinGetSkinName (DeviceType type)
{
	// Get the chosen skin for this device.

	Preference<SkinNameList>	pref(kPrefKeySkins);
	SkinNameList				skinNames = *pref;
	SkinName					name = skinNames[type];

	// If the name is empty or invalid, chose a default skin.

	if (!::PrvValidSkin (type, name))
	{
		Skinfo	skin;
		::PrvGetDefaultSkin (type, skin);
		name = skin.fName;
	}

	return name;
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetSkinNames
 *
 * DESCRIPTION:	Get the list of names of available skins for the given
 *				device.
 *
 * PARAMETERS:	type - the device for which the list of skins should
 *					be returned.  If kDeviceUnspecified, return all
 *					skins for all devices.
 *
 *				results - receives the list of skin names.  Any skin
 *					names are *added* to this list; the list is not
 *					cleared out first.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void SkinGetSkinNames (DeviceType type, SkinNameList& results)
{
	SkinList	skins;
	::PrvGetSkins (type, skins);

	SkinList::iterator	iter = skins.begin ();
	while (iter != skins.end ())
	{
		results.push_back (iter->fName);

		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:	SkinSetSkin
 *
 * DESCRIPTION:	Establish the skin to use, based on the current settings.
 *				All other funtions in this module will then work within
 *				the context of the specified skin.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void SkinSetSkin (void)
{
	Preference<Configuration>	cfgPref (kPrefKeyLastConfiguration);
	Preference<SkinNameList>	skinsPref (kPrefKeySkins);
	Preference<ScaleType>		scalePref (kPrefKeyScale);

	SkinNameList	skins = *skinsPref;

	::SkinSetSkin (cfgPref->fDeviceType, *scalePref, skins[cfgPref->fDeviceType]);
}


void SkinSetSkin (DeviceType type, ScaleType scale, const SkinName& name)
{
	Skinfo		skin;

	if (!::PrvGetNamedSkin (type, name, skin))
	{
		::PrvGetDefaultSkin (type, skin);
	}

	::PrvSetSkin (skin, scale);
}


void PrvSetSkin (const Skinfo& skin, ScaleType scale)
{
	gCurrentSkin = skin;
	gCurrentScale = scale;
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetResourceName
 *
 * DESCRIPTION:	Return the name of the resource containing the image
 *				for the default skin.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Pointer to the resource name of the image to use.
 *
 ***********************************************************************/

const char*		SkinGetResourceName	(void)
{
	return ::SkinGetResourceName (gCurrentScale);
}


const char*		SkinGetResourceName	(ScaleType scale)
{
	if (scale == 1)
		return gCurrentSkin.fImageName1x.c_str();

	return gCurrentSkin.fImageName2x.c_str();
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetImageAsJPEG
 *
 * DESCRIPTION:	Return the name of the resource containing the image
 *				for the default skin.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Pointer to the resource name of the image to use.
 *
 ***********************************************************************/

void		SkinGetImageAsJPEG	(const void*& data, long& length)
{
	::SkinGetImageAsJPEG (gCurrentScale, data, length);
}


void		SkinGetImageAsJPEG	(ScaleType scale, const void*& data, long& length)
{
	data = NULL;
	length = 0;

	string	name (::SkinGetResourceName (scale));

	for (int ii = 0; ii < kNumSkinResources; ++ii)
	{
		if (name == kSkinResources[ii].name)
		{
			data = kSkinResources[ii].image;
			length = (long) kSkinResources[ii].length;
			break;
		}
	}
}


/***********************************************************************
 *
 * FUNCTION:	SkinDoneWithImage
 *
 * DESCRIPTION:	.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void		SkinDoneWithImage	(const void*)
{
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetBackgroundColor
 *
 * DESCRIPTION:	Return the default background color for the current
 *				skin.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The default background color.
 *
 ***********************************************************************/

RGBType		SkinGetBackgroundColor	(void)
{
	return gCurrentSkin.fBackgroundColor;
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetHighlightColor
 *
 * DESCRIPTION:	Return the default highlight color for the current
 *				skin.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	The default highlight color.
 *
 ***********************************************************************/

RGBType		SkinGetHighlightColor	(void)
{
	return gCurrentSkin.fHighlightColor;
}


/***********************************************************************
 *
 * FUNCTION:	SkinTestPoint
 *
 * DESCRIPTION:	Tests the given point to see what skin element it's over.
 *
 * PARAMETERS:	pt - location in the window to test.
 *
 * RETURNED:	If the point is within an element, returns the id for
 *				that element.  If an element was just missed, returns
 *				kElement_None.  Otherwise, returns kElement_Frame.
 *
 ***********************************************************************/

SkinElementType	SkinTestPoint		(const PointType& pt)
{	
	// See if we hit an element.  PrvTestPoint will return either the
	// element hit or kElement_Frame if none were hit.  If an element
	// was hit, return it.

	SkinElementType	result = ::PrvTestPoint (pt, 0);

	if (result != kElement_Frame)
		return result;

	// Test again, this time allowing for some slop around the
	// elements.  If an element was hit this time, then we hit the
	// small "dead area" we allow around the elements.  In that case,
	// we want to return kElement_None.  Otherwise, if no element was
	// hit, signal that the frame was hit.

	result = ::PrvTestPoint (pt, 5);
	if (result != kElement_Frame)
		return kElement_None;

	return kElement_Frame;
}


/***********************************************************************
 *
 * FUNCTION:	SkinWindowToTouchscreen
 *
 * DESCRIPTION:	Convert a point from window coordinates to LCD
 *				coordinates.
 *
 * PARAMETERS:	pt - point in window coordinates to convert.
 *
 * RETURNED:	The point in LCD coordinates (where the topleft corner
 *				of the LCD is 0,0 and the scale is 1x).
 *
 ***********************************************************************/

PointType		SkinWindowToTouchscreen		(const PointType& pt)
{
	PointType	result = pt;

	RectangleType	r = ::SkinGetElementRect (kElement_Touchscreen);

	result.x -= r.topLeft.x;
	result.y -= r.topLeft.y;

	if (result.x < 0)
		result.x = 0;
	
	if (result.y < 0)
		result.y = 0;
	
	if (result.x >= r.extent.x)
		result.x = r.extent.x - 1;
	
	if (result.y >= r.extent.y)
		result.y = r.extent.y - 1;

	result = ::SkinScaleDown (result);

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SkinLCDToWindow
 *
 * DESCRIPTION:	Convert a point from LCD coordinates to window
 *				coordinates.
 *
 * PARAMETERS:	pt - point in LCD coordinates to convert.
 *
 * RETURNED:	The point in window coordinates (where the topleft
 *				corner of the window is 0,0 and the scale is the
 *				scale chosen by the user).
 *
 ***********************************************************************/

PointType		SkinTouchscreenToWindow		(const PointType& lcdPt)
{
	PointType	result = lcdPt;

	RectangleType	r = ::SkinGetElementRect (kElement_Touchscreen);

	result.x += r.topLeft.x;
	result.y += r.topLeft.y;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetElementRect
 *
 * DESCRIPTION:	Get the bounds of the given element.
 *
 * PARAMETERS:	which - element we're querying.
 *
 * RETURNED:	Rectangle of the element's bounds.
 *
 ***********************************************************************/

RectangleType	SkinGetElementRect	(SkinElementType which)
{
	ButtonBoundsList::iterator	iter = gCurrentSkin.fButtons.begin ();
	while (iter != gCurrentSkin.fButtons.end ())
	{
		if (iter->fButton == which)
		{
			RectangleType	result = iter->fBounds;
			result = ::SkinScaleUp (result);
			return result;
		}

		++iter;
	}

	assert (false);

	RectangleType	dummy;
	return dummy;
}


/***********************************************************************
 *
 * FUNCTION:	SkinGetAbsElementRect
 *
 * DESCRIPTION:	Get the bounds of the given element.
 *
 * PARAMETERS:	which - element we're querying.
 *
 * RETURNED:	Rectangle of the element's bounds.
 *
 ***********************************************************************/

AbsRectType		SkinGetAbsElementRect	(SkinElementType which)
{
	AbsRectType		result;
	RectangleType	r = ::SkinGetElementRect (which);

	result.left		= r.topLeft.x;
	result.top		= r.topLeft.y;
	result.right	= r.topLeft.x + r.extent.x;
	result.bottom	= r.topLeft.y + r.extent.y;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SkinScaleDown
 *
 * DESCRIPTION:	Convert a point from 1x or 2x to just 1x.
 *
 * PARAMETERS:	pt - point to change.
 *
 * RETURNED:	Normalized point.
 *
 ***********************************************************************/

PointType	SkinScaleDown	(const PointType& pt)
{
	PointType	result =
	{
		pt.x / gCurrentScale,
		pt.y / gCurrentScale
	};

#if 0
	// Note: I used to do it this way, but this runs afoul of a code
	// generation bug when using egcs 1.1.2 for x86 -- the y coordinate
	// gets trashed.

	result.x			/= gCurrentScale;
	result.y			/= gCurrentScale;
#endif

	return result;
}


RectangleType	SkinScaleDown	(const RectangleType& r)
{
	RectangleType	result = r;

	result.topLeft.x	/= gCurrentScale;
	result.topLeft.y	/= gCurrentScale;
	result.extent.x		/= gCurrentScale;
	result.extent.y		/= gCurrentScale;

	return result;
}


AbsRectType	SkinScaleDown	(const AbsRectType& r)
{
	AbsRectType	result = r;

	result.left			/= gCurrentScale;
	result.top			/= gCurrentScale;
	result.right		/= gCurrentScale;
	result.bottom		/= gCurrentScale;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	SkinScaleUp
 *
 * DESCRIPTION:	Convert a point to 1x or 2x, depending on the scaling
 *				factor.
 *
 * PARAMETERS:	pt - point to change.
 *
 * RETURNED:	Denormalized point.
 *
 ***********************************************************************/

PointType	SkinScaleUp	(const PointType& pt)
{
	PointType	result =
	{
		pt.x * gCurrentScale,
		pt.y * gCurrentScale
	};

	return result;
}


RectangleType	SkinScaleUp	(const RectangleType& r)
{
	RectangleType	result = r;

	result.topLeft.x	*= gCurrentScale;
	result.topLeft.y	*= gCurrentScale;
	result.extent.x		*= gCurrentScale;
	result.extent.y		*= gCurrentScale;

	return result;
}


AbsRectType	SkinScaleUp	(const AbsRectType& r)
{
	AbsRectType	result = r;

	result.left			*= gCurrentScale;
	result.top			*= gCurrentScale;
	result.right		*= gCurrentScale;
	result.bottom		*= gCurrentScale;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	PrvBuildSkinList
 *
 * DESCRIPTION:	Create the full list of skins known to the emulator.
 *				This list of skins includes both the built-in ones and
 *				any found on disk.
 *
 * PARAMETERS:	skins - receives the list of skins.  The list of skins
 *					is *added* to this collection; it is not cleared
 *					out first.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ButtonBounds kPilotButtons [] =
{
	{ kElement_PowerButton,		{ {   5, 287 }, {  14,  22 } } },
	{ kElement_UpButton,		{ { 104, 285 }, {  26,  12 } } },
	{ kElement_DownButton,		{ { 104, 307 }, {  26,  12 } } },
	{ kElement_App1Button,		{ {  30, 285 }, {  25,  25 } } },
	{ kElement_App2Button,		{ {  68, 285 }, {  25,  25 } } },
	{ kElement_App3Button,		{ { 139, 285 }, {  25,  25 } } },
	{ kElement_App4Button,		{ { 178, 285 }, {  25,  25 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  37,  39 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  37,  39 }, { 160, 160 } } }
};

ButtonBounds kPalmPilotButtons [] =
{
	{ kElement_PowerButton,		{ {   5, 287 }, {  14,  22 } } },
	{ kElement_UpButton,		{ { 104, 285 }, {  26,  12 } } },
	{ kElement_DownButton,		{ { 104, 307 }, {  26,  12 } } },
	{ kElement_App1Button,		{ {  30, 285 }, {  25,  25 } } },
	{ kElement_App2Button,		{ {  68, 285 }, {  25,  25 } } },
	{ kElement_App3Button,		{ { 139, 285 }, {  25,  25 } } },
	{ kElement_App4Button,		{ { 178, 285 }, {  25,  25 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  37,  39 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  37,  39 }, { 160, 160 } } }
};

ButtonBounds kPalmIIIButtons [] =
{
	{ kElement_PowerButton,		{ {  10, 295 }, {  16,  24 } } },
	{ kElement_UpButton,		{ { 110, 292 }, {  20,  21 } } },
	{ kElement_DownButton,		{ { 110, 313 }, {  20,  21 } } },
	{ kElement_App1Button,		{ {  37, 295 }, {  23,  25 } } },
	{ kElement_App2Button,		{ {  76, 297 }, {  23,  25 } } },
	{ kElement_App3Button,		{ { 141, 297 }, {  23,  25 } } },
	{ kElement_App4Button,		{ { 180, 294 }, {  23,  25 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  39,  44 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  39,  44 }, { 160, 160 } } }
};

ButtonBounds kPalmIIIxButtons [] =
{
	{ kElement_PowerButton,		{ {  10, 295 }, {  16,  24 } } },
	{ kElement_UpButton,		{ { 110, 292 }, {  20,  21 } } },
	{ kElement_DownButton,		{ { 110, 313 }, {  20,  21 } } },
	{ kElement_App1Button,		{ {  37, 295 }, {  23,  25 } } },
	{ kElement_App2Button,		{ {  76, 297 }, {  23,  25 } } },
	{ kElement_App3Button,		{ { 141, 297 }, {  23,  25 } } },
	{ kElement_App4Button,		{ { 180, 294 }, {  23,  25 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  39,  44 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  39,  44 }, { 160, 160 } } }
};

ButtonBounds kPalmVButtons [] =
{
	{ kElement_PowerButton,		{ { 173,   0 }, {  27,  11 } } },
	{ kElement_UpButton,		{ { 105, 283 }, {  25,  21 } } },
	{ kElement_DownButton,		{ { 105, 304 }, {  25,  21 } } },
	{ kElement_App1Button,		{ {  28, 285 }, {  22,  18 } } },
	{ kElement_App2Button,		{ {  68, 292 }, {  22,  18 } } },
	{ kElement_App3Button,		{ { 143, 292 }, {  22,  18 } } },
	{ kElement_App4Button,		{ { 183, 285 }, {  22,  18 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {  30,   0 }, {  22,  11 } } },
	{ kElement_Touchscreen,		{ {  36,  38 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  36,  38 }, { 160, 160 } } }
};

ButtonBounds kPalmVIIButtons [] =
{
	{ kElement_PowerButton,		{ {  10, 333 }, {  16,  24 } } },
	{ kElement_UpButton,		{ { 110, 329 }, {  20,  21 } } },
	{ kElement_DownButton,		{ { 110, 350 }, {  20,  21 } } },
	{ kElement_App1Button,		{ {  37, 332 }, {  23,  25 } } },
	{ kElement_App2Button,		{ {  76, 334 }, {  23,  25 } } },
	{ kElement_App3Button,		{ { 141, 334 }, {  23,  25 } } },
	{ kElement_App4Button,		{ { 180, 331 }, {  23,  25 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  39,  72 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  39,  72 }, { 160, 160 } } }
};

ButtonBounds kColorDevButtons [] =
{
	{ kElement_PowerButton,		{ {   1, 274 }, {  16,  24 } } },
	{ kElement_UpButton,		{ {  96, 272 }, {  32,  16 } } },
	{ kElement_DownButton,		{ {  96, 293 }, {  32,  16 } } },
	{ kElement_App1Button,		{ {  23, 270 }, {  32,  32 } } },
	{ kElement_App2Button,		{ {  60, 270 }, {  32,  32 } } },
	{ kElement_App3Button,		{ { 131, 270 }, {  32,  32 } } },
	{ kElement_App4Button,		{ { 168, 270 }, {  32,  32 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  32,  32 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  32,  32 }, { 160, 160 } } }
};

ButtonBounds kSymbol1740Buttons [] =
{
	{ kElement_PowerButton,		{ {  45, 460 }, {  16,  24 } } },
	{ kElement_PowerButton,		{ { 215, 460 }, {  16,  24 } } },
	{ kElement_UpButton,		{ {  50,  46 }, {  24,  38 } } },
	{ kElement_UpButton,		{ { 200,  46 }, {  24,  38 } } },
	{ kElement_DownButton,		{ {  37,  82 }, {  20,  28 } } },
	{ kElement_DownButton,		{ { 218,  82 }, {  20,  28 } } },
	{ kElement_App1Button,		{ {  67, 416 }, {  20,  30 } } },
	{ kElement_App2Button,		{ { 107, 427 }, {  20,  30 } } },
	{ kElement_App3Button,		{ { 147, 427 }, {  20,  30 } } },
	{ kElement_App4Button,		{ { 187, 416 }, {  20,  30 } } },
	{ kElement_CradleButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Antenna,			{ {   0,   0 }, {   0,   0 } } },
	{ kElement_ContrastButton,	{ {   0,   0 }, {   0,   0 } } },
	{ kElement_Touchscreen,		{ {  60, 149 }, { 160, 220 } } },
	{ kElement_LCD,				{ {  60, 149 }, { 160, 160 } } }
};

static RGBType	kDefaultBackgroundColor	(0x7B, 0x8C, 0x5A);
static RGBType	kDefaultHighlightColor	(0x64, 0xF0, 0xDC);
static RGBType	kAlmostWhiteColor		(0xF7, 0xF7, 0xF7);
static RGBType	kWhiteColor				(0xFF, 0xFF, 0xFF);
static RGBType	kBlackColor				(0x00, 0x00, 0x00);

void PrvBuildSkinList (SkinList& skins)
{
// -*- NEW DEVICE -*-

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "Pil500016";
		skin.fImageName2x = "Pil500032";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePilot1000);
		skin.fDevices.push_back (kDevicePilot5000);
		for (size_t ii = 0; ii < countof (kPilotButtons); ++ii)
			skin.fButtons.push_back (kPilotButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "PilPro16";
		skin.fImageName2x = "PilPro32";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmPilotPersonal);
		skin.fDevices.push_back (kDevicePalmPilotProfessional);
		skin.fDevices.push_back (kDevicePalmPilotWithUpgrade);
		for (size_t ii = 0; ii < countof (kPalmPilotButtons); ++ii)
			skin.fButtons.push_back (kPalmPilotButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "Palm316";
		skin.fImageName2x = "Palm332";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmIII);
		for (size_t ii = 0; ii < countof (kPalmIIIButtons); ++ii)
			skin.fButtons.push_back (kPalmIIIButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "Palm3x16";
		skin.fImageName2x = "Palm3x32";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmIIIx);
		for (size_t ii = 0; ii < countof (kPalmIIIxButtons); ++ii)
			skin.fButtons.push_back (kPalmIIIxButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Japanese";
		skin.fImageName1x = "IBMWP16";
		skin.fImageName2x = "IBMWP32";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmIIIx);
		for (size_t ii = 0; ii < countof (kPalmIIIxButtons); ++ii)
			skin.fButtons.push_back (kPalmIIIxButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "Palm516";
		skin.fImageName2x = "Palm532";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmV);
		for (size_t ii = 0; ii < countof (kPalmVButtons); ++ii)
			skin.fButtons.push_back (kPalmVButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Japanese";
		skin.fImageName1x = "IBMWPc316";
		skin.fImageName2x = "IBMWPc332";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmV);
		for (size_t ii = 0; ii < countof (kPalmVButtons); ++ii)
			skin.fButtons.push_back (kPalmVButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "Palm716";
		skin.fImageName2x = "Palm732";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmVII);
		skin.fDevices.push_back (kDevicePalmVIIEZ);
		for (size_t ii = 0; ii < countof (kPalmVIIButtons); ++ii)
			skin.fButtons.push_back (kPalmVIIButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Default";
		skin.fImageName1x = "ColorDev16";
		skin.fImageName2x = "ColorDev32";
		skin.fBackgroundColor = kAlmostWhiteColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDeviceAustin);
		for (size_t ii = 0; ii < countof (kColorDevButtons); ++ii)
			skin.fButtons.push_back (kColorDevButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Japanese";
		skin.fImageName1x = "IBMWP16";
		skin.fImageName2x = "IBMWP32";
		skin.fBackgroundColor = kAlmostWhiteColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDeviceAustin);
		for (size_t ii = 0; ii < countof (kPalmIIIxButtons); ++ii)
			skin.fButtons.push_back (kPalmIIIxButtons[ii]);
		skins.push_back (skin);
	}

	{
		Skinfo	skin;
		skin.fName = "Symbol 1740";
		skin.fImageName1x = "Sym174016";
		skin.fImageName2x = "Sym174032";
		skin.fBackgroundColor = kDefaultBackgroundColor;
		skin.fHighlightColor = kDefaultHighlightColor;
		skin.fDevices.push_back (kDevicePalmIII);
		for (size_t ii = 0; ii < countof (kSymbol1740Buttons); ++ii)
			skin.fButtons.push_back (kSymbol1740Buttons[ii]);
		skins.push_back (skin);
	}
}


/***********************************************************************
 *
 * FUNCTION:	PrvGetSkins
 *
 * DESCRIPTION:	Get the list of available skins for the given device.
 *
 * PARAMETERS:	type - the device for which the list of skins should
 *					be returned.  If kDeviceUnspecified, return all
 *					skins for all devices.
 *
 *				results - receives the list of skins.  Any skins are
 *					*added* to this list; the list is not cleared out
 *					first.
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void PrvGetSkins (DeviceType type, SkinList& results)
{
	SkinList	fullSkinList;
	::PrvBuildSkinList (fullSkinList);

	SkinList::iterator	iter = fullSkinList.begin ();
	while (iter != fullSkinList.end ())
	{
		if (type == kDeviceUnspecified ||
			find (iter->fDevices.begin (), iter->fDevices.end (), type) != iter->fDevices.end ())
		{
			results.push_back (*iter);
		}

		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:    PrvGetNamedSkin
 *
 * DESCRIPTION: Find the Skinfo for the given device and that has the
 *				given name.
 *
 * PARAMETERS:  type - the device whose skin we're looking for.
 *
 *				name - the name of the skin to find.
 *
 *				result - reference to the Skinfo in which to place the
 *					found skin information, if any.
 *
 * RETURNED:    True if the skin could be found, false othewise.
 *
 ***********************************************************************/

Bool PrvGetNamedSkin (DeviceType type, const SkinName& name, Skinfo& result)
{
	SkinList	skins;
	::PrvGetSkins (type, skins);

	SkinList::iterator	iter = skins.begin();
	assert (iter != skins.end ());

	Skinfo		skin = *iter;	// Default to the built-in skin

	while (iter != skins.end ())
	{
		if (iter->fName == name)
		{
			result = *iter;
			return true;
		}

		++iter;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:    PrvGetDefaultSkin
 *
 * DESCRIPTION: Return default skin information for the given device.
 *				This function returns information in the case that
 *				a skin with a desired name could not be found.
 *
 * PARAMETERS:  type - the device whose default skin information we want.
 *
 *				skin - reference to the Skinfo in which to place the
 *					default skin information.
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void PrvGetDefaultSkin (DeviceType type, Skinfo& skin)
{
	SkinList	skins;
	::PrvGetSkins (type, skins);

	skin = skins[0];
}


/***********************************************************************
 *
 * FUNCTION:	PrvValidSkin
 *
 * DESCRIPTION:	Returns whether the given device has a skin with the
 *				given name.
 *
 * PARAMETERS:	type - the device type.
 *
 *				skinName - the skin name.
 *
 * RETURNED:	True if the given device has a skin with the given name.
 *				False otherwise.
 *
 ***********************************************************************/

Bool PrvValidSkin (DeviceType type, const SkinName& skinName)
{
	SkinNameList	skins;
	::SkinGetSkinNames (type, skins);

	SkinNameList::iterator	iter = skins.begin ();
	while (iter != skins.end ())
	{
		if (*iter == skinName)
		{
			return true;
		}

		++iter;
	}

	return false;
}


/***********************************************************************
 *
 * FUNCTION:	PrvTestPoint
 *
 * DESCRIPTION:	Test the given point against all of the skin elements.
 *				An optional outset value can be provided which is
 *				used to modify the element bounds before they are
 *				tested.
 *
 * PARAMETERS:	pt - window location to test.
 *
 *				outset - outset value to apply to the bounds of all
 *					the skin elements.
 *
 * RETURNED:	If one contains the given point, return that skin
 *				element.  Otherwise, return kElement_Frame.
 *
 ***********************************************************************/

SkinElementType	PrvTestPoint (const PointType& pt, int outset)
{
	ButtonBoundsList::iterator	iter = gCurrentSkin.fButtons.begin ();
	while (iter != gCurrentSkin.fButtons.end ())
	{
		RectangleType	bounds = iter->fBounds;
		bounds = ::SkinScaleUp (bounds);

		AbsRectType		r;

		r.left = bounds.topLeft.x - outset;
		r.top = bounds.topLeft.y - outset;
		r.right = bounds.topLeft.x + bounds.extent.x + outset;
		r.bottom = bounds.topLeft.y + bounds.extent.y + outset;

		if (::PrvPtInRect (pt, r))
			return iter->fButton;

		++iter;
	}

	return kElement_Frame;
}


/***********************************************************************
 *
 * FUNCTION:	PrvPtInRect
 *
 * DESCRIPTION:	Test if the given point is within the given rectangle.
 *
 * PARAMETERS:	pt - point to test.
 *
 *				r - rectangle to test against.
 *
 * RETURNED:	True if the point is within the rectangle, using a
 *				half-open interval.  False otherwise.
 *
 ***********************************************************************/

Bool	PrvPtInRect		(const PointType& pt, const AbsRectType& r)
{
	return
		pt.x >= r.left &&
		pt.y >= r.top &&
		pt.x < r.right &&
		pt.y < r.bottom;
}
