/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#ifndef _SESSIONFILE_H_
#define _SESSIONFILE_H_

#include "ChunkFile.h"			// ChunkFile
#include "EmulatorTypes.h"		// Configuration, DeviceType
#include "Platform.h"			// Platform


class SessionFile
{
	public:
		typedef uae_u32 BugFixes;
		enum BugFix
		{
			kBugByteswappedStructs = 1	// Fixes bug where SED1375RegsType, HwrDBallEZType, HwrDBallType were
										// stored in host-endian format.
		};

	public:
								SessionFile				(ChunkFile& f);
								~SessionFile			(void);

		// ---------- Reading ----------

		Bool					ReadDeviceType			(DeviceType&);

		Bool					ReadROMFileReference	(FileReference&);

		Bool					ReadDBallRegs			(regstruct&);
		Bool					ReadHwrDBallType		(HwrDBallType&);
		Bool					ReadHwrDBallEZType		(HwrDBallEZType&);
		Bool					ReadDBallState			(Chunk& chunk) { return fFile.ReadChunk (kDBState, chunk); }
		Bool					ReadDBallEZState		(Chunk& chunk) { return fFile.ReadChunk (kDBEZState, chunk); }

		Bool					ReadSED1375RegsType		(SED1375RegsType&);
		Bool					ReadSED1375Image		(void*);
		Bool					ReadSED1375MetaImage	(void*);
		Bool					ReadSED1375Palette		(uae_u16 [256]);

		Bool					ReadPLDRegsType			(HwrJerryPLDType&);

		Bool					ReadGremlinInfo			(Chunk& chunk) { return fFile.ReadChunk (kGremlinInfo, chunk); }
		Bool					ReadDebugInfo			(Chunk& chunk) { return fFile.ReadChunk (kDebugInfo, chunk); }
		Bool					ReadPatchInfo			(Chunk& chunk) { return fFile.ReadChunk (kPatchInfo, chunk); }
		Bool					ReadProfileInfo			(Chunk& chunk) { return fFile.ReadChunk (kProfileInfo, chunk); }
		Bool					ReadLoggingInfo			(Chunk& chunk) { return fFile.ReadChunk (kLoggingInfo, chunk); }

		Bool					ReadPlatformInfo		(Chunk& chunk) { return fFile.ReadChunk (kPlatformInfo, chunk); }
		Bool					ReadPlatformInfoMac		(Chunk& chunk) { return fFile.ReadChunk (kPlatformMac, chunk); }
		Bool					ReadPlatformInfoWin		(Chunk& chunk) { return fFile.ReadChunk (kPlatformWindows, chunk); }
		Bool					ReadPlatformInfoUnix	(Chunk& chunk) { return fFile.ReadChunk (kPlatformUnix, chunk); }

		Bool					ReadRAMImage			(void*);
		Bool					ReadMetaRAMImage		(void*);
		Bool					ReadMetaROMImage		(void*);

		Bool					ReadBugFixes			(BugFixes&);

		// ---------- Writing ----------

		void					WriteDeviceType			(const DeviceType&);

		void					WriteROMFileReference	(const FileReference&);

		void					WriteDBallRegs			(const regstruct&);
		void					WriteHwrDBallType		(const HwrDBallType&);
		void					WriteHwrDBallEZType		(const HwrDBallEZType&);
		void					WriteDBallState			(const Chunk& chunk) { fFile.WriteChunk (kDBState, chunk); }
		void					WriteDBallEZState		(const Chunk& chunk) { fFile.WriteChunk (kDBEZState, chunk); }

		void					WriteSED1375RegsType	(const SED1375RegsType&);
		void					WriteSED1375Image		(const void*, uae_u32);
		void					WriteSED1375MetaImage	(const void*, uae_u32);
		void					WriteSED1375Palette		(const uae_u16 [256]);

		void					WritePLDRegsType		(const HwrJerryPLDType&);

		void					WriteGremlinInfo		(const Chunk& chunk) { fFile.WriteChunk (kGremlinInfo, chunk); }
		void					WriteDebugInfo			(const Chunk& chunk) { fFile.WriteChunk (kDebugInfo, chunk); }
		void					WritePatchInfo			(const Chunk& chunk) { fFile.WriteChunk (kPatchInfo, chunk); }
		void					WriteProfileInfo		(const Chunk& chunk) { fFile.WriteChunk (kProfileInfo, chunk); }
		void					WriteLoggingInfo		(const Chunk& chunk) { fFile.WriteChunk (kLoggingInfo, chunk); }

		void					WritePlatformInfo		(const Chunk& chunk) { fFile.WriteChunk (kPlatformInfo, chunk); }
		void					WritePlatformInfoMac	(const Chunk& chunk) { fFile.WriteChunk (kPlatformMac, chunk); }
		void					WritePlatformInfoWin	(const Chunk& chunk) { fFile.WriteChunk (kPlatformWindows, chunk); }
		void					WritePlatformInfoUnix	(const Chunk& chunk) { fFile.WriteChunk (kPlatformUnix, chunk); }

		void					WriteRAMImage			(const void*, uae_u32);
		void					WriteMetaRAMImage		(const void*, uae_u32);
		void					WriteMetaROMImage		(const void*, uae_u32);

		void					WriteBugFixes			(const BugFixes&);

		// ---------- Other ----------

		Bool					ReadConfiguration		(Configuration&);
		long					GetRAMImageSize			(void);

		// As information is saved to the file, certain parts are recorded
		// here.  That way, this information can be save to the preference
		// file/registry so that newly created sessions can be based on the
		// most recently used settings.

		Configuration			GetConfiguration		(void) { return fCfg; }

		void					SetCanReload			(Bool);
		Bool					GetCanReload			(void);

		void					FixBug					(BugFix);
		Bool					IncludesBugFix			(BugFix);

	private:
		enum CompressionType
		{
			kNoCompression,
			kRLECompression,
			kGzipCompression
		};

		long					ReadChunk				(ChunkFile::Tag tag,
														 void*,
														 CompressionType);
		void					WriteChunk				(ChunkFile::Tag tag,
														 uae_u32,
														 const void*,
														 CompressionType);

		// These functions access kROMAliasTag, kROMNameTag, kROMPathTag
		friend Bool Platform::ReadROMFileReference (ChunkFile&, FileReference&);
		friend void Platform::WriteROMFileReference (ChunkFile&, const FileReference&);

		enum
		{
			kDeviceType			= 'DTyp',	// Device type (Pilot 1000, PalmPilot, Palm III, etc.)

			kROMAliasTag		= 'ROMa',	// Macintosh alias
			kROMNameTag			= 'ROMn',	// Simple file name
			kROMWindowsPathTag	= 'ROMp',	// Full file path (Windows format)
			kROMUnixPathTag	   	= 'ROMu',	// Full file path (Unix format)

			kCPURegs			= 'Creg',	// CPU registers (D0-D7, A0-A7, PC, CCR, SR, etc.).
			kDBRegs				= 'DB  ',	// Memory-mapped registers struct.
			kDBEZRegs			= 'DBEZ',	// Memory-mapped registers struct.
			kDBState			= 'DSt8',	// Extra state stored outside of the memory-mapped registers struct.
			kDBEZState			= 'ESt8',	// Extra state stored outside of the memory-mapped registers struct.

			kSED1375Regs		= '5reg',	// Memory-mapped registers struct.
			kSED1375Image		= '5ram',	// LCD buffer memory.
			kSED1375MetaImage	= '5mrm',	// LCD buffer meta-memory.
			kSED1375Palette		= '5clt',	// LCD color lookup table.

			kPLDRegs			= 'pld ',	// Memory-mapped registers struct.

			kGremlinInfo		= 'grem',	// Gremlin state
			kDebugInfo			= 'dbug',	// Debug state
			kPatchInfo			= 'ptch',	// Trappatch state
			kProfileInfo		= 'prof',	// Profiling state
			kLoggingInfo		= 'log ',	// Standard LogStream data

			kPlatformInfo		= 'plat',	// Information managed by the Platform sub-system that has analogs on all platforms.
			kPlatformMac		= 'mac ',	// Mac-specific information
			kPlatformWindows	= 'wind',	// Windows-specific information
			kPlatformUnix		= 'unix',	// Unix-specific information

			kTimeDelta			= 'Time',	// Delta between the actual time and the time set by
											// the user via the General preference panel.

			kRAMDataTag			= 'zram',	// gzip compressed RAM image
			kMetaRAMDataTag		= 'zmrm',	// gzip compressed meta-RAM image
			kMetaROMDataTag		= 'zmro',	// gzip compressed meta-ROM image

			kBugsTag			= 'bugz',	// bit flags indicating bug fixes in file format
			
			kRLERAMDataTag		= 'cram',	// RLE compressed RAM image - obsolete
			kRLEMetaRAMDataTag	= 'mram',	// RLE compressed meta-RAM image - obsolete
			kUncompRAMDataTag	= 'ram '	// Uncompressed RAM image - obsolete
		};

	private:
		ChunkFile&				fFile;
		Bool					fCanReload;
		Configuration			fCfg;
		bool					fReadBugFixes;
		bool					fChangedBugFixes;
		BugFixes				fBugFixes;
};

#endif	// _SESSIONFILE_H_
