/* -*- mode: C++; tab-width: 4 -*- */
/* ================================================================================== */
/* Copyright (c) 1998-1999 3Com Corporation or its subsidiaries. All rights reserved. */
/* ================================================================================== */

#include "EmulatorCommon.h"
#include "EmRPC.h"

#include "CPU_REG.h"			// Emulator::Suspend
#include "HostControl.h"		// hostErrTimeout
#include "Logging.h"			// LogAppendMsg
#include "Miscellaneous.h"		// CountBits
#include "Platform.h"			// Platform::GetMilliseconds
#include "PreferenceMgr.h"		// Preference
#include "RAM_ROM.h"			// CEnableFullAccess
#include "SLP.h"				// SLP::EventCallback
#include "SocketMessaging.h"	// CTCPSocket
#include "SystemPacket.h"		// SystemPacket::

struct SLPTimeout
{
	// Default constructor for the creation of STL collections.
	// CodeWarrior needed this; VC++ didn't.
	SLPTimeout () :
		fSLP (),
		fStart (0),
		fTimeout (0)
	{
	}

	SLPTimeout (SLP& slp, long timeout) :
		fSLP (slp),
		fStart (Platform::GetMilliseconds()),
		fTimeout (timeout)
	{
	}

	SLP		fSLP;
	uae_u32	fStart;
	uae_u32	fTimeout;
};

typedef vector<SLPTimeout>	SLPTimeoutList;

static SLPTimeoutList		gSLPTimeouts;
static SLP*					gCurrentPacket;
static omni_mutex			gMutex;

#define PRINTF	if (true) ; else LogAppendMsg

/***********************************************************************
 *
 * FUNCTION:	RPC::Startup
 *
 * DESCRIPTION: Create all the sockets we'll need for the application
 *				and start them listening for clients.  Call this once
 *				at application startup.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void RPC::Startup (void)
{
	RPC::CreateNewListener ();
}


/***********************************************************************
 *
 * FUNCTION:	RPC::Shutdown
 *
 * DESCRIPTION: Call this once at application shutdown.
 *
 * PARAMETERS:	None.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void RPC::Shutdown (void)
{
}

/***********************************************************************
 *
 * FUNCTION:    RPC::Idle
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void RPC::Idle (void)
{
	omni_mutex_lock	lock (gMutex);

	uae_u32	now = Platform::GetMilliseconds ();

	SLPTimeoutList::iterator	iter = gSLPTimeouts.begin();

	while (iter != gSLPTimeouts.end())
	{
		SLP&	slp		= iter->fSLP;
		uae_u32	start	= iter->fStart;
		uae_u32	timeout	= iter->fTimeout;

		if (now - start > timeout)
		{
			PRINTF ("RPC::Idle: Timing out.");

			assert (slp.HavePacket());

			slp.DeferReply(false);

			SysPktBodyType&	response = slp.Body();
			if (response.command == sysPktRPCCmd)
			{
				SysPktRPCType2&	response		= (SysPktRPCType2&) slp.Body ();

				response.command				= sysPktRPCRsp;
				response._filler				= 0;

				response.resultD0				= hostErrTimeout;
				response.resultA0				= 0;
			}
			else
			{
				SysPktRPC2Type&	response		= (SysPktRPC2Type&) slp.Body ();

				response.command				= sysPktRPC2Rsp;
				response._filler				= 0;

				response.resultD0				= hostErrTimeout;
				response.resultA0				= 0;
				response.resultException		= 0;
			}

			long	bodySize = slp.GetPacketSize() - (sizeof (SlkPktHeaderType) + sizeof (SlkPktFooterType));

			ErrCode result = slp.SendPacket (&response, bodySize);

			iter = gSLPTimeouts.erase (iter);
			continue;
		}

		++iter;
	}
}


/***********************************************************************
 *
 * FUNCTION:    RPC::SignalWaiters
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

ErrCode RPC::SignalWaiters (HostSignal signal)
{
	omni_mutex_lock	lock (gMutex);

	PRINTF ("RPC::SignalWaiters: Entering");

	Bool						signalledOne = false;
	SLPTimeoutList::iterator	iter = gSLPTimeouts.begin();

	while (iter != gSLPTimeouts.end())
	{
		PRINTF ("RPC::SignalWaiters: Signaling");

		SLP&	slp		= iter->fSLP;
		uae_u32	start	= iter->fStart;
		uae_u32	timeout	= iter->fTimeout;

		Canonical (signal);

		assert (slp.HavePacket());

		slp.DeferReply(false);

		SysPktBodyType&	response = slp.Body();
		if (response.command == sysPktRPCCmd)
		{
			SysPktRPCType2&	response		= (SysPktRPCType2&) slp.Body ();

			response.command				= sysPktRPCRsp;
			response._filler				= 0;

			response.resultD0				= errNone;
			response.resultA0				= 0;

			SysPktRPCParamType*	param		= (SysPktRPCParamType*) response.param;

			*(HostSignal*) param->data		= signal;
		}
		else
		{
			SysPktRPC2Type&	response		= (SysPktRPC2Type&) slp.Body ();

			response.command				= sysPktRPC2Rsp;
			response._filler				= 0;

			response.resultD0				= errNone;
			response.resultA0				= 0;
			response.resultException		= 0;

			int					numRegs		= ::CountBits(response.DRegMask) +
											  ::CountBits(response.ARegMask);
			UInt16*				numParams	= (UInt16*) &response.Regs[numRegs];
			SysPktRPCParamType*	param		= (SysPktRPCParamType*) (numParams + 1);

			*(HostSignal*) param->data		= signal;
		}

		long	bodySize = slp.GetPacketSize() - (sizeof (SlkPktHeaderType) + sizeof (SlkPktFooterType));

		ErrCode result = slp.SendPacket (&response, bodySize);

		if (result == errNone)
		{
			signalledOne = true;
		}

		++iter;
	}

	gSLPTimeouts.clear();

	ErrCode	result = errNone;	// !!! Or set to some "no one signalled" result?
	if (signalledOne)
	{
		result = Emulator::Suspend();
	}

	PRINTF ("RPC::SignalWaiters: Exiting");

	return result;
}


/***********************************************************************
 *
 * FUNCTION:	RPC::HandleNewPacket
 *
 * DESCRIPTION: Completely handle a packet sent from an external
 *				client, setting any state and sending a reply if
 *				necessary.
 *
 * PARAMETERS:	slp - a reference to a SerialLink Protocol object that
 *					contains the packet information and the horse...uh,
 *					socket it rode in on.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

ErrCode RPC::HandleNewPacket (SLP& slp)
{
	ErrCode result = errNone;

	assert (slp.Header ().dest == slkSocketRPC);

	gCurrentPacket = &slp;

	CEnableFullAccess	munge;	// Remove blocks on memory access.

	switch (slp.Body().command)
	{
		case sysPktReadMemCmd:
			result = SystemPacket::ReadMem (slp);
			break;

		case sysPktWriteMemCmd:
			result = SystemPacket::WriteMem (slp);
			break;

		case sysPktRPCCmd:
			result = SystemPacket::RPC (slp);
			break;

		case sysPktRPC2Cmd:
			result = SystemPacket::RPC2 (slp);
			break;

		default:
			break;
	}

	gCurrentPacket = NULL;

	return result;
}


/***********************************************************************
 *
 * FUNCTION:    RPC::HandlingPacket
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

Bool RPC::HandlingPacket (void)
{
	return gCurrentPacket != NULL;
}


/***********************************************************************
 *
 * FUNCTION:    RPC::DeferCurrentPacket
 *
 * DESCRIPTION: DESCRIPTION
 *
 * PARAMETERS:  None
 *
 * RETURNED:    Nothing
 *
 ***********************************************************************/

void RPC::DeferCurrentPacket (long timeout)
{
	omni_mutex_lock	lock (gMutex);

	PRINTF ("RPC::DeferCurrentPacket: Entering");

	assert (gCurrentPacket);

	gCurrentPacket->DeferReply (true);
	gSLPTimeouts.push_back (SLPTimeout(*gCurrentPacket, timeout));

	PRINTF ("RPC::DeferCurrentPacket: Exiting");
}


/***********************************************************************
 *
 * FUNCTION:	RPC::EventCallback
 *
 * DESCRIPTION: Callback function for RPC-related sockets.	When an RPC
 *				socket connects, we instantly create a new listener.
 *				When an RPC socket disconnects, we delete it.
 *
 * PARAMETERS:	s - the socket that connected, disconnected, or received
 *					some data.
 *
 *				event - a code indicating what happened.
 *
 * RETURNED:	Nothing.
 *
 ***********************************************************************/

void RPC::EventCallback (CSocket* s, int event)
{
	switch (event)
	{
		case CSocket::kConnected:
		{
			RPC::CreateNewListener ();
			break;
		}

		case CSocket::kDataReceived:
		{
			break;
		}

		case CSocket::kDisconnected:
		{
			s->Delete();
		}
	}

	SLP::EventCallback (s, event);
}


/***********************************************************************
 *
 * FUNCTION:	RPC::CreateNewListener
 *
 * DESCRIPTION: Create a new socket for listening for RPC clients.
 *
 * PARAMETERS:	None
 *
 * RETURNED:	Nothing
 *
 ***********************************************************************/

void RPC::CreateNewListener (void)
{
	Preference<long>	portPref (kPrefKeyRPCSocketPort);

	if (*portPref != 0)
	{
		CSocket*	rpcSocket = new CTCPSocket (&RPC::EventCallback, *portPref);
		ErrCode		err = rpcSocket->Open ();
		if (err != errNone)
		{
			rpcSocket->Delete();
			rpcSocket = NULL;
		}
	}
}


