;;; jde-widgets.el -- Custom-style widgets used by the JDE
;; $Revision: 1.3 $ $Date: 1999/11/16 05:58:18 $ 

;; Author: Paul Kinnucan <paulk@mathworks.com>
;; Maintainer: Paul Kinnucan
;; Keywords: java, tools

;; Copyright (C) 1997, 1998, 1999 Paul Kinnucan.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This is one of a set of packages that make up the 
;; Java Development Environment (JDE) for Emacs. See the
;; JDE User's Guide for more information.

;; The latest version of the JDE is available at
;; <URL:http://sunsite.auc.dk/jde/>
;; <URL:http://www.geocities.com/SiliconValley/Lakes/1506/>

;; Please send any comments, bugs, or upgrade requests to
;; Paul Kinnucan at paulk@mathworks.com.

;;; Code:

(require 'wid-edit)


;; ----------------------------------------------------------------------
;; The Tree Widget Code:


;;; The `tree' Widget.

(define-widget 'jde-widget-tree-open-button 'item
  "Open node in `jde-tree' widget."
  :button-prefix "["
  :button-suffix "]"
  :tag "+"
  :action 'jde-widget-tree-open-button-callback
  :help-echo "Show subtree."
  :format "%[%t%]")

(defun jde-widget-tree-open-button-callback (widget &optional event)
  ;; Set parent state to open.
  (widget-value-set (widget-get widget :parent) t))

(define-widget 'jde-widget-tree-close-button 'item
  "Close node in `tree' widget."
  :button-prefix "["
  :button-suffix "]"
  :tag "-"
  :action 'jde-widget-tree-close-button-callback
  :help-echo "Hide subtree."
  :format "%[%t%]")

(defun jde-widget-tree-close-button-callback (widget &optional event)
  ;; Set parent state to closed.
  (let* ((parent (widget-get widget :parent))
	(entries (widget-get parent :args))
	(children (widget-get parent :children)))
    (while (and entries children)
      (widget-put (car entries) :value (widget-value (car children)))
      (setq entries (cdr entries)
	    children (cdr children)))
    (widget-value-set parent nil)))

(define-widget 'jde-widget-tree 'default
  "A tree structure widget."
  :convert-widget 'widget-types-convert-widget
  :format "%v"
  :prefix ""
  :prefix-extra "   "
  :prefix-empty " |--- "
  :value-get 'widget-value-value-get
  :value-create 'jde-widget-tree-value-create-callback
  :value-delete 'widget-children-value-delete)

(defun jde-widget-tree-value-create-callback (widget)
  ;; Insert all values
  (let ((open (widget-value widget))
	(tag (widget-get widget :tag))
	(entries (widget-get widget :args))
	children buttons)
    (cond ((null entries)
	   ;; Empty node.
	   (insert (widget-get widget :prefix-empty) tag "\n"))
	  (open
	   ;; Open node.
	   (push (widget-create-child-and-convert widget 'jde-widget-tree-close-button)
		buttons)
	   (insert "-\\ " tag "\n")
	   (let ((prefix (concat (widget-get widget :prefix)
				(widget-get widget :prefix-extra)))
		entry)
	     (while entries 
	       (setq entry (car entries)
		     entries (cdr entries))
	       (insert prefix)
	       (push (if entries
			(widget-create-child-and-convert widget entry 
							  :prefix prefix
							  :prefix-extra " | ")
		       ;; Last entry uses a different prefix.
		       (widget-create-child-and-convert 
			widget entry 
			:prefix prefix
			:prefix-empty " `--- "))
		     children))))
	  (t
	   ;; Closed node.
	   (push (widget-create-child-and-convert widget 'jde-widget-tree-open-button)
		buttons)
	   (insert "-- " tag "\n")))
    (widget-put widget :children children)
    (widget-put widget :buttons buttons)))

;;----------------------------------------------------------------------
;; Eval this to create a small tree.
(defun test-tree ()
  (interactive)
  (switch-to-buffer "*Tree Example*")  
  (kill-all-local-variables)
  ;; (make-local-variable 'widget-example-repeat)  
  (let ((inhibit-read-only t))
    (erase-buffer))  
  (let ((all (overlay-lists)))  
    (mapcar 'delete-overlay (car all))    (mapcar 'delete-overlay (cdr all))) 

  (widget-insert "Test tree widget. \n\n")

;   (setq tree (widget-create 'tree
; 			:tag "Foo"
; 			'(tree :tag "First")
; 			'(tree :tag "Second"
; 			       :value nil
; 			       (tree :tag "Nested"))
; 			'(tree :tag "Third")))

;   (setq tree (widget-create 'jde-widget-tree
; 		  :tag "<test.Foo:139>"
; 		  '(jde-widget-tree :tag "n  int  0")
; 		  '(jde-widget-tree :tag "a  double 5.5")
; 		  '(jde-widget-tree :tag "s  S      <test.S:145>"
; 			 (jde-widget-tree :tag "b   boolean  true"))))
	 
  (let*  ((threads 
	  (list 
	   (list "ThreadGroup" 189 "system"
		 (list
		  (list "Thread" 190 "Signal dispatcher" "runnable" "suspended by debugger")
		  (list "Thread" 191 "Reference Handler" "waiting" "suspended by debugger")
		  (list "Thread" 192 "Finalizer" "waiting" "suspended by debugger")))
	   (list "ThreadGroup" 193 "main" 
		 (list
		  (list "Thread" 1 "main" "runnable" "suspended at breakpoint")) 
		 nil)))
	 (tree (jde-dbs-map-threads-to-tree threads)))
		 
			  
    (apply 'widget-create tree))
    
  (use-local-map widget-keymap)
  (widget-setup))


(defclass jde-dialog ()
  ((title     :initarg :title
	      :type string
	      :initform "JDE Dialog"
	      :documentation
	      "Title of dialog")
   (buf       :initarg :buf
	      :type buffer
	      :documentation
	      "Dialog buffer")
   )
  "Super class of JDE dialogs."
  )

(defmethod initialize-instance ((this jde-dialog) &rest fields)
  "Constructor for trace methods dialog."

  ;; Call parent initializer.
  (call-next-method)

  (oset this buf (get-buffer-create (oref this title)))
  (set-buffer (oref this buf))

  (jde-dialog-create this)

  (widget-put
   (widget-create 
    'push-button
    :notify 
    (lambda (button &rest ignore) (jde-dialog-ok (widget-get button :dialog)))
    "Ok")
   :dialog this)

  (widget-put
   (widget-create 
    'push-button
    :notify (lambda (button &rest ignore) (jde-dialog-cancel (widget-get button :dialog)))
    "Cancel")
   :dialog this)

  (use-local-map widget-keymap)
  (widget-setup)
  )

(defmethod jde-dialog-create ((this jde-dialog)))

(defmethod jde-dialog-ok ((this jde-dialog))
  "Invoked when the user clicks the dialog's okay button. The
default method kills the dialog buffer."
  (kill-buffer (current-buffer)))

(defmethod jde-dialog-cancel ((this jde-dialog))
  "Invoked when the user clicks the dialog's Cancel button. The
default method kills the dialog buffer."
  (kill-buffer (current-buffer)))

(defmethod jde-dialog-show ((this jde-dialog))
   (pop-to-buffer (oref this buf)))



(provide 'jde-widgets)

;; $Log: jde-widgets.el,v $
;; Revision 1.3  1999/11/16 05:58:18  paulk
;; Added trace method commands and skeletons for trace class and cancel
;; trace commands.
;;
;; Revision 1.2  1999/10/13 06:19:57  paulk
;; Add JDEBug->Threads->Show Threads command
;;
;; Revision 1.1  1999/09/28 04:39:42  paulk
;; Initial revision.
;;

;; End of jde-widgets.el
    