// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _SgmlParser_h_
#define _SgmlParser_h_

#include <qobject.h>
#include <qstring.h>
#include <qlist.h>
#include <qstack.h>
#include "Attribute.h"
#include "Dtd.h"
#include "SgmlLexer.h"
#include "Style.h"
#include "StyleSheetManager.h"

//: The STag class is a container for the information associated with an SGML start tag.
//. A start tag has a tag name, and a list of zero or more attributes.  The
//. STag class provides a find method for looking up attribute values.
struct STag {
    struct Attr {
        QString name;
        QString value;
        Attr( QString _name, QString _value ) : name( _name.copy() ), value( _value.copy() ) {}
    };
    QString     name;
    QList<Attr> attrs;

    //. Is this used?!?
    STag() {}

    //. Create an STag given the tag name.
    STag( QString _name ) : name( _name.copy() ) { attrs.setAutoDelete( TRUE ); }

    //. Create a copy of an existing STag.
    STag( STag& src );

    ~STag();

    //. Return the value associated with the given attribute.  If the attribute
    //. is not present on this start tag, return 0.
    const QString* find( QString attr ) const;

    void printAttrs();
private:
    //. We don't allow this!
    STag( const STag& src );
};

//: The SgmlParser class parses a data stream into a normalized sequence of start tags, end tags, and content.
//. The SgmlParser converts a stream of SGML tokens into a normailzed stream of
//. start tags, end tags, and content.  The stream is normalized because for
//. Every star tag there is a corresponding end tag.  This makes the job of
//. rendering the document much easier.
//. <P>
//. This parser is capable of inserting optional end tags, optional start
//. tags, and even empty elements where needed to insure that the document
//. structure given in the DTD is satisfied.  It will even make some pretty
//. good guesses when parsing documents with errors, so that it can continue
//. to parse the rest of the document.
class SgmlParser : public QObject,
                   public StyleSheetReceiver
{
    Q_OBJECT
public:
    struct State {
        SgmlElement*       element;
        ContentModelState* state;
        State( SgmlElement* _element, ContentModelState* _state ) : element( _element ), state( _state ) {}
    };

    struct Op {
        enum Type { Stag, Etag } type;
        Op( Type _type ) : type( _type ) {}
	virtual ~Op() {}
    };
private:
    enum {
        AttrName, AttrEqual, AttrValue,
        ContentState,
        StartTag,
        EndTag, EndTagEat,
        MarkupDeclState
    } _state;
    
    Dtd*           _dtd;
    StyleSheet*    _styleSheet;
    SgmlLexer*     _lexer;
    QString        _tag;
    QList<State>   _stack;
    QString        _leftover;
    QString        _dtdType;
    QString        _dtdName;
    bool           _endOfData;
    STag           _startTag;
    QString        _attrName;
    QString        _attrValue;
    QList<Op>      _ops;

    QList<STag>    _tagStack;
    QList<Style>   _styleStack;
    
    //. Try to fit the given start tag into the parse tree for the current
    //. document instance.
    bool doStartTag( STag& tag, int depth );

    //. Try to fit the given end tag into the parse tree for the current
    //. document instance.
    bool doEndTag( QString tag, int depth );

    //. This is a kludge to cope with documents that have overlapping start and
    //. end tags.  For example
    //. <A HREF="http://www.troll.no/">http://www.troll.no/</A> ;)
    bool doEndTagHack( QString tag );

    //. Emit the given content.
    bool doContent( QString text );

    //. Returns TRUE if the given element is excluded from the content model at
    //. the given stack depth.
    bool excluded( QString tag, int depth );

    //. Returns TRUE if the given element is included in the content model at
    //. the given stack depth.
    bool included( QString tag, int depth );

    //. Returns TRUE if the content model at the given stack depth is in an
    //. accepting state.
    bool satisfied( int depth );

    //. Returns TRUE if the end tag for the element at the given stack depth
    //. is optional.
    bool etagOptional( int depth );

    //. Returns TRUE if the given tag can be directly fit into the content
    //. model of the element at the given stack depth.
    bool fitsModel( QString tag, int depth );

    //. Open a new tag and push it onto the stack.  Just create the operations,
    //. but don't apply them yet.
    void openTag( STag& tag );

    //. Close off all the tags up to, but not including, the tag at the given
    //. stack depth.  Just create the necessary operations, but don't apply
    //. them yet.
    void closeTags( int depth );

    //. See if there is an element with an optional start tag that can be
    //. inserted into the content model for the element at the given stack
    //. depth.  Returns TRUE if successful.
    bool tryOptional( int depth );

    //. We've successfully fit the start/end tag into the content model.  Now
    //. actually emit the start and end tags.
    void commit();

    //. Back out any changes made to the tag stack while trying to fit in a
    //. start or end tag.
    void rollback();

    void logStack();
public:
    //. Create a new SgmlParser.  Create an SgmlLexer to do some of the dirty
    //. work.
    SgmlParser();
    ~SgmlParser();

    const QList<Style>& styleStack() { return _styleStack; }
    const QList<STag>& tagStack() { return _tagStack; }

    void styleSheet( StyleSheet* styleSheet );
public slots:
    //. This slot is activated after the last token has been received.  Try
    //. to close any open tags on the stack.
    void lexerDone();

    //. This slot is activated when the DtdManager has finished processing
    //. our DTD request.  Feed any queued data to the SgmlLexer.
    void dtd( Dtd* dtd );

    //. Forward the data signals to the SgmlLexer.
    void data( const char* bytes, int length );

    //. Forward the endOfData signal to the SgmlLexer.
    void endOfData();

    //. Process a token from the lexer.  Try to parse tokens into start tags,
    //. end tags, and content.  Then normalize the tag stream with doStartTag()
    //. and doEndTag().
    void token( SgmlLexer::Token token, const char* text );
signals:
    //. This signal is used to forward the data signals to the SgmlLexer.
    void fwdData( const char* bytes, int length );

    //. This signal is used to forward the endOfData signal to the SgmlLexer.
    void fwdEndOfData();

    //. This signal is emitted when content is parsed.
    void content( QString text );

    //. This signal is emitted when an end tag is parsed.
    void endTag();

    //. This signal is emitted when a start tag is parsed.  The tag includes
    //. any attributes that were present.
    void startTag();

    //. This signal is emitted when no more tags or content will be emitted.
    void done();

    //. This signal is emitted to inform the browser that we have settled on a
    //. DTD.
    void dtdSelected( QString dtdName );
};

#endif
