// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <qkeycode.h>
#include <qpainter.h>
#include <qscrbar.h>
#include "Canvas.h"
#include "DownloadRenderer.h"
#include "FormRenderer.h"
#include "ImageRenderer.h"
#include "PlainRenderer.h"
#include "Request.h"
#include "SgmlParser.h"
#include "VerticalRenderer.h"

class Clip : public QWidget {
    Canvas* _canvas;
protected:
    void paintEvent( QPaintEvent* e );
    void leaveEvent( QEvent* e );
    void mouseMoveEvent( QMouseEvent* e );
    void mouseReleaseEvent( QMouseEvent* e );
public:
    Clip( Canvas* parent=0, const char* name=0 );
};

Clip::Clip( Canvas* parent, const char* name )
        : QWidget( parent, name ),
          _canvas( parent )
{
    setMouseTracking( TRUE );
}

void Clip::paintEvent( QPaintEvent* e )
{
    if ( !_canvas->renderer() ) return;

    // Create the exposed rectangle for the renderer.
    Rect r( e->rect().x() + _canvas->hsb()->value(), e->rect().y() + _canvas->vsb()->value(), e->rect().width(), e->rect().height() );
    
    QPainter p;
    p.begin( this );

    _canvas->renderer()->repaint( p, r );
    
    p.end();
}

void Clip::leaveEvent( QEvent* )
{
    setCursor( arrowCursor );
}

void Clip::mouseMoveEvent( QMouseEvent* e )
{
    QListIterator<MouseZone> i( _canvas->mouseZones() );
    for ( i.toFirst(); i.current(); ++i ) {
        if ( i.current()->contains( e->pos().x(), e->pos().y() ) ) {
            _canvas->status( i.current()->url );
            setCursor( crossCursor );
            return;
        }
    }
    setCursor( arrowCursor );
}

void Clip::mouseReleaseEvent( QMouseEvent* e )
{
    _canvas->mouseRelease( e );
}

Canvas::Canvas( Browser* browser, const char* name )
        : QWidget( browser, name ),
          _browser( browser ),
          _renderer( 0 ),
          _updateTimerId( 0 ),
          _downloadTimerId( 0 )
{
    setFocusPolicy( QWidget::StrongFocus );
    setFocus();
    
    _vsb = new QScrollBar( QScrollBar::Vertical, this );
    _vsb->setSteps( 10, 500 );
    _vsb->setTracking( TRUE );
    connect( _vsb, SIGNAL( valueChanged( int ) ), this, SLOT( vsbMoved( int ) ) );

    _hsb = new QScrollBar( QScrollBar::Horizontal, this );
    _hsb->setSteps( 10, 500 );
    _hsb->setTracking( TRUE );
    connect( _hsb, SIGNAL( valueChanged( int ) ), this, SLOT( hsbMoved( int ) ) );

    _clip = new Clip( this );
    _clip->move( 0, 0 );
}

Canvas::~Canvas()
{
    clear();
}

void Canvas::clear()
{
    while ( _children.first() ) {
        delete _children.first()->widget;
        delete _children.first();
        _children.remove();
    }

    while ( _mouseZones.first() ) {
        delete _mouseZones.first();
        _mouseZones.remove();
    }

    if ( _renderer ) {
        delete _renderer;
        _renderer = 0;
    }

    _clip->erase();
}

void Canvas::resizeEvent( QResizeEvent* )
{
    _hsb->setGeometry( 0, height() - 16, width() - 16, 16 );
    _vsb->setGeometry( width() - 16, 0, 16, height() - 16 );

    if ( ( _renderer ) && ( _clip->width() != width() - 20 ) ) {
        _renderer->widthChanged( width() - 16 );
        redraw();
    }
    _clip->setGeometry( 0, 0, width() - 16, height() - 16 );
    
    childSizeChanged();
}

void Canvas::mouseRelease( QMouseEvent* e )
{
    if ( e->button() == LeftButton ) {
        QListIterator<MouseZone> i( _mouseZones );
        for ( i.toFirst(); i.current(); ++i ) {
            if ( i.current()->contains( e->pos().x(), e->pos().y() ) ) {
                // Create the url
                Url url( i.current()->url );
                if ( i.current()->renderer->isA( "ImageRenderer" ) ) {
                    ImageRenderer* ir = (ImageRenderer*)i.current()->renderer;
                    int x = e->pos().x();
                    int y = e->pos().y();
                    for ( QObject* o = ir; o; o = o->parent() ) {
                        x -= ((Renderer*)o)->x();
                        y -= ((Renderer*)o)->y();
                    }

                    if ( ir->isSubmit() ) {
                        ir->setX( x );
                        ir->setY( y );
                        if ( ir->form() ) {
                            ir->form()->submit( ir );
                        }
                        return;
                    } else if ( ir->isMap() ) {
                        QString tmp;
                        tmp.sprintf( "%d,%d", x, y );
                        url.setQuery( tmp );
                    }
                }
                
                if ( e->state() & ShiftButton ) {
                    DownloadRenderer* dr = new DownloadRenderer( url );
                    dr->show();
                } else {
                    stop();
                    _browser->open( url );
                }
                return;
            }
        }
    }
}

QWidget* Canvas::clip()
{
    return _clip;
}

void Canvas::hsbMoved( int pos )
{
    if ( _renderer ) {
        Rect r;
        int dx = pos + _renderer->x();
        if ( dx < 0 ) {
            bitBlt( _clip, -dx, 0, _clip, 0, 0, _clip->width() + dx, _clip->height() );
            r.setRect( 0, 0, -dx, _clip->height() );
        } else {
            bitBlt( _clip, 0, 0, _clip, dx, 0, _clip->width() - dx, _clip->height() );
            r.setRect( _clip->width() - dx, 0, dx, _clip->height() );
        }
        
        _renderer->move( -pos, _renderer->y() );
        moveWidgets();
        moveMouseZones();
        _clip->repaint( r.x(), r.y(), r.width(), r.height() );
    }
}

void Canvas::vsbMoved( int pos )
{
    if ( _renderer ) {
        Rect r;
        int dy = pos + _renderer->y();
        if ( dy < 0 ) {
            // Scroll renderer up.
            bitBlt( _clip, 0, -dy, _clip, 0, 0, _clip->width(), _clip->height() + dy );
            r.setRect( 0, 0, _clip->width(), -dy );
        } else {
            // Scroll renderer down.
            bitBlt( _clip, 0, 0, _clip, 0, dy, _clip->width(), _clip->height() - dy );
            r.setRect( 0, _clip->height() - dy, _clip->width(), dy );
        }
        
        _renderer->move( _renderer->x(), -pos );
        moveWidgets();
        moveMouseZones();
        _clip->repaint( r.x(), r.y(), r.width(), r.height() );
    }
}

void Canvas::moveWidgets()
{
    QListIterator<Child> i( _children );
    for ( i.toFirst(); i.current(); ++i ) {
        moveWidget( i.current()->renderer, i.current()->widget );
    }
}

void Canvas::moveMouseZones()
{
    QListIterator<MouseZone> i( _mouseZones );
    for ( i.toFirst(); i.current(); ++i ) {
        moveMouseZone( i.current() );
    }
}

void Canvas::moveWidget( QObject* r, QWidget* w )
{
    int x = 0;
    int y = 0;
    for ( ; r; r = r->parent() ) {
        x += ((Renderer*)r)->x();
        y += ((Renderer*)r)->y();
    }
    w->move( x, y );
    w->show();
}

void Canvas::moveMouseZone( MouseZone* mouseZone )
{
    int x = 0;
    int y = 0;
    QObject* r = mouseZone->renderer;
    for ( ; r; r = r->parent() ) {
        x += ((Renderer*)r)->x();
        y += ((Renderer*)r)->y();
    }
    mouseZone->setRect( x, y, mouseZone->renderer->width(), mouseZone->renderer->height() );
}

void Canvas::registerWidget( Renderer* renderer, QWidget* widget )
{
    _children.append( new Child( renderer, widget ) );
}

void Canvas::registerMouseZone( Renderer* renderer, const QString& url )
{
    _mouseZones.append( new MouseZone( renderer, Url( &_browser->baseUrl(), url ).url() ) );
}

void Canvas::startOfData( Request* request, QString mediaType, QString mediaSubtype, int size )
{
    setFocus();
    
    if ( mediaType == "text" ) {
        if ( ( mediaSubtype == "html" ) || ( mediaSubtype == "sgml" ) ) {
            _browser->startOfData();
            
            SgmlParser* parser = new SgmlParser;
            connect( parser , SIGNAL( done() )                  , this   , SLOT( redraw() ) );
            connect( request, SIGNAL( data( const char*, int ) ), parser , SLOT( data( const char*, int ) ) );
            connect( request, SIGNAL( endOfData() )             , parser , SLOT( endOfData() ) );

            if ( _renderer ) {
                clear();
                childSizeChanged();
            }
    
            _renderer = new VerticalRenderer( this, parser, _clip->width() );
            _renderer->move( 0, 0 );
            connect( parser   , SIGNAL( startTag() )        , _renderer, SLOT( startTag() ) );
            connect( parser   , SIGNAL( endTag() )          , _renderer, SLOT( endTag() ) );
            connect( parser   , SIGNAL( content( QString ) ), _renderer, SLOT( content( QString ) ) );
            connect( _renderer, SIGNAL( resized() )         , this     , SLOT( childSizeChanged() ) );
            return;
        } else if ( mediaSubtype == "plain" ) {
            _browser->startOfData();
            
            if ( _renderer ) {
                clear();
                childSizeChanged();
            }
            
            _renderer = new PlainRenderer( this, _clip->width() );
            _renderer->move( 0, 0 );
            connect( _renderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
            connect( request  , SIGNAL( data( const char*, int ) ), _renderer, SLOT( data( const char*, int ) ) );
            connect( request  , SIGNAL( endOfData() )             , _renderer, SLOT( endOfData() ) );
            return;
        }
    } else if ( mediaType == "image" ) {
        if ( mediaSubtype == "gif" ) {
            _browser->startOfData();
            
            if ( _renderer ) {
                clear();
                childSizeChanged();
            }
            
            _renderer = new ImageRenderer( "", 100, 100, FALSE, FALSE, "", this, 100 );
            _renderer->move( 0, 0 );
            connect( _renderer, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
            ((ImageRenderer*)_renderer)->startOfData( request, mediaType, mediaSubtype, size );
            return;
        }
    }

    // Spawn download renderer after returning from this slot.
    _downloadTimerId = startTimer( 0 );
}

void Canvas::childSizeChanged()
{
    if ( ( _renderer ) && ( _renderer->width() > _clip->width() ) ) {
        _hsb->setRange( 0, _renderer->width() - _clip->width() );
    } else {
        _hsb->setRange( 0, 0 );
    }
        
    if ( ( _renderer ) && ( _renderer->height() > _clip->height() ) ) {
        _vsb->setRange( 0, _renderer->height() - _clip->height() );
    } else {
        _vsb->setRange( 0, 0 );
    }

    _vsb->setSteps( 10, _clip->height() );
    _hsb->setSteps( 10, _clip->width() );
}

void Canvas::redraw()
{
    if ( _renderer ) {
        bool b = _renderer->redraw();

        if ( _renderer->inherits( "SgmlRenderer" ) ) {
            if ( _browser->baseUrl().fragment().length() > 0 ) {
                int x, y;
                if ( ((SgmlRenderer*)_renderer)->findAnchor( _browser->baseUrl().fragment(), x, y ) ) {
                    _vsb->setValue( y );
                }
            }
        }

        if ( b ) {
            _clip->repaint();
            moveWidgets();
            moveMouseZones();
        }
    }
}

void Canvas::setTitle( const QString& title )
{
    _browser->setTitle( title );
}

void Canvas::setBaseUrl( const Url& url )
{
    _browser->setBaseUrl( url );
}

const Url& Canvas::baseUrl()
{
    return _browser->baseUrl();
}

void Canvas::openUrl( const Url& url )
{
    _browser->open( url );
}

void Canvas::registerRequest( Request* req )
{
    _requests.append( req );
    _browser->setTransfer( TRUE );
    if ( !_updateTimerId ) {
        _updateTimerId = startTimer( 2000 );
    }
}

void Canvas::requestDone( Request* req )
{
    if ( _requests.count() == 0 ) {
        return;
    }
    
    _requests.remove( req );

    if ( _requests.count() == 0 ) {
        if ( _updateTimerId ) {
            killTimer( _updateTimerId );
            _updateTimerId = 0;
        }
        _browser->setTransfer( FALSE );
        redraw();
        _browser->status( "done." );
    }
}

void Canvas::timerEvent( QTimerEvent* e )
{
    if ( e->timerId() == _updateTimerId ) {
        redraw();
    } else if ( e->timerId() == _downloadTimerId ) {
        killTimer( _downloadTimerId );
        _downloadTimerId = 0;
        DownloadRenderer* dr = new DownloadRenderer( _browser->baseUrl() );
        dr->show();

        _browser->stop();
    }
}

void Canvas::stop()
{
    for ( _requests.first(); _requests.current(); _requests.next() ) {
        _requests.current()->abort();
    }
}

void Canvas::scrollUp()
{
    _vsb->setValue( _vsb->value() - _vsb->lineStep() );
}

void Canvas::scrollDown()
{
    _vsb->setValue( _vsb->value() + _vsb->lineStep() );
}

void Canvas::scrollLeft()
{
    _hsb->setValue( _hsb->value() - _hsb->lineStep() );
}

void Canvas::scrollRight()
{
    _hsb->setValue( _hsb->value() + _hsb->lineStep() );
}

void Canvas::scrollPrior()
{
    _vsb->setValue( _vsb->value() - _vsb->pageStep() );
}

void Canvas::scrollNext()
{
    _vsb->setValue( _vsb->value() + _vsb->pageStep() );
}

void Canvas::keyPressEvent( QKeyEvent* e )
{
    switch ( e->key() ) {
        case Key_Up:
            scrollUp();
            break;

        case Key_Down:
            scrollDown();
            break;

        case Key_Left:
            scrollLeft();
            break;

        case Key_Right:
            scrollRight();
            break;

        case Key_Prior:
            scrollPrior();
            break;

        case Key_Next:
            scrollNext();
            break;
    }
}

void Canvas::status( QString msg )
{
    _browser->status( msg );
}
