/* 
 * Internal header file.
 * Copyright (c) 1995 Markku Rossi.
 *
 * Author: Markku Rossi <mtr@iki.fi>
 */

/*
 * This file is part of genscript.
 * 
 * Genscript is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * Genscript is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with genscript; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef GSINT_H
#define GSINT_H

/* Version numbers. */
#define MAJOR_VERSION	1
#define MINOR_VERSION	2
#define EDITION_STRING "Edition: 20"

/*
 * Config stuffs.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>

#if STDC_HEADERS

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#else /* no STDC_HEADERS */

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif
char *strchr ();
char *strrchr ();

#ifndef HAVE_MEMCPY
#define memcpy(d, s, n) bcopy((s), (d), (n))
#define memmove(d, s, n) bcopy((s), (d), (n))
#endif

#if HAVE_STDARG_H
#include <stdarg.h>
#else
XXX No stdarg.h, I give up.
#endif

#endif /* no STDC_HEADERS */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <errno.h>
#include <time.h>
#include <assert.h>
#include <sys/types.h>
#include <pwd.h>
#include <ctype.h>
#include <sys/stat.h>

#include "afm.h"
#include "strhash.h"

/* 
 * Types and definitions.
 */

#define MATCH(a, b) (strcmp (a, b) == 0)

#define ISNUMBERDIGIT(ch) \
  (('0' <= (ch) && (ch) <= '9') || (ch) == '.' || (ch) == '-' || (ch) == '+')

/* Constants for output files. */
#define OUTPUT_FILE_NONE   NULL
#define OUTPUT_FILE_STDOUT ((char *) 1)

struct media_entry_st
{
  struct media_entry_st *next;
  char *name;
  int w;
  int h;
  int llx;
  int lly;
  int urx;
  int ury;
};

typedef struct media_entry_st MediaEntry;

typedef enum
{
  HDR_NONE,
  HDR_SIMPLE,
  HDR_FANCY
} HeaderType;


typedef enum
{
  ENC_LATIN1,
  ENC_LATIN2,
  ENC_ASCII,
  ENC_ASCII_SCANDS,
  ENC_IBMPC,
  ENC_MAC,
  ENC_VMS,
  ENC_PS
} InputEncoding;

typedef enum
{
  LABEL_SHORT,
  LABEL_LONG
} PageLabelFormat;

struct file_lookup_ctx_st
{
  char name[256];
  char suffix[256];
  char fullname[512];
};

typedef struct file_lookup_ctx_st FileLookupCtx;

typedef int (*PathWalkProc) (char *path, void *context);

struct input_stream_st
{
  FILE *fp;
  unsigned char buf[4096];
  unsigned int data_in_buf;
  unsigned int bufpos;
  unsigned int nreads;
  int unget_ch;
};

typedef struct input_stream_st InputStream;


/* 
 * Global variables.
 */

extern char *program;
extern FILE *ofp;
extern char version_string[];
extern char date_string[];
extern struct tm run_tm;
extern struct tm mod_tm;
extern struct passwd *passwd;
extern char libpath[];
extern char *afm_path;
extern char afm_path_buffer[];
extern MediaEntry *media_names;
extern MediaEntry *media;
extern char spooler_command[];
extern char queue_param[];
extern char *queue;
extern char queue_buf[];
extern int nl;
extern int bs;

/* Statistics. */
extern int total_pages;
extern int num_truncated_lines;
extern int num_missing_chars;
extern int missing_chars[];

/* Dimensions that are used during PostScript emission. */
extern int d_page_w;
extern int d_page_h;
extern int d_header_w;
extern int d_header_h;
extern int d_output_w;
extern int d_output_h;
extern int d_output_x_margin;
extern int d_output_y_margin;

/* Document needed resources. */
extern StringHashPtr res_fonts;

/* Fonts to download. */
extern StringHashPtr download_fonts;

/* Additional key-value pairs, passed to the generated PostScript code. */
extern StringHashPtr pagedevice;
extern StringHashPtr statusdict;

/* AFM library handle. */
extern AFMHandle afm;

/* Fonts. */
extern char *HFname;
extern double HFpt;
extern char *Fname;
extern double Fpt;
extern double default_Fpt;
extern char *default_Fname;
extern double font_widths[256];
extern char font_ctype[256];
extern int font_is_fixed;

/* Options. */

extern char *queue;
extern int verbose;
extern int num_copies;
extern char *title;
extern int num_columns;
extern int truncate_lines;
extern int quiet;
extern int landscape;
extern HeaderType header;
extern char *fancy_header_name;
extern char fancy_header_default[];
extern int line_indent;
extern char *page_header;
extern char *output_file;
extern unsigned int lines_per_page;
extern InputEncoding encoding;
extern char *media_name;
extern char media_name_buffer[];
extern char *encoding_name;
extern char encoding_name_buffer[];
extern int special_escapes;
extern int tab_size;
extern double baselineskip;
extern double ul_ptsize;
extern double ul_gray;
extern char *ul_font;
extern char *underlay;
extern PageLabelFormat page_label;
extern char *page_label_format;
extern char page_label_format_buf[];
extern int pass_through_ps_files;
extern int line_numbers;

/* 
 * Prototypes for global functions.
 */

/* Print message if <verbose> is >= <verbose_level>. */
void message (int verbose_level, char *fmt, ...);

/* Report continuable error. */
void error (char *fmt, ...);

/* Report fatal error and exit with status 1.  Function never returns. */
void fatal (char *fmt, ...);

/* 
 * Read config file <path, name>.  Returns bool.  If function fails, error
 * is found from errno.
 */
int read_config (char *path, char *name);

/* Print PostScript header to our output stream. */
void dump_ps_header (void);

/* Print PostScript trailer to our output stream. */
void dump_ps_trailer (void);

/* Init InputStream <is> to its start-of-stream state. */
void is_init (InputStream *is);

/*
 * Read next character from the InputStream <is>.  Returns EOF if 
 * EOF was reached.
 */
int is_getc (InputStream *is);

/*
 * Put character <ch> back to the InputStream <is>.  Function returns EOF
 * if character couldn't be unget.
 */
int is_ungetc (int ch, InputStream *is);

/* 
 * Process single input file <fp>.  File's name is given in <fname> and
 * it is used to print headers.
 */
void process_file (char *fname, InputStream *fp);

/* Add a new media to the list of known media. */
void add_media (char *name, int w, int h, int llx, int lly, int urx, int ury);

/* Print a listing of missing characters. */
void do_list_missing_characters (void);

/* 
 * Check if file <name, suffix> exists.  Returns bool.  If function fails
 * error can be found from errno.
 */
int file_existsp (char *name, char *suffix);

/*
 * Paste file <name, suffix> to output stream.  Returns bool. If 
 * function fails, error can be found from errno.
 */
int paste_file (char *name, char *suffix);

/*
 * Do tilde substitution for filename <from> and insert result to <to>.
 * Buffer <to> must be long enought to hold expanded name.
 */
void tilde_subst (char *from, char *to);

/*
 * Parse font spec <spec> and return font's name and size in variables
 * <name_return> and <size_return>.  Returns 1 if <spec> was a valid
 * font spec or 0 otherwise.  Returned name <name_return> is allocated
 * with xcalloc() and must be freed by caller.
 */
int parse_font_spec (char *spec, char **name_return, double *size_return);

/*
 * Read body font's character widths and character codes from AFM files.
 */
void read_font_info (void);

/*
 * Try to download font <name>.
 */
void download_font (char *name);

/*
 * Escape all PostScript string's special characters from string <string>.
 * Returns a xmalloc()ated result.
 */
char *escape_string (char *string);

/*
 * Parses key-value pair <kv> and inserts/deletes key from <set>.
 */
void parse_key_value_pair (StringHashPtr set, char *kv);

/*
 * Count how many non-empty items there are in the key-value set <set>.
 */
int count_key_value_set (StringHashPtr set);

/*
 * Walk through path <path> and call <proc> once for each of its
 * components.  Function returns 0 if all components were accessed.
 * Callback <proc> can interrupt walking by returning a non zero
 * return value.  In that case value is returned as the return value
 * of the pathwalk().
 */
int pathwalk (char *path, PathWalkProc proc, void *context);

/* Lookup file from path.  <context> must point to FileLookupCtx. */
int file_lookup (char *path, void *context);


/* 
 * Non-failing memory allocation.
 */

void *xmalloc (size_t size);

void *xcalloc (size_t num, size_t size);

void *xrealloc (void *ptr, size_t size);

void xfree (void *ptr);

char *xstrdup (char *);

#endif /* not GSINT_H */
