#include "cthugha.h"
#include "sound.h"
#include "options.h"
#include "display.h"
#include "cd_player.h"
#include "net_sound.h"
#include "information.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/soundcard.h>
#include <math.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include <sys/time.h>

/*
 * This file is used for the "normal" cthugha-L and for the cthugha-server.
 * cthugha-server displays no graphic and accepts only a subset of the
 * options cthugha-L supports.
 *
 * If CTH_SERVER is defined, then the cthugha-server version should be 
 * generated.
 */

/* public Variables */

void (* display_wave)(void);			/* Pointer to active wave */
char sound_data[MAX_BUFF_WIDTH][2];		/* Sound-Data in stereo */
int sound_stereo = 1;				/* stereo/mono */
int sound_sample_rate = 44000;			/* sampel rate */
int sound_source = SNDSRC_DEFAULT;		/* default, Line-In, Mic, CD */
int sound_volume_line = -1;		
int sound_volume_mic = -1;		
int sound_volume_cd = -1;		
int sound_volume_rec = -1;

int sound_minnoise = 30;			/* quiet is below this */
int sound_peaklevel = 240;			/* above that is loud */

int sound_wait_quiet = 500;			/* max. quiet interval (5 sec) 
						   then text is displayed */
int sound_quiet_change = 150;			/* change after quiet-pause 
						   1.5 seconds */

int sound_wait_beat = 16;			/* peaks till change */

int sound_wait_min = 500;			/* min time between change 
						   (5 sec) */
int sound_wait_random = 1000;			/* extra random wait-time 
						   (10 sec) */
int sound_wait = 1000;				/* time till change */

int sound_FFT = 0;				/* use FFT or don't */
int sound_use_fft = 1;				/* allow fft-usage */

char wave_first[256] = "";			/* Start with this wave */
int sound_massage_style = 1;			/* massage style */

int sound_quiet_since = 0;			/* used to determine if
						   quiet msg. should be disp.*/
int sound_timer = 0;				/* used for automatic change */
int sound_starttime = 0;

/* private Variables */

int sound_blk_size = 0;				/* Size of DMA-Buffer */
int sound_fmt = AFMT_U8;			/* sound format */
int sound_div = 1;				/* reduction of DMA-Buffer */
int sound_bsize = 0;				/* size of sound-buffer */

char dsp_file[256] = "/dev/dsp";		/* sound-device */
int dsp_des;					/* descriptor to dsp_file */

unsigned char sound_buffer[65536];		/* temporary buffer */

int sound_quiet = 0;				/* quiete since */
int sound_beats = 0;				/* loudness-count */
int sound_count = 0;				/* counter for change */

int sound_sync = 0;				/* reset sound after reading */

/*
 * variables about tables
 */
pal_table tables[MAX_TABLES];			/* Palette-usage-tables */
int nr_tables = 0;
int active_table = 0;
int table_first = CHANGE_RANDOM;		/* table to start with */

/*
 * stuff about massage
 */
int massage_first = CHANGE_RANDOM;		/* massage to start with */

#ifndef CTH_SERVER

int sine[320];					/* sine in 1/320 */
int Bsine[MAX_BUFF_WIDTH];			/* sine in 1/BUFF_WIDTH */
/* for FFT */
#define N 64
#define SCALE 7
static int s[N+N], c[N+N];
#define CTEPI (3.14159265358979323846/(float)N) 
#define TRIG  16 /* trig table lookup precision compatible with S? way above */ 
#endif

int init_sound_mixer();				/* initialize mixer */
int init_sound_dsp();				/* initialize dsp */
int init_sound_FFT();				/* initialize FFT-tables */

/* 
 *  Initialize the Sound-Interface
 */
int init_sound() {
    struct timeval tv;
    struct timezone tz;

    if ( ((sound_source & SNDSRC_CD) && (sound_source > 0)) 
	|| (cd_first_track >= 0) )
	init_cd();

    if( alloc_sound() )
	return 1;

#ifndef CTH_SERVER
    init_sound_FFT();
    init_tables();	
	
    if ( sound_wait_random <= 0)
	sound_wait_random = 1;
#endif

    sound_bsize = max(BUFF_WIDTH, BUFF_HEIGHT);

    /* initialize starttime */
    gettimeofday(&tv, &tz);
    sound_starttime = tv.tv_sec;

    /* set initial wait-time till change */
    sound_wait = sound_wait_min + rand() % sound_wait_random;

    return 0;
}

/*
 * Initialize /dev/mixer
 */
int init_sound_mixer() {
    int mixer_des;
	
    if ( (sound_source == SNDSRC_DEBUG) || (sound_source == SNDSRC_SOCK))
	return 0;
		
    printfv("Setting /dev/mixer...\n");

    /* open mixer-file */	
    if ( (mixer_des = open("/dev/mixer", O_RDWR)) < 0) {
	printfee("Can not open /dev/mixer");
    }

    /* was there ever a -L, -M, -C to select a special input-source */	
    if ( (sound_volume_line >= 0) || (sound_volume_mic >= 0) || 
	 (sound_volume_cd >= 0) ) {
		
	/* Select input-source */
	if(ioctl(mixer_des, SOUND_MIXER_WRITE_RECSRC, &sound_source) < 0) {
	    printfee("Can not set rec-source");
	}
		
	/* Select input-volumes */
		
	/* set volume for Line-in */
	if ( sound_volume_line >= 0) {
	    if ( sound_source & SNDSRC_LINE ) {
		ioctl( mixer_des, SOUND_MIXER_WRITE_LINE, &sound_volume_line);
		printfv("  Using LINE IN as input (volume: %d).\n", 
			sound_volume_line & 255);
	    } else
		printfv("  Disabling LINE IN as input.\n");
	} 

	/* set volume for Mic */	
	if ( sound_volume_mic >= 0) {
	    if ( sound_source & SNDSRC_MIC ) {
		ioctl( mixer_des, SOUND_MIXER_WRITE_MIC, &sound_volume_mic);
		printfv("  Using MIC as input (volume: %d).\n", 
			sound_volume_mic & 255);
	    } else 
		printfv("  Disabling MIC as input.\n");
	}		

	/* set volume for CD */
	if ( sound_volume_cd != 0) {
	    if ( sound_source & SNDSRC_CD) {
		if ( sound_volume_cd > 0) {
		    ioctl( mixer_des, SOUND_MIXER_WRITE_CD, &sound_volume_cd);
		    printfv("  Using CD as input (volume: %d).\n", 
			    sound_volume_cd & 255);
		} else 
		    printfv("  Using CD as input.\n");
	    }
	} else
	    printfv("  Disabling CD as input.\n");
    }
    if ( sound_volume_rec >= 0) {
	ioctl( mixer_des, SOUND_MIXER_WRITE_RECLEV, &sound_volume_rec);
	printfv("  Selected record volume: %d.\n", sound_volume_rec & 255 );
    }
	
    /* close mixer-file */
    close( mixer_des);	
	
    return 0;
}

/*
 * Initialize /dev/dsp
 */
int init_sound_dsp() {
    int fragment;
	
    if ( sound_source == SNDSRC_DEBUG)
	return 0;

    printfv("Setting /dev/dsp...\n");

    /* Get sound-device */
    if ( (dsp_des = open(dsp_file, O_RDONLY ) ) < 0) {
	printfee("Can't open /dev/dsp");
	return 1;
    }
	
    /* Control-Sound */

    /* fragment size of 16 bytes (2^4), two fragments (2) */
    fragment = 0x00020004;	
    if ( ioctl(dsp_des, SNDCTL_DSP_SETFRAGMENT, &fragment) < 0 )
	printfee("ioctl: SETFRAGMENT");

    if ( ioctl(dsp_des, SNDCTL_DSP_SETFMT, &sound_fmt) < 0 )
	printfee("ioctl: SETFMT");

    if ( ioctl(dsp_des, SNDCTL_DSP_STEREO, &sound_stereo) < 0 )
	printfee("ioctl: STEREO");
		
    if ( ioctl(dsp_des, SNDCTL_DSP_SPEED, &sound_sample_rate) < 0 )
	printfee("ioctl: SPEED");

    printfv("  Number of channels: %d\n", sound_stereo+1);
    printfv("  Sample rate: %d\n", sound_sample_rate);
	
    sound_stereo ++;			/* bring to 1 or 2 (nr. of channels) */
	
    return 0;
}

#ifndef CTH_SERVER
/*
 * Initalization for FFT 
 * the sine-table is also used by some other functions
 */
int init_sound_FFT() {
    int i; 
    float xx,ss,cc; 
    int k; 
 
    for (i=0; i<320; i++) 
	sine[i]=(int)(128*sin((float)i*0.03927)); 
	
    for (i=0; i<BUFF_WIDTH; i++)
    	Bsine[i] = (int)(128*sin((float)i/((float)BUFF_WIDTH)*4.0*3.141592654));
 
    for (k=0 ; k<(N+N) ; k++) { 
	xx=CTEPI*k; 
	ss=TRIG*sin(xx); 
	cc=TRIG*cos(xx); 
	if (ss>0.0) ss+=0.5; else ss-=0.5; 
	if (cc>0.0) cc+=0.5; else cc-=0.5; 
	s[k]=(int)ss; /* truncate */ 
	c[k]=(int)cc; /* truncate */ 
    } 
    return 0;
}
#endif

/*
 * Clean up the sound-interface
 */
int exit_sound() {
    if ( (sound_source & SNDSRC_CD) && (sound_source > 0))
	exit_cd();

    return free_sound();
}

/* 
 * get and free sound device
 */
static int sound_alloc=0;
int alloc_sound() {

    if(sound_alloc)
	return 0;

    /* set mixer every time new, 
       When using modules with kerneld, the sound-module might have been 
       release meanwhile, after releading, the mixer can get reset.
       Or: some other program set the mixer to something different
       */
    if(	init_sound_mixer() )
	return 1;

#ifndef CTH_SERVER
    if ( sound_source == SNDSRC_SOCK) 
	init_net_sound();
    else
#endif
	if( init_sound_dsp() )
	    return 1;

    sound_alloc = 1;
    return 0;
}
int free_sound() {

    if(!sound_alloc)
	return 0;

    sound_stereo --;

#ifndef CTH_SERVER
    if ( sound_source == SNDSRC_SOCK)
	exit_net_sound();
#endif

    if ( sound_source != SNDSRC_DEBUG)
	close(dsp_des);

    sound_alloc = 0;
    return 0;
}    

/*****************************************************************************/

#ifndef CTH_SERVER

/*
 *  Select a new display_wave function
 */
int change_display_wave(int to) {
    display_wave = opt_change(to, waves, nr_waves, display_wave)->data;
    update_status();
    return 0;
}	
/*
 *  select what wave-fkts are used 
 */
int select_wave() {
    int i;
	
    i = display_selection( waves, nr_waves, display_wave, "Select Waves\n");
    if ( i >= 0)
	change_display_wave(i);
    return 0;
}

/*
 *  Select a new color-table for display_wave
 */
int change_table(int to) {
    active_table = val_change(to, 0, nr_tables-1, active_table);
    return 0;
}

#endif

/*
 *  Update sample rate, mono/stero
 */
int update_dsp() {
    char str[256];

    alloc_sound();			/* might be freed */

    if ( (sound_source == SNDSRC_DEBUG) || (sound_source == SNDSRC_SOCK))
	return 0;
	
    sound_stereo --;			/* bring to 0, 1 for mono/stereo */
	
    ioctl(dsp_des, SNDCTL_DSP_SYNC);
	
    if ( ioctl(dsp_des, SNDCTL_DSP_STEREO, &sound_stereo) < 0 )
	printfee("ioctl: STEREO");
		
    if ( ioctl(dsp_des, SNDCTL_DSP_SPEED, &sound_sample_rate) <0 )
	printfee("ioctl: SPEED");

    sprintf(str, "rate: %d\n%s", sound_sample_rate, 
	    sound_stereo ? "STEREO" : "MONO");
#ifndef CTH_SERVER
    display_print(str, 0,1);
#else
    printf(str);
#endif
	
    sound_stereo ++;			/* bring back to nr. of channels */
    return 0;
}

/*****************************************************************************/

/*
 *  Create a random sound for those without a sound-card
 */
void sound_random() {
    int x;
    static int v1=0, v2=0, maxdv=2; 

    /* generate random sound */
    sound_data[0][0]=144; 
    sound_data[0][1]=112; 
    for (x=1; x < sound_bsize; x++) { 

	if (rand()%256 > sound_data[x-1][0]) 
	    v1+=rand()%maxdv; 
	else 
	    v1-=rand()%maxdv; 

	if (rand()%256 > sound_data[x-1][1]) 
	    v2+=rand()%maxdv; 
	else 
	    v2-=rand()%maxdv; 

	sound_data[x][0]=sound_data[x-1][0]+v1; 
	sound_data[x][1]=sound_data[x-1][1]+v2; 

    } 

    for(x=0; x < sound_bsize; x++) {
	sound_data[x][0] -= 128;
	sound_data[x][1] -= 128;
    }
}

/*
 *  Get sound from sound-card
 */
void sound_read() {
    int nr_read, i;
    unsigned char * sbuff;

    /* Important information from Jan Kujawa <kujawa@kallisti.montana.com>
       how to do this correctly */
    for(nr_read = 0, sbuff=sound_buffer; 
	nr_read < sound_bsize*sound_stereo; nr_read += 32) {
	if( read(dsp_des, sbuff, 16) < 0)
	    printfee("sound_read < 0"); 
	sbuff += 16;
	if( read(dsp_des, sbuff, 16) < 0)
	    printfee("sound_read < 0");
	sbuff += 16;
    }
    /* this should no longer be necessary */
    if(sound_sync)
    	ioctl(dsp_des, SNDCTL_DSP_RESET);   

    /* Bring the read data to the right place */	
    sbuff = sound_buffer;
    if ( sound_stereo == 2) {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][1] = (int)(*sbuff ++) - 128;
	    sound_data[i][0] = (int)(*sbuff ++) - 128;
	}
    } else {
	for(i=0; i < sound_bsize; i++) {
	    sound_data[i][0] = (int)(*sbuff) - 128;
	    sound_data[i][1] = (int)(*sbuff ++) - 128;
	}	
    }
}

#ifndef CTH_SERVER

/*
 * Get a "line" of sound from the soundcard.
 * Calculate the noisyness. And from that the time since when the 
 * system is silent and if it is time to change the display.
 * returns: 
 *   -1 silentce-message, 
 *    1  it's time to change
 *    0 otherwise
 */
int get_sound() {
    int x;
    char * sdata;
    int noisy, peaknoise;
    int hn, ln;
    struct timeval tv;
    struct timezone tz;
    long now;

    switch( sound_source) {
    case SNDSRC_DEBUG:
	sound_random();		
	break;
    case SNDSRC_SOCK:
	net_sound_read();
	break;
    default:
	sound_read();
    }
	
    /* Calculate weather we have noise or a peaklevel */
    sdata = (char *)sound_data;
    ln = hn = *sdata;
    noisy = 0, peaknoise = 0;
    for( x = sound_bsize * 2; x != 0; x--) {
	if ( *sdata > hn)
	    hn = *sdata;
	if ( *sdata < ln)
	    ln = *sdata;
	if ( (hn-ln) > sound_peaklevel) {
	    peaknoise = 1;	
	    noisy = 1;
	    break;
	} else if ( (hn-ln) > sound_minnoise) 
	    noisy = 1;
	sdata ++;
    }

    /* get current time and convert to usec since program start.
       If the program runs for more than 2 years this computation gives 
       an overfow. */
    gettimeofday( &tv, &tz);
    tv.tv_sec -= sound_starttime;
    now = tv.tv_sec * 100L + tv.tv_usec / 10000L;
	
    /* Check for interrupted silence */
    if ( sound_quiet_change)
	if ( noisy && ( (now - sound_quiet_since) > sound_quiet_change) ) {
	    sound_quiet_since = now;		/* from now on no not quiet */
	    return 1;				/* ret: interrupted silence */
	}
	
    /* Check for long quietness */
    if ( !noisy ) {				
	if( (now - sound_quiet_since) > sound_wait_quiet) {
	    sound_quiet_since = now;
	    return -1;				/* ret: long quiet */
	}
    } else
	sound_quiet_since = now;		/* not quiet now */
		
    /* Check for peaksound */
    if ( sound_wait_beat)
	if ( peaknoise ) {
	    if ( sound_beats ++ > sound_wait_beat) {
		sound_beats = 0;
		return 1;
	    }
	} 

    /* nothing special happend */
    if( (now - sound_timer) > sound_wait) {	/* same for long enough */
	sound_timer = now;
	sound_wait = sound_wait_min + rand() % max(1,sound_wait_random);
	return 1;
    }
    return 0;
}

/*****************************************************************************/

/* Macros for compatibility with DOS-version */
#define stereo sound_data
#define massageStyle sound_massage_style

/* 
   from orignal CTHUGHA 
   I have no idea what this function really does. But I think is is
   a usefull function (at least the output looks better with this function). 
*/
int massage_audio() {
    int temp,x; 
    int temp2; 
 
    switch (massageStyle) { 
    case 0: 
    default: 
	break; 
    case 1: 
	temp=stereo[0][1]; 
	temp2=stereo[0][0]; 
	for (x=1; x < sound_bsize; x++) { 
	    if ((stereo[x][1]-temp)>10) { 
		stereo[x][1]=temp+10; 
	    } else if ((stereo[x][1]-temp)<-10) { 
		stereo[x][1]=temp-10; 
	    } 
	    if ((stereo[x][0]-temp2)>10) { 
		stereo[x][0]=temp2+10; 
	    } else if ((stereo[x][0]-temp2)<-10) { 
		stereo[x][0]=temp2-10; 
	    } 
	    temp=stereo[x][1]; 
	    temp2=stereo[x][0]; 
	} 
	break; 
    case 2: 
	temp=stereo[0][1]; 
	temp2=stereo[0][0]; 
	for (x=1; x < sound_bsize; x++) { 
	    if ((stereo[x][1]-temp)>3) { 
		stereo[x][1]=temp+3; 
	    } else if ((stereo[x][1]-temp)<-3) { 
		stereo[x][1]=temp-3; 
	    } 
	    if ((stereo[x][0]-temp2)>3) { 
		stereo[x][0]=temp2+3; 
	    } else if ((stereo[x][0]-temp2)<-3) { 
		stereo[x][0]=temp2-3; 
	    } 
	    temp2=stereo[x][0]; 
	    temp=stereo[x][1]; 
	} 
	break; 
    } 
    return 0;
}

int change_massage_style(int to) {
    sound_massage_style = val_change(to, 0, 2, sound_massage_style);
    return 0;
}

/*****************************************************************************/

/* 
 * This is the old FFT-code
 */
#if 0
/* 
   compute fast short integer FT 
*/ 
void FFT_channel(int channel) { 

    static int kl,a[N],b[N],fff[N]; 
    static int k,l,f[N+N]; 
    static int level;
 
    for (k=0 ; k<(N+N) ; k++) { 
	f[k]=stereo[k*2][channel]; 
    } 
 
    for (l=1 ; l<N ; l++) 
	a[l]=b[l]=0; 
    for (k=0 ; k<(N+N) ; k++)  
	for (l=1 ; l<N ; l++) { 
	    kl=(l*k)%(N+N); 
	    a[l]+=f[k]*c[kl]; 
	    b[l]+=f[k]*s[kl]; 
	} 
    for (l=1 ; l<N ; l++) { 
	fff[l]=(abs(a[l])>>SCALE)+(abs(b[l])>>SCALE); 
    } 
 
    for (k=N*4-3; k<BUFF_WIDTH; k++)  /* Clean stuff past the end of the fft */
	stereo[k][channel]=0; 
    stereo[0][channel]=0; 
 
    for (l=1; l<N ; l++) { 
	level=fff[l]; 
 
	if (level>255) { 
	    for (k=0; k<4; k++) 
		stereo[l*4+k][channel]=127; 
	} else { 
	    for (k=0; k<4; k++) 
		stereo[l*4+k][channel]=(level>>1); 
	} 
    } 
} 
 
int FFT() {
    if( sound_FFT && sound_use_fft && (sound_bsize > 256) ) {
	FFT_channel(0);
	FFT_channel(1);
    } 
    return 0;
}
#endif

/*
 * Now comes the new FFT-code 
 */
int FFT() {
    int channel;
    unsigned char *p, *q;
    static palette Pal;
    static unsigned int slab[256];
    int dir,currentd,lastgot,i,lens,got, lastd=1,temp;
    int lastcapc,nextonec,amt;
    int curpal;
    
    
    if( !sound_FFT || !sound_use_fft )
	return 0;

    lens=0;
    for (i=0; i < 256; i++) 
	slab[i] = 0;

    lastgot=128;
    lens=0;
    lastd=1;
    lastcapc = 128;
    nextonec = 1;

    
    for (channel=0; channel<2; channel++)
	for (i=0; i < sound_bsize; i++) {
	    got=stereo[i][channel];
	    dir=got-lastgot;
	    
	    if (dir>1)
		currentd=1;
	    else if (dir<-1)
		currentd=-1;
	    else
		currentd=lastd;
	    
	    if (currentd!=lastd) {
		if (lens>255)
		    lens=255;
		lens=lens>>3;
		
		slab[lens]+=(lens>>1);
		
		lens=0;
		
		if (nextonec == 1) {
		    nextonec=0;
		    lastcapc=got;
		} else {
		    nextonec = 1;
		    amt=abs(lastcapc-got);
		    if (amt>127)
			amt=127;
		    slab[amt>>2]++;
		}
	    } else
		lens++;
	    
	    
	    lastd=currentd;
	    lastgot=got;
	}

    curpal = opt_number( active_palette, palettes, nr_palettes);
    p = (unsigned char *)Pal;
    q = (unsigned char *)(palettes[curpal].data);
    
    for (i=0; i < 256; i++) {
	temp=slab[(255-i)>>3];
	
	if (temp>6)
	    temp=6;
	
	q = (unsigned char *)(palettes[(curpal+temp)%nr_palettes].data)+i*3;
	
	*p++ = *q++;
	*p++ = *q++;
	*p++ = *q++;
	
    }

    cth_setpalette(Pal, 1);

    return 0;
}

int change_FFT(int to) {
    sound_FFT = val_change(to, 0, 1, sound_FFT);
    return 0;
}

#endif

