# Samizdat message content model
#
#   Copyright (c) 2002-2007  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

require 'samizdat/engine'
require 'fileutils'

begin
  require 'redcloth'

  # get rid of <del>, '-' belongs to text, not markup
  RedCloth::QTAGS.delete_if {|rc, ht, re, rtype| 'del' == ht }
rescue LoadError
  class RedCloth < String
    def to_html   # revert to text/plain
      "<pre>#{CGI.escapeHTML(self)}</pre>"
    end
  end
end

class Content

  def initialize(id, login, title=nil, format=nil, body=nil)
    @id = id if id.kind_of? Integer
    @login = login

    if @id.kind_of? Integer
      title ||= rdf.get_property(@id, 'dc::title')
      format ||= rdf.get_property(@id, 'dc::format')
    end

    @title = title
    self.format = format   # also sets @inline and @cacheable

    if cacheable? and @id.kind_of? Integer
      @html_full = rdf.get_property(@id, 's::htmlFull')
      @html_short = rdf.get_property(@id, 's::htmlShort')
    end

    if inline?
      @body = body
      @body.gsub!(/\r\n/, "\n") if @body

      if @id.kind_of? Integer
        @body ||= rdf.get_property(@id, 's::content')
      end
    end
  end

  attr_reader :title, :format

  attr_reader :html_full, :html_short

  # content body (+nil+ if not inline)
  attr_reader :body

  def Content.format_extension(format)
    config['file_extension'][format] or format.sub(%r{\A.*/}, '')
  end

  # relative path to file holding multimedia message content (+nil+ if inline)
  #
  def file_path(id = @id)
    return nil if inline?

    id = upload_id unless id.kind_of? Integer

    # security: keep format and creator login controlled (see untaint in
    # upload_filename())
    '/' + @login + '/' + id.to_s + '.' + Content.format_extension(@format)
  end

  def file_size(request, id = @id)
    return nil if inline?

    id = upload_id unless id.kind_of? Integer

    filename = upload_filename(request, @id)
    return nil unless File.exists? filename

    File.size(filename)
  end

  # true if content is rendered by Samizdat and not linked to a file
  #
  def inline?
    @inline
  end

  # HTML rendering of all inline messages except RDF queries is cached in
  # database
  #
  def cacheable?
    @cacheable
  end

  # check if size is within configured limits
  #
  def Content.validate_size(size)
    # todo: fine-grained size limits
    if size > config['limit']['content']
      raise UserError, sprintf(
        _('Uploaded file is larger than %s bytes limit'),
        config['limit']['content'])
    end
  end

  # if content is uploaded from file, perform all necessary file operations,
  # set @body to +nil+
  #
  # returns +true+ if content is handled as upload, +nil+ if it's inline
  #
  # todo: extract file-handling methods dependent on request#filename into a
  # separate class
  #
  def upload(request)
    @id.kind_of?(Integer) and raise RuntimeError,
      'Content#upload called for existing message content'

    file = request.cgi_params['file'][0]   # just a file object, not its contents

    return nil unless
      (file.kind_of? StringIO or file.kind_of? Tempfile) and file.size > 0

    Content.validate_size(file.size)
    set_format_from_upload(file)

    if inline?
      @body = file.read   # transform to inline message
      return nil
    else
      request.upload_enabled? or raise UserError,
        _('Multimedia upload is disabled on this site')
      save_as(file, upload_filename(request, upload_id))
      @body = nil
      return true
    end
  end

  # check whether uploaded file is in place
  #
  def validate_upload(request)
    if 'true' == request['file'] and request.has_key? 'confirm'
      inline? and raise UserError, 'Unexpected inline upload confirm'
      File.exists?(upload_filename(request, upload_id))
    else
      raise UserError, 'Unexpected upload state'
    end
  end

  # move uploaded file from one id to another
  #
  def move_upload(request, from_id = upload_id, to_id = @id)
    return nil if inline?

    File.rename(
      upload_filename(request, from_id),
      upload_filename(request, to_id)
    )
  end

  # update html_full and html_short for a message
  # (running inside transaction is assumed)
  #
  def update_html
    return unless cacheable? and @id.kind_of? Integer

    @html_full = Content.render_cacheable(@body, @format)
    db.do 'UPDATE Message SET html_full = ? WHERE id = ?', @html_full, @id

    body_short = limit_string(@body, config['limit']['short'])
    if body_short and body_short.size < @body.size
      @html_short = Content.render_cacheable(body_short, @format)
      db.do 'UPDATE Message SET html_short = ? WHERE id = ?', @html_short, @id
    else
      @html_short = nil
      db.do 'UPDATE Message SET html_short = NULL WHERE id = ?', @id
    end
  end

  # re-render html_full and html_short for a single messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Content.regenerate_html(1)'
  #
  def Content.regenerate_html(id)
    db.transaction do |db|
      message = Message.cached(id)
      Content.new(message.id, message.creator.login).update_html
    end   # transaction
  end

  # re-render html_full and html_short for all messages
  #
  # command line:
  #
  # SAMIZDAT_SITE=samizdat SAMIZDAT_URI=/ ruby -r samizdat/engine -e 'Content.regenerate_all_html'
  #
  def Content.regenerate_all_html
    db.select_all('SELECT id FROM Message WHERE content IS NOT NULL') do |id,|
      Content.regenerate_html(id)
    end
    cache.flush
  end

  AUTOURL_SCHEMES = %w[http https ftp]

  # format inline content according to _format_
  #
  def Content.render_cacheable(body, format)
    return nil if body.nil?
    html =
      case format
      when 'text/plain'   # inline verbatim text
        "<pre>#{CGI.escapeHTML(body)}</pre>"
      when 'text/html'   # limited HTML text
        '<div>' + body + '</div>'
      when 'text/textile'   # textile formatted text
        RedCloth.new(body).to_html
      else   # default text rendering
        CGI.escapeHTML(body).split(/^\s*$/).collect {|p|
          '<p>' + p.gsub(URI::ABS_URI_REF) {|url|
            scheme, host = $1, $4   # see URI::REGEXP::PATTERN::X_ABS_URI
            if AUTOURL_SCHEMES.include?(scheme) and not host.nil?
              url =~ /\A(.*?)([.,;:?!()]+)?\z/   # don't grab punctuation
              url, tail = $1, $2
              %{<a href="#{url}">#{url}</a>#{tail}}
            else
              url
            end
          } + "</p>\n"
        }.join
      end
    Samizdat::Sanitize.new(config.xhtml).sanitize(html)
  end

  private

  # id component of the file path of a new upload
  #
  def upload_id
    'upload'
  end

  # multimedia message content filename
  #
  def upload_filename(request, id)
    (id.kind_of?(Integer) or upload_id == id) or raise RuntimeError,
      'Unexpected file upload id'

    request.content_dir + file_path(id).untaint
  end

  # detect and validate format from upload
  #
  def set_format_from_upload(file)
    if file.methods.include? 'content_type' and file.size > 0
      format = file.content_type.strip   # won't work with FastCGI?
      format = 'image/jpeg' if format == 'image/pjpeg'   # MSIE...
      format = 'image/png' if format == 'image/x-png'
    end

    format.nil? and raise UserError,
      _('Content type of the uploaded file was not provided')

    config['format'].values.flatten.include?(format) or
      raise UserError,
        sprintf(_("Format '%s' is not supported"), CGI.escapeHTML(format))

    self.format = format
  end

  # store checks for format, inline? and cacheable?
  #
  def format=(format)
    @format = format
    @format = nil unless config['format'].values.flatten.include? @format
    @inline = (@format.nil? or config['format']['inline'].include?(@format))
    @cacheable = (@inline and @format != 'application/x-squish')
  end

  def save_as(file, destination)
    File.exists?(File.dirname(destination)) or
      FileUtils.mkdir_p(File.dirname(destination))

    case file
    when Tempfile   # copy large files directly
      FileUtils.cp(file.path, destination)
    when StringIO
      File.open(destination, 'w') {|f| f.write(file.read) }
    else
      raise RuntimeError, "Unexpected file class '#{file.class}'"
    end
  end
end
